//
//  A simple sketch for the remote load of a Mk2 PV Router.  Displays each 
//  RF message received, and drives the trigger accordingly.  The triac 
//  is turned off if transmission is lost.  Various LEDs are supported.
// 
//  Wriiten to support the associated basestation sketch, Mk2i_rev5.ino
//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  March 2013
//


#define LOAD_ON LOW   // to match Tx protocol
#define LOAD_OFF HIGH


const byte transmissionLost_LEDpin = 4; // active high (red)
const byte transmissionOK_LEDpin = 5;   // active high (yellow)
const byte loadOn_LEDpin = 6;           // active high (green)
const byte triggerPin = 7; // suitable for driving a Fairchild MOC 3041 trigger 
                           // in active low mode.  This device ensures that the 
                           // associated triac is only turned on and off at 
                           // zero-crossing points.
const byte onBoard_LEDpin = 9; // active high (duplicates the LED at pin 6)

/* frequency options are RF12_433MHZ, RF12_868MHZ or RF12_915MHZ
 */
#define freq RF12_868MHZ // Use the freq to match the module you have.

const int TXnodeID = 10;
const int myNode = 15;
const int networkGroup = 210; 
const int UNO = 1; // Set to 0 if you're not using the UNO bootloader 

#include <JeeLib.h>      // Download JeeLib: http://github.com/jcw/jeelib


typedef struct { byte dumpState; int msgNumber;} Rx_struct;    //  data for RF comms
Rx_struct receivedData;

unsigned long timeAtLastMessage = 0;
int lastMsgNumber = 0;
unsigned long timeAtLastTransmissionLostDisplay;



void setup() 
{
  Serial.begin(9600);
  delay(5000);
  Serial.println("Remote load controller for Mk2 PV Router");
  Serial.print("Node: "); 
  Serial.print(myNode); 
  Serial.print(" Freq: "); 
  if (freq == RF12_433MHZ) Serial.print("433Mhz");
  if (freq == RF12_868MHZ) Serial.print("868Mhz");
  if (freq == RF12_915MHZ) Serial.print("915Mhz"); 
  Serial.print(" Network: "); 
  Serial.println(networkGroup);
  
  pinMode(transmissionLost_LEDpin, OUTPUT);
  digitalWrite(transmissionLost_LEDpin, LOW);
  pinMode(transmissionOK_LEDpin, OUTPUT);
  digitalWrite(transmissionOK_LEDpin, LOW);
  pinMode(loadOn_LEDpin, OUTPUT);
  digitalWrite(loadOn_LEDpin, LOW);
  pinMode(onBoard_LEDpin, OUTPUT);
  digitalWrite(onBoard_LEDpin, LOW);
  pinMode(triggerPin, OUTPUT);
  digitalWrite(triggerPin, LOAD_OFF);
    
  delay(1000);
  rf12_set_cs(10); //emonTx, emonGLCD, NanodeRF, JeeNode

  rf12_initialize(myNode, freq, networkGroup);                      
}

void loop() 
{ 
  unsigned long timeNow = millis();
  
  if (rf12_recvDone())
  {      
    if (rf12_crc == 0 && (rf12_hdr & RF12_HDR_CTL) == 0)
    {
      int node_id = (rf12_hdr & 0x1F);
      byte n = rf12_len;
    
      if (node_id == TXnodeID)
      {
        receivedData = *(Rx_struct*) rf12_data;
        byte dumpState = receivedData.dumpState;
        int msgNumber = receivedData.msgNumber;
        
	digitalWrite(triggerPin, dumpState); // active low, same as Tx protocol

	if (receivedData.dumpState == LOAD_ON) {
	  digitalWrite(loadOn_LEDpin, HIGH); 
	  digitalWrite(onBoard_LEDpin, HIGH); }
	else {
	  digitalWrite(loadOn_LEDpin, LOW); 
	  digitalWrite(onBoard_LEDpin, LOW);  }
       
	Serial.print(dumpState); 
	Serial.print(",  ");
	Serial.println(msgNumber); 

        if (msgNumber != lastMsgNumber + 1)
        {
          Serial.println("Message numbering error!");
        }
        
        timeAtLastMessage = timeNow;
        lastMsgNumber = msgNumber;
	}	
      }	
      else
      {
        Serial.println("Corrupt message!");
      }
   }
   
   if ((timeNow - timeAtLastMessage) > 3000)
   {
     // transmission has been lost!
     digitalWrite(triggerPin, HIGH); // turn off the triac which is active low
     digitalWrite(transmissionLost_LEDpin, HIGH);  // LED on      
     digitalWrite(transmissionOK_LEDpin, LOW);     // LED off      
     
     if(timeNow > timeAtLastTransmissionLostDisplay + 1000)
     {
       Serial.println("transmission lost!");
       timeAtLastTransmissionLostDisplay = timeNow;
     }
   }
   else
   {
     // transmission is OK
     digitalWrite(transmissionOK_LEDpin, HIGH);   // LED on       
     digitalWrite(transmissionLost_LEDpin, LOW);  // LED off       
   }
}
