/*
 * Tool to capture the raw samples generated by an Arduino during several mains
 * cycles.  This data is displayed on the Serial Monitor, and is also available
 * for subsequent processing using a spreadsheet. 
 *
 * Pauses after each set of measurements has been taken.  Press 'g', then [cr], 
 * to repeat.
 *
 *      Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *      December 2012
 */
 
#define POSITIVE 1
#define NEGATIVE 0
#define ON 0  // the external trigger device is active low
#define OFF 1

byte voltageSensorPin = 2;
byte currentSensorPin = 1;
const byte outputPinForTrigger = 9; // digital

long cycleCount = 0;
int samplesRecorded = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

byte polarityNow; 
boolean beyondStartUpPhase = false;
byte currentStateOfTriac;

int lastSampleV;     // stored value from the previous loop (HP filter is for voltage samples only)         
float lastFilteredV;  //  voltage values after HP-filtering to remove the DC offset
byte polarityOfLastSampleV; // for zero-crossing detection
  
boolean recordingNow;
boolean recordingComplete;
byte cycleNumberBeingRecorded;
byte noOfCyclesToBeRecorded;

unsigned long recordingMayStartAt;
boolean firstLoop = true;
int settlingDelay = 5; // <<---  settling time (seconds) for HPF 

char blankLine[82];
char newLine[82];
int storedSampleV[250]; // sufficient for 3 mains cycles' worth
int storedSampleI[250];
float VOLTAGECAL;

void setup()
{  
  Serial.begin(9600);
  pinMode(outputPinForTrigger, OUTPUT);  
  VOLTAGECAL = (float)679 / 471; // Units are Volts per ADC-level.
  
  // initialise each character of the display line
  blankLine[0] = '|';
  blankLine[80] = '|';
  
  for (int i = 1; i < 80; i++)
  {
    blankLine[i] = ' ';
  }
  
  blankLine[40] = '.';
}


/*  Allow the system to run for several seconds so that the filtered 
 *  voltage waveform can settle down.  This info is needed for determining 
 *  the start of each new mains cycle.  During this period, a countdown 
 *  is displayed.
 *
 *  After the settling period has expired, raw samples taken during 
 *  one complete mains cycle are stored in an array.  The capacity of the 
 *  array needs to be sufficient for the number of sample pairs that may
 *  appear.  A 100 x 2 integer array will probably suffice.
 *
 *  At the start of the following cycle, the data collected during the 
 *  previous cycle data is sent to the Serial window.  
 */
void loop() // each iteration of loop is for one pair of measurements only
{
  if(firstLoop)
  {
    unsigned long timeNow = millis();
    Serial.print ("millis() now = ");
    Serial.println (timeNow);
    
    recordingMayStartAt = timeNow + (settlingDelay * 1000);
    Serial.print ("recordingMayStartAt ");
    Serial.println (recordingMayStartAt);
    
    recordingNow = false;
    firstLoop = false;
    recordingComplete = false;
    noOfCyclesToBeRecorded = 3;
    cycleNumberBeingRecorded = 0;
    samplesRecorded = 0;    
  }
  
  int sampleV = analogRead(voltageSensorPin);                 //Read in raw voltage signal
  int sampleI = analogRead(currentSensorPin);                 //Read in raw current signal

  float filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV); 

  byte polarityOfThisSampleV;
  if(filteredV > 0)
  {
    polarityOfThisSampleV = POSITIVE;
    
    if (polarityOfLastSampleV != POSITIVE) 
    {
      // This is the start of a new mains cycle
      cycleCount++; 
         
      if (recordingNow == true) {
        if (cycleNumberBeingRecorded >= noOfCyclesToBeRecorded) {
          Serial.print ("No of cycles recorded = ");
          Serial.println (cycleNumberBeingRecorded);      
          dispatch_recorded_data(); } 
        else {
          cycleNumberBeingRecorded++; } }    

      else
      if((cycleCount % 50) == 1) {  
        unsigned long timeNow = millis();   
        if (timeNow > recordingMayStartAt) {
           recordingNow = true;
           cycleNumberBeingRecorded++; } 
        else  {
          Serial.println((int)(recordingMayStartAt - timeNow) / 1000); } }    
    } // end of specific processing for first +ve reading in each mains cycle
    
  } // end of specific processing of +ve cycles
  else
  {
    polarityOfThisSampleV = NEGATIVE; 
    
    if (polarityOfLastSampleV != NEGATIVE) 
    {
      // at the start of a new negative half cycle
    }
  }
  
  if (recordingNow == true)
  {
    storedSampleV[samplesRecorded] = sampleV;
    storedSampleI[samplesRecorded] = sampleI;
    samplesRecorded++;
  }
    
  polarityOfLastSampleV = polarityOfThisSampleV;  
  lastSampleV=sampleV;                       
  lastFilteredV = filteredV;                  
} // end of loop()

void dispatch_recorded_data()
{      
  // display raw samples via the Serial Monitor
  // ------------------------------------------ 

  Serial.print("cycleCount ");
  Serial.print(cycleCount);
  Serial.print(",  samplesRecorded ");
  Serial.println(samplesRecorded);

  int voltage, current;
  int minVoltage = 1023, minCurrent = 1023;
  int maxVoltage = 0, maxCurrent = 0;
  byte triacState;
      
  for (int index = 0; index < samplesRecorded; index++) 
  {
    strcpy(newLine, blankLine);
    voltage = storedSampleV[index]; 
    current = storedSampleI[index]; 
 
    if (voltage < minVoltage){minVoltage = voltage;}
    if (voltage > maxVoltage){maxVoltage = voltage;}
    if (current < minCurrent){minCurrent = current;}
    if (current > maxCurrent){maxCurrent = current;}
    
    newLine[map(voltage, 0, 1023, 0, 80)] = 'v'; 
    newLine[map(current, 0, 1023, 0, 80)] = 'c'; 
              
    if ((index % 2) == 0) // change this to "% 1" for full resolution
    {
      Serial.println(newLine);
    }
  }
    
  Serial.print("minVoltage ");  Serial.print(minVoltage);
  Serial.print(",  maxVoltage ");  Serial.print(maxVoltage);
  Serial.print(",  minCurrent ");  Serial.print(minCurrent);
  Serial.print(",  maxCurrent ");  Serial.println(maxCurrent);
  Serial.println();
  Serial.println();
      
  // despatch raw samples via the Serial Monitor 
  // ------------------------------------------- 
      
  Serial.println("Raw data from stored cycle: <Vsample>,<Isample>[cr]");
  Serial.print(samplesRecorded);
  Serial.println(", <<< No of sample pairs");

  for (int index = 0; index < samplesRecorded; index++) 
  {
    Serial.print (storedSampleV[index]); 
    Serial.print(','); 
    Serial.println (storedSampleI[index]);  
  }

  recordingNow = false;
  firstLoop = true;
  pause();
}      

void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') done++;
    }
  }    
}



