//------------------------------------------------------------------------------------------------------------------------------------------------
// emonGLCD Solar PV monitor example - compatible with Arduino 1.0 

// ******************************************   Brian's Version  **************************************************************************
// Revision 1 compiles to 31,048 of 32,256 12/01/2013
// Rev 2 compiles to 27,508 19/02/2013

// The following changes have been made to the original:

// 1.
// The LED's have been mounted on the non-component side of the PCB so they project through a diffusing aperture in the front panel
// This allows the two LED colours to mix and the relationship between generation and use is represented by colour.
// The light intensity remains constant. Previously this was a variable

// HIGH IMPORT.........................BALANCE..............................HIGH GENERATION
//    RED         Orange                Amber             Yellow             Green
//
// There are 256 colour steps between Red and Green

// 2.
// When the energy bucket overflows this indicates that power is being wasted to the grid therefore the code has been changed to flash
// the LED's which indicates that it's time to switch on the dishwasher or washing machine

// 3.
// The temperature quadrant on the display is now used to show diverted power. This is a parameter transmitted by the emonTx in the 
// power3 parameter.

// 4.
// Diverted power is now integrated over several emonTx transmissions to provide an integrated reading

// 5.
// A #define for the GLC_unit is now included. This provides a single entry for the unit ID and transmit delay to be controlled from
// This is only required if the system supports more than one emonGLCD

// 6.
// Removed the secondary diplay pages during testing and because I have never used them have not restored them

// 7.
// Display Voltage and frequency sent from the emonTx


// *******************************************************************************************
// emonGLCD documentation http://openEnergyMonitor.org/emon/emonglcd
// solar PV monitor build documentation: http://openenergymonitor.org/emon/applications/solarpv

// For use with emonTx setup with 2CT with CT 1 monitoring consumption/grid and CT 2 monitoring PV generation .
// The CT's should be clipped on with the orientation so grid reading is postive when importing and negative when exporting. Generation reading should always be positive. 

// Correct time is updated via NanodeRF which gets time from internet, this is used to reset Kwh/d counters at midnight. 

// Temperature recorded on the emonglcd is also sent to the NanodeRF for online graphing

// this sketch is currently setup for type 1 solar PV monitoring where CT's monitor generation and consumption separately
// The sketch assumes emonx.power1 is consuming/grid power and emontx.power2 is solarPV generation
// to use this sketch for type 2 solar PV monitoring where CT's monitor consumption and grid import/export using an AC-AC adapter to detect current flow direction 
// change type to '2' on emonGLCD setup section below.    

// GLCD library by Jean-Claude Wippler: JeeLabs.org
// 2010-05-28 <jcw@equi4.com> http://opensource.org/licenses/mit-license.php
//
// History page by vworp https://github.com/vworp
//
// Authors: Glyn Hudson and Trystan Lea
// Part of the: openenergymonitor.org project
// Licenced under GNU GPL V3
// http://openenergymonitor.org/emon/license

// Libraries in the standard arduino libraries folder:
//
//	- OneWire library	http://www.pjrc.com/teensy/td_libs_OneWire.html
//	- DallasTemperature	http://download.milesburton.com/Arduino/MaximTemperature
//                           or https://github.com/milesburton/Arduino-Temperature-Control-Library
//	- JeeLib		https://github.com/jcw/jeelib
//	- RTClib		https://github.com/jcw/rtclib
//	- GLCD_ST7565		https://github.com/jcw/glcdlib
//
// Other files in project directory (should appear in the arduino tabs above)
//	- icons.ino
//	- templates.ino
//
//-------------------------------------------------------------------------------------------------------------------------------------------------

#include <JeeLib.h>
#include <GLCD_ST7565.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
GLCD_ST7565 glcd;

#include <OneWire.h>		    // http://www.pjrc.com/teensy/td_libs_OneWire.html
#include <DallasTemperature.h>      // http://download.milesburton.com/Arduino/MaximTemperature/ (3.7.2 Beta needed for Arduino 1.0)
#include <RTClib.h>                 // Real time clock (RTC) - used for software RTC to reset kWh counters at midnight
#include <Wire.h>                   // Part of Arduino libraries - needed for RTClib
RTC_Millis RTC;

// The following #define allows for three GLCD units. Number one, number two and number three
// the update intervals are prime numbers to reduce transmission allignment
// the node assignment values are not very important

#define GLC_unit 1

#if GLC_unit == 1
  #define MYNODE 20
  #define slow_update_time 7000
#endif

#if GLC_unit == 2
  #define MYNODE 25
  #define slow_update_time 5000
#endif

#if GLC_unit == 3
  #define MYNODE 27
  #define slow_update_time 11000
#endif  

//--------------------------------------------------------------------------------------------
// RFM12B Settings
//--------------------------------------------------------------------------------------------
// This is the original #define for the node ID
// #define MYNODE 20            // Should be unique on network, node ID 30 reserved for base station
#define freq RF12_433MHZ     // frequency - match to same frequency as RFM12B module (change to 868Mhz or 915Mhz if appropriate)
#define group 210            // network group, must be same as emonTx and emonBase
#define transmitloops 2 // Number of transmissions (+1) to integrate

//---------------------------------------------------
// Data structures for transfering data between units
//---------------------------------------------------
typedef struct { int power1, power2, power3, Vrms, temp, frequency; } PayloadTX;         // neat way of packaging data for RF comms
PayloadTX emontx;

typedef struct { int temperature; } PayloadGLCD;
PayloadGLCD emonglcd;

//---------------------------------------------------
// emonGLCD SETUP
//---------------------------------------------------
//#define emonGLCDV1.3               // un-comment if using older V1.3 emonGLCD PCB - enables required internal pull up resistors. Not needed for V1.4 onwards 
const int SolarPV_type=2;            // Select solar PV wiring type - Type 1 is when use and gen can be monitored seperatly. Type 2 is when gen and use can only be monitored together, see solar PV application documentation for more info
const int maxgen=3000;              // peak output of soalr PV system in W - used to calculate when to change cloud icon to a sun
const int PV_gen_offset=25;         // When generation drops below this level generation will be set to zero - used to force generation level to zero at night

const int greenLED=6;               // Green tri-color LED
const int redLED=9;                 // Red tri-color LED
const int LDRpin=4;    		    // analog pin of onboard lightsensor 
const int switch1=15;               // Push switch digital pins (active low for V1.3, active high for V1.4)
const int switch2=16;
const int switch3=19;
const int threshold=10;
const int OFF=0;

//---------------------------------------------------
// emonGLCD variables 
//---------------------------------------------------
int hour = 0, minute = 0;
double usekwh = 0, genkwh = 0;

double use_history[7], gen_history[7];
double divertedpower;
double divertedpower1;
double housepower;
double housepower1;
double divertedkwh;
int cval_use, cval_gen;
int waste =0;
byte page = 1;
int LED_colour=128;
int flash = 0;
int transmitcounter = 2;
int voltage = 0;

//---------------------------------------------------
// Temperature Sensor Setup
//---------------------------------------------------
#define ONE_WIRE_BUS 5              // temperature sensor connection - hard wired 
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature sensors(&oneWire);
double temp, maxtemp,mintemp,controllertemp;

//-------------------------------------------------------------------------------------------- 
// Flow control
//-------------------------------------------------------------------------------------------- 
unsigned long last_emontx;                   // Used to count time from last emontx update
unsigned long last_emonbase;                   // Used to count time from last emontx update
boolean last_switch_state, switch_state; 
unsigned long fast_update, slow_update;


//--------------------------------------------------------------------------------------------
// Setup
//--------------------------------------------------------------------------------------------
void setup()
{
  Serial.begin(9600);
  rf12_initialize(MYNODE, freq,group);
  glcd.begin(0x20);
//  glcd.backLight(200);
  
  sensors.begin();                         // start up the DS18B20 temp sensor onboard  
  sensors.requestTemperatures();
  temp = (sensors.getTempCByIndex(0));     // get inital temperture reading
  mintemp = temp; maxtemp = temp;          // reset min and max

  pinMode(greenLED, OUTPUT); 
  pinMode(redLED, OUTPUT); 
  
  #ifdef emonGLCDV1.3                      //enable internal pull up resistors for push switches on emonGLCD V1.3 (old) 
  pinMode(switch1, INPUT); pinMode(switch2, INPUT); pinMode(switch2, INPUT);
  digitalWrite(switch1, HIGH); digitalWrite(switch2, HIGH); digitalWrite(switch3, HIGH); 
  #endif

}

//--------------------------------------------------------------------------------------------
// Loop
//--------------------------------------------------------------------------------------------
void loop()
{
 
  if (rf12_recvDone())
  {
    if (rf12_crc == 0 && (rf12_hdr & RF12_HDR_CTL) == 0)  // and no rf errors
    {
      int node_id = (rf12_hdr & 0x1F);
       
      if (node_id == 10)
      {
     
  // Here to process a transmission from the emontx, these arrive every few seconds
        
        emontx = *(PayloadTX*) rf12_data;
        last_emontx = millis();
        voltage=emontx.Vrms/100; // Read the mains voltage
        
        // The LCD will be updated every 200mS but we want the diverted power and house power from emontx to be averaged over several transmissions
        // before it is passed to the display.
      
        housepower1 += emontx.power1; // Add in the latest transmitted house power
        divertedpower1 += emontx.power3; // Add in the latest transmitted diverted power
        transmitcounter--; // Decrement the counter
        if (transmitcounter <= 0) //Check if the last one
        {
          divertedpower=divertedpower1/(transmitloops+1); // Calculate the average
          divertedpower1=emontx.power3; // Rewind divertedpower1
          housepower=housepower1/(transmitloops+1); // Calculate the average
          housepower1=emontx.power1; // Rewind housepower1
          transmitcounter=transmitloops; // Get ready for another set of data
        }
      }
// This will be useful when a base station is available      
//      if (node_id == 15)
//      {
//        RTC.adjust(DateTime(2012, 1, 1, rf12_data[1], rf12_data[2], rf12_data[3]));
//        last_emonbase = millis();
//      } 
    }
  }

  //--------------------------------------------------------------------------------------------
  // Display update every 200ms
  //--------------------------------------------------------------------------------------------
  if ((millis()-fast_update)>200)
  {
    fast_update = millis();
    
    DateTime now = RTC.now();
    int last_hour = hour;
    hour = now.hour();
    minute = now.minute();
    usekwh += ((emontx.power1 + emontx.power2) * 0.2) / 3600000; //Calculate power usage
    genkwh += (emontx.power2 * 0.2) / 3600000; // Calculate power generated
    divertedkwh += (emontx.power3 * 0.2) / 3600000; // Calculate power diverted
    
    if (last_hour == 23 && hour == 00) 
    { 
//      int i; for (i=6; i>0; i--) gen_history[i] = gen_history[i-1]; 
      genkwh = 0;
//      for(i=6; i>0; i--) use_history[i] = use_history[i-1];
      usekwh = 0;
      divertedkwh = 0;
    }
    
    cval_use += (housepower + (emontx.power2) - cval_use) * 0.5;
//    cval_use = cval_use + ((emontx.power1+emontx.power2) - cval_use)*0.50;
    cval_gen = cval_gen + (emontx.power2 - cval_gen)*0.50;

    if (cval_gen<PV_gen_offset) cval_gen=0;                  // Set generation to zero when generation level drops below PV_gen_offset
 
                          //use, usekwh, gen,    maxgen, genkwh, temp, mintemp, maxtemp, hour, minute, last_emontx, last_emonbase)
      draw_solar_page(cval_use, usekwh, cval_gen, maxgen, genkwh, divertedpower, (emontx.temp), maxtemp, voltage, emontx.frequency, last_emontx, divertedkwh);

      glcd.refresh();
    
    int LDR = analogRead(LDRpin);                     // Read the LDR Value so we can work out the light level in the room.
    int LDRbacklight = map(LDR, 0, 1023, 50, 250);    // Map the data from the LDR from 0-1023 (Max seen 1000) to var GLCDbrightness min/max
    LDRbacklight = constrain(LDRbacklight, 0, 255);   // Constrain the value to make sure its a PWM value 0-255
    glcd.backLight(LDRbacklight); //BD
 //Serial.print("Colour: "); Serial.print(LED_colour); Serial.println(" ");
 //Serial.print("Flash: "); Serial.print(flash); Serial.println(" ");
  if(flash>0)
  {
    analogWrite(greenLED,OFF);
    flash--;
  }
  if(flash==0)
  {
  LED_colour=constrain((map(cval_use-(cval_gen*2),0,4000,0,255)+128),0,254);
  
   analogWrite(redLED,LED_colour);
   analogWrite(greenLED,map(LED_colour,0,255,255,0));
  
   // Now is a good time to see if we are wasting power to the grid
  }
  if ((cval_gen-cval_use)>1000)waste++;
  else waste=0;
  
  if(flash==0)
  {
   if(waste > 20)
   {
   // Here if wasting power to the grid
   // flash the green LED's
   waste=20;
   flash=2;
   }
  }
 
 } 
  
  if ((millis()-slow_update)>slow_update_time)
  {
    slow_update = millis();

    sensors.requestTemperatures();
    double rawtemp = (sensors.getTempCByIndex(0));
    if ((rawtemp>-20) && (rawtemp<50)) temp=rawtemp;                  //is temperature within reasonable limits?
    if (temp > maxtemp) maxtemp = temp;
    if (temp < mintemp) mintemp = temp;
   
    emonglcd.temperature = (int) (temp * 100);                          // set emonglcd payload

    rf12_recvDone();

    
    if (rf12_canSend())

    {

     rf12_sendStart(0, &emonglcd, sizeof emonglcd);

     rf12_sendWait(0);
    }
  }
}

