/* emonTxV3_4_as_Mk2Router.ino
 *
 * Loosely based on Mk2_PV_Router_rev6a.ino, this sketch has been restructed in order
 * to make better use of the ISR.  The underlying Mk2 control activity now runs 
 * entirely within the ISR so is unaffected by slower activities in loop() such 
 * as Serial, RF tranmissions and temperature sensing.  This code uses the ADC in its 
 * free-running mode.
 *
 * This sketch is intended to run on the emonTx V3.4 platform  with an 
 * RFM69CW module.
 *
 * CT4 is for the primary sensor channel, this being necessary because of the increased 
 * sensitivity that this channel can provide.
 *
 * CT3 is intended for monitoring the diverted power.  This channel (CT3)has some additional 
 * processing in order to calculate the total amount of diverted energy for "today".  This 
 * value is cleared after a pre-set period of inactivity, e.g. ten hours.  CT2 and CT1 are
 * available for any general datalogging purpose.
 *
 * The IO port which drives the output stage is digital 2 which is available at element 4
 * of the terminal block on the emonTx V3.  The other wire from the output stage needs to
 * be connected to 3.3V, at element 2 of the same terminal block.  As with all previous 
 * versions of the Mk2 PV Router, the output stage is driven "active-low".
 *
 * A Dallas DS18B20 sensor can be supported and is connected to elements 3, 5 and 6 of the 
 * same terminal block.
 *
 * As noted in section 8 of my online article at openenergymonitor.org/emon/mk2/build ,
 * a more sensitive trigger device (MOC 3043) should be used when the emonTx V3 is powered 
 * only from an AC/AC source.  The series resistor should also be increased to 180R.  By 
 * treating the output stage as active-low, as noted above, none of the connectors in the
 * terminal block will require more than one wire.
 *
 * Pre-built output stages for switching 230V AC loads are available from my website at 
 * www.mk2pvrouter.co.uk
 *
 *      Robin Emley (calypso_rae on OEM forum)
 *      February 2015
 */

#include <Arduino.h> 

//#include <RFu_JeeLib.h> // for the RF capability (RFM12B on RFu sub-assembly)  
#define RF69_COMPAT 1
#include <JeeLib.h> // for the RFM69CW

#include <OneWire.h> // for use with Dallas sensor

// Physical constants, please do not change!
#define SECONDS_PER_MINUTE 60
#define MINUTES_PER_HOUR 60
#define JOULES_PER_WATT_HOUR 3600 //  (0.001 kWh = 3600 Joules)

// -----------------------------------------------------
// Change these values to suit the local mains frequency and supply meter
#define CYCLES_PER_SECOND 50 
#define SWEETZONE_IN_JOULES 3600 
#define REQUIRED_EXPORT_IN_WATTS 0 // when set to a negative value, this acts as a PV generator 

// --------------------------
// Dallas DS18B20 commands
#define SKIP_ROM 0xcc 
#define CONVERT_TEMPERATURE 0x44
#define READ_SCRATCHPAD 0xbe
#define BAD_TEMPERATURE 30000 // this value (300C) is sent if no sensor is present

// ----------------
// general literals
#define DATALOG_PERIOD_IN_MAINS_CYCLES  250 
#define DISPLAY_SHUTDOWN_IN_HOURS 10 // auto-reset after this period of inactivity
//#define DISPLAY_SHUTDOWN_IN_HOURS 0.01 // for testing that the display clears after 36 seconds

// to prevent the diverted energy total from 'creeping'
#define ANTI_CREEP_LIMIT 5 // in Joules per mains cycle (has no effect when set to 0)
long antiCreepLimit_inIEUperMainsCycle;

// -------------------------------
// definitions of enumerated types
enum polarities {NEGATIVE, POSITIVE};
enum loadStates {LOAD_ON, LOAD_OFF}; // the trigger is driven active low
enum LEDstates {LED_OFF, LED_ON};  // the on-board LED is active high
enum outputModes {ANTI_FLICKER, NORMAL}; // retained for compatibility with previous versions.

/* ----  Output mode selection -----
 * When using the emonTx V3, it is not obvious how an external switch could be fitted which
 * would allow the output mode to be selected at run-time.  For compatibility with previous 
 * versions, the ability to select the outputMode in this way has been retained.  When such a 
 * switch is not available, the output mode can only be set at compile-time, as below:
 */
enum outputModes outputMode = ANTI_FLICKER;                                                   

/* --------------------------------------
 * RF configuration (for the RFM12B module)
 * frequency options are RF12_433MHZ, RF12_868MHZ or RF12_915MHZ
 */
#define freq RF12_433MHZ // Use the freq to match the module you have.

const int nodeID = 10; 
const int networkGroup = 210;  // wireless network group - needs to be same for all nodes 
const int UNO = 1;  // for when the processor contains the UNO bootloader.

// ----------------------------
//  data structure for RF comms 
typedef struct { 
 int power_grid_Watts; // uses CT4
 int power_diverted_Watts; // uses CT3
 int power_CT2_Watts;  
 int power_CT1_Watts;  
 int diverted_energy_WattHours; // uses CT3
 int Vrms_timesTen;  
 int temperature_times100;  
} tx_struct;    
tx_struct tx_data; 


/* --------------------------
 * allocation of digital pins
 *
 *  (These port allocations are for use with an emonTx V3.2)
 */
//const byte triggerControl = 2; // <-- for RF12B version
const byte triggerControl = 3; // <-- for RF69CW version
const byte tempSensorPin = 5; // <-- data input from the DS18B20 sensor 
const byte LEDpin = 6; // <-- used as an output for the on-board LED

/* ---------------------------
 * allocation of analogue pins
 */
const byte voltageSensor = 0;         //  A0 is for the voltage sensor
const byte currentSensor_CT1 = 1;     //  CT1 can be used for any purpose (low sensitivity)
const byte currentSensor_CT2 = 2;     //  CT2 can be used for any purpose (low sensitivity)
const byte currentSensor_diverted = 3;  //   CT3 is for diverted power (low sensitivity)
const byte currentSensor_grid = 4;    //  CT4 is for the grid sensor (high sensitivity)
const byte powerForDallasSensor = 19; //  A5 (dig 19) supplies 3.3 V to the DS18B20 sensor 

const byte delayBeforeSerialStarts = 5;  // in seconds, to allow Serial window to be opened
const byte startUpPeriod = 3;  // in seconds, to allow LP filter to settle
const int DCoffset_I = 512;    // nominal mid-point value of ADC @ x1 scale

/* -------------------------------------------------------------------------------------
 * Global variables that are used in multiple blocks so cannot be static.
 * For integer maths, many variables need to be 'long'
 */
boolean beyondStartUpPhase = false;     // start-up delay, allows things to settle
long energyInBucket_long; // in Integer Energy Units (for controlling the dump-load) 
long capacityOfEnergyBucket_long;  // depends on powerCal, frequency & the 'sweetzone' size.
long lowerEnergyThreshold_long;    // for turning load off
long upperEnergyThreshold_long;    // for turning load on
// int phaseCal_grid_int;             // to avoid the need for floating-point maths
// int phaseCal_spareChannel_int;         // to avoid the need for floating-point maths
long DCoffset_V_long;              // <--- for LPF
long DCoffset_V_min;               // <--- for LPF
long DCoffset_V_max;               // <--- for LPF
int phaseCal_int_CT1;                  // to avoid the need for floating-point maths
int phaseCal_int_CT2;                  // to avoid the need for floating-point maths
int phaseCal_int_CT3;                  // to avoid the need for floating-point maths
int phaseCal_int_CT4;                  // to avoid the need for floating-point maths

long divertedEnergyRecent_IEU = 0; // Hi-res accumulator of limited range
unsigned int divertedEnergyTotal_Wh = 0; // WattHour register of 63K range
long IEU_per_Wh; // depends on powerCal, frequency & the 'sweetzone' size.
unsigned long EDDshutdown_inMainsCycles; 
unsigned long absenceOfDivertedEnergyCount = 0;

float offsetOfEnergyThresholdsInAFmode = 0.1; // <-- not wise to exceeed 0.4

int sampleSetsDuringThisCycle;    // for normalising the power during each mains cycle
long sumP_forEnergyBucket;        // for per-cycle summation of 'real power' contributions
long sumP_forDivertedEnergy;      // for per-cycle summation of 'real power' contributions
long sumP_grid;      // for datalog summation of 'grid' power contributions (uses CT4)
long sumP_diverted;  // for datalog summation of 'diverted' power contributions (uses CT3)
long sumP_CT2;       // for datalog summation of power contributions on CT2
long sumP_CT1;       // for datalog summation of power contributions on CT1
long sum_Vsquared;                // for summation of V^2 values during datalog period            
long sampleSetsDuringThisDatalogPeriod; // for power and Vrms datalogging

long cum_VdeltasThisCycle_long;    // for the LPF which determines DC offset (voltage)
long lastSampleV_minusDC_long;     //    for the phaseCal algorithm
byte cycleCountForDatalogging = 0;  
long sampleVminusDC_long;
long requiredExportPerMainsCycle_inIEU;

// for interaction between the main code and the ISR
volatile boolean datalogEventPending;
volatile boolean newMainsCycle = false;

long copyOf_sumP_grid; // uses data from CT4        
long copyOf_sumP_CT1;  // uses data from CT3        
long copyOf_sumP_CT2;          
long copyOf_sumP_diverted;          
int copyOf_lowestNoOfSampleSetsPerMainsCycle;
long copyOf_sampleSetsDuringThisDatalogPeriod;
long copyOf_sum_Vsquared;


// For temperature sensing
OneWire oneWire(tempSensorPin);
int tempTimes100;


// For an enhanced polarity detection mechanism, which includes a persistence check
#define POLARITY_CHECK_MAXCOUNT 1
enum polarities polarityNow;   
enum polarities polarityConfirmed;  // for zero-crossing detection
enum polarities polarityConfirmedOfLastSampleV;  // for zero-crossing detection

// For a mechanism to check the integrity of this code structure
int lowestNoOfSampleSetsPerMainsCycle;
unsigned long timeAtLastDelay;


// Calibration values (not important for the Router's basic operation)
//-------------------
// For accurate calculation of real power/energy, two calibration values are 
// used: powerCal and phaseCal.  With most hardware, the default values are 
// likely to work fine without need for change.  A full explanation of each 
// of these values now follows:
//   
// powerCal is a floating point variable which is used for converting the 
// product of voltage and current samples into Watts.
//
// The correct value of powerCal is dependent on the hardware that is 
// in use.  For best resolution, the hardware should be configured so that the 
// voltage and current waveforms each span most of the ADC's usable range.  For 
// many systems, the maximum power that will need to be measured is around 3kW. 
//
// My sketch "MinAndMaxValues.ino" provides a good starting point for 
// system setup.  First arrange for the CT to be clipped around either core of a  
// cable which supplies a suitable load; then run the tool.  The resulting values 
// should sit nicely within the range 0-1023.  To allow some room for safety, 
// a margin of around 100 levels should be left at either end.  This gives a 
// output range of around 800 ADC levels, which is 80% of its usable range.
//
// My sketch "RawSamplesTool.ino" provides a one-shot visual display of the
// voltage and current waveforms.  This provides an easy way for the user to be 
// confident that their system has been set up correctly for the power levels 
// that are to be measured.  Any pre-built system that I supply will have been
// checked with this tool to ensure that the input sensors are working correctly.
//
// The ADC has an input range of 0-5V and an output range of 0-1023 levels.
// The purpose of each input sensor is to convert the measured parameter into a 
// low-voltage signal which fits nicely within the ADC's input range. 
//
// In the case of 240V mains voltage, the numerical value of the input signal 
// in Volts is likely to be fairly similar to the output signal in ADC levels.  
// 240V AC has a peak-to-peak amplitude of 679V, which is not far from the ideal 
// output range.  Stated more formally, the conversion rate of the overall system 
// for measuring VOLTAGE is likely to be around 1 ADC-step per Volt (RMS).
//
// In the case of AC current, however, the situation is very different.  At
// mains voltage, a power of 3kW corresponds to an RMS current of 12.5A which 
// has a peak-to-peak range of 35A.  This is smaller than the output signal by 
// around a factor of twenty.  The conversion rate of the overall system for 
// measuring CURRENT is therefore likely to be around 20 ADC-steps per Amp.
//
// When calculating "real power", which is what this code does, the individual 
// conversion rates for voltage and current are not of importance.  It is 
// only the conversion rate for POWER which is important.  This is the 
// product of the individual conversion rates for voltage and current.  It 
// therefore has the units of ADC-steps squared per Watt.  Most systems will
// have a power conversion rate of around 20 (ADC-steps squared per Watt).
// 
// powerCal is the RECIPR0CAL of the power conversion rate.  A good value 
// to start with is therefore 1/20 = 0.05 (Watts per ADC-step squared)
//
const float powerCal_grid = 0.045;  // for grid power at CT4 (burden = 120R)
const float powerCal_diverted = 0.25;  // for diverted power at CT3 (burden = 22R)
const float powerCal_CT2 = 0.25;  // for real power at CT2 (burden = 22R)
const float powerCal_CT1 = 0.25;  // for real power at CT1 (burden = 22R)
 
                        
// phaseCal is used to alter the phase of the voltage waveform relative to the
// current waveform.  This mechanism can be used to offset any difference in 
// phase delay between the voltage and current sensors.  The algorithm interpolates 
// between the most recent pair of voltage samples according to the phaseCal value. 
//
//    With phaseCal = 1, the most recent sample is used.  
//    With phaseCal = 0, the previous sample is used
//    With phaseCal = 0.5, the mid-point (average) value is used
//
// Values ouside the 0 to 1 range involve extrapolation, rather than interpolation
// and are not recommended.  By altering the order in which V and I samples are 
// taken, and for how many loops they are stored, it should always be possible to
// arrange for the optimal value of phaseCal to lie within the range 0 to 1. 
//
const float  phaseCal_CT1 = 1;
const float  phaseCal_CT2 = 1;
const float  phaseCal_CT3 = 1;
const float  phaseCal_CT4 = 1;


// For datalogging purposes, voltageCal has been included too.  When running at 230 V AC, the
// the range of ADC values will be similar to the actual range of volts, so the optimal value 
// for this cal factor will be close to unity.  
//
const float voltageCal = 0.875; // <-- using a Fluke 77 multimeter for my own setup - RAE

boolean EDD_isActive = false; // energy diversion detection (start in 'night' mode)
// boolean EDD_isActive = true; // energy diversion detection ('day' mode, for test only)


void setup()
{  
  delay(delayBeforeSerialStarts * 1000); // allow time to open Serial monitor     
 
  Serial.begin(9600);
  Serial.println();
  Serial.println("-------------------------------------");
  Serial.println("Sketch ID: emonTxV3_4_as_Mk2Router.ino");
  Serial.println();
       
  pinMode(LEDpin, OUTPUT);  
  digitalWrite (LEDpin, LED_OFF); // the on-board LED is active high

  pinMode(triggerControl, OUTPUT);  
  digitalWrite (triggerControl, LOAD_OFF); // the external trigger is active low

  pinMode(powerForDallasSensor, OUTPUT);  // 3.3 V power for the temperature sensor
  digitalWrite (powerForDallasSensor, HIGH); 
 
  // When using integer maths, calibration values that have supplied in floating point 
  // form need to be rescaled.  
  //
  phaseCal_int_CT1 = phaseCal_CT1 * 256; // for integer maths
  phaseCal_int_CT2 = phaseCal_CT2 * 256; // for integer maths
  phaseCal_int_CT3 = phaseCal_CT3 * 256; // for integer maths
  phaseCal_int_CT4 = phaseCal_CT4 * 256; // for integer maths
  
  // When using integer maths, the SIZE of the ENERGY BUCKET is altered to match the
  // scaling of the energy detection mechanism that is in use.  This avoids the need 
  // to re-scale every energy contribution, thus saving processing time.  This process 
  // is described in more detail in the function, processLatestContribution(), just before 
  // the energy bucket is updated at the start of each new cycle of the mains.
  //
  // An electricity meter has a small range over which energy can ebb and flow without 
  // penalty.  This has been termed its "sweet-zone".  For optimal performance, the energy
  // bucket of a PV Router should match this value.  The sweet-zone's value is therefore 
  // included in the calculation below.
  //
  // For the flow of energy at the 'grid' connection point (CT4) 
  capacityOfEnergyBucket_long = 
     (long)SWEETZONE_IN_JOULES * CYCLES_PER_SECOND * (1/powerCal_grid);

  // For recording the accumulated amount of diverted energy data (using CT2), a similar 
  // calibration mechanism is required.  Rather than a bucket with a fixed capacity, the 
  // accumulator for diverted energy just needs to be scaled in a known way.  As soon as its 
  // value exceeds 1 Wh, an associated WattHour register is incremented, and the 
  // accumulator's value is decremented accordingly. The calculation below is to determine
  // the scaling for this accumulator.     
  IEU_per_Wh = 
     (long)JOULES_PER_WATT_HOUR * CYCLES_PER_SECOND * (1/powerCal_diverted); 
 
  // to avoid the diverted energy accumulator 'creeping' when the load is not active
  antiCreepLimit_inIEUperMainsCycle = (float)ANTI_CREEP_LIMIT * (1/powerCal_diverted);

  // for the Energy Diversion Detector
  long mainsCyclesPerHour = (long)CYCLES_PER_SECOND * 
                             SECONDS_PER_MINUTE * MINUTES_PER_HOUR;                           
  EDDshutdown_inMainsCycles = DISPLAY_SHUTDOWN_IN_HOURS * mainsCyclesPerHour;                           
  requiredExportPerMainsCycle_inIEU = (long)REQUIRED_EXPORT_IN_WATTS * (1/powerCal_grid); 

  // Define operating limits for the LP filter which identifies DC offset in the voltage 
  // sample stream.  By limiting the output range, the filter always should start up 
  // correctly.
  DCoffset_V_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale  
  DCoffset_V_min = (long)(512L - 100) * 256; // mid-point of ADC minus a working margin
  DCoffset_V_max = (long)(512L + 100) * 256; // mid-point of ADC plus a working margin

  Serial.println ("ADC mode:       free-running");
  
  // Set up the ADC to be free-running 
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable the ADC 
  
  ADCSRA |= (1<<ADATE);  // set the Auto Trigger Enable bit in the ADCSRA register.  Because 
                         // bits ADTS0-2 have not been set (i.e. they are all zero), the 
                         // ADC's trigger source is set to "free running mode".
                         
  ADCSRA |=(1<<ADIE);    // set the ADC interrupt enable bit. When this bit is written 
                         // to one and the I-bit in SREG is set, the 
                         // ADC Conversion Complete Interrupt is activated. 

  ADCSRA |= (1<<ADSC);   // start ADC manually first time 
  sei();                 // Enable Global Interrupts  

  Serial.print ( "Output mode:    ");
  if (outputMode == NORMAL) {
    Serial.println ( "normal"); }
  else 
  {  
    Serial.println ( "anti-flicker");
    Serial.print ( "  offsetOfEnergyThresholds  = ");
    Serial.println ( offsetOfEnergyThresholdsInAFmode);    
  }
    
  Serial.print ( "powerCal_grid (CT4) =     "); Serial.println (powerCal_grid, 4);
  Serial.print ( "powerCal_diverted (CT3) = "); Serial.println (powerCal_diverted, 4);
  Serial.print ( "powerCal_CT2  =  "); Serial.println (powerCal_CT2, 4);
  Serial.print ( "powerCal_CT1  =  "); Serial.println (powerCal_CT1, 4);
  Serial.print ( "voltageCal =  "); Serial.println (voltageCal, 4); 
 
  Serial.print(">>free RAM = ");
  Serial.println(freeRam());  // a useful value to keep an eye on
  configureParamsForSelectedOutputMode(); 
  Serial.println ("----");    
  
  rf12_initialize(nodeID, freq, networkGroup);      // initialize RFM12B
//  rf12_sleep(RF12_SLEEP); <- the RFM12B now stays awake throughout
}

/* None of the workload in loop() is time-critical.  All the processing of 
 * ADC data is done within the ISR.
 */
void loop()             
{ 
  static byte perSecondTimer = 0;
//  
  // The ISR provides a 50 Hz 'tick' which the main code is free to use.
  if (newMainsCycle)
  {
    newMainsCycle = false;
    perSecondTimer++;
    
    if(perSecondTimer >= CYCLES_PER_SECOND) 
    {       
      perSecondTimer = 0; 
      
      // After a pre-defined period of inactivity, the Energy Diversion Detector  
      // needs to close down in readiness for the next's day's data. 
      //
      if (absenceOfDivertedEnergyCount > EDDshutdown_inMainsCycles)
      {
        // Clear the accumulators for diverted energy.  These are the "genuine" 
        // accumulators that are used by ISR rather than the copies that are 
        // regularly made available for use by the main code.
        //
        divertedEnergyTotal_Wh = 0;
        divertedEnergyRecent_IEU = 0;
        EDD_isActive = false; // the Energy Diversion Detector is now inactive
      }
    }     
  }
  
  if (datalogEventPending)   
  {
    datalogEventPending= false; 
    tx_data.power_grid_Watts = copyOf_sumP_grid * powerCal_grid / copyOf_sampleSetsDuringThisDatalogPeriod;
    tx_data.power_diverted_Watts = copyOf_sumP_diverted * powerCal_diverted / copyOf_sampleSetsDuringThisDatalogPeriod;
    tx_data.power_CT2_Watts = copyOf_sumP_CT2 * powerCal_CT2 / copyOf_sampleSetsDuringThisDatalogPeriod;
    tx_data.power_CT1_Watts = copyOf_sumP_CT1 * powerCal_CT1 / copyOf_sampleSetsDuringThisDatalogPeriod;
    tx_data.diverted_energy_WattHours = divertedEnergyTotal_Wh;
    tx_data.Vrms_timesTen = 10 * sqrt(copyOf_sum_Vsquared / copyOf_sampleSetsDuringThisDatalogPeriod) * voltageCal;
    tx_data.temperature_times100 = readTemperature();
    send_rf_data();    
    
    Serial.print("Watts@CT4(grid):");  Serial.print(tx_data.power_grid_Watts);
    Serial.print(", CT3:");  Serial.print(tx_data.power_diverted_Watts);
    Serial.print(", CT2:");  Serial.print(tx_data.power_CT2_Watts);
    Serial.print(", CT1:");  Serial.print(tx_data.power_CT1_Watts);
    Serial.print(", Wh@CT3:");  Serial.print(tx_data.diverted_energy_WattHours);
    Serial.print(", Vrms:");  Serial.print((float)tx_data.Vrms_timesTen / 10, 1);
    Serial.print(", temp:");  Serial.print((float)tx_data.temperature_times100 / 100, 1);
    Serial.print(", min#ofSS/MC:");  Serial.print(copyOf_lowestNoOfSampleSetsPerMainsCycle);
    Serial.print(", #ofSS:"); Serial.print(copyOf_sampleSetsDuringThisDatalogPeriod);
    Serial.println();
    convertTemperature(); // for use next time around
  }  
}


ISR(ADC_vect)  
/*
 * This Interrupt Service Routine looks after the acquisition and processing of
 * raw samples from the ADC sub-processor.  By means of various helper functions, all of 
 * the time-critical activities are processed within the ISR.  The main code is notified
 * by means of a flag to show when fresh copies of loggable data are available.
 */
{                                         
  static unsigned char sample_index = 0;
  static long sampleV_minusDC_long;
  int rawSample;
  long filtV_div4;
  long filtI_div4;
  long instP;
  long inst_Vsquared;
  long sampleIminusDC_long;
  long phaseShiftedSampleV_minusDC_long;
   
  switch(sample_index)
  {
    case 0:
      rawSample = ADC;                    // store the ADC value (this one is for Voltage)
      ADMUX = 0x40 + currentSensor_diverted;  // the conversion for current_grid (CT4) is already under way
      sample_index++;                   // increment the control flag
      //
      sampleVminusDC_long = ((long)rawSample<<8) - DCoffset_V_long; 
      if(sampleVminusDC_long > 0) { 
        polarityNow = POSITIVE; }
      else { 
        polarityNow = NEGATIVE; }
      confirmPolarity();
      //  
      checkProgress(); // deals with aspects that only occur at particular stages of each mains cycle
      //      
      // for the Vrms calculation (for datalogging only)
      filtV_div4 = sampleVminusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
      inst_Vsquared = filtV_div4 * filtV_div4; // 32-bits (now x4096, or 2^12)
      inst_Vsquared = inst_Vsquared>>12;     // scaling is now x1 (V_ADC x I_ADC)
      sum_Vsquared += inst_Vsquared; // cumulative V^2 (V_ADC x I_ADC)
      sampleSetsDuringThisDatalogPeriod++; 
      //      
      // store items for use during next loop
      cum_VdeltasThisCycle_long += sampleVminusDC_long; // for use with LP filter
      lastSampleV_minusDC_long = sampleVminusDC_long;  // required for phaseCal algorithm
      polarityConfirmedOfLastSampleV = polarityConfirmed;  // for identification of half cycle boundaries
      sampleSetsDuringThisCycle++;  // for real power calculations
    break;
    case 1:
      rawSample = ADC;               // store the ADC value (this one is for Grid Current on CT4)
      ADMUX = 0x40 + currentSensor_CT2;  // the conversion for current_diverted (CT3) is already under way
      sample_index++;                   // increment the control flag
      //
      // remove most of the DC offset from the current sample (the precise value does not matter)
      sampleIminusDC_long = ((long)(rawSample-DCoffset_I))<<8;
      //
      // phase-shift the voltage waveform so that it aligns with the current waveform at CT4
      phaseShiftedSampleV_minusDC_long = lastSampleV_minusDC_long
             + (((sampleVminusDC_long - lastSampleV_minusDC_long)*phaseCal_int_CT4)>>8);  
      //                                                            
      // calculate the "real power" in this sample pair and add to the accumulated sum
      filtV_div4 = phaseShiftedSampleV_minusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
      filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
      instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
      instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
      sumP_grid +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
      sumP_forEnergyBucket+=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
      break;
    case 2:
      rawSample = ADC;    // store the ADC value (this one is for diverted current on CT3)
      ADMUX = 0x40 + currentSensor_CT1;  // the conversion for current_CT2 is already under way
      sample_index++;                   // increment the control flag
      //
      // remove most of the DC offset from the current sample (the precise value does not matter)
      sampleIminusDC_long = ((long)(rawSample-DCoffset_I))<<8;
      //
      // phase-shift the voltage waveform so that it aligns with the current waveform at CT3
      phaseShiftedSampleV_minusDC_long = lastSampleV_minusDC_long
             + (((sampleVminusDC_long - lastSampleV_minusDC_long)*phaseCal_int_CT3)>>8);  
      //                                                            
      // calculate the "real power" in this sample pair and add to the accumulated sum
      filtV_div4 = phaseShiftedSampleV_minusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
      filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
      instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
      instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
      sumP_diverted +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
      sumP_forDivertedEnergy+=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)    
      break;
    case 3:
      rawSample = ADC;               // store the ADC value (this one is for current_CT2)
      ADMUX = 0x40 + voltageSensor;  // the conversion for current_CT1 is already under way
      sample_index++;                   // increment the control flag
      //
      // remove most of the DC offset from the current sample (the precise value does not matter)
      sampleIminusDC_long = ((long)(rawSample-DCoffset_I))<<8;
      //
      // phase-shift the voltage waveform so that it aligns with the current waveform at CT2
      phaseShiftedSampleV_minusDC_long = lastSampleV_minusDC_long
             + (((sampleVminusDC_long - lastSampleV_minusDC_long)*phaseCal_int_CT2)>>8);  
      //                                                            
      // calculate the "real power" in this sample pair and add to the accumulated sum
      filtV_div4 = phaseShiftedSampleV_minusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
      filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
      instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
      instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
      sumP_CT2 +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
      break;
    case 4:
      rawSample = ADC;               // store the ADC value (this one is for current_CT1)
      ADMUX = 0x40 + currentSensor_grid;  // the conversion for Voltage is already under way
      sample_index = 0;                 // reset the control flag
      //       
      // remove most of the DC offset from the current sample (the precise value does not matter)
      sampleIminusDC_long = ((long)(rawSample-DCoffset_I))<<8;
      //
      // phase-shift the voltage waveform so that it aligns with the current waveform at CT1
      phaseShiftedSampleV_minusDC_long = lastSampleV_minusDC_long
             + (((sampleVminusDC_long - lastSampleV_minusDC_long)*phaseCal_int_CT1)>>8);  
      //                                                            
      // calculate the "real power" in this sample pair and add to the accumulated sum
      filtV_div4 = phaseShiftedSampleV_minusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
      filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
      instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
      instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
      sumP_CT1 +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
      break;
     default:
      sample_index = 0;                 // to prevent lockup (should never get here)      
  }
}

/* -----------------------------------------------------------
 * Start of various helper functions which are used by the ISR 
 */
 
void checkProgress()
/* 
 * This routine is called by the ISR when each voltage sample becomes available. 
 * At the start of each new mains cycle, another helper function is called.  
 * Processing at other stages within the mains cycle is done by this function.
 */
{  
  static enum loadStates nextStateOfLoad; // must be static for Anti-Flicker mode
  static enum LEDstates nextStateOfLED;   // must be static for Anti-Flicker mode
  
  if (polarityConfirmed == POSITIVE) 
  { 
    if (beyondStartUpPhase)
    {     
      if (polarityConfirmedOfLastSampleV != POSITIVE)
      {
        // this is the start of a new mains cycle
        //
        // a simple routine for checking the performance of this new ISR structure     
        if (sampleSetsDuringThisCycle < lowestNoOfSampleSetsPerMainsCycle) {
          lowestNoOfSampleSetsPerMainsCycle = sampleSetsDuringThisCycle; }
          
        processLatestContribution(); // updates the various energy accumulators
                                     // and clears the per-mains cycle variables
                                     
      } // end of processing that is specific to the first Vsample in each +ve half cycle 
  
      // still processing samples where the voltage is POSITIVE ...
      //
      if (sampleSetsDuringThisCycle == 5) // part way through the +ve half cycle
      {              
        if (energyInBucket_long < lowerEnergyThreshold_long)
        {
          nextStateOfLoad = LOAD_OFF; 
          nextStateOfLED = LED_OFF; 
        }
        else
        if (energyInBucket_long > upperEnergyThreshold_long)
        {
          nextStateOfLoad = LOAD_ON; 
          nextStateOfLED = LED_ON; 
        }
        
        // set the output pin and LED state accordingly
        digitalWrite(triggerControl, nextStateOfLoad);   
        digitalWrite(LEDpin, nextStateOfLED);   
      
        // update the Energy Diversion Detector
        if (nextStateOfLoad == LOAD_ON)
        {
          absenceOfDivertedEnergyCount = 0; 
          EDD_isActive = true;
        }            
        else
        {
          absenceOfDivertedEnergyCount++; 
        }          
      }    
    }
    else
    {  
      // wait until the DC-blocking filters have had time to settle
      if(millis() > (delayBeforeSerialStarts + startUpPeriod) * 1000) 
      {
        beyondStartUpPhase = true;
        sumP_grid = 0; // uses data from CT4
        sumP_diverted = 0; // uses data from CT3
        sumP_CT2 = 0;
        sumP_CT1 = 0;
        sampleSetsDuringThisCycle = 0;
        lowestNoOfSampleSetsPerMainsCycle = 999;
        sampleSetsDuringThisDatalogPeriod =0;
      }
    }    
  } // end of processing that is specific to samples where the voltage is positive
  
  else // the polatity of this sample is negative
  {     
    if (polarityConfirmedOfLastSampleV != NEGATIVE)
    {
      // This is the start of a new -ve half cycle (just after the zero-crossing point)      
      // which is a convenient point to update the Low Pass Filter for DC-offset removal
      //
      long previousOffset = DCoffset_V_long;
      DCoffset_V_long = previousOffset + (cum_VdeltasThisCycle_long>>6); // faster than * 0.01
      cum_VdeltasThisCycle_long = 0;
      
      // To ensure that the LPF will always start up correctly when 240V AC is available, its
      // output value needs to be prevented from drifting beyond the likely range of the 
      // voltage signal.  This avoids the need to use a HPF as was done for initial Mk2 builds.
      //
      if (DCoffset_V_long < DCoffset_V_min) {
        DCoffset_V_long = DCoffset_V_min; }
      else  
      if (DCoffset_V_long > DCoffset_V_max) {
        DCoffset_V_long = DCoffset_V_max; }
           
    } // end of processing that is specific to the first Vsample in each -ve half cycle
  } // end of processing that is specific to samples where the voltage is negative
} //  end of checkProgress()

void confirmPolarity()
{
  /* This routine prevents a zero-crossing point from being declared until 
   * a certain number of consecutive samples in the 'other' half of the 
   * waveform have been encountered.  It forms part of the ISR.
   */ 
  static byte count = 0;
  if (polarityNow != polarityConfirmedOfLastSampleV) { 
    count++; }  
  else {
    count = 0; }
    
  if (count >= POLARITY_CHECK_MAXCOUNT)
  {
    count = 0;
    polarityConfirmed = polarityNow;
  }
}


void processLatestContribution()
/* 
 * This routine runs once per mains cycle.  It forms part of the ISR.
 */
{
  newMainsCycle = true; // <--  a 50 Hz 'tick' for use by the main code 
  
  // Various sumP_xxx accumulators contain the sum of many individual contributions of
  // instantaneous power.  Two of these accumulators run for just one mains cycle, and are
  // processed first.  The remaining accumulators, which run for the entire datalogging 
  // period, are processed last.
  //
  // For the mechanism which controls the diversion of surplus power, the AVERAGE power 
  // at the 'grid' point during the previous mains cycle must be quantified. The first 
  // stage in this process is for sumP_forEnergyBucket to be divided by the number of 
  // samples that have contributed to its value.
  //
  // The next stage would normally be to apply a calibration factor so that real power 
  // can be expressed in Watts.  That's fine for floating point maths, but it's not such
  // a good idea when integer maths is being used.  To keep the numbers large, and also 
  // to save time, calibration of power is omitted at this stage.  Real Power (stored as 
  // a 'long') is therefore (1/powerCal) times larger than the actual power in Watts.
  //
  long realPower_for_energyBucket  = sumP_forEnergyBucket / sampleSetsDuringThisCycle; 
  long realPower_forDivertedEnergy = sumP_forDivertedEnergy / sampleSetsDuringThisCycle; 
  
  // The per-mainsCycle variables can now be reset for ongoing use 
  sumP_forEnergyBucket = 0;
  sumP_forDivertedEnergy = 0;
  sampleSetsDuringThisCycle = 0;
  
  // Next, the energy content of this power rating needs to be determined.  Energy is 
  // power multiplied by time, so the next step is normally to multiply the measured
  // value of power by the time over which it was measured.
  //   Instanstaneous power is calculated once every mains cycle. When integer maths is 
  // being used, a repetitive power-to-energy conversion seems an unnecessary workload.  
  // As all sampling periods are of similar duration, it is more efficient simply to 
  // add all of the power samples together, and note that their sum is actually 
  // CYCLES_PER_SECOND greater than it would otherwise be.
  //   Although the numerical value itself does not change, I thought that a new name 
  // may be helpful so as to minimise confusion.  
  //   The 'energy' variable below is CYCLES_PER_SECOND * (1/powerCal) times larger than 
  // the actual energy in Joules.
  //
  long realEnergy_for_energyBucket = realPower_for_energyBucket; 
  long realEnergy_diverted = realPower_forDivertedEnergy; 
          
  // The latest energy contribution from the grid connection point can now be added
  // to the energy bucket which determines the state of the dump-load.  
  //
  energyInBucket_long += realEnergy_for_energyBucket;   
  energyInBucket_long -= requiredExportPerMainsCycle_inIEU; // <- useful for PV simulation
         
  // Apply max and min limits to the bucket's level.  This is to ensure correct operation
  // when conditions change, i.e. when import changes to export, and vici versa.
  //
  if (energyInBucket_long > capacityOfEnergyBucket_long) { 
    energyInBucket_long = capacityOfEnergyBucket_long; } 
  else         
  if (energyInBucket_long < 0) {
    energyInBucket_long = 0; }           
 
  if (EDD_isActive) // Energy Diversion Display
  {
    // For diverted energy, the latest contribution needs to be added to an 
    // accumulator which operates with maximum precision.  To avoid the displayed
    // value from creeping, any small contributions which are likely to be 
    // caused by noise can be ignored.
    //   
    if (realEnergy_diverted > antiCreepLimit_inIEUperMainsCycle) {
      divertedEnergyRecent_IEU += realEnergy_diverted; }
      
    // Whole Watt-Hours are then recorded separately
    if (divertedEnergyRecent_IEU > IEU_per_Wh)
    {
      divertedEnergyRecent_IEU -= IEU_per_Wh;
      divertedEnergyTotal_Wh++;
    }  
  }
               
  /* Instantaneous power contributions are summed in accumulators during each 
   * datalogging period.  At the end of each period, copies are made of their 
   * content for use by the main code.  The accumulators are then reset for use 
   * during the next period.
   */       
  cycleCountForDatalogging ++;       
  if (cycleCountForDatalogging  >= DATALOG_PERIOD_IN_MAINS_CYCLES ) 
  { 
    cycleCountForDatalogging = 0;
    
    copyOf_sumP_grid = sumP_grid; // uses data from CT4
    copyOf_sumP_diverted = sumP_diverted; // uses data from CT3
    copyOf_sumP_CT2 = sumP_CT2;
    copyOf_sumP_CT1 = sumP_CT1;
    copyOf_sum_Vsquared = sum_Vsquared; 
    copyOf_lowestNoOfSampleSetsPerMainsCycle = lowestNoOfSampleSetsPerMainsCycle; // (for diags only)
    copyOf_sampleSetsDuringThisDatalogPeriod = sampleSetsDuringThisDatalogPeriod; // (for diags only)  
    
    sumP_grid = 0; // uses data from CT4
    sumP_diverted = 0; // uses data from CT3
    sumP_CT2 = 0;
    sumP_CT1 = 0;
    sum_Vsquared = 0;
    lowestNoOfSampleSetsPerMainsCycle = 999;
    sampleSetsDuringThisDatalogPeriod = 0;
    datalogEventPending = true;
  }
} 
/* End of helper functions which are used by the ISR
 * -------------------------------------------------
 */

void configureParamsForSelectedOutputMode()
// retained for compatibility with previous versions
{
  if (outputMode == ANTI_FLICKER)
  {
    // settings for anti-flicker mode
    lowerEnergyThreshold_long = 
       capacityOfEnergyBucket_long * (0.5 - offsetOfEnergyThresholdsInAFmode); 
    upperEnergyThreshold_long = 
       capacityOfEnergyBucket_long * (0.5 + offsetOfEnergyThresholdsInAFmode);   
  }
  else
  { 
    // settings for normal mode
    lowerEnergyThreshold_long = capacityOfEnergyBucket_long * 0.5; 
    upperEnergyThreshold_long = capacityOfEnergyBucket_long * 0.5;   
  }
  
  // display relevant settings for selected output mode
  Serial.print("  capacityOfEnergyBucket_long = ");
  Serial.println(capacityOfEnergyBucket_long);
  Serial.print("  lowerEnergyThreshold_long   = ");
  Serial.println(lowerEnergyThreshold_long);
  Serial.print("  upperEnergyThreshold_long   = ");
  Serial.println(upperEnergyThreshold_long);
  
  Serial.print(">>free RAM = ");
  Serial.println(freeRam());  // a useful value to keep an eye on
}


void convertTemperature()
{
  oneWire.reset();
  oneWire.write(SKIP_ROM);
  oneWire.write(CONVERT_TEMPERATURE);
//  oneWire.write(CONVERT_TEMPERATURE,1); // trial (RAE)
}

int readTemperature()
{
  byte buf[9];
  int result;
  
  oneWire.reset();
  oneWire.write(SKIP_ROM);
  oneWire.write(READ_SCRATCHPAD);
  for(int i=0; i<9; i++) buf[i]=oneWire.read();
  if(oneWire.crc8(buf,8)==buf[8])
  {
    result=(buf[1]<<8)|buf[0];
    // result is temperature x16, multiply by 6.25 to convert to temperature x100
    result=(result*6)+(result>>2);
  }
  else result=BAD_TEMPERATURE;
  return result;
}

void send_rf_data()
{
  // if ready to send + exit route if it gets stuck 
  int i = 0; 
  while (!rf12_canSend() && i<10)
  { 
    rf12_recvDone(); 
    i++;
  }
  rf12_sendNow(0, &tx_data, sizeof tx_data);
}

int freeRam () {
  extern int __heap_start, *__brkval; 
  int v; 
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval); 
}




