/* 
 * Tool to display the raw samples generated by an Arduino.  Starts in' debug' mode
 * which uses synthesized values for voltage and current.  For normal operation, 
 * commment out the '#define DEBUG' statement a few lines down.
 *
 * Pauses after each set of measurements has been taken.  Press spacebar, then [cr], 
 * to repeat.
 *
 *      Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *      July 2012
 */

#include "EmonLib.h"      // Include Emon Library
EnergyMonitor emon1;      // Create an instance of the EnergyMonitor class
                          //  (only used here for the function, readVcc() )

// for normal operation, the next line should be commented out
// -----------------------------------------------------------
//
#define DEBUG
 
#define POSITIVE 1
#define NEGATIVE 0

// define the input pins
byte voltageSensorPin = 2;
byte currentSensorPin = 1;

long cycleCount = 0;
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

long noOfSamplePairs = 0;
byte polarityNow = NEGATIVE; // probably not important, but better than being indeterminate
boolean recordingNow;

unsigned long timeNow;
unsigned long recordingMayStartAt;

int settlingDelay = 5; // <<---  settling time (seconds)for dc-blocking filters 
boolean firstLoop = true;

int SUPPLYVOLTAGE; // measured once in setup().  

// Local declarations of items for code that's been lifted from EmonLib. 
//
#ifdef DEBUG
// to ensure accurate maths for synthesised operation
float lastSampleV,sampleV;   // raw Voltage and Current samples, 
float lastSampleI,sampleI;   //   and their equivalent values from the previous loop                   
#else
int lastSampleV,sampleV;   // raw Voltage and Current samples, 
int lastSampleI,sampleI;   //   and their equivalent values from the previous loop                   
#endif

double lastFilteredV,filteredV;  // raw V & I values after filtering to remove the DC offset,
double lastFilteredI, filteredI; //   and their equivalent values from the previous loop             
double sumP;   //  cumulative sum of power calculations within this mains cycles

/* The following calibration constants need to be defined for local use rather than being
 * passed into the relevant EnergyMonitor receptor functions.  
 */
double VCAL = 295;
double ICAL = 78;
double PHASECAL = 1.7; 

double V_RATIO;
double I_RATIO;

char blankLine[82];
char newLine[82];

int storedSampleV[100];
int storedSampleI[100];

#ifdef DEBUG
// some additional components for use in DEBUG mode
float powerRating_4debug = 1234; // <<<------------ power to be 'measured' in debug mode 
byte noOfVoltageSamplesPerCycle_4debug = 65;
int voltageSamples_4debug[65];
byte vsIndex_4debug = 0;
#endif


void setup()
{  
  Serial.begin(9600);
    
#if defined DEBUG 
  Serial.println("In DEBUG mode ..."); 
  Serial.println(); 
  
  // clear these calibration values when in DEBUG mode
  V_RATIO = 1.0;
  I_RATIO = 1.0;
  PHASECAL = 0;
  
  /*  populate the voltage sample array for DEBUG use
   */
  float amplitude = 240 * sqrt(2); // in ADC units
  float angleIncrement = (2 * PI) / noOfVoltageSamplesPerCycle_4debug;
  float angleOffset = 0.01; // to avoid sample right at zero crossing point
  float angle, voltage;
  
  byte index;
  
  for (index = 0; index < noOfVoltageSamplesPerCycle_4debug; index++)
  {
    angle = (index * angleIncrement) + angleOffset;
    voltage = amplitude * sin(angle);
    voltageSamples_4debug[index] = (int)voltage;
  } 
  
#else
  Serial.println("In NORMAL mode ..."); 
  Serial.println(); 

  SUPPLYVOLTAGE = emon1.readVcc();
  V_RATIO = VCAL *((SUPPLYVOLTAGE/1000.0) / 1023.0);
  I_RATIO = ICAL *((SUPPLYVOLTAGE/1000.0) / 1023.0);

  Serial.print("SUPPLYVOLTAGE = "); 
  Serial.print(SUPPLYVOLTAGE); 
  Serial.print(", V_RATIO = "); 
  Serial.print(V_RATIO); 
  Serial.print(", I_RATIO = "); 
  Serial.println(I_RATIO); 
#endif

  blankLine[0] = '|';
  blankLine[80] = '|';
  
  // initialise each character
  for (int i = 1; i < 80; i++)
  {
    blankLine[i] = ' ';
  }
}


/*  Allow the system to run for several seconds so that the filtered 
 *  voltage waveform can settle down.  This info is needed for determining 
 *  the start of each new mains cycle.  During this period, a countdown 
 *  is displayed.
 *
 *  After the settling period has expired, raw samples taken during 
 *  one complete mains cycle are stored in an array.  The capacity of the 
 *  array needs to be sufficient for the number of sample pairs that may
 *  appear.  A 100 x 2 integer array will probably suffice.
 *
 *  At the start of the following cycle, the data collected during the 
 *  previous cycle data is sent to the Serial window.  
 */
void loop() // each iteration of loop is for one pair of measurements only
{
  lastSampleV=sampleV;                          //Used for digital high pass filter
  lastSampleI=sampleI;                          //Used for digital high pass filter
    
  lastFilteredV = filteredV;                    //Used for offset removal
  lastFilteredI = filteredI;                    //Used for offset removal 


  if(firstLoop)
  {
    timeNow = millis();
    Serial.print ("millis() now = ");
    Serial.println (timeNow);
    
    recordingMayStartAt = timeNow + (settlingDelay * 1000);
    Serial.print ("recordingMayStartAt ");
    Serial.println (recordingMayStartAt);
    
    recordingNow = false;
    firstLoop = false;
  }

  //-----------------------------------------------------------------------------
  //  Read in the raw voltage and current sample 
  //     (values are synthesised when in DEBUG mode)
  //-----------------------------------------------------------------------------
#ifdef DEBUG
  sampleV = getNextVoltageSample(); 
  
  // The current sample in DEBUG mode can now be synthesized.  Because the basic system is
  // linear, the value of the current sample can be based on that of the voltage sample.
  // First remove the known amount of DC-offset
   float voltage = sampleV - 500;
   
  // Next, calculate the current for the required amount of current and instantaneous voltage.
  // For 1kW, the rms current at 240V is (1000W / 240V) Amps.  For different values of 
  // voltage and power, this value should be adjusted accordingly
  //
  float current = (1000.0/240.0) * (voltage/240.0) * (powerRating_4debug/1000.0);
  //     Amps   =   Watts/Volts  *   Volts/Volts   *   Watts/Watts
  //
  // Now multipy by an arbitrary factor to give a nice looking waveform at 3kW 
  current *= 23.7;
  // 
  // Finally, add DC offset as the Arduino's ADC would do
  sampleI = current + 500;
  
#else    
  sampleV = analogRead(voltageSensorPin);                 //Read in raw voltage signal
  sampleI = analogRead(currentSensorPin);                 //Read in raw current signal
#endif

  //------------------------------------------------------------
  // B1) Apply digital high pass filter to remove 2.5V DC offset  
    //----------------------------------------------------------
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;
 

  if ((polarityNow == POSITIVE) && (polarityOfLastReading == NEGATIVE))
  {
    // This is the start of a new mains cycle
    cycleCount++; 
#ifdef DEBUG
    delay(14); //  <<----- to make the 'debug' rate match 50Hz    
#endif
    timeNow = millis();
    
    if (recordingNow == true)
    {
      // this is the start of the cycle AFTER the one where samples were collected, so
      // now display the data that was collected
      recordingNow = false;
      firstLoop = true;
      
      Serial.print("cycleCount ");
      Serial.print(cycleCount);
      Serial.print(",  samplesDuringThisMainsCycle ");
      Serial.println(samplesDuringThisMainsCycle);

      int voltage, current;
      int minVoltage = 1023, minCurrent = 1023;
      int maxVoltage = 0, maxCurrent = 0;
      
      for (int index = 0; index < samplesDuringThisMainsCycle; index++) 
      {
        strcpy(newLine, blankLine);
        voltage = storedSampleV[index]; 
        current = storedSampleI[index]; 

        if (voltage < minVoltage){minVoltage = voltage;}
        if (voltage > maxVoltage){maxVoltage = voltage;}
        if (current < minCurrent){minCurrent = current;}
        if (current > maxCurrent){maxCurrent = current;}
    
        newLine[map(voltage, 0, 1023, 0, 80)] = 'v';
        newLine[map(current, 0, 1023, 0, 80)] = 'c';
        
        if ((index % 2) == 0) // change this to "% 1" for full resolution
        {
          Serial.println(newLine);
        }
      }
      
      Serial.print("minVoltage ");  Serial.print(minVoltage);
      Serial.print(",  maxVoltage ");  Serial.print(maxVoltage);
      Serial.print(",  minCurrent ");  Serial.print(minCurrent);
      Serial.print(",  maxCurrent ");  Serial.println(maxCurrent);
      pause();
      
    }
    
    if((cycleCount % 50) == 1)
    {     
      if (timeNow > recordingMayStartAt)    
      {
        // we're off!
        Serial.println( " <recording data ...> ");
        Serial.println();
        recordingNow = true;
      } 
      else
      {
        Serial.println((int)(recordingMayStartAt - timeNow) / 1000);
//        Serial.print("  ");
//        Serial.println(cycleCount);
      }  
    }
    
    samplesDuringThisMainsCycle = 0;
    
  } // end of processing for first +ve reading in each mains cycle
  
  if (recordingNow == true)
  {
    storedSampleV[samplesDuringThisMainsCycle] = sampleV;
    storedSampleI[samplesDuringThisMainsCycle] = sampleI;

    if ((polarityNow == NEGATIVE) && (polarityOfLastReading == POSITIVE))
    {
    // uncomment this block to show the effect of Serial operations (approx 2mS delay)
/*
      Serial.print ("I've recorded ");
      Serial.print (samplesDuringThisMainsCycle);
      Serial.println (" samples so far ... ");
*/
    }    
  }
  samplesDuringThisMainsCycle++;
} // end of loop()


#ifdef DEBUG
// function to synthesive voltage samples for DEBUG mode only
float getNextVoltageSample()
{
  float voltageSample;

  voltageSample = voltageSamples_4debug[vsIndex_4debug];
  voltageSample += 500;
  
  vsIndex_4debug++;  
  if (vsIndex_4debug >= noOfVoltageSamplesPerCycle_4debug) 
  {
    vsIndex_4debug = 0;
  }

  return voltageSample;
}
#endif

// When called, this next function will cause the code to pause 
// until any key, followed by [C/R], is pressed.  
//

void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      done++;
    }
  }    
}

