/*
	© 2007 ROBO Design
	http://www.robodesign.ro
*/

// Last update: 2007-09-21

var RD_game = new (function (no_autoinit)
{
	var _me = this;

	// allow us to easily change the targeted document
	var doc = document;

	// default number of slots, configurable within UI
	_me.nslots = 9;

	// split the slots on to the next row, after N slots (columns)
	_me.cols = 3;

	// slot width and height
	_me.slotw = 34;
	_me.sloth = 34;

	// slot spacing (width and height)
	_me.slotpw = 4;
	_me.slotph = 4;

	// slot width, height and padding when shown as a bonus combination in the table
	_me.bcombow = 10;
	_me.bcomboh = 10;
	_me.bcombop = 2;

	// limit the number of turns/moves, such that players are forced to think of a quick solution
	// this also limits playing for an infinite amount of time
	_me.maxMoves = 100;

	// end the game after N turns, if after N turns none of the following happens:
	// - a reduction of variants
	// - a bonus combination is scored
	_me.maxMovesNoChange = 5;

	// the properties for the objects. any number of properties and values can be added, just follow the same scheme.
	// when adding/deleting shapes you must also update the SVG document.
	// note that the order for the values is important
	_me.props = {
		// 'property-id' : [value1, value2, ...]
		'shape' : ['circle', 'square', 'triangle'],
		'color' : ['red','yellow', 'blue']
	};

	// how many previous moves are to be disallowed
	_me.disallow_nmoves = 2;

	// control game mode: networked (not yet implemented) or local mode
	// in networked game the local user can't make moves for other players, only for _me.local_player
	_me.local_mode = true;

	// show form controls?
	// if not, only voice can be used to control the game
	_me.showFormControls = false;

	// this stores the name of each player
	_me.players = ['user 1', 'user 2'];

	_me.messages = {
		'init-error' : 'The game cannot be initialized!',
		'player-move' : 'Player %nr% (%name%) make your move!',
		'player-won' : 'Player %nr% (%name%) won the game!',
		'game-over-tie' :  'Game over! It is a tie!',
		'game-end-maxMoves' : 'Game ended after %nr% moves. Restart the game to try again.',
		'game-end-nochange' : 'Game ended after %nr% moves which did not change anything important (no reduction of variations, no bonus combination). Restart the game to try again.',
		'move-not-allowed' : 'You are not allowed to make the move!',
		'no-objects-found' : 'No object matched your query!',
		'internal-error' : 'Internal error',
		'previous-move' : 'You are trying to repeat a previous move! You cannot repeat the previous %nr% move(s).',
		'false-move' : 'You are trying to make a move which does not change anything!',

		// voice messages
		'voice-prompt' : 'Player %nr% (%name%) make your move!',
		'voice-help' : 'Just say the objects you want to change, and what you want to do with them. For example: change red circles to blue squares.',
		'voice-nomatch' : 'I did not understand what you want.',
		'voice-noinput' : 'If you do not know what to do, ask for help.'
	};

	const ns_svg = 'http://www.w3.org/2000/svg',
		ns_xlink = 'http://www.w3.org/1999/xlink';

	// the ID of the local player
	_me.local_player = 0;

	_me.playing = false;
	_me.initialized = false;

	// this is only run once, on page load
	_me.init = function ()
	{
		// containers for slots, bonus combinations, previous moves and the SVG document itself
		_me.svgc = doc.getElementById('svg-slots');
		_me.svg_bcombos = doc.getElementById('svg-bonusCombos');
		_me.svg_bgrcombos = doc.getElementById('svg-bgrCombos');
		_me.prevmc = doc.getElementById('svg-prevMoves');
		_me.svgdoc = doc.getElementById('mysvg');

		_me.animSlots = doc.getElementById('anim-slots');

		// game status info
		_me.out_gStatus = doc.getElementById('out_gStatus');

		// show variations
		_me.out_variations = doc.getElementById('out_variations');

		// the list of players, in the GUI, with inputs for allowing the user to change the name
		_me.out_players = doc.getElementById('out_players');

		_me.in_nslots = doc.getElementById('in_nslots');
		_me.in_cols = doc.getElementById('in_cols');

		var btn_usrCmd = doc.getElementById('btn_usrCommand'),
			btn_rGame = doc.getElementById('btn_restartGame'),
			formControl = doc.getElementById('formControl');

		// if the elements were not found, the game cannot be initialized
		if(!_me.svgc || !_me.in_nslots || !btn_usrCmd || !btn_rGame || !_me.out_variations || !_me.out_players || !_me.in_cols || !_me.svgdoc || !_me.prevmc || !_me.svg_bcombos || !_me.svg_bgrcombos)
		{
			alert(_me.messages['init-error'] || 'Internal error');
			return false;
		}

		// setup the events
		btn_usrCmd.addEventListener('click', formCommand, false);
		btn_rGame.addEventListener('click', _me.restartGame, false);
		_me.in_nslots.addEventListener('change', onChange_nslots, false);
		_me.in_cols.addEventListener('change', onChange_cols, false);

		// calculate the maximum number of slots, rows and columns
		var svgw = _me.svgdoc.width.baseVal.value,
			svgh = _me.svgdoc.height.baseVal.value,
			transform = _me.svgc.transform.baseVal.getItem(0),
			sw = _me.slotw+_me.slotpw,
			sh = _me.sloth+_me.slotph;

		if(transform.matrix && transform.matrix.e)
			svgw -= transform.matrix.e*2;

		if(transform.matrix && transform.matrix.f)
			svgh -= transform.matrix.f*2;

		svgw -= 160; // the HUD
		svgh -= 16; // the interface has a drop shadow of 16px...

		_me.max_cols = Math.ceil( svgw/sw );
		_me.max_rows = Math.ceil( svgh/sh );
		_me.max_slots = _me.max_rows * _me.max_cols;

		// set the defaults of in_cols and of in_nslots, then validate the values
		_me.in_nslots.value = _me.nslots;
		_me.in_cols.value = _me.cols;
		onChange_nslots();
		onChange_cols();

		// we need this to be able to check if the animation is playing or not
		if(_me.animSlots)
		{
			_me.animSlots.isPlaying = false;
			_me.animSlots.addEventListener('beginEvent', function () { this.isPlaying = true; }, false);
			_me.animSlots.addEventListener('endEvent', function () { this.isPlaying = false; }, false);

			// also make the svg slots invisible, during initialization
			_me.svgc.setAttributeNS( null, 'opacity', 0 );
		}

		var i, elem_tr, elem_td, elem, elem_p;

		// add the list of players, with text field inputs (player names) and scores
		remChilds(_me.out_players, 1);
		for(i in _me.players)
		{
			elem_tr = doc.createElement('tr');
			elem_tr._pscore = 0;
			elem_tr._pid = parseInt(i);

			elem_td = doc.createElement('td');
			elem_p = doc.createElement('p');

			// the input
			elem = doc.createElement('input');
			elem.type = 'text';
			elem.value = _me.players[i];
			elem.addEventListener('change', onChange_playerName, false);

			elem_p.appendChild(elem);
			elem_td.appendChild(elem_p);
			elem_tr.appendChild(elem_td);

			// the score
			elem = doc.createElement('td');
			elem.appendChild( doc.createTextNode('0') );

			elem_tr.appendChild(elem);

			_me.out_players.appendChild(elem_tr);
		}

		// calculate the maximum number of variations possible (based on the available slot properties and values)
		_me.max_variations = 1;
		_me.min_props = 1000;
		for(i in _me.props)
		{
			if(_me.props[i].length < _me.min_props)
				_me.min_props = _me.props[i].length;

			_me.max_variations *= _me.props[i].length;
		}
		_me.max_variations--;
		_me.min_variations = calculateMinVar();

		var y, elem_sel, elem_dest;
		_me.formFields = {};
		for(i in _me.props)
		{
			elem_sel = doc.getElementById('sel_' + i);
			elem_dest = doc.getElementById('dest_' + i);

			if(!elem_sel || !elem_dest)
			{
				alert(_me.messages['init-error'] || 'Internal error');
				return false;
			}

			elem_sel.value = '-';
			elem_dest.value = '-';
			_me.formFields['sel_' + i] = elem_sel;
			_me.formFields['dest_' + i] = elem_dest;

			for(y in _me.props[i])
			{
				elem = doc.createElement('option');
				elem.value = y;

				elem.appendChild( doc.createTextNode(_me.props[i][y]) );

				elem = elem_sel.appendChild(elem);

				elem_dest.appendChild( elem.cloneNode(true) );
			}
		}

		// show or hide the form control based on configuration
		if(formControl)
		{
			if(_me.showFormControls)
				formControl.style.display = 'block';
			else
				formControl.style.display = 'none';
		}

		var res = _me.restartGame();
		if(res)
			_me.initialized = true;
		else
			alert(_me.messages['internal-error'] || 'Internal error');

		return res;
	};

	// used for (re)starting the game
	_me.restartGame = function ()
	{
		if(_me.animSlots && _me.animSlots.isPlaying)
			return false;

		var do_reposition = false;

		// update the number of slots per column, from the GUI
		if(_me.in_cols.value && _me.cols != _me.in_cols.value)
		{
			do_reposition = true;
			_me.cols = parseInt(_me.in_cols.value);
		}

		// update the number of slots, from the GUI
		if(_me.in_nslots.value && _me.nslots != _me.in_nslots.value)
		{
			_me.nslots = parseInt(_me.in_nslots.value);
			_me.min_variations = calculateMinVar();
		}

		// the actual function which restarts the game
		var doRestart = function ()
		{
			// set the ID of the local player if in local_mode or if the game is not yet initialized
			if(_me.local_mode)
				_me.local_player = 0;

			_me.winner = -1;
			_me.move_by = 0;
			_me.variations = false;
			_me.prev_moves = [];
			_me.bonus_combos = [];
			_me.moves = 0;
			_me.movesNoChange = 0;
			_me.playing = false;

			// reset player scores
			_me.scores = [];
			var i;

			for(i in _me.players)
			{
				_me.scores[i] = 0;
			}

			var randProps = {}, y, elem, n, p;

			// remove the current set of slots
			_me.slots = [];
			remChilds(_me.svgc);

			// this removes all the current UI shapes that indicate the previous moves
			remChilds(_me.prevmc);

			// remove all the bonus combo shapes
			remChilds(_me.svg_bcombos);

			// update the players table
			n = _me.out_players.childNodes.length;
			for(i=0; i < n; i++)
			{
				elem = _me.out_players.childNodes.item(i);
				if(typeof elem._pid == 'undefined')
					continue;

				// update the score
				remChilds(elem.lastChild);
				elem.lastChild.appendChild( doc.createTextNode('0') );
				elem._pscore = 0;

				// update bonus combos and remove the .player_bonus_combos className
				elem = elem.childNodes[0];
				if(elem.className)
					elem.className = '';
			}

			if(do_reposition || !_me.initialized)
				repositionSvg();

			var r = 0, c = 0, score, l = 0;

			// add the new slots
			for(i = 0; i < _me.nslots; i++, c++)
			{
				// randomly pick the slot properties
				randProps = {};
				for(p in _me.props)
				{
					n = _me.props[p].length-1;
					randProps[p] = Math.round(n*Math.random());
				}

				// draw the slot
				if(drawSlot(r, c, randProps) < -1)
					return false;

				// go to the next row
				if( (c+1) == _me.cols )
				{
					r++;
					c = -1;
				}

				// restart if the number of variations is too low
				if((i+1) == _me.nslots)
				{
					_me.variations = calculateVariations();

					// we do not want an endless loop
					if(l > 10)
						continue;

					// if there are enough variations, also check for bonus combinations
					if(_me.variations >= _me.min_variations)
					{
						_me.bonus_combos = [];
						score = calculateScore();
					}

					if(_me.variations < _me.min_variations || score > 0)
					{
						i = -1;
						r = 0;
						c = -1;
						l++;
					}

					if(!_me.variations || score == -1)
						return false;
				}
			}

			// fade in for the objects
			if(_me.initialized && _me.animSlots && _me.animSlots.beginElement)
			{
				_me.animSlots.setAttributeNS(null, 'from', 0);
				_me.animSlots.setAttributeNS(null, 'to', 1);
				_me.animSlots.removeEventListener('endEvent', doRestart, false);
			}

			if(_me.animSlots && _me.animSlots.beginElement)
				_me.animSlots.beginElement();

			_me.playing = true;

			syncSVGnHUD();

			if(!_me.updateDisplay())
				return false;

			return true;
		};

		// fade out for the objects
		if(_me.initialized && _me.animSlots && _me.animSlots.beginElement)
		{
			_me.animSlots.setAttributeNS(null, 'from', 1);
			_me.animSlots.setAttributeNS(null, 'to', 0);
			_me.animSlots.addEventListener('endEvent', doRestart, false);
			_me.animSlots.beginElement();
		} else
			return doRestart();

		return true;
	};

	// display the current information
	_me.updateDisplay = function ()
	{
		var elem, msg = ''; 

		if(_me.playing)
			msg = getMsg('player-move', {'nr' : _me.move_by+1, 'name' : _me.players[_me.move_by]});
		else
		{
			if(_me.winner > -1)
				msg = getMsg('player-won', {'nr' : _me.winner+1, 'name' : _me.players[_me.winner]});
			else
				msg = getMsg('game-over-tie');
		}

		elem = doc.createTextNode(msg);
		remChilds(_me.out_gStatus);
		_me.out_gStatus.appendChild(elem);

		var n = _me.out_players.childNodes.length, i, score_node, pid;
		for(i=0; i < n; i++)
		{
			elem = _me.out_players.childNodes.item(i);
			pid = elem._pid;
			if(typeof pid == 'undefined')
				continue;

			if(pid == _me.move_by)
				elem.className = 'move_by';
			else if(pid == _me.local_player)
				elem.className = 'local_player';
			else if(elem.className)
				elem.className = '';

			if(_me.scores[pid] != elem._pscore)
			{
				remChilds(elem.lastChild);

				score_node = doc.createTextNode( _me.scores[pid] );
				elem.lastChild.appendChild(score_node);
				elem._pscore = _me.scores[pid];
			}
		}

		elem = doc.createTextNode(_me.variations);
		remChilds(_me.out_variations);
		_me.out_variations.appendChild(elem);

		return true;
	};

	// this is the general-purpose function that does the move in the game
	// @src = an object {name: value, ...}. These are used as a query to find the matching slots.
	// @dest = an object {name: value, ...}. These are set to the slots found.
	// @who = (optional) the ID of the player who makes the move. In order to be allowed to make the move who must be equal to _me.move_by
	//        if undefined, @who is default _me.local_player
	// if _me.local_mode is true, then _me_.local_player will be set to _me.move_by (the next player). this is to allow multiple players to make moves on a single computer
	// returns:
	//    4 if the limit of maxMovesNoChange has been reached, after making the move
	//    3 if the limit of maxMoves has been reached, after making the move
	//    2 if the player won
	//    1 on success
	//   -1 if the player is not allowed to make the move
	//   -2 if no objects found matching the query
	//   -3 if the user wanted to repeat a previous move
	//   -4 if the user tried to make a move that does not change any slot
	//   -5 on general error
	_me.makeMove = function (src, dest, who)
	{
		if(!src || !dest) 
			return -4;

		// make the move for the local_player if who is undefined or not found
		if(isNaN(who) || !_me.players[who])
			who = _me.local_player;

		who = parseInt(who); // it's not always a number...

		// the player cannot make a move if it's not his/her turn
		// we also disallow making moves if the game is finished (e.g. someone won)
		if(!_me.playing || who != _me.move_by)
			return -1;

		var i, found_diff = false;

		// check for differences between src and dest
		for(i in src)
		{
			if(typeof dest[i] == 'undefined' || src[i] != dest[i])
			{
				found_diff = true;
				break;
			}
		}

		for(i in dest)
		{
			if(typeof src[i] == 'undefined' || src[i] != dest[i])
			{
				found_diff = true;
				break;
			}
		}

		// if no difference found, then we do not continue
		if(!found_diff)
			return -4;

		// find the slots, the function returns an array of array [row, col] for each slot found
		var src_slots = findSlots(src);

		// no slots found? return the error
		if(!src_slots)
			return -2;

		var str_move = arrToStr(src_slots), str;

		// check if this move is remembered, so we deny it
		for(i in _me.prev_moves)
		{
			str = arrToStr(_me.prev_moves[i]);

			if(str == str_move)
				return -3;
		}

		// update the slots
		var n = src_slots.length, drawres, changes_made = false;
		for(i = 0; i < n; i++)
		{
			drawres = drawSlot(src_slots[i][0], src_slots[i][1], dest);

			if(!changes_made && drawres == 1)
				changes_made = 1;
			else if(drawres < -1)
				return -5; // general error
		}
		if(!changes_made)
			return -4;

		// store the move and, if needed, remove the oldest move
		if(_me.prev_moves.unshift(src_slots) > _me.disallow_nmoves)
			_me.prev_moves.pop();

		_me.moves++;
		_me.movesNoChange++;

		var variations = calculateVariations();
		if(!variations)
			return -5; // general error, calculateVariations should never return false

		// reset the counter for moves without a change, if the number of variations has been reduced
		if(variations < _me.variations)
			_me.movesNoChange = 0;

		_me.variations = variations;

		// calculateScore for the current setup of slots
		// if there are any bonus points (same figure / same color), then add them

		var score = calculateScore();
		if(score == -1)
			return -5; // general error, calculateScore should never return -1
		else if(score > 0)
		{
			_me.scores[who] += score;

			// again, reset the counter for moves without any changes, if the user scored a bonus combination
			_me.movesNoChange = 0;

			renderBonusCombos();
		}

		var res = 1;

		// if all objects are the same, the player won
		if(_me.variations == 1)
		{
			_me.winner = who;
			_me.playing = false;
			res = 2;
		} else if(_me.maxMoves == _me.moves)
		{
			_me.playing = false;
			res = 3;
		} else if(_me.maxMovesNoChange == _me.movesNoChange)
		{
			_me.playing = false;
			res = 4;
		} else
		{
			// cycle to the next player
			_me.move_by++;
			if(_me.move_by == _me.players.length)
				_me.move_by = 0;

			// allow the local players to make the next move
			if(_me.local_mode)
				_me.local_player = _me.move_by;
		}

		// determine the winner by score
		if(res == 3 || res == 4)
		{
			var winner = -1, max_score = 0;
			for(i in _me.scores)
			{
				if(_me.scores[i] > max_score)
				{
					winner = i;
					max_score = _me.scores[i];
				} else if(_me.scores[i] == max_score)
					winner = -1;
			}

			_me.winner = parseInt(winner);
		}

		if(!_me.updateDisplay() || !renderPrevMoves())
			return -5; // oops

		return res;
	};


	// this calculates the minimum number of variations for the game initialization
	function calculateMinVar ()
	{
		// calculate the minimum number of variations
		var min = Math.ceil ( _me.nslots / 2 );
		if(min < _me.min_props)
			min = _me.min_props;

		// we might have more slots than possible unique combinations
		if(min > _me.max_variations)
			min = _me.max_variations;

		return min;
	};

	// this function checks the slots, for any bonus combos (same figure/same color, same property, on the same row/column)
	// for performance optimisation, this could be merged into calculateVariations()
	// returns: the bonus score (integer) which can be 0, -1 on error
	function calculateScore ()
	{
		var score = 0, rows = [], cols = [], r, c, slot, i, p, variation_str = '', variation_obj = {};

		for(r in _me.slots)
		{
			for(c in _me.slots[r])
			{
				// skip if the current row and column do not match
				if(rows[r] == -1 && cols[c] == -1)
					continue;

				slot = _me.slots[r][c];
				variation_str = '';
				variation_obj = {};

				for(p in _me.props)
				{
					if(typeof slot[p] == 'undefined')
						return -1;

					variation_str += ' ' + slot[p];
					variation_obj[p] = slot[p];
				}

				// if this combo is already remembered, we can skip it
				if(_me.bonus_combos[variation_str])
				{
					rows[r] = -1;
					cols[c] = -1;
					continue;
				}

				// remember the slot props for this row
				if(!rows[r])
					rows[r] = {'str' : variation_str, 'obj' : variation_obj};

				// check if the slot on this column matches the previous slot(s) on the same row
				if(rows[r] != -1 && rows[r]['str'] != variation_str)
					rows[r] = -1;

				// same as above, but for columns
				if(!cols[c])
					cols[c] = {'str' : variation_str, 'obj' : variation_obj};

				if(cols[c] != -1 && cols[c]['str'] != variation_str)
					cols[c] = -1;
			}
		}

		// calculate the score

		for(r in rows)
		{
			if(rows[r] == -1)
				continue;

			variation_str = rows[r]['str'];
			if(_me.bonus_combos[variation_str])
				continue;

			score += 1;
			_me.bonus_combos[variation_str] = rows[r]['obj'];
			_me.bonus_combos[variation_str]['_pid'] = parseInt(_me.move_by);
		}

		for(c in cols)
		{
			if(cols[c] == -1)
				continue;

			variation_str = cols[c]['str'];
			if(_me.bonus_combos[variation_str])
				continue;

			score += 1;
			_me.bonus_combos[variation_str] = cols[c]['obj'];
			_me.bonus_combos[variation_str]['_pid'] = parseInt(_me.move_by);
		}

		return score;
	};

	// this function takes the list of bonus combinations found by the players and renders the figures in the interface.
	function renderBonusCombos ()
	{
		var i, pid, elem_td, elem_tr, xpos = 0, ypos = 0, elem, elem_trs, shape, color, bcombos_per_player = [];

		// the initial Y-axis position for the figures
		ypos = _me.out_players.offsetTop;

		remChilds(_me.svg_bcombos);

		elem_trs = _me.out_players.getElementsByTagName('tr');

		for(i in _me.bonus_combos)
		{
			props = _me.bonus_combos[i];
			pid = props['_pid'];

			elem_tr = elem_trs[ (pid+1) ];
			if(!elem_tr || elem_tr._pid != pid)
				continue;

			if(!bcombos_per_player[pid])
				bcombos_per_player[pid] = 0;

			bcombos_per_player[pid]++;

			elem_td = elem_tr.firstChild;
			if(!elem_td.className)
				elem_td.className = 'player_bonus_combos';

			xpos = bcombos_per_player[pid]*(_me.bcombow+_me.bcombop);

			// the slot cannot be (re)drawn if the shape and color properties are missing
			if(isNaN(props['shape']) || isNaN(props['color']))
				return false;

			// the values of the properties
			shape = _me.props.shape[props['shape']];
			color = _me.props.color[props['color']];

			// add the SVG element
			elem = doc.createElementNS(ns_svg, 'svg:use');
			elem.setAttributeNS(ns_xlink, 'xlink:href', '#shape-' + shape );
			elem.setAttributeNS( null, 'fill', 'url(#fill-' + color + ')' );
			elem.setAttributeNS( null, 'x', xpos );
			elem.setAttributeNS( null, 'width', _me.bcombow );
			elem.setAttributeNS( null, 'height', _me.bcomboh );

			elem = _me.svg_bcombos.appendChild(elem);
			elem._pid = pid;
		}

		// update the Y position of the objects
		var n = _me.svg_bcombos.childNodes.length;
		for(i = 0; i < n; i++)
		{
			elem = _me.svg_bcombos.childNodes.item(i);
			pid = elem._pid;

			elem_tr = elem_trs[ (pid+1) ];

			elem.setAttributeNS( null, 'y', ypos + elem_tr.offsetTop + elem_tr.clientHeight - _me.bcomboh-4 );
		}

		syncSVGnHUD();

		return true;
	};

	function syncSVGnHUD ()
	{
		var svgh = _me.svgdoc.height.baseVal.value,
			hudh = _me.out_players.parentNode.clientHeight;

		// if the SVG container is smaller in height than the "heads-up" display (the right-side of the GUI), we must change the height
		if(svgh != hudh)
			_me.svgdoc.setAttributeNS(null, 'height', hudh);

		var trs = _me.out_players.getElementsByTagName('tr');

		// update the rectangle which is the background of the figures
		_me.svg_bgrcombos.setAttributeNS(null, 'height', _me.out_players.clientHeight - trs[0].clientHeight);
		_me.svg_bgrcombos.setAttributeNS(null, 'y', _me.out_players.offsetTop + trs[0].clientHeight);

		return true;
	};

	// this function takes the list of previous moves and adds UI indicators to the specific slots
	function renderPrevMoves ()
	{
		var i, y, slot, id, str, elem, pNode,
			sx = _me.slotw + _me.slotpw,
			sy = _me.sloth + _me.slotph;

		// this removes all the current UI shapes that indicate the previous moves
		remChilds(_me.prevmc);

		// do not continue if someone won the game
		if(_me.winner > -1)
			return true;

		for(i in _me.prev_moves)
		{
			i = parseInt(i); // needed... hmm
			for(y in _me.prev_moves[i])
			{
				id = _me.prev_moves[i][y];
				if(!_me.slots[id[0]] || !_me.slots[id[0]][id[1]])
					return false;
				slot = _me.slots[id[0]][id[1]];
				if(!slot.elem)
					return false;

				// add the SVG element
				elem = doc.createElementNS(ns_svg, 'svg:use');
				elem.setAttributeNS(ns_xlink, 'xlink:href', '#prevMove' + (i+1) );
				elem.setAttributeNS( null, 'x', id[1] * sx );
				elem.setAttributeNS( null, 'y', id[0] * sy );
				elem.setAttributeNS( null, 'width', _me.slotw );
				elem.setAttributeNS( null, 'height', _me.sloth );

				_me.prevmc.appendChild(elem);
			}
		}

		return true;
	};

	// this finds out how many unique slots are displayed, based on their properties
	// e.g. red triangle is different from red circle, also a blue square is different from a yellow square
	// if the return value is 1, the game is in a "winning" state
	function calculateVariations ()
	{
		var variations = {}, n = 0, props = [], slot, r, c, i, p = 0, v = '';

		for(r in _me.slots)
		{
			for(c in _me.slots[r])
			{
				slot = _me.slots[r][c];
				v = '';


				for(p in _me.props)
				{
					if(typeof slot[p] == 'undefined')
						return -1;

					v += ' ' + slot[p];
				}

				if(!variations[v])
				{
					variations[v] = 1;
					n++;
				}
			}
		}

		// n must not be 0
		if(n == 0)
			return -1;

		return n;
	};

	// this returns the list of slots found having the given properties (query)
	// all properties in the "query" must match
	function findSlots (query)
	{
		if(!query)
			return false;

		var i, y, z, list = [], found, slot;

		for(i in _me.slots)
		{
			for(y in _me.slots[i])
			{
				found = true;
				slot = _me.slots[i][y];

				for(z in query)
				{
					if(slot[z] != query[z])
					{
						found = false;
						break;
					}
				}

				if(found)
				{
					list.push([i, y]);
				}
			}
		}

		if(!list[0])
			return false;

		return list;
	};

	// this is called when the user changes the player name
	function onChange_playerName ()
	{
		var pid = this.parentNode.parentNode.parentNode._pid;

		if(typeof pid == 'undefined' || !_me.players[pid])
			return false;

		// if not in local mode, do not allow the local user to change other player names
		// if the value is empty, but the initial name back
		if(!this.value || (!_me.local_mode && _me.local_player != pid))
			this.value = _me.players[pid];
		else
		{
			_me.players[pid] = this.value;
			_me.updateDisplay();
		}

		return true;
	};

	// validate the value of in_nslots form field
	// the function also automatically updates in_cols.value if needed
	function onChange_nslots ()
	{
		var n = _me.in_nslots.value;
		if(!n)
			n = _me.nslots;

		if(isNaN(n))
			n = _me.nslots;

		n = parseInt(n);
		var	c = parseInt(_me.in_cols.value);

		if(n < _me.min_props)
			n = _me.min_props;

		if(n > _me.max_slots)
			n = _me.max_slots;

		var max_currently = c * _me.max_rows;

		if(n > max_currently)
			c = Math.ceil( n / _me.max_rows );

		if(n < c)
			c = n;

		_me.in_nslots.value = n;
		_me.in_cols.value = c;
	};

	// validate the value of in_cols form field
	// the function also automatically updates in_nslots.value if needed
	function onChange_cols ()
	{
		var c = _me.in_cols.value;
		if(!c || isNaN(c) || c < 1)
			c = _me.cols;

		c = parseInt(c);

		var n = parseInt(_me.in_nslots.value);

		if(c > _me.max_cols)
			c = _me.max_cols;

		var max_currently = c * _me.max_rows;

		if(n > max_currently)
			n = max_currently;

		if(c > n)
			n = c;

		_me.in_cols.value = c;
		_me.in_nslots.value = n;
	};

	// this updates the translation of the svg container, to center the objects in the view
	function repositionSvg ()
	{
		// the 140 pixels are for the #HUD on the right
		var svgw = _me.svgdoc.width.baseVal.value-140,
			transform1 = _me.svgc.transform.baseVal.getItem(0),
			transform2 = _me.prevmc.transform.baseVal.getItem(0),
			w = (_me.slotw+_me.slotpw)*_me.cols;

		transform1.setTranslate( Math.round(svgw/2)-Math.round(w/2) , transform1.matrix.f );
		transform2.setTranslate( Math.round(svgw/2)-Math.round(w/2) , transform2.matrix.f );

		return true;
	};

	// this function draws a slot (an object), given a shape and a color
	// @row and @col define the location of the slot (integer >= 0)
	// @props must be an object which defines the properties for the new slot {'shape': 'whatever1', 'color' : 'whatever2'}
	// return 1 on success, -1 if nothing changes, -2 on error
	function drawSlot (row, col, props)
	{
		if(isNaN(row) || isNaN(col) || !props)
			return -2;

		if(!_me.slots[row])
			_me.slots[row] = [];

		if(!_me.slots[row][col])
			_me.slots[row][col] = {};

		// check if there are new properties for this slot
		var i, found_new = false;
		for(i in props)
		{
			if(_me.slots[row][col][i] != props[i])
			{
				found_new = true;
				_me.slots[row][col][i] = props[i];
			} 
		}

		// if nothing new, no need to update the slot
		if(!found_new)
			return -1;

		// if there's a shape in the slot, remove it
		if( _me.slots[row][col].elem)
		{
			var pNode = _me.slots[row][col].elem.parentNode;
			pNode.removeChild(_me.slots[row][col].elem);
		}

		// the IDs of the properties
		var shape = _me.slots[row][col]['shape'],
			color = _me.slots[row][col]['color'];

		// the slot cannot be (re)drawn if the shape and color properties are missing
		if(isNaN(shape) || isNaN(color))
			return -2;

		// the values of the properties
		shape = _me.props.shape[shape];
		color = _me.props.color[color];

		var x = _me.slotw + _me.slotpw,
			y = _me.sloth + _me.slotph,
			strokes = {'red' : '#c10000', 'yellow' : '#d2cb00', 'blue' : '#050081'};

		// add the SVG element
		var elem = doc.createElementNS(ns_svg, 'svg:use');
		elem.setAttributeNS(ns_xlink, 'xlink:href', '#shape-' + shape );
		elem.setAttributeNS( null, 'fill', 'url(#fill-' + color + ')' );
		elem.setAttributeNS( null, 'stroke', strokes[color]);
		elem.setAttributeNS( null, 'x', col * x );
		elem.setAttributeNS( null, 'y', row * y );
		elem.setAttributeNS( null, 'width', _me.slotw );
		elem.setAttributeNS( null, 'height', _me.sloth );

		_me.slots[row][col].elem = _me.svgc.appendChild(elem);

		return 1;
	};

	// this is used to get all the messages in the game
	// @id - message ID (must be in _me.messages)
	// @vars - an object {'param' : 'value', ...}. This pair is used for substitution in the message, so variables can be used within the messages.
	function getMsg (id, vars)
	{
		if(!id || !_me.messages || !_me.messages[id])
			return _me.messages['internal-error'] || 'Internal error';

		var msg = _me.messages[id], re, i;

		for(i in vars)
		{
			re = new RegExp('%' + i + '%', 'g');
			msg = msg.replace(re, vars[i]);
		}

		return msg;
	};

	// this function removes child nodes based on simple rules
	// @elem is the DOM element node from which the function does remove the child nodes
	// @skip tells the function to skip N element nodes before starting to remove nodes (counting only nodes of type 1)
	function remChilds (elem, skip, clean)
	{
		if(!elem || !elem.childNodes)
			return false;

		var nodes = [], child, i, y = 0, n = elem.childNodes.length;

		if(!skip)
			var skip = 0;

		for(i = 0; i < n; i++)
		{
			child = elem.childNodes.item(i);

			if(child.nodeType == 1)
				y++;

			if(y > skip || child.nodeType != 1)
				nodes.push(child);
		}

		for(i in nodes)
		{
			elem.removeChild(nodes[i]);
		}

		return true;
	};

	// used by _me.makeMove() when comparing current move with previous moves
	function arrToStr (arr)
	{
		if(!arr || !(arr instanceof Array))
			return false;

		var i, n = arr.length, r = '';
		for(i=0; i<n; i++)
		{
			r += "\n" + i + ' : ';
			if(arr[i] instanceof Array)
				r += arrToStr(arr[i]);
			else
				r += arr[i];
		}

		return r;
	};

	// this uses the available HTML form inputs to control the game
	function formCommand ()
	{
		var src = {},
			dest = {}, i;

		for(i in _me.props)
		{
			if(!_me.formFields['sel_' + i] || !_me.formFields['dest_' + i])
			{
				alert(_me.messages['internal-error'] || 'Internal error');
				return false;
			}

			if(_me.formFields['sel_' + i].value != '-')
				src[i] = _me.formFields['sel_' + i].value;

			if(_me.formFields['dest_' + i].value != '-')
				dest[i] = _me.formFields['dest_' + i].value;
		};

		// execute the command
		var res = _me.makeMove(src, dest);

		// can't move because the game is finished (we have a winner!), maxMoves reached, or it's not his/her turn to move
		if(res == -1)
			return alert(getMsg('move-not-allowed'));
		else if(res == -2)
			return alert(getMsg('no-objects-found'));
		else if(res == -3)
			return alert(getMsg('previous-move', {'nr' : _me.disallow_nmoves}));
		else if(res == -4)
			return alert(getMsg('false-move'));
		else if(res < -4)
			return alert(_me.messages['internal-error'] || 'Internal error');

		for(i in _me.formFields)
		{
			_me.formFields[i].selectedIndex = 0;
		}

		if(res == 2)
			alert(getMsg('player-won', {'nr' : _me.winner+1, 'name' : _me.players[_me.winner]}));
		else if(res == 3)
			alert(getMsg('game-end-maxMoves', {'nr' : _me.maxMoves}));
		else if(res == 4)
			alert(getMsg('game-end-nochange', {'nr' : _me.maxMovesNoChange}));

		return true;
	};

	// ------------------
	// voice layer start

	// this is used to nicely generate dynamic prompts in the voice form
	_me.voiceMessage = function (type)
	{
		if(!type)
			return _me.messages['internal-error'] || 'Internal error';

		type = 'voice-' + type;

		var msg = '';
		if(type == 'voice-prompt')
			msg = getMsg(type, {'nr' : _me.move_by+1, 'name' : _me.players[_me.move_by]});
		else
			msg = getMsg(type);

		return msg;
	};

	// this takes the voice command and executes makeMove appropriately
	_me.voiceCommand = function (vres)
	{
		if(!vres || !vres.interpretation)
			return false;

		var src = {},
			dest = {},
			p, found_src = false, found_dest = false,

			// semantic interpretation
			si = vres.interpretation;

		// get the properties from the semantic interpretation object
		for(p in _me.props)
		{
			if((si['src_' + p] || si['src_' + p] == 0) && si['src_' + p] != -1)
			{
				src[p] = si['src_' + p];
				found_src = true;
			}

			if((si['dest_' + p] || si['dest_p' + p] == 0) && si['dest_' + p] != -1)
			{
				dest[p] = si['dest_' + p];
				found_dest = true;
			}
		}

		if(!found_src || !found_dest)
			return 'event-nomatch';

		var res = {'action' : 'prompt-value'};

		// execute the command
		var gres = _me.makeMove(src, dest);

		// can't move because the game is finished (we have a winner!), maxMoves reached, or it's not his/her turn to move
		if(gres == -1)
			res['message'] = getMsg('move-not-allowed');
		else if(gres == -2)
			res['message'] = getMsg('no-objects-found');
		else if(gres == -3)
			res['message'] = getMsg('previous-move', {'nr' : _me.disallow_nmoves});
		else if(gres == -4)
			res['message'] = getMsg('false-move');
		else if(gres < -4)
			res['message'] = _me.messages['internal-error'] || 'Internal error';
		else if(gres == 2)
			res['message'] = getMsg('player-won', {'nr' : _me.winner+1, 'name' : _me.players[_me.winner]});
		else if(gres == 3)
			res['message'] = getMsg('game-end-maxMoves', {'nr' : _me.maxMoves});
		else if(gres == 4)
			res['message'] = getMsg('game-end-nochange', {'nr' : _me.maxMovesNoChange});

		if(!res['message'])
			return 'clear-cmd';
		else
			return res;
	};

	// voice layer end
	// ------------------

	if(!no_autoinit)
		window.addEventListener('load', _me.init, false);

	return _me;
})();

