#!/usr/bin/python3
"""Convert Aeon Timeline 2 project data to Obsidian Markdown fileset. 

usage: aeon2obsidian.py Sourcefile

positional arguments:
  Sourcefile  The path of the .aeonzip file.

Copyright (c) 2024 Peter Triesberger
For further information see https://github.com/peter88213/aeon2obsidian
Published under the MIT License (https://opensource.org/licenses/mit-license.php)
"""
import os
import sys
import codecs
import json
import zipfile


def open_timeline(filePath):
    """Unzip the project file and read 'timeline.json'.

    Positional arguments:
        filePath -- Path of the .aeonzip project file to read.
        
    Return a Python object containing the timeline structure.
    Raise the "Error" exception in case of error. 
    """
    with zipfile.ZipFile(filePath, 'r') as myzip:
        jsonBytes = myzip.read('timeline.json')
        jsonStr = codecs.decode(jsonBytes, encoding='utf-8')
    if not jsonStr:
        raise ValueError('No JSON part found in timeline data.')
    jsonData = json.loads(jsonStr)
    return jsonData

from datetime import datetime
from datetime import timedelta


class Event:
    DATE_LIMIT = (datetime(1, 1, 1) - datetime.min).total_seconds()
    # Dates before 1-01-01 can not be displayed properly in novelibre

    def __init__(self):
        self.title: str = None
        # single line text

        self.relationships: dict[str, list[str]] = None
        # key: role ID, value: list of entity IDs

        self.values: dict[str, str] = None
        # key: property ID, value: text

        self.tags: list[str] = None

        self.date: str = None
        # ISO date string

        self.time: str = None
        # ISO time string

        self.lastsDays: int = None
        self.lastsHours: int = None
        self.lastsMinutes: int = None

    def read(self, jsonEvent: dict, tplDateGuid: str):
        """Read an event from Aeon 2 JSON."""
        self.read_title(jsonEvent)
        self.read_date(jsonEvent, tplDateGuid)
        self.read_relationships(jsonEvent)
        self.read_values(jsonEvent)
        self.read_tags(jsonEvent)

    def read_date(self, jsonEvent: dict, tplDateGuid: str):
        """Set date/time/duration from Aeon 2 JSON event dictionary."""
        timestamp = 0
        for evtRgv in jsonEvent['rangeValues']:
            if evtRgv['rangeProperty'] == tplDateGuid:
                timestamp = evtRgv['position']['timestamp']
                if timestamp >= self.DATE_LIMIT:
                    # Restrict date/time calculation to dates within novelibre's range
                    eventStart = datetime.min + timedelta(seconds=timestamp)
                    startDateTime = eventStart.isoformat().split('T')
                    self.date, self.time = startDateTime

                    # Calculate duration
                    if 'years' in evtRgv['span'] or 'months' in evtRgv['span']:
                        endYear = eventStart.year
                        endMonth = eventStart.month
                        if 'years' in evtRgv['span']:
                            endYear += evtRgv['span']['years']
                        if 'months' in evtRgv['span']:
                            endMonth += evtRgv['span']['months']
                            while endMonth > 12:
                                endMonth -= 12
                                endYear += 1
                        eventEnd = datetime(endYear, endMonth, eventStart.day)
                        eventDuration = eventEnd - datetime(eventStart.year, eventStart.month, eventStart.day)
                        lastsDays = eventDuration.days
                        lastsHours = eventDuration.seconds // 3600
                        lastsMinutes = (eventDuration.seconds % 3600) // 60
                    else:
                        lastsDays = 0
                        lastsHours = 0
                        lastsMinutes = 0
                    if 'weeks' in evtRgv['span']:
                        lastsDays += evtRgv['span']['weeks'] * 7
                    if 'days' in evtRgv['span']:
                        lastsDays += evtRgv['span']['days']
                    if 'hours' in evtRgv['span']:
                        lastsDays += evtRgv['span']['hours'] // 24
                        lastsHours += evtRgv['span']['hours'] % 24
                    if 'minutes' in evtRgv['span']:
                        lastsHours += evtRgv['span']['minutes'] // 60
                        lastsMinutes += evtRgv['span']['minutes'] % 60
                    if 'seconds' in evtRgv['span']:
                        lastsMinutes += evtRgv['span']['seconds'] // 60
                    lastsHours += lastsMinutes // 60
                    lastsMinutes %= 60
                    lastsDays += lastsHours // 24
                    lastsHours %= 24
                    self.lastsDays = lastsDays
                    self.lastsHours = lastsHours
                    self.lastsMinutes = lastsMinutes
                break

    def read_relationships(self, jsonEvent: dict):
        """Set relationships from Aeon 2 JSON event list."""
        self.relationships = {}
        for relationship in jsonEvent['relationships']:
            role = relationship.get('role', None)
            if role:
                if not role in self.relationships:
                    self.relationships[role] = []
                entity = relationship.get('entity', None)
                if entity:
                    self.relationships[role].append(entity)

    def read_tags(self, jsonEvent: dict):
        """Set tags from Aeon 2 JSON event list."""
        self.tags = []
        for tag in jsonEvent['tags']:
            self.tags.append(tag.strip())

    def read_title(self, jsonEvent: dict):
        """Set title from Aeon 2 JSON event."""
        self.title = jsonEvent['title'].strip()

    def read_values(self, jsonEvent: dict):
        """Set values from Aeon 2 JSON event list."""
        self.values = {}
        for eventValue in jsonEvent['values']:
            eventProperty = eventValue.get('property', None)
            if eventProperty:
                val = eventValue.get('value', None)
                if val:
                    self.values[eventProperty] = val.strip()



class Entity:

    def __init__(self):
        self.name: str = None
        # single line text

        self.entityType: str = None
        # type ID

        self.notes: str = None
        # multiline text

    def read(self, jsonEntity: dict):
        """Read a property from Aeon 2 JSON."""
        self.name = jsonEntity['name']
        self.entityType = jsonEntity['entityType']
        self.notes = jsonEntity['notes']



class Timeline:

    def __init__(self):

        # Strategies:
        self.entityClass = Entity
        self.eventClass = Event

        self.types: dict[str, str] = {}
        # key: ID, value: name

        self.roles: dict[str, str] = {}
        # key: ID, value: name

        self.properties: dict[str, str] = {}
        # key: ID, value: name

        self.entities: dict[str, Entity] = {}
        # key: ID, value: entityClass instance

        self.events: dict[str, Event] = {}
        # key: ID, value: eventClass instance

        self.entitiesByType: dict[str, list[str]] = {}
        # key: type ID, value: list of entity IDs


class Aeon2File:

    def __init__(self, filePath: str):
        """Set the Aeon 2 project file path."""
        self.filePath = filePath
        self.timeline: Timeline = None

    def read(self) -> str:
        """Read the Aeon 2 project file.
        
        Return a success message.
        """

        #--- Read the aeon file and get a JSON data structure.
        jsonData = open_timeline(self.filePath)

        #--- Get the date definition.
        for tplRgp in jsonData['template']['rangeProperties']:
            if tplRgp['type'] == 'date':
                for tplRgpCalEra in tplRgp['calendar']['eras']:
                    if tplRgpCalEra['name'] == 'AD':
                        tplDateGuid = tplRgp['guid']
                        break

        #--- Read type and role names.
        for jsonType in jsonData['template']['types']:
            uid = jsonType['guid']
            name = jsonType['name']
            self.timeline.types[uid] = name
            self.timeline.entitiesByType[uid] = []
            for jsonRole in jsonType['roles']:
                uid = jsonRole['guid']
                name = jsonRole['name']
                self.timeline.roles[uid] = name

        #--- Read property names.
        for jsonProperty in jsonData['template']['properties']:
            uid = jsonProperty['guid']
            name = jsonProperty['name']
            self.timeline.properties[uid] = name

        #--- Read entities.
        for jsonEntity in jsonData['entities']:
            uid = jsonEntity['guid']
            self.timeline.entities[uid] = self.timeline.entityClass()
            self.timeline.entities[uid].read(jsonEntity)
            self.timeline.entitiesByType[jsonEntity['entityType']].append(uid)

        #--- Read events.
        eventTitles = {}
        for jsonEvent in jsonData['events']:
            uid = jsonEvent['guid']
            self.timeline.events[uid] = self.timeline.eventClass()
            self.timeline.events[uid].read(jsonEvent, tplDateGuid)
            title = self.timeline.events[uid].title
            if title in eventTitles:
                print(f'Multiple event title: {title}')
                number = eventTitles[title] + 1
                eventTitles[title] = number
                self.timeline.events[uid].title = f'{title}({number})'
            else:
                eventTitles[title] = 0

        return 'Aeon 2 file successfully read.'
from re._compiler import isstring


class ObsidianFiles:
    FORBIDDEN_CHARACTERS = ('\\', '/', ':', '*', '?', '"', '<', '>', '|')
    # set of characters that filenames cannot contain

    def __init__(self, folderPath:str):
        """Set the Obsidian folder."""
        self.folderPath = folderPath
        self.timeline = None

    def write(self) -> str:
        """Create a set of Markdown files in the Obsidian folder.
        
        Return a success message.
        """
        os.makedirs(self.folderPath, exist_ok=True)
        self._build_index()

        for uid in self.timeline.entities:
            entity = self.timeline.entities[uid]
            name = self._strip_title(entity.name)
            text = self._to_markdown(entity.notes)
            self._write_file(f'{self.folderPath}/{name}.md', text)

        for uid in self.timeline.events:
            event = self.timeline.events[uid]
            title = self._strip_title(event.title)
            text = self._build_content(event)
            self._write_file(f'{self.folderPath}/{title}.md', text)

        return 'Obsidian files successfully written.'

    def _build_content(self, event:Event) -> str:
        """Return a string with the Markdown file content.
        
        Positional arguments:
            event: Event instance.
        """
        lines = []

        #--- Event properties.
        for propertyId in event.values:
            propertyName = self.timeline.properties[propertyId]
            lines.append(f'### {propertyName}')
            eventValue = event.values[propertyId]
            if eventValue and type(eventValue) == str:
                lines.append(self._to_markdown(eventValue))

        #--- Links to entities.
        for roleId in event.relationships:
            roleName = self.timeline.roles[roleId]
            for entityId in event.relationships[roleId]:
                entityName = self.timeline.entities[entityId].name
                link = self._strip_title(entityName)
                lines.append(f'- {roleName}: [[{link}]]')

        #--- Tags.
        for tag in event.tags:
            lines.append(f"#{tag.replace(' ', '_')}")

        #--- Date and time.
        lines.append(event.date)
        lines.append(event.time)

        #--- Duration.
        durationList = []
        if event.lastsDays:
            durationList.append(f'{event.lastsDays} days')
        if event.lastsHours:
            durationList.append(f'{event.lastsHours} hours')
        if event.lastsMinutes:
            durationList.append(f'{event.lastsMinutes} minutes')
        if durationList:
            durationStr = ', '.join(durationList)
            lines.append(durationStr)
        return '\n\n'.join(lines)

    def _build_index(self):
        """Create index pages."""
        mainIndexlines = []

        #--- Create an index file with the events.
        mainIndexlines.append(f'- [[__events]]')
        lines = []
        for uid in self.timeline.events:
            eventTitle = self.timeline.events[uid].title
            lines.append(f'- [[{self._strip_title(eventTitle)}]]')
        text = '\n'.join(lines)
        self._write_file(f'{self.folderPath}/__events.md', text)

        for typeUid in self.timeline.entitiesByType:
            entityType = f'_{self._strip_title(self.timeline.types[typeUid])}'
            mainIndexlines.append(f'- [[{entityType}]]')
            entityUidList = self.timeline.entitiesByType[typeUid]

            #--- Create an index file with the entities of the type.
            lines = []
            for entityUid in entityUidList:
                entityName = self.timeline.entities[entityUid].name
                lines.append(f'- [[{self._strip_title(entityName)}]]')
            text = '\n'.join(lines)
            self._write_file(f'{self.folderPath}/{entityType}.md', text)

        #--- Create a main index file with the types and event link.
        text = '\n'.join(mainIndexlines)
        self._write_file(f'{self.folderPath}/__index.md', text)

    def _strip_title(self, title: str) -> str:
        """Return title with characters removed that must not appear in a file name."""
        for c in self.FORBIDDEN_CHARACTERS:
            title = title.replace(c, '')
        return title

    def _to_markdown(self, text: str) -> str:
        """Return text with double linebreaks."""
        return text.replace('\n', '\n\n')

    def _write_file(self, filePath: str, text: str) -> str:
        """Write a single file and create a backup copy, if applicable.
        
        Positional arguments:
            filePath: str -- Path of the file to write.
            text: str -- File content.
            
        Return a success message.
        """
        backedUp = False
        if os.path.isfile(filePath):
            try:
                os.replace(filePath, f'{filePath}.bak')
                backedUp = True
            except Exception as ex:
                raise Exception(f'Error: Cannot overwrite "{os.path.normpath(filePath)}": {str(ex)}.')

        try:
            with open(filePath, 'w', encoding='utf-8') as f:
                f.write(text)
        except Exception as ex:
            if backedUp:
                os.replace(f'{filePath}.bak', self.filePath)
            raise Exception(f'Error: Cannot write "{os.path.normpath(filePath)}": {str(ex)}.')

        return f'"{os.path.normpath(filePath)}" written.'



def main(sourcePath):
    """Convert an .aeonzip source file to a set of Markdown files.
    
    Positional arguments:
        sourcePath -- str: The path of the .aeonzip file.
    """

    # Create an Aeon 2 file object and read the data.
    aeon2File = Aeon2File(sourcePath)
    aeon2File.timeline = Timeline()
    print(aeon2File.read())

    # Define the output directory.
    aeonDir, aeonFilename = os.path.split(sourcePath)
    projectName = os.path.splitext(aeonFilename)[0]
    obsidianFolder = os.path.join(aeonDir, projectName)

    # Create an Obsidian fileset object and write the data.
    obsidianFiles = ObsidianFiles(obsidianFolder)
    obsidianFiles.timeline = aeon2File.timeline
    print(obsidianFiles.write())


if __name__ == '__main__':
    main(sys.argv[1])
