#!/usr/bin/python3
"""Convert Aeon Aeon3Data 3 project data to Obsidian Markdown fileset. 

Version 2.2.0
Requires Python 3.6+

usage: aeon3obsidian.py Sourcefile

positional arguments:
  Sourcefile  The path of the .aeon file.

Copyright (c) 2025 Peter Triesberger
For further information see https://github.com/peter88213/aeon3obsidian
License: GNU GPLv3 (https://www.gnu.org/licenses/gpl-3.0.en.html)

This program is free software: you can redistribute it and/or modify 
it under the terms of the GNU General Public License as published by 
the Free Software Foundation, either version 3 of the License, or 
(at your option) any later version.

This program is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of 
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
GNU General Public License for more details.
"""
import os
import sys



class Aeon3Data:

    def __init__(self):
        self.items = {}
        # item instances by item UIDs
        self.itemTypes = {}
        # item type instances by type UIDs
        self.itemIndex = {}
        # item UIDs by type labels
        self.narrative = {}
        # tree of item UIDs

    def sort_items_by_date(self, itemList):
        """Return a list of item UIDs, sorted by date including the era.
        
        Positional arguments:
            itemList -- List of item UIDs.
        
        UIDs of items with the same date are sorted by uniqueLabel.
        UIDs of undated items are placed last, sorted by uniqueLabel.
        Invalid UIDs are discarded.
        """
        datedItems = []
        undatedItems = []
        for uid in itemList:
            try:
                if self.items[uid].timestamp is not None:
                    datedItems.append(uid)
                else:
                    undatedItems.append(uid)
            except KeyError:
                continue

        datedItems.sort(key=lambda e: (
            self.items[e].era,
            self.items[e].timestamp,
            self.items[e].uniqueLabel
            ))
        undatedItems.sort(key=lambda e: (
            self.items[e].uniqueLabel
            ))
        return datedItems + undatedItems

import codecs
import json



class Aeon3Calendar:

    ISO_ERAS = ('AD')

    def __init__(self, calendarDefinitions):

        #--- Era enumerations.
        self.eraShortNames = []
        self.eraNames = []
        for era in calendarDefinitions['eras']:
            self.eraShortNames.append(era['shortName'])
            self.eraNames.append(era['name'])

        #--- Month enumerations.
        self.monthShortNames = []
        self.monthNames = []
        for month in calendarDefinitions['months']:
            self.monthShortNames.append(month['shortName'])
            self.monthNames.append(month['name'])

        #--- Weekday enumerations.
        self.weekdayShortNames = []
        self.weekdayNames = []
        for weekday in calendarDefinitions['weekdays']:
            self.weekdayShortNames.append(weekday['shortName'])
            self.weekdayNames.append(weekday['name'])

    def get_day(self, itemDates):
        """Return an integer day or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not None:
            return startDate.get('day', None)

    def get_duration_str(self, itemDates):
        """Return a string with comma-separated elements of the duration."""
        durationList = []
        durationDict = itemDates.get('duration', None)
        if durationDict:
            durations = list(durationDict)
            for unit in durations:
                if durationDict[unit]:
                    durationList.append(f'{durationDict[unit]} {unit}')
        durationStr = ', '.join(durationList)
        return durationStr

    def get_era(self, itemDates):
        """Return a tuple: (era as an integer, era's short name, era's name)."""
        startDate = itemDates.get('startDate', None)
        if startDate is None:
            return

        era = startDate.get('era', None)
        if era is None:
            return

        try:
            return era, self.eraShortNames[era], self.eraNames[era]
        except:
            return

    def get_hour(self, itemDates):
        """Return an integer hour or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not  None:
            return  startDate.get('hour', None)

    def get_iso_date(self, itemDates):
        """Return a date string formatted acc. to ISO 8601, if applicable. 
        
        Return None, if the date isn't within the range specified by ISO 8601, 
        or in case of error.
        """
        try:
            startDate = itemDates['startDate']
            era = startDate['era']
            eraName = self.eraNames[era]
            if eraName not in self.ISO_ERAS:
                return

            year = startDate['year']
            month = startDate['month']
            day = startDate['day']
        except:
            return

        return f'{year:04}-{month:02}-{day:02}'

    def get_iso_time(self, itemDates):
        """Return a time string formatted acc. to ISO 8601. 
        
        Return None in case of error.
        """
        try:
            startDate = itemDates['startDate']
            hour = startDate['hour']
            minute = startDate['minute']
            second = startDate['second']
        except:
            return

        return f'{hour:02}:{minute:02}:{second:02}'

    def get_minute(self, itemDates):
        """Return an integer minute or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not None:
            return startDate.get('minute', None)

    def get_month(self, itemDates):
        """Return a tuple: (month's order, month's short name, month's name)."""
        startDate = itemDates.get('startDate', None)
        if startDate is None:
            return

        month = startDate.get('month', None)
        if month is None:
            return

        try:
            return month, self.monthShortNames[month - 1], self.monthNames[month - 1]
        except:
            return

    def get_second(self, itemDates):
        """Return an integer second or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not None:
            return startDate.get('second', None)

    def get_timestamp(self, itemDates):
        """Return an integer timestamp or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not None:
            return startDate.get('timestamp', None)

    def get_weekday(self, itemDates):
        """Return a tuple: (weekday as an integer, weekday's short name, weekday's name)."""
        startDate = itemDates.get('startDate', None)
        if startDate is None:
            return

        weekday = startDate.get('weekday', None)
        if weekday is None:
            return

        try:
            return weekday, self.weekdayShortNames[weekday], self.weekdayNames[weekday]
        except:
            return

    def get_year(self, itemDates):
        """Return an integer year or None."""
        startDate = itemDates.get('startDate', None)
        if startDate is not None:
            return startDate.get('year', None)



class Aeon3Item:

    def __init__(
            self,
            uniqueLabel,
            displayId,
            typeUid,
            label=None,
            shortLabel=None,
            summary=None,
            properties=[],
            tags=None,
            timestamp=None,
            isoDate=None,
            isoTime=None,
            era=None,
            weekday=None,
            month=None,
            year=None,
            day=None,
            hour=None,
            minute=None,
            second=None,
            durationStr=None,
            relationships=[],
            children=[],
            ):

        self.uniqueLabel = uniqueLabel
        self.displayId = displayId
        self.typeUid = typeUid
        self.label = label
        self.shortLabel = shortLabel
        self.summary = summary
        self.properties = properties
        self.tags = tags
        self.timestamp = timestamp
        self.isoDate = isoDate
        self.isoTime = isoTime
        if era:
            self.era, self.eraShortName, self.eraName = era
        else:
            self.era = self.eraShortName = self.eraName = None
        if weekday:
            self.weekday, self.weekdayShortName, self.weekdayName = weekday
        else:
            self.weekday = self.weekdayShortName = self.weekdayName = None
        if month:
            self.month, self.monthShortName, self.monthName = month
        else:
            self.month = self.monthShortName = self.monthName = None
        self.year = year
        self.day = day
        self.hour = hour
        self.minute = minute
        self.second = second
        self.duration = durationStr
        self.relationships = relationships
        self.children = children



class Aeon3Type:

    def __init__(self, label, isNarrativeFolder):
        self.label = label
        self.isNarrativeFolder = isNarrativeFolder


class Aeon3File:

    def __init__(self, filePath):
        self.filePath = filePath
        self.data = None
        self._labelCounts = {}

    def read(self):
        """Read the Aeon 3 project file.
        
        Store the relevant data in the data model.
        Return a success message.
        """

        #--- Read the aeon file and get a JSON data structure.
        print(f'Reading file "{os.path.normpath(self.filePath)}" ...')
        jsonStr = self._get_json_string()
        jsonData = json.loads(jsonStr)

        print(f'Found file version: "{jsonData.get("fileVersion", "Unknown")}".')

        #--- Create lookup dictionaries (labels by UID).
        itemLabelLookup, uniqueItemLabelLookup = self._get_item_label_lookup(jsonData)
        tagLookup = self._get_tag_lookup(jsonData)
        relationshipTypeLookup = self._get_relationship_type_lookup(jsonData)
        propertyTypeLookup = self._get_property_type_lookup(jsonData)
        propertyEnumLookup = self._get_property_enum_lookup(jsonData)

        #--- Set the item types of the data model.
        self.data.itemTypes = self._get_item_types(jsonData)

        #--- Build the items of the data model.
        calendar = Aeon3Calendar(jsonData['core']['definitions']['calendar'])

        for itemUid in uniqueItemLabelLookup:
            label = itemLabelLookup[itemUid]
            uniqueLabel = uniqueItemLabelLookup[itemUid]

            # Get properties.
            jsonItem = jsonData['core']['data']['itemsById'][itemUid]
            displayId = jsonItem.get('displayId', None)
            typeUid = jsonItem.get('type', None)
            shortLabel = jsonItem.get('shortLabel', None)
            summary = jsonItem.get('summary', None)
            tags = []
            for tagUid in jsonItem['tags']:
                tags.append(tagLookup[tagUid])
            properties = []
            for propertyTypeUid in jsonItem['propertyValues']:
                propertyValues = jsonItem['propertyValues'][propertyTypeUid]
                customProperty = (
                    propertyTypeLookup[propertyTypeUid],
                    propertyEnumLookup.get(propertyValues, propertyValues)
                )
                properties.append(customProperty)

            # Get relationships.
            relationships = []
            jsonRelationshipDict = jsonData['collection']['relationshipIdsByItemId'][itemUid]
            for relUid in jsonRelationshipDict:
                if not jsonRelationshipDict[relUid]:
                    # target might be deleted
                    continue

                relationship = jsonData['core']['data']['relationshipsById'][relUid]
                itemRelationship = (
                    uniqueItemLabelLookup[relationship['object']],
                    relationshipTypeLookup[relationship['reference']]
                )
                relationships.append(itemRelationship)

            # Get children.
            children = []
            jsonChildrenDict = jsonData['core']['data']['superAndChildOrderById'][itemUid]
            for childUid in jsonChildrenDict['childOrder']:
                if not childUid in uniqueItemLabelLookup:
                    # child might be deleted
                    continue

                children.append(uniqueItemLabelLookup[childUid])

            # Get date/time/duration.
            itemDates = jsonData['core']['data']['itemDatesById'][itemUid]
            era = calendar.get_era(itemDates)
            weekday = calendar.get_weekday(itemDates)
            month = calendar.get_month(itemDates)
            year = calendar.get_year(itemDates)
            day = calendar.get_day(itemDates)
            hour = calendar.get_hour(itemDates)
            minute = calendar.get_minute(itemDates)
            second = calendar.get_second(itemDates)
            timestamp = calendar.get_timestamp(itemDates)
            isoDate = calendar.get_iso_date(itemDates)
            isoTime = calendar.get_iso_time(itemDates)

            # Instantiate the item object.
            self.data.items[itemUid] = Aeon3Item(
                uniqueLabel,
                displayId,
                typeUid,
                label=label,
                shortLabel=shortLabel,
                summary=summary,
                properties=properties,
                tags=tags,
                timestamp=timestamp,
                isoDate=isoDate,
                isoTime=isoTime,
                era=era,
                weekday=weekday,
                month=month,
                year=year,
                day=day,
                hour=hour,
                minute=minute,
                second=second,
                relationships=relationships,
                children=children,
                )

        #--- Create an item index.
        itemIndex = {}
        jsonItemIndex = jsonData['collection']['itemIdsByType']
        for typeUid in jsonItemIndex:
            itemIndex[typeUid] = []
            for itemUid in jsonItemIndex[typeUid]:
                itemIndex[typeUid].append(itemUid)
        self.data.itemIndex = itemIndex

        #--- Get the narrative tree.
        self.data.narrative = self._get_narrative_tree(jsonData)
        return 'Aeon 3 file successfully read.'

    def _get_item_label_lookup(self, jsonData):
        # Return a lookup dictionary with unique item labels by UID.
        uniqueItemLabelLookup = {}
        itemLabelLookup = {}
        jsonItems = jsonData['core']['data']['itemsById']
        for itemUid in jsonItems:
            if not jsonData['collection']['allItemIds'][itemUid]:
                # item might be deleted
                continue

            jsonItem = jsonItems[itemUid]
            aeonLabel = jsonItem.get('label', None)
            if aeonLabel is not None:
                itemLabelLookup[itemUid] = aeonLabel
                uniqueLabel = self._get_unique_label(aeonLabel.strip())
                uniqueItemLabelLookup[itemUid] = uniqueLabel
        return itemLabelLookup, uniqueItemLabelLookup

    def _get_item_types(self, jsonData):
        # Return a dictionary with item types by UID.
        itemTypes = {}
        jsonTypes = jsonData['core']['definitions']['types']['byId']
        for typeUid in jsonTypes:
            label = jsonTypes[typeUid].get('label', '').strip()
            isNarrativeFolder = jsonTypes[typeUid].get('isNarrativeFolder', None)
            itemTypes[typeUid] = Aeon3Type(label, isNarrativeFolder)
        return itemTypes

    def _get_json_string(self):
        # Return a string containing the JSON part of the aeon file.
        with open(self.filePath, 'rb') as f:
            binInput = f.read()

        # JSON part: all characters between the first and last curly bracket.
        chrData = []
        opening = ord('{')
        closing = ord('}')
        level = 0
        for c in binInput:
            if c == opening:
                level += 1
            if level > 0:
                chrData.append(c)
                if c == closing:
                    level -= 1
                    if level == 0:
                        break
        if level != 0:
            raise ValueError('Error: Corrupted data.')

        jsonStr = codecs.decode(bytes(chrData), encoding='utf-8')
        if not jsonStr:
            raise ValueError('Error: No JSON part found.')

        return jsonStr

    def _get_narrative_tree(self, jsonData):
        # Return a tree of narrative item UIDs.

        def get_branch(uid, branch):
            for child in jsonNarrative[uid]['children']:
                branch[child] = {}
                get_branch(child, branch[child])

        narrativeTree = {}
        jsonNarrative = jsonData['collection'].get('narrativeSacById', None)
        if jsonNarrative is None:
            return narrativeTree

        for folderUid in jsonNarrative:
            if jsonNarrative[folderUid]['super'] is None:
                break

        narrativeTree[folderUid] = {}
        get_branch(folderUid, narrativeTree[folderUid])
        return narrativeTree

    def _get_property_enum_lookup(self, jsonData):
        # Return a lookup dictionary with allowed property enum labels by UID.
        propertyEnumLookup = {}
        propertyTypes = jsonData['core']['definitions']['properties']['byId']
        for propertyTypeUid in propertyTypes:
            propertyType = propertyTypes[propertyTypeUid]
            for enumUid in propertyType['allowed']:
                enumLabel = propertyType['allowed'][enumUid]['label']
                propertyEnumLookup[enumUid] = enumLabel
        return propertyEnumLookup

    def _get_property_type_lookup(self, jsonData):
        # Return a lookup dictionary with property type labels by UID.
        propertyTypeLookup = {}
        propertyTypes = jsonData['core']['definitions']['properties']['byId']
        for propertyTypeUid in propertyTypes:
            propertyType = propertyTypes[propertyTypeUid]
            propertyTypeLabel = propertyType['label']
            propertyTypeLookup[propertyTypeUid] = propertyTypeLabel.strip()
        return propertyTypeLookup

    def _get_relationship_type_lookup(self, jsonData):
        # Return a lookup dictionary with relationship type labels by UID.
        relationshipTypeLookup = {}
        relationshipTypes = jsonData['core']['definitions']['references']['byId']
        for relationshipTypeUid in relationshipTypes:
            relationshipType = relationshipTypes[relationshipTypeUid]['label']
            relationshipTypeLookup[relationshipTypeUid] = relationshipType.strip()
        return relationshipTypeLookup

    def _get_tag_lookup(self, jsonData):
        # Return a lookup dictionary with tags by UID.
        tagLookup = {}
        tags = jsonData['core']['data']['tags']
        for tagUid in tags:
            tagName = tags[tagUid]
            tagLookup[tagUid] = tagName.strip()
        return tagLookup

    def _get_unique_label(self, aeonLabel):
        # Return a unique item label.
        if aeonLabel in self._labelCounts:
            print(f'Multiple aeonLabel: {aeonLabel}')
            counts = self._labelCounts[aeonLabel] + 1
            self._labelCounts[aeonLabel] = counts
            aeonLabel = f'{aeonLabel}({counts})'
        else:
            self._labelCounts[aeonLabel] = 0
        return aeonLabel
import re


class ObsidianFiles:

    HIDDEN_ERAS = ('AD')

    def __init__(self, folderPath):
        """Set the Obsidian folder."""
        self.folderPath = folderPath
        self.data = None

    def write(self):
        """Create a set of Markdown files in the Obsidian folder.
        
        Return a success message.
        """
        os.makedirs(self.folderPath, exist_ok=True)

        #--- Create one file per item.
        for uid in self.data.items:
            item = self.data.items[uid]
            title = self._sanitize_title(item.uniqueLabel)
            text = self._get_item_page_yaml(item)
            text = f'{text}{self._get_item_page_markdown(item)}'
            self._write_file(f'{self.folderPath}/{title}.md', text)

        self._create_index_page()
        self._create_narrative_page()
        return 'Obsidian files successfully written.'

    def _create_index_page(self):
        # Write a file containing of item links grouped by item types.
        mainIndexlines = ['\n']
        for typeUid in self.data.itemIndex:
            typeLabel = self.data.itemTypes[typeUid].label
            mainIndexlines.append(f'- [[_{typeLabel}]]')
            lines = []
            sortedItems = self.data.sort_items_by_date(self.data.itemIndex[typeUid])
            for itemUid in sortedItems:
                lines.append(f'- [[{self.data.items[itemUid].uniqueLabel}]]')
            text = '\n'.join(lines)
            self._write_file(f'{self.folderPath}/_{typeLabel}.md', text)
        text = '\n'.join(mainIndexlines)
        self._write_file(f'{self.folderPath}/__Index.md', text)

    def _create_narrative_page(self):

        def get_branch(branch, level):
            level += 1
            for uid in branch:
                if uid in self.data.items:
                    link = self._sanitize_title(self.data.items[uid].uniqueLabel)
                    typeUid = self.data.items[uid].typeUid
                    if self.data.itemTypes[typeUid].isNarrativeFolder:
                        linkPrefix = f"{'#' * level} "
                    else:
                        linkPrefix = ''
                    lines.append(f"{linkPrefix}[[{link}]]")
                get_branch(branch[uid], level)

        lines = []
        get_branch(self.data.narrative, 1)
        text = '\n\n'.join(lines)
        self._write_file(f'{self.folderPath}/__Narrative.md', text)

    def _get_date_str(self, item):
        # Return a formatted string with the date to display.
        dateStr = ''
        if item.weekdayName is not None:
            dateStr = f'{item.weekdayName}'
        if item.day is not None:
            dateStr = f'{dateStr} {item.day}'
        if item.monthName is not None:
            dateStr = f'{dateStr} {item.monthName}'
        if item.year is not None:
            dateStr = f'{dateStr} {item.year}'
        if item.eraShortName is not None:
            if item.eraShortName not in self.HIDDEN_ERAS:
                dateStr = f'{dateStr} {item.eraShortName}'
        return dateStr

    def _get_item_page_markdown(self, item):
        # Return the Markdown part of an item page as a single string.
        lines = ['\n']

        #--- Summary between rulers.
        if item.summary:
            lines.append('---')
            lines.append(self._to_markdown(item.summary))
            lines.append('---')

        #--- Date and time in a row.
        dateTimeStr = ''
        dateStr = self._get_date_str(item)
        if dateStr:
            dateTimeStr = f'- **When** : {dateStr} {self._get_time_str(item)}\n'

        #--- Duration.
        if item.duration:
            dateTimeStr = f'{dateTimeStr}- **Lasts** : {item.duration}'
        lines.append(dateTimeStr)

        #--- List of properties.
        if item.properties:
            propertyStr = ''
            for reference , customProperty in item.properties:
                propertyStr = f'{propertyStr}- **{reference}** : {self._to_markdown_list_element(customProperty)}\n'
            lines.append(propertyStr)

        #--- List of relationships.
        if item.relationships:
            relationshipStr = ''
            for target, reference in item.relationships:
                relationshipStr = f'{relationshipStr}- **{reference}** : [[{self._sanitize_title(target)}]]\n'
            lines.append(relationshipStr)

        #--- List of children.
        if item.children:
            childrenStr = ''
            for child in item.children:
                childrenStr = f'{childrenStr}- [[{self._sanitize_title(child)}]]\n'
            lines.append(childrenStr)

        return '\n\n'.join(lines)

    def _get_item_page_yaml(self, item):
        # Return the YAML part of an item page as a single string.
        obsidianProperties = {}

        #--- Label.
        if item.label:
            obsidianProperties['label'] = f'{item.label}'

        #--- Short label as Alias.
        if item.shortLabel:
            obsidianProperties['aliases'] = f'\n  - {item.shortLabel}'

        #--- Type.
        obsidianProperties['type'] = f'{self.data.itemTypes[item.typeUid].label}'

        #--- Display ID.
        if item.displayId:
            obsidianProperties['ID'] = f'{item.displayId}'

        #--- Date and time in ISO format ("AD" era only).
        if item.isoDate:
            obsidianProperties['date'] = item.isoDate
            if item.isoTime:
                obsidianProperties['time'] = f'{item.isoDate}T{item.isoTime}'

        #--- List of tags.
        if item.tags:
            tags = []
            for tag in item.tags:
                tags.append(f'  - {self._sanitize_tag(tag)}')
            tagStr = '\n'.join(tags)
            obsidianProperties['tags'] = f'\n{tagStr}'

        if not obsidianProperties:
            return ''

        yamlLines = ['---']
        for propertyLabel in obsidianProperties:
            yamlLines.append(f'{propertyLabel}: {obsidianProperties[propertyLabel]}')
        yamlLines.append('---')
        return '\n'.join(yamlLines)

    def _get_time_str(self, item):
        # Return a formatted string with the time to display.
        timeStr = ''
        if item.hour is not None:
            timeStr = f'{item.hour}'
        if item.minute is not None:
            timeStr = f'{timeStr}:{item.minute:02}'
        if item.second:
            timeStr = f'{timeStr}:{item.second:02}'
        return timeStr

    def _sanitize_tag(self, tag):
        # Return tag with non-alphanumeric characters replaced.
        return re.sub(r'\W+', '_', tag)

    def _sanitize_title(self, title):
        # Return title with disallowed characters removed.
        return re.sub(r'[\\|\/|\:|\*|\?|\"|\<|\>|\|]+', '', title)

    def _to_markdown(self, text):
        # Return a string with double linebreaks.
        while '\n\n' in text:
            text = text.replace('\n\n', '\n')
        return text.replace('\n', '\n\n')

    def _to_markdown_list_element(self, text):
        # Return a string with single linebreaks and indented lines.
        while '\n\n' in text:
            text = text.replace('\n\n', '\n')
        return text.replace('\n', '\n  ')

    def _write_file(self, filePath, text):
        # Write text to a single file specified by filePath.
        # Create a backup copy, if applicable.
        backedUp = False
        if os.path.isfile(filePath):
            try:
                os.replace(filePath, f'{filePath}.bak')
                backedUp = True
            except Exception as ex:
                raise Exception(f'Error: Cannot overwrite "{os.path.normpath(filePath)}": {str(ex)}.')

        try:
            with open(filePath, 'w', encoding='utf-8') as f:
                f.write(text)
        except Exception as ex:
            if backedUp:
                os.replace(f'{filePath}.bak', self.filePath)
            raise Exception(f'Error: Cannot write "{os.path.normpath(filePath)}": {str(ex)}.')

        return f'"{os.path.normpath(filePath)}" written.'



def main(sourcePath):
    """Convert an .aeon source file to a set of Markdown files.
    
    Positional arguments:
        sourcePath -- str: The path of the .aeon file.
    """
    print('aeon3obsidian version 2.2.0')

    # Create an Aeon 3 file object and read the data.
    aeon3File = Aeon3File(sourcePath)
    aeon3File.data = Aeon3Data()
    print(aeon3File.read())

    # Define the output directory.
    aeonDir, aeonFilename = os.path.split(sourcePath)
    projectName = os.path.splitext(aeonFilename)[0]
    obsidianFolder = os.path.join(aeonDir, projectName)

    # Create an Obsidian fileset object and write the data.
    obsidianFiles = ObsidianFiles(obsidianFolder)
    obsidianFiles.data = aeon3File.data
    print(obsidianFiles.write())


if __name__ == '__main__':
    main(sys.argv[1])
