######################################################################
# Author: Info2-Team
# MatNr: 012345678
# Description: Reference solution
# Comments: -
######################################################################

from bokeh.models import Band, ColumnDataSource, HoverTool
from bokeh.plotting import figure
from geopy.geocoders import ArcGIS

from .coordinates import Coordinates


class Sensor:
    geolocator = ArcGIS()
    def __init__(self, sensor_id, city, coords, data):
        self._id = sensor_id
        self._city = city
        self._coords = coords
        self._distance_to_centre = Coordinates.distance(self.city.coords, self.coords)
        self._data = data
        self._address = None

    @property
    def id(self):
        return self._id

    @property
    def city(self):
        return self._city

    @property
    def coords(self):
        return self._coords

    @property
    def distance_to_centre(self):
        return self._distance_to_centre

    @property
    def data(self):
        return self._data

    @property
    def address(self):
        if self._address is None:
            self._address = self.geolocator.reverse((self.coords.lat, self.coords.lon)).address
        return self._address


    def create_bokeh_plot(self, smooth=False):
        data = self.data.copy(deep=True)

        if smooth:
            data[['P1', 'P2']] = data.rolling(smooth, on="timestamp").mean()[['P1', 'P2']]

        source = ColumnDataSource(data)

        fig = figure(title=f"Sensor ID: {self.id}",
                     plot_width=400,
                     plot_height=300,
                     x_axis_type="datetime")

        p1 = fig.line(x="timestamp", y="P1", source=source, line_color="blue")
        p2 = fig.line(x="timestamp", y="P2", source=source, line_color="orange")

        hover = HoverTool(tooltips=[("Date", "@timestamp{%F}"),
                                    ("PM 10", "@P1"),
                                    ("PM 2.5", "@P2")],
                          formatters={"timestamp": "datetime"},
                          mode="vline",
                          point_policy="follow_mouse",
                          renderers=[p1]
                         )

        p1_band = Band(base='timestamp', lower=0, upper='P1', source=source, level='underlay', fill_alpha=.8, fill_color='blue')
        p2_band = Band(base='timestamp', lower=0, upper='P2', source=source, level='underlay', fill_alpha=.8, fill_color='orange')
        fig.add_layout(p1_band)
        fig.add_layout(p2_band)

        fig.add_tools(hover)
        fig.xaxis.major_label_orientation = 0.75

        return fig
