######################################################################
# Author: Info2-Team
# MatNr: 012345678
# Description: Reference solution
# Comments: -
######################################################################

import folium
import numpy as np
import pandas as pd
from bokeh.embed import file_html
from bokeh.resources import CDN
from folium.plugins import MarkerCluster

from .city import City
from .coordinates import Coordinates
from .sensor import Sensor


def read_cities(city_path, countries=[]):
    colors = ['red', 'blue', 'gray', 'darkred', 'lightred', 'orange', 'beige',
              'green', 'darkgreen', 'lightgreen', 'darkblue', 'lightblue',
              'purple', 'darkpurple', 'pink', 'cadetblue', 'lightgray', 'black']

    countries = [c.lower() for c in countries]

    cities_df = pd.read_csv(city_path)
    cities = []
    for i_color, (_, city) in enumerate(cities_df.iterrows()):
        if countries and city["country"].lower() not in countries:
            continue

        new_city = City(name=city["city"],
                        coords=Coordinates(lat=city["lat"], lon=city["lng"]),
                        country=city["country"],
                        population=city["population"],
                        colorcode=colors[i_color % len(colors)]
                        )

        cities.append(new_city)

    return cities


def find_nearest_city(cities, coords):
    nearest_city = None
    nearest_distance = 10
    for city in cities:
        new_distance = Coordinates.distance(city.coords, coords)
        if new_distance < nearest_distance:
            nearest_city = city
            nearest_distance = new_distance
    return nearest_city


def parse_sensors(data_path, cities):
    df = pd.read_csv(data_path, parse_dates=["timestamp"])
    for sensor_id, entries in df.groupby("sensor_id"):
        lat, lon = entries["lat"].values[0], entries["lon"].values[0]
        sensor_coords = Coordinates(lat, lon)
        nearest_city = find_nearest_city(cities, sensor_coords)

        if nearest_city is None:
            continue
        nearest_city.add_sensor(Sensor(sensor_id=sensor_id,
                                       city=nearest_city,
                                       coords=sensor_coords,
                                       data=entries[["P1", "P2", "timestamp"]].reset_index(drop=True)
                                       ))


def create_map(plot_list, filename, plot_sensor_values=False, smooth=False, zoom=6):
    sensors_to_plot = []
    for element in plot_list:
        if isinstance(element, City):
            sensors_to_plot.extend(element.sensors)
        elif isinstance(element, Sensor):
            sensors_to_plot.append(element)
        else:
            raise TypeError("elements to plot have to be from class City or Sensor!")

    lat = np.median([sensor.coords.lat for sensor in sensors_to_plot])
    lon = np.median([sensor.coords.lon for sensor in sensors_to_plot])

    sensor_map = folium.Map(location=(lat, lon), width=1000, height=1000, tiles="OpenStreetMap", zoom_start=zoom)
    marker_cluster = MarkerCluster(overlay=True, control=True)
    for sensor in sensors_to_plot:
        if plot_sensor_values:
            popup = folium.Popup(
                        folium.IFrame(
                            folium.Html(
                                file_html(sensor.create_bokeh_plot(smooth=smooth), CDN),
                                script=True),
                            height=325,
                            width=425),
                        max_width=425)
        else:
            popup = folium.Popup(folium.Html(f"<b>Sensor ID:</b> {sensor.id} <b>City:</b> {sensor.city.name}<br>"
                                             f"<b>Population:</b> {sensor.city.population}<br>"
                                             f"<b>Address:</b> {sensor.address}<br>"
                                             f"<b>Distance to centre:</b> {sensor.distance_to_centre:.2f}km",
                                             script=True,
                                             width=300))

        marker = folium.Marker(location=(sensor.coords.lat, sensor.coords.lon),
                               popup=popup,
                               icon=folium.Icon(color=sensor.city.colorcode)
                               )
        marker_cluster.add_child(marker)

    sensor_map.add_child(marker_cluster)
    sensor_map.save(filename)
