/**
 * Simple Debug Logging for Extension Tatkal
 * Provides basic logging functionality for proxy manager compatibility
 */

// Create a simple logger that mimics the DragonLogger interface
const createSimpleLogger = (component = 'tatkal') => {
    const log = (level, message, data = {}) => {
        const timestamp = new Date().toISOString();
        const emoji =
            {
                info: 'ℹ️',
                debug: '🔍',
                warn: '⚠️',
                error: '❌',
                success: '✅',
            }[level] || '📝';

        const contextStr =
            Object.keys(data).length > 0 ? ` | ${JSON.stringify(data)}` : '';
        const formattedMessage = `${emoji} [${timestamp}] [${level.toUpperCase()}] [${component}] ${message}${contextStr}`;

        console[level === 'success' ? 'info' : level](formattedMessage);

        // Store to Chrome storage for debugging
        try {
            chrome.storage.local.get(['debug_logs'], result => {
                const logs = result.debug_logs || [];
                logs.push({
                    timestamp,
                    level,
                    component,
                    message,
                    data,
                });

                // Keep only last 500 logs
                if (logs.length > 500) {
                    logs.splice(0, logs.length - 500);
                }

                chrome.storage.local.set({ debug_logs: logs });
            });
        } catch (e) {
            // Silently ignore storage errors
        }
    };

    return {
        info: (message, data) => log('info', message, data),
        debug: (message, data) => log('debug', message, data),
        warn: (message, data) => log('warn', message, data),
        error: (message, data) => log('error', message, data),
        success: (message, data) => log('success', message, data),
    };
};

// Create global logger instance
window.DragonLogger = {
    logger: createSimpleLogger('tatkal-popup'),
};

console.log('🔧 Simple Debug Logging initialized for Extension Tatkal');
console.log('🔧 DragonLogger exported to window:', !!window.DragonLogger);
