/**
 * Booking Timer Module
 * Handles compact booking clock widget in top-right corner, synced with extension clock
 * EXACT implementation from original backup for line-by-line compatibility
 */

// Module state - exact variable names from original
let clockInterval = null;
let bookingStartTime = null;
let isBookingActive = false;
let overlayElement = null;
let clockElement = null;
let isSyncedWithExtension = false;

/**
 * Creates the booking clock widget - compact top-right corner (EXACT from original)
 */
function createBookingOverlay() {
    console.log(
        '🎯 [Voltas-CLOCK] Creating booking clock widget - compact top-right corner'
    );

    // Remove existing overlay if any
    const existingOverlay = document.getElementById('Voltas-booking-overlay');
    if (existingOverlay) {
        console.log('🎯 [Voltas-CLOCK] Removing existing overlay');
        existingOverlay.remove();
    }

    // Create a compact clock widget instead of full overlay
    overlayElement = document.createElement('div');
    overlayElement.id = 'Voltas-booking-overlay';
    overlayElement.className = 'booking-clock-widget';

    console.log(
        '🎯 [Voltas-CLOCK] Created clock widget element:',
        overlayElement
    );

    // Add compact CSS styles for top-right corner widget
    const style = document.createElement('style');
    style.textContent = `
    .booking-clock-widget {
      position: fixed !important;
      top: 20px !important;
      right: 20px !important;
      width: 280px !important;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
      padding: 15px !important;
      border-radius: 12px !important;
      color: white !important;
      box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3) !important;
      z-index: 999999 !important;
      font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif !important;
      font-size: 12px !important;
      line-height: 1.4 !important;
      backdrop-filter: blur(10px) !important;
      border: 1px solid rgba(255, 255, 255, 0.2) !important;
      transition: all 0.3s ease !important;
    }
    
    .booking-clock-widget:hover {
      transform: translateY(-2px) !important;
      box-shadow: 0 12px 35px rgba(0, 0, 0, 0.4) !important;
    }
    
    .booking-widget-title {
      font-size: 14px !important;
      font-weight: bold !important;
      margin-bottom: 5px !important;
      text-align: center !important;
    }
    
    .booking-widget-subtitle {
      font-size: 11px !important;
      margin-bottom: 10px !important;
      opacity: 0.9 !important;
      text-align: center !important;
    }
    
    .booking-clock {
      font-size: 24px !important;
      font-weight: bold !important;
      margin: 8px 0 !important;
      font-family: 'Courier New', monospace !important;
      text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.3) !important;
      text-align: center !important;
      background: rgba(255, 255, 255, 0.1) !important;
      padding: 8px !important;
      border-radius: 6px !important;
      border: 1px solid rgba(255, 255, 255, 0.2) !important;
    }
    
    .booking-widget-status {
      font-size: 10px !important;
      margin-top: 8px !important;
      opacity: 0.8 !important;
      text-align: center !important;
    }
    
    .booking-widget-spinner {
      width: 16px !important;
      height: 16px !important;
      border: 2px solid rgba(255, 255, 255, 0.3) !important;
      border-top: 2px solid white !important;
      border-radius: 50% !important;
      animation: spin 1s linear infinite !important;
      margin: 5px auto !important;
      display: inline-block !important;
    }
    
    @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }
  `;

    document.head.appendChild(style);
    console.log('🎯 [Voltas-CLOCK] Added CSS styles to head');

    // Create compact HTML structure for top-right corner
    overlayElement.innerHTML = `
    <div class="booking-widget-title">🐉 Voltas Booking</div>
    <div class="booking-widget-subtitle">Automated booking in progress...</div>
    <div class="booking-widget-spinner"></div>
    <div class="booking-clock" id="Voltas-booking-clock">00:00:00</div>
    <div class="booking-widget-status" id="Voltas-booking-status">Initializing...</div>
  `;

    console.log('🎯 [Voltas-CLOCK] Set widget innerHTML');

    document.body.appendChild(overlayElement);
    console.log('🎯 [Voltas-CLOCK] Appended widget to body');

    // Get reference to clock element
    clockElement = document.getElementById('Voltas-booking-clock');
    console.log('🎯 [Voltas-CLOCK] Clock element reference:', clockElement);

    // Verify widget is in DOM and visible
    const widgetInDOM = document.getElementById('Voltas-booking-overlay');
    console.log('🎯 [Voltas-CLOCK] Widget in DOM:', widgetInDOM);
    console.log(
        '🎯 [Voltas-CLOCK] Widget computed style:',
        widgetInDOM ? window.getComputedStyle(widgetInDOM).display : 'not found'
    );

    console.log(
        '🎯 [Voltas-CLOCK] Booking clock widget created - compact top-right corner'
    );
    return overlayElement;
}

/**
 * Starts the booking timer (EXACT from original)
 */
function startBookingTimer() {
    console.log(
        '🎯 [Voltas-CLOCK] Starting booking clock widget in top-right corner'
    );

    // Don't start independent clock if we're syncing with extension
    if (isSyncedWithExtension) {
        console.log(
            '🎯 [Voltas-CLOCK] Synced with extension, not starting independent clock'
        );
        return;
    }

    // Reset state and force creation of widget
    if (isBookingActive) {
        console.log(
            '🎯 [Voltas-CLOCK] Clock was already active, resetting state'
        );
        stopBookingTimer();
    }

    // Always create/recreate widget to ensure it's visible
    createBookingOverlay();

    // Widget is automatically visible (no need to activate like overlay)
    if (overlayElement) {
        console.log('🎯 [Voltas-CLOCK] Clock widget created and visible');
    } else {
        console.error(
            '🎯 [Voltas-CLOCK] ERROR: Widget element not found after creation!'
        );
        return;
    }

    // Initialize state
    isBookingActive = true;
    bookingStartTime = Date.now();

    // Update initial status
    updateBookingTimerStatus('Booking in progress...');

    // Start the clock update function
    function updateClock() {
        if (!isBookingActive) return;

        const now = new Date();
        const timeString = now.toLocaleTimeString('en-US', { hour12: false });

        if (clockElement) {
            clockElement.textContent = timeString;
        }

        // Sync with extension
        try {
            chrome.runtime.sendMessage(
                window.MessageHandler.getMsg('syncClockUpdate', {
                    timeString: timeString,
                    timestamp: now.getTime(),
                    isActive: isBookingActive,
                })
            );
        } catch (error) {
            // Ignore sync errors
        }
    }

    updateClock();
    clockInterval = setInterval(updateClock, 1000);

    console.log(
        '🎯 [Voltas-CLOCK] Booking clock widget started in top-right corner'
    );
}

/**
 * Stops the booking timer (EXACT from original)
 */
function stopBookingTimer() {
    console.log('🎯 [Voltas-CLOCK] Stopping booking clock widget');

    isBookingActive = false;

    if (clockInterval) {
        clearInterval(clockInterval);
        clockInterval = null;
    }

    // Update status to show completion - widget remains visible
    const statusElement = document.getElementById('Voltas-booking-status');
    if (statusElement) {
        statusElement.textContent = 'Booking completed!';
    }

    // Change widget background to indicate completion
    if (overlayElement) {
        overlayElement.style.background =
            'linear-gradient(135deg, #4CAF50 0%, #45a049 100%)';
    }

    console.log(
        '🎯 [Voltas-CLOCK] Booking clock widget stopped - widget remains visible'
    );
}

/**
 * Updates the booking timer status (EXACT from original)
 */
function updateBookingTimerStatus(status) {
    console.log('🎯 [Voltas-CLOCK] Updating status:', status);
    const statusElement = document.getElementById('Voltas-booking-status');
    if (statusElement) {
        statusElement.textContent = status;
    }
}

/**
 * Syncs clock start with extension (EXACT from original)
 */
function syncClockStart(startTime, status) {
    console.log(
        '🎯 [Voltas-CLOCK] Syncing clock start with extension:',
        startTime,
        status
    );

    // Stop any independent clock
    if (clockInterval) {
        clearInterval(clockInterval);
        clockInterval = null;
    }

    isSyncedWithExtension = true;

    // Create widget if it doesn't exist
    if (!overlayElement || !document.getElementById('Voltas-booking-overlay')) {
        createBookingOverlay();
    }

    // Widget is automatically visible (no need to activate)
    isBookingActive = true;
    bookingStartTime = startTime;

    // Update status
    updateBookingTimerStatus(status);

    console.log('🎯 [Voltas-CLOCK] Clock sync started successfully');
}

/**
 * Syncs clock update with extension (EXACT from original)
 */
function syncClockUpdate(timeString, timestamp, isActive) {
    if (clockElement && isActive) {
        clockElement.textContent = timeString;
    }
}

/**
 * Syncs clock stop with extension (EXACT from original)
 */
function syncClockStop(status) {
    console.log('🎯 [Voltas-CLOCK] Syncing clock stop with extension:', status);

    isBookingActive = false;
    isSyncedWithExtension = false;

    if (clockInterval) {
        clearInterval(clockInterval);
        clockInterval = null;
    }

    // Update status
    updateBookingTimerStatus(status);

    // Change widget background to indicate completion
    if (overlayElement) {
        overlayElement.style.background =
            'linear-gradient(135deg, #4CAF50 0%, #45a049 100%)';
    }

    console.log('🎯 [Voltas-CLOCK] Clock sync stopped successfully');
}

/**
 * Syncs status update (EXACT from original)
 */
function syncStatusUpdate(status) {
    updateBookingTimerStatus(status);
}

/**
 * Legacy timer functions for compatibility (EXACT from original)
 */
function syncTimerStart(startTime, status) {
    syncClockStart(startTime, status);
}

function syncTimerUpdate(timeString, elapsed, isActive) {
    // Convert to clock format for compatibility
    const now = new Date();
    const timeString2 = now.toLocaleTimeString('en-US', { hour12: false });
    syncClockUpdate(timeString2, now.getTime(), isActive);
}

function syncTimerStop(status) {
    syncClockStop(status);
}

/**
 * Test function to manually show overlay (for debugging) (EXACT from original)
 */
function testShowOverlay() {
    console.log('🧪 [TEST] Manually showing clock widget for testing');
    createBookingOverlay();
    if (overlayElement) {
        console.log(
            '🧪 [TEST] Clock widget should now be visible in top-right corner'
        );
    }
}

// Export functions for use in other modules
window.BookingTimer = {
    startBookingTimer,
    stopBookingTimer,
    updateBookingTimerStatus,
    createBookingOverlay,
    syncTimerStart,
    syncTimerUpdate,
    syncTimerStop,
    syncStatusUpdate,
    syncClockStart,
    syncClockUpdate,
    syncClockStop,
    testShowOverlay,
};

// Make test function available globally for debugging (EXACT from original)
window.testShowOverlay = testShowOverlay;

// Auto-show overlay for testing when page loads on IRCTC (EXACT from original)
if (window.location.href.includes('irctc.co.in')) {
    console.log(
        '🎯 [Voltas-CLOCK] IRCTC page detected, creating test widget in 2 seconds'
    );
    setTimeout(() => {
        console.log('🎯 [Voltas-CLOCK] Creating test widget now');
        testShowOverlay();
    }, 2000);
}
