/**
 * Proxy Background Handler Module
 * Handles proxy configuration, testing, and authentication for Tatkal extension
 * Extracted from background_script.js for better modularity
 */

/**
 * Initialize proxy background handler
 * @param {Object} logger - Logger instance from background script
 */
function initializeProxyBackground(logger) {
    if (!logger) {
        throw new Error('Logger is required for proxy background handler');
    }

    // Store logger reference for use in proxy functions
    self.proxyLogger = logger;

    logger.info('🔧 Proxy Background Handler initialized');

    // Apply proxy settings on initialization
    applyProxySettings();

    return {
        applyProxySettings,
        testProxyConnectivity,
        handleProxyAuthentication,
    };
}

/**
 * Apply proxy settings from Chrome storage
 * Replaces the embedded proxy logic in background_script.js
 */
function applyProxySettings() {
    const logger = self.proxyLogger;
    logger.debug('🌐 Applying proxy settings from storage');

    chrome.storage.local.get('proxy_settings', result => {
        const settings = result.proxy_settings;

        if (!settings || settings.activeProxy === 'disabled') {
            logger.info('🔄 Clearing proxy settings - disabled or no settings');
            chrome.proxy.settings.clear(
                {
                    scope: 'regular',
                },
                () => {
                    logger.info('✅ Proxy settings cleared successfully');
                }
            );
            return;
        }

        const activeIndex = parseInt(settings.activeProxy, 10) - 1;

        if (
            isNaN(activeIndex) ||
            !settings.proxies ||
            !settings.proxies[activeIndex]
        ) {
            logger.error('❌ Invalid proxy configuration', {
                activeIndex,
                hasProxies: !!settings.proxies,
                proxiesLength: settings.proxies?.length || 0,
            });

            chrome.proxy.settings.clear({
                scope: 'regular',
            });
            return;
        }

        const activeProxy = settings.proxies[activeIndex];

        if (!activeProxy.ip || !activeProxy.port) {
            logger.warn('⚠️ Active proxy missing IP or Port - clearing proxy', {
                proxyIndex: activeIndex,
                hasIp: !!activeProxy.ip,
                hasPort: !!activeProxy.port,
            });

            chrome.proxy.settings.clear({
                scope: 'regular',
            });
            return;
        }

        const config = {
            mode: 'fixed_servers',
            rules: {
                singleProxy: {
                    scheme: 'http',
                    // Assuming HTTP proxy, adjust if SOCKS5 or HTTPS is needed
                    host: activeProxy.ip,
                    port: parseInt(activeProxy.port, 10),
                },
                bypassList: ['<local>'],
            },
        };

        chrome.proxy.settings.set(
            {
                value: config,
                scope: 'regular',
            },
            () => {
                if (chrome.runtime.lastError) {
                    logger.error('❌ Failed to set proxy settings', {
                        error: chrome.runtime.lastError.message,
                        proxyConfig: {
                            ip: activeProxy.ip,
                            port: activeProxy.port,
                        },
                    });
                } else {
                    logger.info('🌐 Proxy settings applied successfully', {
                        proxyHost: `${activeProxy.ip}:${activeProxy.port}`,
                        proxyIndex: activeIndex,
                    });
                }
            }
        );
    });
}

/**
 * Test proxy connectivity
 * Temporarily sets proxy and tests with a simple endpoint
 * @param {Object} proxy - Proxy configuration object
 * @returns {Promise<Object>} Test result with success/failure info
 */
async function testProxyConnectivity(proxy) {
    const logger = self.proxyLogger;
    logger.info('🧪 Testing proxy connectivity', {
        proxyHost: `${proxy.ip}:${proxy.port}`,
    });

    const config = {
        mode: 'fixed_servers',
        rules: {
            singleProxy: {
                scheme: 'http',
                // Assuming HTTP proxy
                host: proxy.ip,
                port: parseInt(proxy.port, 10),
            },
            bypassList: ['<local>'],
        },
    };
    let success = false;
    let message = '';

    try {
        await new Promise((resolve, reject) => {
            chrome.proxy.settings.set(
                {
                    value: config,
                    scope: 'regular',
                },
                () => {
                    if (chrome.runtime.lastError) {
                        const error = new Error(
                            `Failed to set proxy settings: ${chrome.runtime.lastError.message}`
                        );
                        logger.error('❌ Proxy test setup failed', {
                            error: chrome.runtime.lastError.message,
                            proxyHost: `${proxy.ip}:${proxy.port}`,
                        });
                        reject(error);
                    } else {
                        logger.debug('✅ Temporary proxy set for testing', {
                            proxyHost: `${proxy.ip}:${proxy.port}`,
                        });
                        resolve();
                    }
                }
            );
        });

        // Use a reliable, simple endpoint for testing
        // Returns 204 No Content, good for connectivity check
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout

        try {
            const response = await fetch(
                'https://www.google.com/generate_204',
                {
                    signal: controller.signal,
                }
            );
            clearTimeout(timeoutId);

            if (response.status === 204) {
                success = true;
                message = 'Proxy is working.';
                logger.info('✅ Proxy connectivity test passed', {
                    proxyHost: `${proxy.ip}:${proxy.port}`,
                    responseStatus: response.status,
                });
            } else {
                success = false;
                message = `Proxy returned unexpected status: ${response.status}`;
                logger.warn('⚠️ Proxy test returned unexpected status', {
                    proxyHost: `${proxy.ip}:${proxy.port}`,
                    status: response.status,
                });
            }
        } catch (fetchError) {
            clearTimeout(timeoutId);

            if (fetchError.name === 'AbortError') {
                message = 'Proxy test timed out (10 seconds).';
                logger.warn('⏰ Proxy test timed out', {
                    proxyHost: `${proxy.ip}:${proxy.port}`,
                    timeout: '10 seconds',
                });
            } else if (
                fetchError.message.includes('ERR_NO_SUPPORTED_PROXIES')
            ) {
                message =
                    'Proxy requires authentication or is unreachable. Please check username/password or IP/port.';
                logger.error('🔐 Proxy authentication or connectivity issue', {
                    proxyHost: `${proxy.ip}:${proxy.port}`,
                    error: fetchError.message,
                });
            } else {
                message = `Network error or proxy unreachable: ${fetchError.message}`;
                logger.error('🌐 Network error during proxy test', {
                    proxyHost: `${proxy.ip}:${proxy.port}`,
                    error: fetchError.message,
                });
            }
            success = false;
        }
    } catch (proxyError) {
        success = false;
        message = proxyError.message;
        logger.error('❌ Proxy test failed with error', {
            proxyHost: `${proxy.ip}:${proxy.port}`,
            error: proxyError.message,
        });
    } finally {
        // Always clear the proxy settings after testing
        await new Promise(resolve => {
            chrome.proxy.settings.clear(
                {
                    scope: 'regular',
                },
                () => {
                    logger.debug(
                        '🧹 Temporary proxy settings cleared after test'
                    );
                    resolve();
                }
            );
        });
    }

    logger.info('🧪 Proxy connectivity test completed', {
        proxyHost: `${proxy.ip}:${proxy.port}`,
        success,
        message,
    });

    return {
        success,
        message,
    };
}

/**
 * Handle proxy authentication
 * Sets up the webRequest listener for proxy authentication
 * @param {Function} callbackFn - Chrome webRequest callback function
 */
function handleProxyAuthentication(details, callbackFn) {
    const logger = self.proxyLogger;
    logger.info('🔐 Proxy authentication required', {
        url: details.url,
        challenger: details.challenger,
    });

    chrome.storage.local.get('proxy_settings', result => {
        const settings = result.proxy_settings;

        if (settings && settings.activeProxy !== 'disabled') {
            const activeIndex = parseInt(settings.activeProxy, 10) - 1;
            const proxy = settings.proxies[activeIndex];

            if (proxy && proxy.user && proxy.pass) {
                logger.info('✅ Providing proxy authentication credentials', {
                    proxyIndex: activeIndex,
                    hasUser: !!proxy.user,
                    hasPass: !!proxy.pass,
                });

                callbackFn({
                    authCredentials: {
                        username: proxy.user,
                        password: proxy.pass,
                    },
                });
            } else {
                logger.warn(
                    '❌ Proxy authentication required but credentials missing',
                    {
                        proxyIndex: activeIndex,
                        hasUser: !!proxy?.user,
                        hasPass: !!proxy?.pass,
                    }
                );

                callbackFn({
                    cancel: true,
                });
            }
        } else {
            logger.info('🚫 Proxy authentication required but proxy disabled');
            callbackFn({
                cancel: true,
            });
        }
    });
}

/**
 * Setup proxy authentication listener
 * Call this from background script to set up the authentication handler
 */
function setupProxyAuthListener() {
    const logger = self.proxyLogger;

    chrome.webRequest.onAuthRequired.addListener(
        (details, callbackFn) => {
            handleProxyAuthentication(details, callbackFn);
            return { cancel: false };
        },
        {
            urls: ['<all_urls>'],
        },
        ['asyncBlocking']
    );

    logger.info('🔧 Proxy authentication listener set up');
}

// Export the module functions
self.TatkalProxyBackground = {
    initializeProxyBackground,
    applyProxySettings,
    testProxyConnectivity,
    handleProxyAuthentication,
    setupProxyAuthListener,
};

// Debug: Module loaded
console.log(
    'TatkalProxyBackground module loaded and exported to self.TatkalProxyBackground'
);
