/**
 * Tatkal Congratulation Handler Module
 * Handles success messages and booking completion notifications for Tatkal extension
 * Adapted from the main Dragon extension congratulation handler
 */

// Initialize logger for congratulation handler
const congratulationLogger = window.DragonLogging
    ? window.DragonLogging.createLogger({
          component: 'congratulation-handler',
          level: 'debug',
          enableStorage: true,
          environment: 'development',
      })
    : {
          info: (...args) => console.log('🎉 [TATKAL-CONGRATULATION]', ...args),
          debug: (...args) =>
              console.log('🎯 [TATKAL-CONGRATULATION]', ...args),
          warn: (...args) =>
              console.warn('⚠️ [TATKAL-CONGRATULATION]', ...args),
          error: (...args) =>
              console.error('❌ [TATKAL-CONGRATULATION]', ...args),
      };

/**
 * Centralized function to check for existing congratulation messages
 * @returns {boolean} True if congratulation message already exists
 */
function hasExistingCongratulationMessage() {
    const congratulationSelectors = [
        '#dragon-tatkal-congratulation',
        '.dragon-tatkal-congratulation',
        "[id*='dragon-tatkal-congratulation']",
        "[class*='dragon-tatkal-congratulation']",
    ];

    for (const selector of congratulationSelectors) {
        const existingElement = document.querySelector(selector);
        if (existingElement) {
            congratulationLogger.debug(
                'Existing congratulation message found',
                {
                    selector,
                    element: existingElement.tagName,
                }
            );
            return true;
        }
    }
    return false;
}

/**
 * Function to show congratulatory message after Tatkal booking completion
 */
function showCongratulationMessage() {
    congratulationLogger.info('showCongratulationMessage function started', {
        timestamp: new Date().toISOString(),
        url: window.location.href,
        operation: 'show-congratulation',
    });

    // Stop the booking timer when congratulation is shown
    if (window.BookingTimer) {
        window.BookingTimer.stopBookingTimer();
    }

    // Check if congratulation element already exists
    if (hasExistingCongratulationMessage()) {
        congratulationLogger.warn(
            'Congratulation message already displayed, exiting',
            {
                operation: 'show-congratulation',
            }
        );
        return;
    }

    // Additional safety check - only show on booking confirmation page
    const currentUrl = window.location.href.toLowerCase();
    if (!currentUrl.includes('nget/booking-confirm')) {
        congratulationLogger.debug(
            'Not on booking confirmation page, skipping congratulation',
            {
                url: currentUrl,
                operation: 'show-congratulation',
            }
        );
        return;
    }

    // Check for success indicators in page content
    const pageText = document.body.innerText.toLowerCase();
    const hasSuccessIndicators =
        pageText.includes('thank you') ||
        pageText.includes('congratulations') ||
        pageText.includes('successfully booked');

    if (!hasSuccessIndicators) {
        congratulationLogger.debug(
            'Success indicators not found in page content, skipping congratulation',
            {
                pageTextLength: pageText.length,
                operation: 'show-congratulation',
            }
        );
        return;
    }

    // Create the congratulatory element with enhanced styling
    const congratulationDiv = document.createElement('div');
    congratulationDiv.id = 'dragon-tatkal-congratulation';
    congratulationDiv.style.cssText = `
    position: relative !important;
    top: 0 !important;
    left: 0 !important;
    right: 0 !important;
    bottom: auto !important;
    width: 100% !important;
    max-width: 100% !important;
    margin: 20px auto !important;
    padding: 20px !important;
    background: linear-gradient(135deg, #4CAF50, #45a049) !important;
    color: white !important;
    border-radius: 12px !important;
    box-shadow: 0 8px 32px rgba(0,0,0,0.3) !important;
    font-family: 'Arial', sans-serif !important;
    font-size: 16px !important;
    text-align: center !important;
    z-index: 999999 !important;
    animation: slideInFromTop 0.8s ease-out !important;
    border: 3px solid #2E7D32 !important;
    backdrop-filter: blur(10px) !important;
    transform: none !important;
    display: block !important;
    visibility: visible !important;
    opacity: 1 !important;
  `;

    congratulationDiv.innerHTML = `
    <style>
      @keyframes slideInFromTop {
        0% { transform: translateY(-100px); opacity: 0; }
        100% { transform: translateY(0); opacity: 1; }
      }
      @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.05); }
      }
      #dragon-tatkal-congratulation-close {
        position: absolute !important;
        top: 10px !important;
        right: 15px !important;
        background: rgba(255,255,255,0.2) !important;
        border: none !important;
        color: white !important;
        font-size: 24px !important;
        font-weight: bold !important;
        cursor: pointer !important;
        border-radius: 50% !important;
        width: 35px !important;
        height: 35px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.3s ease !important;
      }
      #dragon-tatkal-congratulation-close:hover {
        background: rgba(255,255,255,0.3) !important;
        transform: scale(1.1) !important;
      }
    </style>
    <button id="dragon-tatkal-congratulation-close" onclick="this.parentElement.remove()">&times;</button>
    <div style="font-size: 28px; font-weight: bold; margin-bottom: 15px; animation: pulse 2s infinite;">
      🎉 CONGRATULATIONS! 🎉
    </div>
    <div style="font-size: 18px; margin-bottom: 10px;">
      Your train ticket has been successfully booked!
    </div>
    <div style="font-size: 14px; opacity: 0.9;">
      Thank you for using Dragon IRCTC Extension
    </div>
    <div style="margin-top: 15px; font-size: 12px; opacity: 0.8;">
      This message will auto-close in 10 seconds
    </div>
  `;

    // Find the best insertion point
    let insertionTarget = null;
    const insertionSelectors = [
        '.container-fluid',
        '.container',
        'main',
        '#main',
        '.main-content',
        'body > div:first-child',
        'body',
    ];

    for (const selector of insertionSelectors) {
        const element = document.querySelector(selector);
        if (element && element.offsetParent !== null) {
            insertionTarget = element;
            break;
        }
    }

    if (insertionTarget) {
        if (insertionTarget.tagName.toLowerCase() === 'body') {
            insertionTarget.insertBefore(
                congratulationDiv,
                insertionTarget.firstChild
            );
        } else {
            insertionTarget.insertBefore(
                congratulationDiv,
                insertionTarget.firstChild
            );
        }
        congratulationLogger.info(
            'Congratulation message displayed successfully',
            {
                insertionTarget: insertionTarget.tagName,
                operation: 'show-congratulation',
            }
        );
    } else {
        document.body.insertBefore(congratulationDiv, document.body.firstChild);
        congratulationLogger.info(
            'Congratulation message inserted into body as fallback',
            {
                operation: 'show-congratulation',
            }
        );
    }
}

/**
 * Force show congratulation message with enhanced insertion strategies for Tatkal
 */
function forceShowCongratulationMessage() {
    congratulationLogger.info(
        'forceShowCongratulationMessage function started',
        {
            timestamp: new Date().toISOString(),
            url: window.location.href,
            operation: 'force-show-congratulation',
        }
    );

    // Stop the booking timer when congratulation is shown
    if (window.BookingTimer) {
        window.BookingTimer.stopBookingTimer();
    }

    // Check if congratulation element already exists
    if (hasExistingCongratulationMessage()) {
        congratulationLogger.warn(
            'Congratulation message already displayed, exiting',
            {
                operation: 'force-show-congratulation',
            }
        );
        return;
    }

    // Create the congratulatory element with enhanced styling
    const congratulationDiv = document.createElement('div');
    congratulationDiv.id = 'dragon-tatkal-congratulation';
    congratulationDiv.style.cssText = `
    position: fixed !important;
    top: 20px !important;
    left: 50% !important;
    transform: translateX(-50%) !important;
    width: 90% !important;
    max-width: 600px !important;
    margin: 0 !important;
    padding: 25px !important;
    background: linear-gradient(135deg, #FF6B35, #F7931E) !important;
    color: white !important;
    border-radius: 15px !important;
    box-shadow: 0 10px 40px rgba(0,0,0,0.4) !important;
    font-family: 'Arial', sans-serif !important;
    font-size: 16px !important;
    text-align: center !important;
    z-index: 2147483647 !important;
    animation: forceSlideIn 1s ease-out !important;
    border: 4px solid #E55100 !important;
    backdrop-filter: blur(15px) !important;
    display: block !important;
    visibility: visible !important;
    opacity: 1 !important;
  `;

    congratulationDiv.innerHTML = `
    <style>
      @keyframes forceSlideIn {
        0% { transform: translateX(-50%) translateY(-200px); opacity: 0; }
        100% { transform: translateX(-50%) translateY(0); opacity: 1; }
      }
      @keyframes forcePulse {
        0%, 100% { transform: translateX(-50%) scale(1); }
        50% { transform: translateX(-50%) scale(1.05); }
      }
    </style>
    <button id="dragon-tatkal-congratulation-close" onclick="this.parentElement.remove()" style="
      position: absolute !important;
      top: 10px !important;
      right: 15px !important;
      background: rgba(255,255,255,0.2) !important;
      border: none !important;
      color: white !important;
      font-size: 24px !important;
      font-weight: bold !important;
      cursor: pointer !important;
      border-radius: 50% !important;
      width: 35px !important;
      height: 35px !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      transition: all 0.3s ease !important;
    ">&times;</button>
    <div style="font-size: 32px; font-weight: bold; margin-bottom: 15px; animation: forcePulse 2s infinite;">
      🚀 FORCE CONGRATULATIONS! 🚀
    </div>
    <div style="font-size: 20px; margin-bottom: 10px;">
      Your booking process has been completed!
    </div>
    <div style="font-size: 16px; opacity: 0.9; margin-bottom: 15px;">
      Dragon Extension has successfully automated your booking
    </div>
    <div style="font-size: 12px; opacity: 0.8;">
      This message will auto-close in 15 seconds
    </div>
  `;

    // Force insertion into document body
    document.body.insertBefore(congratulationDiv, document.body.firstChild);
    congratulationLogger.info(
        'Force congratulation message displayed successfully',
        {
            operation: 'force-show-congratulation',
        }
    );

    // Auto-remove after 15 seconds
    setTimeout(() => {
        if (document.getElementById('dragon-tatkal-congratulation')) {
            document.getElementById('dragon-tatkal-congratulation').remove();
            congratulationLogger.debug(
                'Auto-removed force congratulation message after 15 seconds',
                {
                    operation: 'auto-remove-force-congratulation',
                }
            );
        }
    }, 15000);
}

// Tatkal-specific success checking functions
function checkForTatkalSuccessIndicators() {
    const tatkalSuccessIndicators = [
        'booking successful',
        'ticket booked',
        'payment successful',
        'booking confirmed',
        'pnr',
        'transaction successful',
        'tatkal',
        'confirmed',
        'thank you',
        'congratulations',
    ];

    const pageText = document.body.innerText.toLowerCase();
    return tatkalSuccessIndicators.some(indicator =>
        pageText.includes(indicator)
    );
}

function createCongratulationPopup() {
    // This is handled by the main showCongratulationMessage function
    congratulationLogger.debug('Using main congratulation function', {
        operation: 'create-congratulation-popup',
    });
}

function showGenericSuccessMessage() {
    congratulationLogger.info('Showing generic success message', {
        operation: 'show-generic-success',
    });
    forceShowCongratulationMessage();
}

function playSuccessSound() {
    // Sound functionality (optional)
    congratulationLogger.debug('Success sound would play here', {
        operation: 'play-success-sound',
    });
}

function sendBookingNotification() {
    // Notification functionality (optional)
    congratulationLogger.debug('Booking notification would be sent here', {
        operation: 'send-booking-notification',
    });
}

function extractBookingDetails() {
    // Details extraction (optional)
    congratulationLogger.debug('Booking details would be extracted here', {
        operation: 'extract-booking-details',
    });
    return {};
}

// Export functions for use in other modules
window.TatkalCongratulationHandler = {
    hasExistingCongratulationMessage,
    showCongratulationMessage,
    forceShowCongratulationMessage,
    checkForTatkalSuccessIndicators,
    createCongratulationPopup,
    showGenericSuccessMessage,
    playSuccessSound,
    sendBookingNotification,
    extractBookingDetails,
};

// ===== TATKAL CONGRATULATION SYSTEM INITIALIZATION =====
// Initialize congratulation system when page loads
congratulationLogger.info(
    'Tatkal Congratulation System initialization started',
    {
        timestamp: new Date().toISOString(),
        url: window.location.href,
        pageTitle: document.title,
        operation: 'module-initialization',
    }
);

// Set up a delayed initialization to ensure page is fully loaded
setTimeout(() => {
    congratulationLogger.debug('Delayed initialization completed', {
        timestamp: new Date().toISOString(),
        documentReadyState: document.readyState,
        bodyExists: !!document.body,
        operation: 'delayed-initialization',
    });
}, 3000);
