function demo1
% Simple example of the use of Hidden Semi-Markov Model (HSMM) to learn 
% and reproduce a movement represented as a combination of linear systems 
% with a velocity dx computed iteratively as dx = sum_i h_i (A_i x + b_i), 
% where h_i is a weight defined by HSMM. A_i and b_i form a matrix and 
% vector associated with state i of the HSMM.
%
% Authors:	Sylvain Calinon, Antonio Pistillo, 2014
%           http://programming-by-demonstration.org/
%
% This source code is given for free! However, we would be grateful if
% you refer to the following paper in any academic publication that 
% uses this code or part of it: 
%
% @inproceedings{Calinon11IROS,
%   author="Calinon, S. and Pistillo, A. and Caldwell, D. G.",
%   title="Encoding the time and space constraints of a task in explicit-duration hidden {M}arkov model",
%   booktitle="Proc. {IEEE/RSJ} Intl Conf. on Intelligent Robots and Systems ({IROS})",
%   year="2011",
%   month="September",
%   address="San Francisco, CA, USA",
%   pages="3413--3418"
% }

%% Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
nbStates = 4; %Number of states in the HSMM
nbSamples = 3; %Number of demonstrations
nbDataRepro = 220; %Number of datapoints in each reproduction sample
nbD = 80; %Number of maximum duration step to consider in the HSMM
nbVar = 2; %Dimensionality of the data
dt = 0.01; %Duration of one time step

%% Load data 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Data=[]; DataTS=[];
for n=1:nbSamples
  load(['data/sample' num2str(n,'%.2d') '.mat']); %Load trajectory in a temporary 'DataSample' variable
  s(n).Data = DataSample; %Copy trajectory in a structure 's(n)' 
  s(n).DataVel = gradient(s(n).Data) / dt; %Compute velocity data
  s(n).nbData = size(s(n).Data,2);
  Data = [Data [s(n).Data; s(n).DataVel]]; %Concatenation of all demonstrations (with position+velocity)
  DataTS = [DataTS [1:s(n).nbData; s(n).Data]]; %Concatenation of all demonstrations (with time+position)
end


%% Estimation of HSMM parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Estimation of the HSMM parameters with EM algorithm...');
%Initialization
[MuTmp, SigmaTmp] = EM_init_regularTiming(DataTS, nbStates);
m.Mu = MuTmp(2:end,:);
m.Sigma = SigmaTmp(2:end,2:end,:);
m.Trans = ones(nbStates,nbStates) * 0.01;
for i=1:nbStates-1
  m.Trans(i,i) = 0.9;
  m.Trans(i,i+1) = 0.08;
end
m.Trans(nbStates,nbStates) = 0.97;
m.StatesPriors = ones(nbStates,1) * .1;
m.StatesPriors(1) = .7;
m.Pd = repmat(gaussPDF(0:nbD-1, nbD/2, nbD)', nbStates, 1);
%Normalization
m.Trans = m.Trans ./ repmat(sum(m.Trans,2),1,nbStates);
m.StatesPriors = m.StatesPriors / sum(m.StatesPriors);
m.Pd = m.Pd ./ repmat(sum(m.Pd,2), 1, nbD);

%Estimate HSMM parameters with EM algorithm 
m = EM_HSMM(s, m);
disp('States priors:'); disp(m.StatesPriors);
disp('Transition probabilities:'); disp(m.Trans);

%Estimation of A_i and b_i
disp('Estimation of the associated matrices A_i and vectors b_i...');
H=[];
for n=1:nbSamples
  [bmx, ALPHA, S] = hsmm_fwd_init_hsum(s(n).Data(:,1), m);
  for t=1:s(n).nbData
    [bmx, ALPHA, S, alpha] = hsmm_fwd_step_hsum(s(n).Data(:,t), m, bmx, ALPHA, S);
    H = [H; alpha'];
  end
end
H = H ./ repmat(sum(H,2),1,nbStates);

%Weighted least-squares estimate (stability is not guaranteed in this example)
posId=[1:nbVar]; velId=[nbVar+1:2*nbVar]; 
X = [Data(posId,:); ones(1,size(Data,2))];
Y = [Data(velId,:)];
for i=1:nbStates
  m.s(i).Ab = [pinv(X * diag(H(:,i).^2) * X') * X * diag(H(:,i).^2) * Y']'; %dx=Ax+b -> dx=[Ab]*[x;1]
end


%% Reproduction
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Reproduction...');
for n=1:nbSamples
  currPos = s(n).Data(:,1);
  currVel = s(n).DataVel(:,1); %zeros(size(currPos));
  r(n).Data(:,1) = [currPos; currVel];
  %Initialize HSMM weights
  %[ALPHA, S, r(n).h(:,1)] = hsmm_fwd_init_ts(m);
	[bmx, ALPHA, S, r(n).h(:,1)] = hsmm_fwd_init_hsum(currPos, m);
  for t=2:nbDataRepro 
		%[ALPHA, S, r(n).h(:,t)] = hsmm_fwd_step_ts(m, ALPHA, S);
    [bmx, ALPHA, S, r(n).h(:,t)] = hsmm_fwd_step_hsum(currPos, m, bmx, ALPHA, S);
    %Compute velocity command
    for i=1:nbStates
      velTmp(:,i) = m.s(i).Ab * [currPos; 1]; 
    end
    %Update position
    currVel = velTmp * r(n).h(:,t);
    currPos = currPos + currVel * dt;
    r(n).Data(:,t) = [currPos; currVel];
  end
end


%% Plots
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Plots...');
figure('Position',[20 200 800 400]); 
clrmap = colormap('Jet');
clrmap = clrmap(round(linspace(1,64,nbStates)),:);
%Self-transition probability plot
subplot(1,2,1); hold on; box on;
for i=1:nbStates-1
  patch([1 1:nbD nbD], [0 m.Pd(i,:) 0], min(clrmap(i,:)+0.4,1), 'lineWidth', 1, 'EdgeColor', clrmap(i,:));
end
axis([1 nbD 0 max(max(m.Pd(1:end-1,:)))+0.01]); 
xlabel('d'); ylabel('p_i');
%Model and trajectories plot
subplot(1,2,2); hold on; box on;
for i=1:nbStates
  plotGMM(m.Mu(:,i),m.Sigma(:,:,i),clrmap(i,:),1);
end
for n=1:nbSamples
  plot(s(n).Data(1,1),s(n).Data(2,1),'.','markersize',14,'color',[.6 .6 .6]);
  plot(s(n).Data(1,:),s(n).Data(2,:),'-','linewidth',2,'color',[.6 .6 .6]);
end
for n=1:nbSamples
  plot(r(n).Data(1,1),r(n).Data(2,1),'.','markersize',14,'color',[.8 0 0]);
  plot(r(n).Data(1,:),r(n).Data(2,:),'-','linewidth',2,'color',[.8 0 0]);
end
xlabel('x_1'); ylabel('x_2');

pause;
close all;

