// Training of a task-parameterized Gaussian mixture model (GMM) based on candidate frames of reference.
// The proposed task-parameterized GMM approach relies on the linear transformation and product properties of
// Gaussian distributions to derive an expectation-maximization (EM) algorithm to train the model.
// The proposed approach is contrasted with an implementation of the approach proposed by Wilson and Bobick in
// 1999, with an implementation applied to GMM (that we will call PGMM) and following the model described in
// "Parametric Hidden Markov Models for Gesture Recognition", IEEE Trans. on Pattern Analysis and Machine
// Intelligence.
// In contrast to the standard PGMM approach, the new approach that we propose allows the parameterization of
// both the centers and covariance matrices of the Gaussians. It has been designed for targeting problems in
// which the task parameters can be represented in the form of coordinate systems, which is for example the
// case in robot manipulation problems.
//
// Authors:	Tohid Alizadeh and Sylvain Calinon, 2012
//         	http://programming-by-demonstration.org/
//
// This source code is given for free! In exchange, we would be grateful if you cite
// the following reference in any academic publication that uses this code or part of it:
//
// @inproceedings{Calinon12Hum,
//   author="Calinon, S. and Li, Z. and Alizadeh, T. and Tsagarakis, N. G. and Caldwell, D. G.",
//   title="Statistical dynamical systems for skills acquisition in humanoids",
//   booktitle="Proc. {IEEE} Intl Conf. on Humanoid Robots ({H}umanoids)",
//   year="2012",
//   address="Osaka, Japan"
// }

#include "PGMM.h"

//--------------------------------------------------------------
pGMM::pGMM(){
  //pGMM(2,1);
}

//--------------------------------------------------------------
pGMM::pGMM(int n, int f, int s){
  nbSamples = 0;
  nbRSamples = 0;
  model.nbStates = s;   //Number of States of the model
  model.nbVar = n;  //2 for (x,y) or 3 for (t, x, y) or ...
  model.nbFrames = f; // Number of frames of the model
};

//--------------------------------------------------------------
int pGMM::load(string folderName, int nbDemo, int ParamType){
  mat ParamM, DataM;
  stringstream pathTmp;
  string path;

  int idList[]={1,2,3,4};
  for (int i=0;i<nbDemo;i++){
    pathTmp<<idList[i];
    path = folderName + "/param_obs0" + pathTmp.str() + ".txt";
    ParamM.load(path, raw_ascii);
    path = folderName + "/data_obs0" + pathTmp.str() + ".txt";
    DataM.load(path, raw_ascii);
    pathTmp.str("");
    createParams(ParamM, ParamType);
    addDemo(DataM);
  }
  cout<<endl<<"Loading Finished"<<endl;
}

//--------------------------------------------------------------
int pGMM::createParams(mat ParamM, int ParamType, int flag){
  Sample sTmp;
  sTmp.ParamM = ParamM;
  if (ParamType == 1){
    sTmp.angle = ParamM(0,1+3*model.nbVar);
    sTmp.scaling = ParamM(1,1+3*model.nbVar);
  }
  //The following two if statements could be unified probably ?????
  if (ParamType==1){
    sTmp.p.resize(model.nbFrames);
    for(int i=0; i<model.nbFrames; i++){
      sTmp.p[i].A.set_size(model.nbVar,model.nbVar);
      sTmp.p[i].D.set_size(model.nbVar,model.nbVar);
      sTmp.p[i].b = ParamM.col(i*(1+3*model.nbVar));
      sTmp.p[i].E = ParamM.submat(0, i*(1+3*model.nbVar)+1, model.nbVar-1, i*(1+3*model.nbVar)+model.nbVar);
      sTmp.p[i].D = ParamM.submat(0, i*(1+3*model.nbVar)+model.nbVar+1, model.nbVar-1, i*(1+3*model.nbVar)+2*model.nbVar);
      sTmp.p[i].A = ParamM.submat(0, i*(1+3*model.nbVar)+2*model.nbVar+1, model.nbVar-1, i*(1+3*model.nbVar)+3*model.nbVar);
    }
  }
  else if (ParamType==2){
    sTmp.nbData = 200;
    sTmp.p.resize(model.nbFrames*sTmp.nbData);
    for (int i=0; i<sTmp.nbData; i++){
      for (int m=0; m<model.nbFrames; m++){
        sTmp.p[i+m*sTmp.nbData].A.set_size(model.nbVar,model.nbVar);
        sTmp.p[i+m*sTmp.nbData].invA.set_size(model.nbVar,model.nbVar);
        sTmp.p[i+m*sTmp.nbData].D.set_size(model.nbVar,model.nbVar);
        sTmp.p[i+m*sTmp.nbData].b = ParamM.submat(m*model.nbVar, i*(4*model.nbVar+1), (m+1)*model.nbVar-1, i*(4*model.nbVar+1));
        sTmp.p[i+m*sTmp.nbData].E = ParamM.submat(m*model.nbVar, i*(4*model.nbVar+1)+1, (m+1)*model.nbVar-1, i*(4*model.nbVar+1)+model.nbVar);
        sTmp.p[i+m*sTmp.nbData].D = ParamM.submat(m*model.nbVar, i*(4*model.nbVar+1)+model.nbVar+1, (m+1)*model.nbVar-1, i*(4*model.nbVar+1)+2*model.nbVar);
        sTmp.p[i+m*sTmp.nbData].A = ParamM.submat(m*model.nbVar, i*(4*model.nbVar+1)+2*model.nbVar+1, (m+1)*model.nbVar-1, i*(4*model.nbVar+1)+3*model.nbVar);
        sTmp.p[i+m*sTmp.nbData].invA = ParamM.submat(m*model.nbVar, i*(4*model.nbVar+1)+3*model.nbVar+1, (m+1)*model.nbVar-1, i*(4*model.nbVar+1)+4*model.nbVar);
      }
    }
  }
  if (flag==0){
    smpl.resize(nbSamples+1);
    smpl[nbSamples]=sTmp;
  }
  else{
    rnew.resize(nbRSamples+1);
    rnew[nbRSamples]=sTmp;
  }
  return 1;
}

//--------------------------------------------------------------
int pGMM::addDemo(mat DataM){
  smpl[nbSamples].nbData = DataM.n_rows;
  smpl[nbSamples].Data = trans(DataM.cols(0,model.nbVar-1));
  smpl[nbSamples].Data0 = trans(DataM.cols(model.nbVar-1, DataM.n_cols-1));
  //Concatenate data
  nbData_Tot=0;
  for (int n=0; n<=nbSamples;n++){
    nbData_Tot += smpl[n].nbData;
  }
  Data.set_size(model.nbVar, nbData_Tot);  //"Data" will contain all data from all samples
  for (int n=0;n<=nbSamples;n++){
    int ind=0;
    for (int i=0;i<=n;i++){
      ind += smpl[i].nbData;
    }
    Data.submat(0, ind-smpl[n].nbData, model.nbVar-1, ind-1) = smpl[n].Data;
  }
  GAMMA.set_size(model.nbStates, nbData_Tot);

  nbSamples++;
  return 1;
}

//--------------------------------------------------------------
int pGMM::repro(string path){
  stringstream pathTmp;

  //Reproduction for the set of parameters used to train the model
  cout<<endl<<"Reproductions .... "<<endl;
  rsmpl.resize(nbSamples);
  for (int n=0; n<nbSamples; n++){
    rsmpl[n] = reproduction_DSGMR(smpl[0].Data.row(0), smpl[n], smpl[n].Data.submat(1,0,2,0));
    pathTmp<<n+1;
    mat Dtmp=trans(rsmpl[n].Data);
    Dtmp.save(path + "/data_rep_0" + pathTmp.str() + ".txt", raw_ascii);
    pathTmp.str("");
  }
  cout<<endl<<"Reproductions for the same set of task parameters are generated"<<endl;
  return 1;
}

//--------------------------------------------------------------
int pGMM::repro(vec id, vec w, string path){
  if(model.nbStates<1)
    return 0;
  Sample rTmp;
  stringstream pathTmp;
  mat currPos = zeros(2,1);
  rTmp = rnew[nbRSamples];
  currPos = smpl[id(0)].Data.submat(1,0,2,0)*w(0) + smpl[id(1)].Data.submat(1,0,2,0)*w(1);
  cout<<"New reproduction #"<<rnew.size()<<" using DSGMR..."<<endl;
  rnew[nbRSamples] = reproduction_DSGMR(smpl[0].Data.row(0), rTmp, currPos);
  rnew[nbRSamples].nbData = rnew[nbRSamples].Data.n_cols;
  pathTmp<<nbRSamples+1;
  mat Dtmp=trans(rnew[nbRSamples].Data);
  Dtmp.save(path + "/data_rep_new0" + pathTmp.str() + ".txt", raw_ascii);
  pathTmp.str("");
  nbRSamples++;
  return 1;
}

//--------------------------------------------------------------
Sample pGMM::reproduction_DSGMR(mat DataIn, Sample rr, vec DataInit){
  Sample r;
  cResGaussOut cresGout;
  vec in, out;
  int nbVarIn, nbVarOut;
  mat currVel, currPos, currTar, currAcc, MuTmp;
  r.M.resize(rr.nbData);
  r.S.set_size(model.nbStates, rr.nbData);
  r.H = zeros(model.nbStates, rr.nbData);
  in.set_size(DataIn.n_rows);
  out.set_size(model.nbVar-DataIn.n_rows);
  in = linspace(0, DataIn.n_rows-1, DataIn.n_rows);
  out = linspace(in(in.n_rows-1)+1 , model.nbVar-1, out.n_rows);
  nbVarIn = in.n_rows;
  nbVarOut = out.n_rows;
  MuTmp = zeros(nbVarOut, model.nbStates);
  currPos = DataInit;
  r.Data = zeros(nbVarOut*3+nbVarIn,rr.nbData);
  r.Data.rows(0, nbVarIn-1) = DataIn;
  cresGout = computeResultingGaussians(rr.p);
  r.p = cresGout.pp;
  r.prodRes = cresGout.prodr;

  for(int t=0; t<rr.nbData; t++){
    r.M[t] = zeros(model.nbVar, model.nbStates);
    r.M[t] = r.prodRes[t].mu;
    for (int i=0; i<model.nbStates; i++){
      r.S(i,t) = r.prodRes[t].Sigma[i];
    }
  }
  r.Mu = r.M[0];
  currVel = zeros(nbVarOut,1);
  for(int n=0; n<rr.nbData; n++){
    //Compute Activation weight
    for(int i=0; i<model.nbStates; i++){
      r.H(i,n) = model.priors(i)*as_scalar(GaussPDFfast(DataIn.col(n), cresGout.prodr[n].mu.submat(in(0),i,in(nbVarIn-1),i), cresGout.prodr[n].invSigma[i].submat(in(0), in(0), in(nbVarIn-1), in(nbVarIn-1)), cresGout.prodr[n].detSigma(i)));
    }
    r.H.col(n) /= sum(r.H.col(n));
    //Evaluate the current target (see Eq. (2) Calinon et al, Humanoids'2012 paper)
    currTar = zeros(nbVarOut,1);
    for(int i=0; i<model.nbStates; i++){
      MuTmp.col(i) = cresGout.prodr[n].mu.submat(out(0),i,out(nbVarOut-1),i)+cresGout.prodr[n].Sigma[i].submat(out(0),in(0),out(nbVarOut-1),in(nbVarIn-1))*inv(cresGout.prodr[n].Sigma[i].submat(in(0),in(0),in(nbVarIn-1),in(nbVarIn-1)))* (DataIn.col(n)-cresGout.prodr[n].mu.submat(in(0),i,in(nbVarIn-1),i));
      currTar += r.H(i,n)*MuTmp.col(i);
    }
    //Compute acceleration (see Eq. (1) Calinon et al, Humanoids'2012 paper)
    currAcc = model.kP * (currTar-currPos) - model.kV * currVel;
    //Update velocity and position
    currVel = currVel + currAcc * model.dt;
    currPos = currPos + currVel * model.dt;
    //Keep a trace of data
    r.Data(0,n) = n*model.dt;
    r.Data.submat(1,n,nbVarOut,n) = currPos;
    r.Data.submat(nbVarOut+1,n,2*nbVarOut,n) = currVel;
    r.Data.submat(2*nbVarOut+1,n,3*nbVarOut,n) = currAcc;
  }
  return r;
}

//--------------------------------------------------------------
vec pGMM::GaussPDF(mat M, vec Mu, mat Sigma){
  mat D_Tmp = trans(M) - repmat(trans(Mu),M.n_cols,1);
  mat invTmp;
  if (inv(invTmp,Sigma)){
    vec Probs = sum(D_Tmp%((D_Tmp)*(invTmp)), 1);
    Probs = exp(-0.5*Probs) / (sqrt(pow((2*PI),Sigma.n_cols) * (abs(det(Sigma))+REALMIN))+REALMIN);
    return Probs;
  }
  else{
    cout<<"Error: INV failed!!!2"<<endl;
    vec Probs(M.n_cols);
    Probs=REALMIN;
    return Probs;
  }
}

//--------------------------------------------------------------
double pGMM::GaussPDF(vec M, vec Mu, mat Sigma){
  mat D_Tmp = trans(M)-trans(Mu);
  mat invTmp=Sigma;
  if (inv(invTmp,Sigma)){
    double Probs = as_scalar(sum(D_Tmp%((D_Tmp)*(invTmp)), 1));
    Probs = exp(-0.5*Probs) / (sqrt(pow((2*PI),Sigma.n_cols) * (abs(det(Sigma))+REALMIN))+REALMIN);
    return Probs;
  }
  else{
    cout<<"Sigma ="<<endl<<Sigma<<endl;
    cout<<"Error: INV failed!!!11"<<endl;
    return REALMIN;
  }
}

//--------------------------------------------------------------
double pGMM::GaussPDF(double M, double Mu, double Sigma){
  double D_Tmp = M-Mu;
  double Probs = abs((D_Tmp*D_Tmp)/((Sigma)+REALMIN));
  Probs = exp(-0.5*Probs) / (sqrt(abs(pow((2*PI),model.nbVar)*(Sigma)))+REALMIN);
  return Probs;
}

//--------------------------------------------------------------
int pGMM::EMpGMMStandard(){
  if(model0.nbStates<1 || nbSamples<1)
    return -1;
  //Thresholds for the EM iterations
  int nbMaxSteps = 100;  //maximum # of iterations
  int nbMinSteps = 10;  //minimum # of iterations
  float maxDiffLL = 1E-50, sumTmp;
  float diagRegularizationFactor = 1E-4;
  int nbVarParams, nbData;
  vector <mat> M;
  vec Probs;
  int nTmp;
  mat LL_Tmp, sumTmp1;
  mat GAMMA_Tmp, DataTmp, LL(1,nbMaxSteps);
  mat MuTmp(model0.nbVar,1),SigmaTmp(2*model0.nbVar,2*model0.nbVar), SigmaTmp2(model0.nbVar,model0.nbVar);
  SigmaTmp = zeros(2*model0.nbVar,2*model0.nbVar);
  Probs = zeros(smpl0[0].nbData);
  nbVarParams = smpl0[0].OmegaMu.n_rows;
  nbData = 0;
  for (int n=0; n<nbSamples; n++){
    nbData += smpl0[n].nbData;
  }

  cout<<"START TO lEARN using Standard PGMM"<<endl;
  for(int nbIter=0;  nbIter<nbMaxSteps; nbIter++){
    // ------------ E step -----------------
    int j=0, k=0;
    for(int n=0; n<nbSamples;n++){
      smpl0[n].GAMMAO.set_size(model0.nbStates, smpl0[n].nbData);
      for (int i=0; i<model0.nbStates; i++){
        MuTmp = model0.Zmu[i]*smpl0[n].OmegaMu;
        Probs = GaussPDF(smpl0[n].Data, MuTmp, model0.Sigma[i]);
        smpl0[n].GAMMAO.row(i) = trans(model0.priors(i) * Probs);
      }
      k += smpl0[n].nbData;
      GAMMA.cols(j,k-1) = smpl0[n].GAMMAO;
      j = k;
    }
    //Normalization
    GAMMA = GAMMA/repmat(sum(GAMMA, 0)+REALMIN,model0.nbStates,1);
    //Data reshape
    nTmp = 0;
    for(int n=0;n<nbSamples;n++){
      smpl0[n].GAMMA = GAMMA.cols(nTmp, nTmp+smpl0[n].nbData-1);
      nTmp += smpl0[n].nbData;
    }
    // ----------- M step -------------
    for(int i=0;i<model0.nbStates;i++){
      //Update priors
      model0.priors(i)=sum(GAMMA.row(i))/nbData;
      //Update Zmu
      model0.Zmu[i] = zeros(model0.nbVar, nbVarParams);
      sumTmp1 = zeros(nbVarParams, nbVarParams);
      for (int n=0; n<nbSamples; n++){
        model0.Zmu[i] += (smpl0[n].Data * diagmat(smpl0[n].GAMMA.row(i))) * repmat(trans(smpl0[n].OmegaMu), smpl0[n].nbData,1);
        sumTmp1 += smpl0[n].OmegaMu * trans(smpl0[n].OmegaMu) * sum(smpl0[n].GAMMA.row(i));
      }
      model0.Zmu[i] *= pinv(sumTmp1 + eye(nbVarParams, nbVarParams)*diagRegularizationFactor);
      //Update Sigma
      model0.Sigma[i] = zeros(model0.nbVar, model0.nbVar);
      sumTmp = 0;
      for (int n=0; n<nbSamples; n++){
       MuTmp = model0.Zmu[i] * smpl0[n].OmegaMu;
       DataTmp = smpl0[n].Data - repmat(MuTmp, 1, smpl0[n].nbData);
       model0.Sigma[i] += DataTmp * diagmat(smpl0[n].GAMMA.row(i)) * trans(DataTmp);
       sumTmp += sum(smpl0[n].GAMMA.row(i));
      }
      model0.Sigma[i] = (model0.Sigma[i]+eye(model0.nbVar, model0.nbVar)*diagRegularizationFactor)/(sumTmp+REALMIN);
    }
    // ====== Computing the average log-likelihood through the ALPHA scaling factors
    LL(nbIter)=0;
    for(int n=0; n<nbSamples; n++){
      LL_Tmp = log(sum(smpl0[n].GAMMAO,0));
      LL(nbIter) += as_scalar(sum(LL_Tmp,1));
    }
    LL(nbIter)/= nbData;
    //Stop the algorithm if EM converged
    if(nbIter>nbMinSteps && (LL(nbIter)-LL(nbIter-1))<maxDiffLL){
      cout<<"EM converged after  "<< nbIter<<"  iterations."<<endl;
      break;
    }
    if(nbIter==nbMaxSteps-1)
      cout<<"The maximum number of  "<<nbMaxSteps<< "  EM iterations has been reached."<<endl;
  }
  return 1;
}

//--------------------------------------------------------------
int pGMM::saveModelStandard(string path){
  // Saving the parameters of the model in txt files in the given path
  mat ZMu = zeros(model0.nbVar, model0.Zmu[0].n_cols*model0.nbStates);
  mat Sigma = zeros(model0.nbVar, model0.nbVar*model0.nbStates);
  for(int i=0; i<model0.nbStates; i++){
    ZMu.cols(i*model0.Zmu[0].n_cols, (i+1)*model0.Zmu[0].n_cols -1) = model0.Zmu[i];
    Sigma.cols(i*model0.nbVar, (i+1)*model0.nbVar-1) = model0.Sigma[i];
  }
  ZMu.save(path + "/ZMuStandard.txt", raw_ascii);
  Sigma.save(path + "/SigmaStandard.txt", raw_ascii);
  model0.priors.save(path + "/priorsStandard.txt", raw_ascii);
  cout<<"Parameters of the standard PGMM model are saved now"<<endl;
  return 1;
}
//--------------------------------------------------------------
//int pGMM::init_proposedPGMM(){
int pGMM::init_PGMM(){
  model.kP = 150;
  model.kV = 20;
  model.dt = 0.01;
  if(model.nbStates<1 || nbSamples<1)
    return 0;
  model.priors.set_size(model.nbStates);
  model.ref.resize(model.nbFrames);
  GAMMA.set_size(model.nbStates, nbData_Tot);
  return 1;
}

//--------------------------------------------------------------
int pGMM::init_standardPGMM(){
  if(model.nbStates<1 || nbSamples<1)
    return 0;

  //model.priors.set_size(model.nbStates);
  //model.ref.resize(model.nbFrames);
  //GAMMA.set_size(model.nbStates, nbData_Tot);
  init_PGMM();

  smpl0.resize(nbSamples);
  for (int n = 0; n<nbSamples; n++){
    smpl0[n] = smpl[n];
    smpl0[n].OmegaMu << smpl[n].p[0].b(0) <<smpl[n].p[0].b(1) << smpl[n].angle << smpl[n].scaling << 1;
  }
  for (int m=0; m<model.nbFrames; m++){
    model.ref[m].ZMu.set_size(model.nbVar, model.nbStates);
    model.ref[m].ZSigma.resize(model.nbStates);
    for(int i=0;i<model.nbStates;i++){
      model.ref[m].ZMu.col(i) = randu(model.nbVar);
      model.ref[m].ZSigma[i] = eye(model.nbVar, model.nbVar);
    }
  }
  model0 = model;
  model0.Zmu.resize(model.nbStates);
  model0.Sigma.resize(model.nbStates);
  for (int i=0; i < model.nbStates; i++){
    model0.Zmu[i] = randu(model.nbVar, smpl0[0].OmegaMu.n_rows);
    model0.Sigma[i] = eye(model.nbVar, model.nbVar);
  }
  model0.priors = ones<vec>(model.nbStates)/model.nbStates;

  return 1;
}

//--------------------------------------------------------------
int pGMM::init_proposedPGMM_kmeans(){
  int k;
  KmeansOut kmeansCl;
  mat Mu, Sigma, Stmp, DataAll, DataTmp;
  vec Priors = zeros<vec>(model.nbStates), idList;
  Mu.set_size(model.nbVar, model.nbStates);
  DataAll = zeros(model.nbFrames*model.nbVar, nbData_Tot);
  Stmp = zeros(model.nbVar, model.nbStates*model.nbVar);
  idList = zeros(nbData_Tot,1);

  init_PGMM();

  for (int m=0; m<model.nbFrames; m++){
    DataTmp = zeros(model.nbVar, nbSamples*smpl[0].nbData);
    for(int n=0; n<nbSamples; n++){
      DataTmp.submat(0, n*smpl[n].nbData, model.nbVar-1, (n+1)*smpl[n].nbData -1) =
          inv(smpl[n].p[m*smpl[n].nbData].A)*(smpl[n].Data - repmat(smpl[n].p[m*smpl[n].nbData].b,1,smpl[n].nbData));
    }
    DataAll.submat(m*model.nbVar, 0, (m+1)*model.nbVar-1, nbData_Tot-1) = DataTmp;
  }

  kmeansCl = kmeansClust(DataAll, model.nbStates);
  Mu = kmeansCl.Centers;
  idList = kmeansCl.id;

  DataTmp = zeros(DataAll.n_rows, DataAll.n_cols*2);
  for(int i=0; i<model.nbStates; i++){
    k =0;
    for (int j=0; j<nbData_Tot; j++){
      if(idList(j)==i){
        DataTmp.col(k) = DataAll.col(j);
        k+=1;
      }
    }
    DataTmp.cols(k,2*k-1) = DataTmp.cols(0,k-1);
    Priors(i) = k;
    Stmp.submat(0, i*model.nbVar*model.nbFrames, model.nbFrames*model.nbVar-1, (i+1)*model.nbFrames*model.nbVar-1) =
          cov(trans(DataTmp.cols(0,2*k-1)));
  }
  Priors = Priors/sum(Priors);
  cout<<"Initial parameters for GMM are loaded now!!"<<endl;
  for (int m=0; m<model.nbFrames; m++){
    model.ref[m].ZSigma.resize(model.nbStates);
    //model.ref[m].ZMu = zeros(model.nbFrames*model.nbVar, model.nbStates); //There was a  bug here. ZMu should have model.nbVar rows
    model.ref[m].ZMu = zeros(model.nbVar, model.nbStates);
    for (int i=0; i<model.nbStates; i++){
      Sigma = Stmp.submat(0, i*model.nbVar, model.nbVar-1, (i+1)*model.nbVar-1);
      model.ref[m].ZMu.col(i) = Mu.submat(m*model.nbVar, i, (m+1)*model.nbVar-1, i);
      model.ref[m].ZSigma[i] = Sigma.submat(m*model.nbVar, m*model.nbVar, (m+1)*model.nbVar-1, (m+1)*model.nbVar-1);
    }
  }
  model.priors = Priors;
  return 1;
}

//--------------------------------------------------------------
KmeansOut pGMM::kmeansClust(mat Data, int nbClust){
// Implementing Kmeans clustering
  int nbMaxSteps = 100, k;  //maximum # of iterations
  float maxDiffDist = 1E-20;
  double cumDist, cumDistOld = -REALMAX;
  KmeansOut KMout;
  mat CTmp, distTmp, vTmp, vTmpCol;
  vTmpCol = zeros(nbData_Tot,1);
  distTmp = zeros(nbData_Tot, model.nbStates);
  vec idList = zeros(nbData_Tot,1);
  //Performing randperm to choose randomly some data point as initial centers
  uvec indices = sort_index(randn<vec>(nbData_Tot));
  mat Centers = Data.cols(indices.rows(0, nbClust-1));
  for(int nbIter=0; nbIter<nbMaxSteps; nbIter++){
    //Compute distances
    for (int i=0; i<nbClust; i++){
      distTmp.col(i) = trans(sum((Data-repmat(Centers.col(i),1,nbData_Tot))%(Data-repmat(Centers.col(i),1,nbData_Tot))));
    }
    vTmp = sort(distTmp,0,1);
    vTmpCol = vTmp.col(0);
    for(int i=0; i<nbClust; i++){
      k =0;
      CTmp = zeros(model.nbVar*model.nbFrames,1);
      for (int j=0; j<nbData_Tot; j++){
        uvec q = find(distTmp.row(j)==min(distTmp.row(j)));
        idList(j) = as_scalar(q(0));
        if(idList(j)==i){
          k +=1;
          CTmp += Data.col(j);
        }
      }
      Centers.col(i) = CTmp/k;
    }
    cumDist = as_scalar(sum(vTmpCol));
    //Stop the algorithm if EM converged
    if(abs(cumDist-cumDistOld)<maxDiffDist){
      cout<<"Kmeans initialization converged after  "<< nbIter<<"  iterations."<<endl;
      break;
    }
    cumDistOld = cumDist;
    if(nbIter==nbMaxSteps-1)
      cout<<"The maximum number of  "<<nbMaxSteps<< "  iterations has been reached for Kmenas initialization."<<endl;
  }
  KMout.id = idList;
  KMout.Centers = Centers;
  return KMout;
}

//--------------------------------------------------------------
int pGMM::init_proposedPGMM_timeBased(){
  //Initialization of the model with clusters equally split in time.
  mat Mu, Sigma, Stmp, DataAll, DataTmp, M, MuTmp;
  mat p = zeros(model.nbStates,1);
  vec Priors = zeros<vec>(model.nbStates), TimingSep = zeros<vec>(model.nbStates+1);
  DataAll = zeros(model.nbFrames*model.nbVar, nbData_Tot);
  Mu.set_size(model.nbVar*model.nbFrames, model.nbStates);
  Stmp = zeros(model.nbVar*model.nbFrames, model.nbStates*model.nbVar*model.nbFrames);

  init_PGMM();

  for (int m=0; m<model.nbFrames; m++){
    DataTmp = zeros(model.nbVar, nbSamples*smpl[0].nbData);
    for(int n=0; n<nbSamples; n++){
      DataTmp.submat(0, n*smpl[n].nbData, model.nbVar-1, (n+1)*smpl[n].nbData -1) =
          inv(smpl[n].p[m*smpl[n].nbData].A)*(smpl[n].Data - repmat(smpl[n].p[m*smpl[n].nbData].b,1,smpl[n].nbData));
    }
    DataAll.submat(m*model.nbVar, 0, (m+1)*model.nbVar-1, nbData_Tot-1) = DataTmp;
  }
  TimingSep = linspace(min(DataAll.row(0)) ,max(DataAll.row(0)) , model.nbStates+1);
  vec IdTmp=zeros<vec>(nbData_Tot);
  int k;
  DataTmp = zeros(DataAll.n_rows, DataAll.n_cols);
  for (int i=0; i<model.nbStates; i++){
    MuTmp = zeros(model.nbVar*model.nbFrames,1);
    k =0;
    for (int j=0; j<nbData_Tot; j++){
      if(DataAll(0,j)>=TimingSep(i)&&DataAll(0,j)<TimingSep(i+1)){
        IdTmp(k) = j;
        DataTmp.col(k) = DataAll.col(j);
        MuTmp += DataAll.col(j);
        k+=1;
      }
    }
    Mu.col(i) = MuTmp/k;
    Stmp.submat(0, i*model.nbVar*model.nbFrames, model.nbFrames*model.nbVar-1, (i+1)*model.nbFrames*model.nbVar-1) =
          cov(trans(DataTmp.cols(0,k-1))) + 1E-6*eye(model.nbVar*model.nbFrames,model.nbVar*model.nbFrames);
    Priors(i) = k;
  }
  Priors = Priors/sum(Priors);
  for (int m=0; m<model.nbFrames; m++){
    model.ref[m].ZSigma.resize(model.nbStates);
    model.ref[m].ZMu = zeros(model.nbVar, model.nbStates);
    for (int i=0; i<model.nbStates; i++){
      Sigma = Stmp.submat(0, i*model.nbVar*model.nbFrames, model.nbFrames*model.nbVar-1, (i+1)*model.nbFrames*model.nbVar-1);
      model.ref[m].ZMu.col(i) = Mu.submat(m*model.nbVar, i, (m+1)*model.nbVar-1, i);
      model.ref[m].ZSigma[i] = Sigma.submat(m*model.nbVar, m*model.nbVar, (m+1)*model.nbVar-1, (m+1)*model.nbVar-1);
      model.ref[m].ZSigma[i] += 1E-6*eye(model.ref[m].ZSigma[i].n_rows, model.ref[m].ZSigma[i].n_cols);
    }
  }
  model.priors = Priors;
  return 1;
}

//--------------------------------------------------------------
int pGMM::EMpGMMProposed(){
  if(model.nbStates<1 || nbSamples<1)
    return -1;
   //Thresholds for the EM iterations in the Proposed PGMM approach
  int nbMaxSteps = 25;  //maximum # of iterations
  int nbMinSteps = 5;  //minimum # of iterations
  float maxDiffLL = 1E-50, sumTmp;
  int nbVarParams, nbData, ind;
  vector <mat> M;
  cResGaussOut crgOut;
  vector <ProdR> pr;
  int nTmp;
  mat LL_Tmp, sumTmp1;
  mat GAMMA_Tmp, GAMMAO, DataTmp, LL(1,nbMaxSteps);
  mat MuTmp(model.nbVar,1);
  nbVarParams = smpl[0].OmegaMu.n_rows;
  nbData = 0;
  for (int n=0; n<nbSamples; n++){
    nbData += smpl[n].nbData;
  }
  cout<<"Start to learn using Proposed PGMM"<<endl;
  for(int nbIter=0;  nbIter<nbMaxSteps; nbIter++){
    // ------------ E step -----------------
    int j=0, k=0;
    GAMMAO = zeros(GAMMA.n_rows, GAMMA.n_cols);
    for(int n=0; n<nbSamples;n++){
      crgOut = computeResultingGaussians(smpl[n].p);
      pr = crgOut.prodr;
      smpl[n].GAMMAO.set_size(model.nbStates, smpl[n].nbData);
      for(int i=0; i<model.nbStates; i++){
        if (smpl[n].p.size()==1){   //frame is fixed during each demonstration
          smpl[n].GAMMAO.row(i) = model.priors(i) * trans(GaussPDFfast(smpl[n].Data, pr[0].mu.col(i), pr[0].invSigma[i], pr[0].detSigma(i)));
        }
        else{     //frame is moving during demonstration
          for (int t=0; t<smpl[n].nbData; t++){
            smpl[n].GAMMAO(i,t) = model.priors(i)*as_scalar(GaussPDFfast(smpl[n].Data.col(t), pr[t].mu.col(i), pr[t].invSigma[i], pr[t].detSigma(i)));
          }
        }
      }
      k += smpl[n].nbData;
      GAMMAO.cols(j,k-1) = smpl[n].GAMMAO;
      j = k;
    }
    //Normalization
    GAMMA = GAMMAO/repmat(sum(GAMMAO, 0)+REALMIN,model.nbStates,1);
    //Data reshape
    nTmp = 0;
    for(int n=0;n<nbSamples;n++){
      smpl[n].GAMMA = GAMMA.cols(nTmp, nTmp+smpl[n].nbData-1);
      nTmp += smpl[n].nbData;
    }
    // ----------- M step -------------
    model.Zmu.resize(model.nbStates);
    for(int i=0;i<model.nbStates;i++){
      //Update priors
      model.priors(i)=sum(GAMMA.row(i))/nbData;
      //Update Zmu  (see Eq. (4) Calinon et al, Humanoids'2012 paper)
      for(int m=0; m<model.nbFrames; m++){
        model.Zmu[i] = zeros(model.nbVar, nbVarParams);
        sumTmp = 0;
        for (int n=0; n<nbSamples; n++){
          if (smpl[n].p.size()==1){   //frame is fixed during each demonstration
            DataTmp = inv(smpl[n].p[m].A) * (smpl[n].Data - repmat(smpl[n].p[m].b, 1, smpl[n].nbData));
          }
          else{   //frame is moving during demonstration
            DataTmp = zeros(model.nbVar, smpl[n].nbData);
            for (int t=0; t<smpl[n].nbData; t++){
              DataTmp.col(t) = smpl[n].p[t + m*smpl[n].p.size()/model.nbFrames].invA *
              (smpl[n].Data.col(t) - smpl[n].p[t + m*smpl[n].p.size()/model.nbFrames].b);
            }
          }
          //Update
          model.ref[m].ZMu.col(i) += DataTmp* trans(smpl[n].GAMMA.row(i));
          sumTmp += sum(smpl[n].GAMMA.row(i));
        }
        model.ref[m].ZMu.col(i) /= (sumTmp);
      }
      //Update Zcov (see Eq. (4) Calinon et al, Humanoids'2012 paper)
      for (int m=0; m<model.nbFrames; m++){
        model.ref[m].ZSigma[i] = zeros(model.nbVar, model.nbVar);
        sumTmp = 0;
        for (int n=0; n<nbSamples; n++){
          if (smpl[n].p.size()==1){   //frame is fixed during each demonstration
            MuTmp = smpl[n].p[m].A * model.ref[m].ZMu.col(i) + smpl[n].p[m].b;
            DataTmp = (inv(smpl[n].p[m].A) * (smpl[n].Data - repmat(MuTmp,1,smpl[n].nbData)));
          }
          else{   //frame is moving during demonstration
            DataTmp = zeros(model.nbVar, smpl[n].nbData);
            for (int t=0; t<smpl[n].nbData; t++){
              ind = t + m*smpl[n].nbData;
              MuTmp = smpl[n].p[ind].A * model.ref[m].ZMu.col(i) + smpl[n].p[ind].b;
              DataTmp.col(t) = smpl[n].p[ind].invA * (smpl[n].Data.col(t) - MuTmp);
            }
          }
          //Update
          model.ref[m].ZSigma[i] += DataTmp * diagmat(smpl[n].GAMMA.row(i)) * trans(DataTmp);
          sumTmp += sum(smpl[n].GAMMA.row(i));
        }
        model.ref[m].ZSigma[i] /= (sumTmp);
        //Regularization term (optional)
        model.ref[m].ZSigma[i] += eye(model.ref[m].ZSigma[i].n_rows,model.ref[m].ZSigma[i].n_cols)*1E-6;
      }
    }
    // ====== Computing the average log-likelihood through the ALPHA scaling factors
    LL(nbIter)=0;
    for(int n=0; n<nbSamples; n++){
      LL_Tmp = log(sum(smpl[n].GAMMAO,0));
      LL(nbIter) += as_scalar(sum(LL_Tmp,1));
    }
    LL(nbIter)/= nbData;
    //Stop the algorithm if EM converged
    if(nbIter>nbMinSteps && (LL(nbIter)-LL(nbIter-1))<maxDiffLL){
      cout<<"EM converged after  "<< nbIter<<"  iterations."<<endl;
      break;
    }
    if(nbIter==nbMaxSteps-1)
      cout<<"The maximum number of  "<<nbMaxSteps<< "  EM iterations has been reached."<<endl;
  }
  return 1;
}

//--------------------------------------------------------------
int pGMM::saveModelProposed(string path){
  //Saving the parameters of the model in txt files in the given path
  mat ZMu = zeros(model.nbVar, model.nbFrames*model.nbStates);
  mat ZSigma = zeros(model.nbVar, model.nbVar*model.nbStates*model.nbFrames);
  for(int m=0; m<model.nbFrames; m++){
    ZMu.cols(m*model.nbStates, (m+1)*model.nbStates-1) = model.ref[m].ZMu;
    for(int i=0; i<model.nbStates; i++){
      ZSigma.cols((i+model.nbStates*m)*model.nbVar, (i+model.nbStates*m+1)*model.nbVar-1) = model.ref[m].ZSigma[i];
    }
  }
  ZMu.save(path + "/ZMuProposed.txt", raw_ascii);
  ZSigma.save(path + "/ZSigmaProposed.txt", raw_ascii);
  model.priors.save(path + "/priorsProposed.txt", raw_ascii);
  cout<<"Parameters of the proposed PGMM model are saved"<<endl;
  return 1;
}

//--------------------------------------------------------------
vec pGMM::GaussPDFfast(mat D, vec Mu, mat invSig, double detSig){
  vec prob;
  int nbData = D.n_cols;
  D = trans(D) - repmat(trans(Mu), nbData,1);
  prob = sum(D*invSig%D,1);
  prob = exp(-.5*prob)/sqrt(pow(2*PI,model.nbVar) *(detSig+REALMIN));
  return prob;
}

//--------------------------------------------------------------
cResGaussOut pGMM::computeResultingGaussians(vector<Params> p){
  cResGaussOut resGout;
  vector <ProdR> pr;
  mat SigmaTmp, MuTmp;
  int ind;
  //Projections in the different candidate frames (see Eq. (3) Calinon et al, Humanoids'2012 paper)
  pr.resize(p.size());
  for(int t=0; t<p.size()/model.nbFrames; t++){
    for (int i=0; i<model.nbStates; i++){
      for (int m=0; m<model.nbFrames; m++){
        ind = t + m*p.size()/model.nbFrames;
        p[ind].Mu.set_size(model.nbVar, model.nbStates);
        p[ind].Sigma.resize(model.nbStates);
        p[ind].Mu.col(i) = p[ind].A * model.ref[m].ZMu.col(i) + p[ind].b;
        p[ind].Sigma[i] = p[ind].A * model.ref[m].ZSigma[i] * trans(p[ind].A);
      }
    }
  }
  //GMM products (see Eq. (3) Calinon et al, Humanoids'2012 paper)
  for (int t=0; t<p.size()/model.nbFrames; t++){
    pr[t].invSigma.resize(model.nbStates);
    pr[t].Sigma.resize(model.nbStates);
    pr[t].detSigma = zeros(model.nbStates);
    pr[t].mu = zeros(model.nbVar, model.nbStates);
    for (int i=0; i<model.nbStates; i++){
      SigmaTmp = zeros(model.nbVar, model.nbVar);
      MuTmp = zeros(model.nbVar,1);
      for (int m=0; m<model.nbFrames; m++){
        ind = t + m*p.size()/model.nbFrames;
        p[ind].Mu.col(i) = p[ind].A * model.ref[m].ZMu.col(i) + p[ind].b;
        p[ind].Sigma[i] = p[ind].A * model.ref[m].ZSigma[i] * trans(p[ind].A);
        SigmaTmp += inv(p[ind].Sigma[i]);
        MuTmp += inv(p[ind].Sigma[i])*p[ind].Mu.col(i);
      }
      pr[t].invSigma[i] = SigmaTmp;
      pr[t].Sigma[i] = inv(SigmaTmp);
      pr[t].detSigma(i) = det(pr[t].Sigma[i]);
      pr[t].mu.col(i) = pr[t].Sigma[i] * MuTmp;
    }
  }
  resGout.prodr = pr;
  resGout.pp = p;
  return resGout;
}

//--------------------------------------------------------------
int pGMM::repro_new(string path){
  Sample smplTmp;
  mat ParamM;
  stringstream pathTmp;

  for (int i=0; i<4; i++){
    vec id=floor(randu(2)*nbSamples);
    vec w=randu(2); w=w/sum(w);
    if(nbSamples>1){
      pathTmp<<nbRSamples+1;
      smplTmp.ParamM = smpl[id(0)].ParamM*w(0) + smpl[id(1)].ParamM*w(1);
      smplTmp.ParamM.save(path + "/param_rep_new0" + pathTmp.str() + ".txt",raw_ascii);
      createParams(smplTmp.ParamM,2,1);
      repro(id, w, path);
      pathTmp.str("");
    }
  }
  return 0;
}

