// Training of a task-parameterized Gaussian mixture model (GMM) based on candidate frames of reference.
// The proposed task-parameterized GMM approach relies on the linear transformation and product properties of
// Gaussian distributions to derive an expectation-maximization (EM) algorithm to train the model.
// The proposed approach is contrasted with an implementation of the approach proposed by Wilson and Bobick in
// 1999, with an implementation applied to GMM (that we will call PGMM) and following the model described in
// "Parametric Hidden Markov Models for Gesture Recognition", IEEE Trans. on Pattern Analysis and Machine
// Intelligence.
// In contrast to the standard PGMM approach, the new approach that we propose allows the parameterization of
// both the centers and covariance matrices of the Gaussians. It has been designed for targeting problems in
// which the task parameters can be represented in the form of coordinate systems, which is for example the
// case in robot manipulation problems.
//
// Authors:	Tohid Alizadeh and Sylvain Calinon, 2012
//         	http://programming-by-demonstration.org/
//
// This source code is given for free! In exchange, we would be grateful if you cite
// the following reference in any academic publication that uses this code or part of it:
//
// @inproceedings{Calinon12Hum,
//   author="Calinon, S. and Li, Z. and Alizadeh, T. and Tsagarakis, N. G. and Caldwell, D. G.",
//   title="Statistical dynamical systems for skills acquisition in humanoids",
//   booktitle="Proc. {IEEE} Intl Conf. on Humanoid Robots ({H}umanoids)",
//   year="2012",
//   address="Osaka, Japan"
// }

#ifndef _PGMM
#define _PGMM

#include <string.h>
#include <cstdlib>
#include <cstdio>
#include <sys/time.h>
#include <iostream>

#include "armadillo" //use #include </usr/include/armadillo> if the compiler could not find the library.
#define REALMIN 2.2251e-200
#define REALMAX 1.7977e200
#define PI 3.14

using namespace arma;
using namespace std;

struct Params{
  vec b;
  mat A, E, D, Mu, invA;
  vector <mat> Sigma;
};

struct ProdR{
  mat mu;
  vec detSigma;
  vector<mat> invSigma, Sigma;
};

struct Sample{
  int nbData;
  double scaling, angle;
  mat Data, GAMMAO, GAMMA, rData, detTraj, ParamM, Data0, Mu, H;
  field<mat> S;
  vector<mat> M, Sigma;
  vector<ProdR> prodRes;
  vector<Params> p;
  vec DataParam, OmegaMu;
};
struct refer{
  mat ZMu;
  vector<mat> ZSigma;
};

struct Model{
  int nbStates, nbFrames, nbVar;
  double kP, kV, dt;
  vec priors;
  vector<refer> ref;
  vector<mat> ZSigma, Zmu, Sigma;
};

struct cResGaussOut{
  vector<ProdR> prodr;
  vector<Params> pp;
};

struct KmeansOut{   //This structure will contain the outputs of kmeansclustering which are ID and Centers
  vec id;
  mat Centers;
};

class pGMM{
  public:
    pGMM();
    pGMM(int n, int f, int s);
    int createParams(mat paramM, int ParamType, int flag=0);
    void GaussPDF();
    vec GaussPDF(mat M, vec Mu, mat Sigma);
    double GaussPDF(vec M, vec Mu, mat Sigma);
    double GaussPDF(double M, double Mu, double Sigma);
    vec GaussPDFfast(mat D, vec Mu, mat invSig, double detSig);

    int EMpGMMStandard();
    int saveModelStandard(string path);
    int EMpGMMProposed();
    int saveModelProposed(string path);
    int init_proposedPGMM_kmeans();
    int init_proposedPGMM_timeBased();
    cResGaussOut computeResultingGaussians(vector<Params> p);
    Sample reproduction_DSGMR(mat DataIn, Sample rr, vec currPos);
    KmeansOut kmeansClust(mat Data, int nbClust);

    int init_PGMM();
    int init_standardPGMM();
    int init_proposedPGMM();

    int load(string folderName, int nbDemo, int ParamType);
    //int learn();
    int repro_new(string path);
    int repro(vec id, vec w, string path);
    int repro(string path);
    int addDemo(mat DataM);

    mat Data,GAMMA;
    vector<Sample> smpl, rsmpl, smpl0, rnew;
    int nbSamples, nbData_Tot, nbRSamples;

    Model model, model0;
  private:
};
#endif
