function demo1
%
% Training of a task-parameterized Gaussian mixture model (GMM) based on candidate frames of reference. 
% The proposed task-parameterized GMM approach relies on the linear transformation and product properties of
% Gaussian distributions to derive an expectation-maximization (EM) algorithm to train the model. 
% The proposed approach is contrasted with an implementation of the approach proposed by Wilson and Bobick in
% 1999, with an implementation applied to GMM (that we will call PGMM) and following the model described in 
% "Parametric Hidden Markov Models for Gesture Recognition", IEEE Trans. on Pattern Analysis and Machine 
% Intelligence. 
% In contrast to the standard PGMM approach, the new approach that we propose allows the parameterization of 
% both the centers and covariance matrices of the Gaussians. It has been designed for targeting problems in
% which the task parameters can be represented in the form of coordinate systems, which is for example the
% case in robot manipulation problems.  
%
% DEMO1: Simple example of task-parameterized GMM learning and comparison with standard PGMM
%
% This example uses 3 trajectories demonstrated in a frame of reference that varies from one demonstration 
% to the other. A model of 3 Gaussian components is used to encode the data in the different frames, by 
% providing the parameters of the corrdinate systems as inputs (transformation matrix A and offset vector b).   
%
% Author:	Sylvain Calinon, 2013
%         http://programming-by-demonstration.org/SylvainCalinon
%
% This source code is given for free! In exchange, I would be grateful if you cite  
% the following reference in any academic publication that uses this code or part of it: 
%
% @inproceedings{Calinon12Hum,
%   author="Calinon, S. and Li, Z. and Alizadeh, T. and Tsagarakis, N. G. and Caldwell, D. G.",
%   title="Statistical dynamical systems for skills acquisition in humanoids",
%   booktitle="Proc. {IEEE} Intl Conf. on Humanoid Robots ({H}umanoids)",
%   year="2012",
%   address="Osaka, Japan",
%   pages="323--329"
% }

%% Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.nbStates = 3; %Number of components in the model
model.nbFrames = 1; %Number of candidate frames of reference
model.nbVar = 2; %Dimension of the datapoints in the dataset (here: x1,x2)

%% Load data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The MAT file below contains a matrix 'Data' containing the raw data and a structure 's' containing the different
% demonstration samples. 's(n).Data' contains the data for sample n (with 's(n).nbData' datapoints).
% 's(n).p(m).b' and 's(n).p(m).A' contain the position and transformation matrix of the m-th candidate coordinate system
% of sample n. In this example, the transformation matrices represent rotated and homogeneoulsy scaled
% canonical systems, which can alternatively be represented in the 2D case as a rotation angle 's(n).angle'
% and scaling factor 's(n).scaling'.
load('data/Data.mat');
nbSamples = length(s); %Number of samples in the MAT file

%% Parameters estimation for the standard PGMM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('Parameters estimation of standard PGMM:');
%Use the same model parameters and dataset
model0 = model;
s0 = s;

%Definition of task parameters
for n=1:nbSamples
  %Parameterization with rotation angle and scaling factor
  s0(n).OmegaMu = [s(n).p(1).b; s(n).angle; s(n).scaling; 1]; 

%     %Parameterization with the elements of the tranformation matrix
%     s0(n).OmegaMu = [s(n).p(1).b; reshape(s0(n).p(1).A,model.nbVar^2,1); 1]; 

end

%Random initialization of the Zmu parameters 
%(Sigma initialized as large circular covariance matrix, and Priors with equal coefficients) 
for i=1:model.nbStates
  model0.ZMu(:,:,i) = rand(model.nbVar,size(s0(1).OmegaMu,1));
  model0.Sigma(:,:,i) = eye(model.nbVar);
end
model0.Priors = ones(1,model.nbStates)/model.nbStates;

%   %Initialization assuming that the task parameters do not influence the GMM parameters
%   [Priors0,Mu0,Sigma0] = EM_init_kmeans(Data,model.nbStates);
%   [Mu0,Sigma0,Priors0] = EM_GMM(s0,Mu0,Sigma0,Priors0);
%   for i=1:model.nbStates
%     model0.ZMu(:,:,i) = zeros(model.nbVar,size(s0(1).OmegaMu,1));
%     model0.ZMu(:,end,i) = Mu0(:,i);
%     model0.Sigma(:,:,i) = Sigma0(:,:,i);
%   end

[s0, model0] = EM_standardPGMM(s0, model0);
%Computation of the resulting Gaussians for each sample (for display purpose)
for n=1:nbSamples
  for i=1:model0.nbStates
    s0(n).Mu(:,i) = model0.ZMu(:,:,i) * s0(n).OmegaMu;
  end
  s0(n).Sigma = model0.Sigma;
end

%% Parameters estimation for the proposed PGMM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('Parameters estimation of proposed PGMM:');
model = init_proposedPGMM_kmeans(s, model); %Initialization with k-means clustering
[s, model, LL] = EM_proposedPGMM(s, model);
%Computation of the resulting Gaussians for each sample (for display purpose)
for n=1:nbSamples
  %Computes resulting Gaussians after projections and products of Gaussians estimation
  s(n).prodRes = computeResultingGaussians(model, s(n).p);
end

%% Plots
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure('position',[20,50,1100,700]);
%posList = [-0.9 0.4 -0.3; -0.9 -0.3 0.3];
limAxes = [-1 1 -1 1];
if model.nbStates==3
  colGMM = eye(3)*0.5+0.5;
else
  colGMM = rand(model.nbStates,3);
end

%Plot standard PGMM
subplot(1,2,1); hold on; box on; title('Standard PGMM');
for n=1:nbSamples
  for i=1:model0.nbStates
    plotGMM(s0(n).Mu(:,i), s0(n).Sigma(:,:,i), colGMM(i,:), 1);  
  end
  for m=1:model.nbFrames
    draw2DArrow(s(n).p(m).b, s(n).p(m).A(:,1), [.6 .6 .6]);
    draw2DArrow(s(n).p(m).b, s(n).p(m).A(:,2), [.6 .6 .6]);
    plot(s(n).p(m).b(1),s(n).p(m).b(2),'.','markersize',20,'color',[.6 .6 .6]);
  end
  plot(s(n).Data(1,:), s(n).Data(2,:),'-','linewidth',1.5,'color',[.2 .2 .2]);
end
axis(limAxes); axis square; set(gca,'xtick',[],'ytick',[]);

%Plot proposed PGMM
subplot(1,2,2); hold on; box on; title('Proposed PGMM');
for n=1:nbSamples
  for i=1:model.nbStates
    plotGMM(s(n).prodRes(1).Mu(:,i), s(n).prodRes(1).Sigma(:,:,i), colGMM(i,:), 1);  
  end
  for m=1:model.nbFrames
    draw2DArrow(s(n).p(m).b, s(n).p(m).A(:,1), [.6 .6 .6]);
    draw2DArrow(s(n).p(m).b, s(n).p(m).A(:,2), [.6 .6 .6]);
    plot(s(n).p(m).b(1),s(n).p(m).b(2),'.','markersize',20,'color',[.6 .6 .6]);
  end
  plot(s(n).Data(1,:), s(n).Data(2,:),'-','linewidth',1.5,'color',[.2 .2 .2]);
end
axis(limAxes); axis square; set(gca,'xtick',[],'ytick',[]);

%print('-dpng','demo1.png'); 

