function [s, model, LL] = EM_proposedPGMM(s, model)
%
% Training of a task-parameterized Gaussian mixture model (GMM) based on candidate frames of reference. 
% The proposed task-parameterized GMM approach relies on the linear transformation and product properties of
% Gaussian distributions to derive an expectation-maximization (EM) algorithm to train the model. 
% The proposed approach is contrasted with an implementation of the approach proposed by Wilson and Bobick in
% 1999, with an implementation applied to GMM (that we will call PGMM) and following the model described in 
% "Parametric Hidden Markov Models for Gesture Recognition", IEEE Trans. on Pattern Analysis and Machine 
% Intelligence. 
% In contrast to the standard PGMM approach, the new approach that we propose allows the parameterization of 
% both the centers and covariance matrices of the Gaussians. It has been designed for targeting problems in
% which the task parameters can be represented in the form of coordinate systems, which is for example the
% case in robot manipulation problems.   
%
% Author:	Sylvain Calinon, 2013
%         http://programming-by-demonstration.org/SylvainCalinon
%
% This source code is given for free! In exchange, I would be grateful if you cite  
% the following reference in any academic publication that uses this code or part of it: 
%
% @inproceedings{Calinon12Hum,
%   author="Calinon, S. and Li, Z. and Alizadeh, T. and Tsagarakis, N. G. and Caldwell, D. G.",
%   title="Statistical dynamical systems for skills acquisition in humanoids",
%   booktitle="Proc. {IEEE} Intl Conf. on Humanoid Robots ({H}umanoids)",
%   year="2012",
%   address="Osaka, Japan"
% }

%Parameters of the EM algorithm
nbMinSteps = 5; %Minimum number of iterations allowed
nbMaxSteps = 25; %Maximum number of iterations allowed
maxDiffLL = 1E-8; %Likelihood increase threshold to stop the algorithm

nbSamples = length(s);
nbData=0;
for n=1:nbSamples
  nbData = nbData + s(n).nbData;
end

for nbIter=1:nbMaxSteps
  fprintf('.');
  
  %E-step
  [s, GAMMA] = computeGamma(s, model); %See 'computeGamma' function below 
  
  %M-step
  for i=1:model.nbStates 
    
    %Update Priors
    model.Priors(i) = sum(GAMMA(i,:))/nbData;
    
    %Update Zmu (see Eq. (4) Calinon et al, Humanoids'2012 paper)
    for m=1:model.nbFrames
      model.ref(m).ZMu(:,i)=0; sumTmp=0;
      for n=1:nbSamples
        if size(s(n).p,2)==1 %frame is fixed during each demonstration 
          DataTmp = inv(s(n).p(m).A) * (s(n).Data - repmat(s(n).p(m).b,1,s(n).nbData)); 
        else %frame is moving during demonstration
          for t=1:s(n).nbData
            DataTmp(:,t) = s(n).p(m,t).invA * (s(n).Data(:,t) - s(n).p(m,t).b);
          end
        end
        %Update
        model.ref(m).ZMu(:,i) = model.ref(m).ZMu(:,i) + DataTmp * s(n).GAMMA(i,:)'; 
        sumTmp = sumTmp + sum(s(n).GAMMA(i,:));
      end
      model.ref(m).ZMu(:,i) = model.ref(m).ZMu(:,i)/sumTmp;
    end

    %Update Zcov (see Eq. (4) Calinon et al, Humanoids'2012 paper)
    for m=1:model.nbFrames
      model.ref(m).ZSigma(:,:,i)=0; sumTmp=0;
      for n=1:nbSamples
        if size(s(n).p,2)==1 %frame is fixed during each demonstration 
          MuTmp = s(n).p(m).A * model.ref(m).ZMu(:,i) + s(n).p(m).b;
          DataTmp = (inv(s(n).p(m).A) * (s(n).Data - repmat(MuTmp,1,s(n).nbData)));
        else %frame is moving during demonstration
          DataTmp = [];
          for t=1:s(n).nbData
            MuTmp = s(n).p(m,t).A * model.ref(m).ZMu(:,i) + s(n).p(m,t).b;
            DataTmp(:,t) = s(n).p(m,t).invA * (s(n).Data(:,t) - MuTmp);
          end
        end
        %Update
        model.ref(m).ZSigma(:,:,i) = model.ref(m).ZSigma(:,:,i) + DataTmp * diag(s(n).GAMMA(i,:)) * DataTmp';
        sumTmp = sumTmp + sum(s(n).GAMMA(i,:));
      end         
      model.ref(m).ZSigma(:,:,i) = model.ref(m).ZSigma(:,:,i)/sumTmp;
      %Regularization term (optional)
      model.ref(m).ZSigma(:,:,i) = model.ref(m).ZSigma(:,:,i) + eye(size(model.ref(m).ZSigma(:,:,i)))*1E-6;
    end

  end

  %Compute the average log-likelihood through the ALPHA scaling factors
  LL(nbIter)=0; sz=0;
  for n=1:nbSamples
    LL(nbIter) = LL(nbIter) + sum(log(sum(s(n).GAMMA0,1)));
    sz = sz + size(s(n).GAMMA0,2);
  end
  LL(nbIter) = LL(nbIter)/sz;
  %Stop the algorithm if EM converged (small change of LL)
  if nbIter>nbMinSteps
    if LL(nbIter)-LL(nbIter-1)<maxDiffLL || nbIter==nbMaxSteps-1
      disp(['EM converged after ' num2str(nbIter) ' iterations.']); 
      return;
    end
  end
end
disp(['The maximum number of ' num2str(nbMaxSteps) ' EM iterations has been reached.']); 

end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [s, GAMMA] = computeGamma(s, model)
  nbSamples = length(s); 
  nbStates = size(model.ref(1).ZSigma,3); 
  nbVar = size(s(1).Data,1);
  %Observation probabilities 
  GAMMA0=[]; 
  for n=1:nbSamples
    %Computes resulting Gaussians after projections and products of Gaussians estimation
    prodRes = computeResultingGaussians(model, s(n).p);
    for i=1:nbStates
      if size(s(n).p,2)==1 %frame is fixed during each demonstration 
        s(n).GAMMA0(i,:) = model.Priors(i) * gaussPDFfast(s(n).Data, prodRes(1).Mu(:,i), prodRes(1).invSigma(:,:,i), prodRes(1).detSigma(i));
      else %frame is moving during demonstration
        for t=1:s(n).nbData
          s(n).GAMMA0(i,t) = model.Priors(i) * gaussPDFfast(s(n).Data(:,t), prodRes(t).Mu(:,i), prodRes(t).invSigma(:,:,i), prodRes(t).detSigma(i));
          %s(n).GAMMA0(i,t) = model.Priors(i) * gaussPDF(s(n).Data(:,t), prodRes(t).Mu(:,i), prodRes(t).Sigma(:,:,i));
          for m=1:model.nbFrames
            DataTmp = inv(s(n).p(m,t).A) * (s(n).Data(:,t) - s(n).p(m,t).b);
            P(m) = model.Priors(i) * gaussPDF(DataTmp, model.ref(m).ZMu(:,i), model.ref(m).ZSigma(:,:,i));
            d(m) = 1/det(model.ref(m).ZSigma(:,:,i));
          end
        end
      end
    end
    GAMMA0 = [GAMMA0 s(n).GAMMA0];
  end
  %Normalization
  GAMMA = GAMMA0 ./ repmat(sum(GAMMA0,1)+realmin,size(GAMMA0,1),1);
  %Data reshape
  nTmp=1;
  for n=1:nbSamples
    s(n).GAMMA = GAMMA(:,[nTmp:nTmp+size(s(n).GAMMA0,2)-1]);
    nTmp = nTmp+size(s(n).GAMMA,2);
  end
end




