function demo2_movingFrames
%
% Training of a task-parameterized Gaussian mixture model (GMM) based on candidate frames of reference. 
% The proposed task-parameterized GMM approach relies on the linear transformation and product properties of
% Gaussian distributions to derive an expectation-maximization (EM) algorithm to train the model. 
% The proposed approach is contrasted with an implementation of the approach proposed by Wilson and Bobick in
% 1999, with an implementation applied to GMM (that we will call PGMM) and following the model described in 
% "Parametric Hidden Markov Models for Gesture Recognition", IEEE Trans. on Pattern Analysis and Machine 
% Intelligence. 
% In contrast to the standard PGMM approach, the new approach that we propose allows the parameterization of 
% both the centers and covariance matrices of the Gaussians. It has been designed for targeting problems in
% which the task parameters can be represented in the form of coordinate systems, which is for example the
% case in robot manipulation problems.
%
% DEMO2: Example of task-parameterized movement learning with DS-GMR (statistical dynamical systems based on
% Gaussian mixture regression)  
% This demo shows how the approach can be combined with the DS-GMR model to learn movements modulated 
% with respect to different frames of reference. The DS-GMR model is a statistical dynamical system approach 
% to learn and reproduce movements with a superposition of virtual spring-damper systems
% retrieved by Gaussian mixture regression (GMR). For more details, see the 'DMP-learned-by-GMR-v1.0' example 
% code downloadable from the website below. This example considers the general case of moving frames (see also 
% demo2_inanimateFrames for faster computation).
%
% Author:	Sylvain Calinon, 2013
%         http://programming-by-demonstration.org/SylvainCalinon
%
% This source code is given for free! In exchange, I would be grateful if you cite  
% the following reference in any academic publication that uses this code or part of it: 
%
% @inproceedings{Calinon12Hum,
%   author="Calinon, S. and Li, Z. and Alizadeh, T. and Tsagarakis, N. G. and Caldwell, D. G.",
%   title="Statistical dynamical systems for skills acquisition in humanoids",
%   booktitle="Proc. {IEEE} Intl Conf. on Humanoid Robots ({H}umanoids)",
%   year="2012",
%   address="Osaka, Japan",
%   pages="323--329"
% }

%% Parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.nbStates = 3; %Number of Gaussians in the model
model.nbFrames = 2; %Number of candidate frames of reference
model.nbVar = 3; %Dimension of the datapoints in the dataset (here: t,x1,x2)
model.kP = 150; %Stiffness gain (for DS-GMR)
model.kV = 20; %Damping gain (for DS-GMR)
model.dt = 0.01; %Time step (for DS-GMR)

%% Load data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The MAT file below contains a matrix 'Data' containing the raw data and a structure 's' containing the different
% demonstration samples. 's(n).Data' contains the data for sample n (with 's(n).nbData' datapoints).
% 's(n).p(m).b' and 's(n).p(m).A' contain the offset and transformation matrix of the m-th candidate coordinate system
% of sample n. In this example, the offsets represent the augmented positions of the candidate frames (zero for
% the time component). The transformation matrices represent augmented rotation matrices (identity for the 
% time component). s(n).Data0 represents the trajectory of the end-effector of the n-th demonstration. 
% s(n).Data represents the trajectory of the virtual spring attractor point (see paper for details). 
disp('Load data...');
load(['data/Data2.mat']);
nbSamples = length(s);
nbRepros = nbSamples;

%% Learning
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fprintf('Parameters estimation of proposed PGMM (moving frames):');
model = init_proposedPGMM_timeBased(s, model); %Initialization 
[s, model, LL] = EM_proposedPGMM(s, model); 

%% Reproduction for the set of parameters used to train the model
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Reproductions...');
for n=1:nbSamples
  r(n) = reproduction_DSGMR(s(1).Data(1,:), model, s(n), s(n).Data0(2:3,1));
end

%% Reproduction for new sets of parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('New reproductions...');
for n=1:nbRepros 
  for m=1:model.nbFrames
    %Generate new task parameters artificially 
    id=ceil(rand(2,1)*nbSamples);
    w=rand(2); w=w/sum(w);
    for t=1:s(n).nbData
      rTmp.p(m,t).b = s(id(1)).p(m,t).b * w(1) + s(id(2)).p(m,t).b * w(2);
      rTmp.p(m,t).A = s(id(1)).p(m,t).A * w(1) + s(id(2)).p(m,t).A * w(2);
    end
  end
  currPos = s(id(1)).Data0(2:3,1) * w(1) + s(id(2)).Data0(2:3,1) * w(2);
  rnew(n) = reproduction_DSGMR(s(1).Data(1,:), model, rTmp, currPos);
end

%% Plots
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
figure('position',[20,50,1100,600]);
xx = round(linspace(1,64,nbSamples));
clrmap = colormap('jet');
clrmap = min(clrmap(xx,:),.95);
limAxes = [-1.2 0.8 -1.1 0.9];
colPegs = [[.9,.5,.9];[.5,.9,.5]];

%DEMOS
subplot(1,3,1); hold on; box on; title('Demonstrations');
for n=1:nbSamples
  %Plot frames
  for m=1:model.nbFrames
    plot([s(n).p(m).b(2) s(n).p(m).b(2)+s(n).p(m).A(2,3)], [s(n).p(m).b(3) s(n).p(m).b(3)+s(n).p(m).A(3,3)], '-','linewidth',6,'color',colPegs(m,:));
    plot(s(n).p(m).b(2), s(n).p(m).b(3),'.','markersize',30,'color',colPegs(m,:)-[.05,.05,.05]);
  end
  %Plot trajectories
  plot(s(n).Data0(2,1), s(n).Data0(3,1),'.','markersize',12,'color',clrmap(n,:));
  plot(s(n).Data0(2,:), s(n).Data0(3,:),'-','linewidth',1.5,'color',clrmap(n,:));
  axis(limAxes); axis square; set(gca,'xtick',[],'ytick',[]);
end

%REPROS
subplot(1,3,2); hold on; box on; title('Reproductions');
for n=1:nbSamples
  %Plot frames
  for m=1:model.nbFrames
    plot([r(n).p(m).b(2) r(n).p(m).b(2)+r(n).p(m).A(2,3)], [r(n).p(m).b(3) r(n).p(m).b(3)+r(n).p(m).A(3,3)], '-','linewidth',6,'color',colPegs(m,:));
    plot(r(n).p(m).b(2), r(n).p(m).b(3),'.','markersize',30,'color',colPegs(m,:)-[.05,.05,.05]);
  end
  %Plot trajectories
  plot(r(n).Data(2,1), r(n).Data(3,1),'.','markersize',12,'color',clrmap(n,:));
  plot(r(n).Data(2,:), r(n).Data(3,:),'-','linewidth',1.5,'color',clrmap(n,:));
  %Plot Gaussians
  plotGMM(r(n).Mu(2:3,:,1), r(n).Sigma(2:3,2:3,:,1), [.7 .7 .7], 1);
  axis(limAxes); axis square; set(gca,'xtick',[],'ytick',[]);
end

%NEW REPROS
subplot(1,3,3); hold on; box on; title('New reproductions');
for n=1:nbRepros
  %Plot frames
  for m=1:model.nbFrames
    plot([rnew(n).p(m).b(2) rnew(n).p(m).b(2)+rnew(n).p(m).A(2,3)], [rnew(n).p(m).b(3) rnew(n).p(m).b(3)+rnew(n).p(m).A(3,3)], '-','linewidth',6,'color',colPegs(m,:));
    plot(rnew(n).p(m).b(2), rnew(n).p(m).b(3), '.','markersize',30,'color',colPegs(m,:)-[.05,.05,.05]);
  end
  %Plot trajectories
  plot(rnew(n).Data(2,1), rnew(n).Data(3,1),'.','markersize',12,'color',[.2 .2 .2]);
  plot(rnew(n).Data(2,:), rnew(n).Data(3,:),'-','linewidth',1.5,'color',[.2 .2 .2]);
  %Plot Gaussians
  plotGMM(rnew(n).Mu(2:3,:,1), rnew(n).Sigma(2:3,2:3,:,1), [.7 .7 .7], 1);
  axis(limAxes); axis square; set(gca,'xtick',[],'ytick',[]);
end

%print('-dpng','demo2.png'); 




