package com.sc.call;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyManager;


/**
 * CallListener.java: 来电状态监听、去电监听。
 * 用法：创建CallListener实例，并重写电话相关的四个abstract抽象函数即可。
 * new CallListener(context){...};
 * 
 * 需要添加权限：
 * uses-permission android:name="android.permission.READ_PHONE_STATE" 
 * uses-permission android:name="android.permission.PROCESS_OUTGOING_CALLS"
 * -----
 * 2019-1-9 上午10:16:06
 * scimence 
 */
public abstract class CallListener
{
	CallListener Instance;
	
	CallInListener callin;		// 来电监听
	OutCallListener callout;	// 拨出监听
	
	public CallListener(Context context)
	{
		Instance = this;
		callin = new CallInListener(context)
		{
			@Override
			public void Ringing(String phoneNumber)
			{
				Instance.Ringing(phoneNumber);
			}
			
			@Override
			public void OffHook(String phoneNumber)
			{
				Instance.OffHook(phoneNumber);
			}
			
			@Override
			public void HungUp(String phoneNumber)
			{
				Instance.HungUp(phoneNumber);
			}
		};
		
		callout = new OutCallListener(context)
		{
			@Override
			public void OutCall(String phoneNumber)
			{
				Instance.OutCall(phoneNumber);
			}
		};
	}
	
	/** 响铃时执行逻辑 */
	public abstract void Ringing(String phoneNumber);
	
	/** 接听时执行逻辑 */
	public abstract void OffHook(String phoneNumber);
	
	/** 挂断时执行逻辑 */
	public abstract void HungUp(String phoneNumber);
	
	/** 去电时执行逻辑 */
	public abstract void OutCall(String phoneNumber);
}

/**
 * 来电状态监听。
 * 
 * <uses-permission android:name="android.permission.READ_PHONE_STATE" />
 * */
abstract class CallInListener extends PhoneStateListener
{
	TelephonyManager phoneManager;
	public CallInListener(Context context)
	{
		phoneManager = (TelephonyManager)context.getSystemService(Context.TELEPHONY_SERVICE);
		Listen();
	}
	
	/** 监听来电状态 */
	public void Listen()
	{
		phoneManager.listen(this, PhoneStateListener.LISTEN_CALL_STATE);	// 在系统服务上,为当前Listener注册CALL_STATE状态监听
	}
	
	/** 不再监听 */
	public void ClearListen()
	{
		phoneManager.listen(this, PhoneStateListener.LISTEN_NONE);
	}
	
	/** 重写监听来电状态 */
	public void onCallStateChanged(int state, String phoneNumber)
	{
		if(state == TelephonyManager.CALL_STATE_IDLE)			// 空闲/挂断
		{
			HungUp(phoneNumber);
		}
		else if(state == TelephonyManager.CALL_STATE_RINGING)	// 响铃
		{
			Ringing(phoneNumber);
		}
		else if(state == TelephonyManager.CALL_STATE_OFFHOOK)	// 接听
		{
			OffHook(phoneNumber);
		}
	}
	
	/** 响铃时执行逻辑 */
	public abstract void Ringing(String phoneNumber);
	
	/** 接听时执行逻辑 */
	public abstract void OffHook(String phoneNumber);
	
	/** 挂断时执行逻辑 */
	public abstract void HungUp(String phoneNumber);
	
}


/**
 * 去电监听。
 * 
 * <uses-permission android:name="android.permission.PROCESS_OUTGOING_CALLS" />
 * */
abstract class OutCallListener extends BroadcastReceiver
{
	Context context;
	public OutCallListener(Context context)
	{
		this.context = context;
		Listen();
	}

	/** 监听来电状态 */
	public void Listen()
	{
		IntentFilter intentFilter = new IntentFilter(Intent.ACTION_NEW_OUTGOING_CALL);
		context.registerReceiver(this, intentFilter);
	}
	
	/** 不再监听 */
	public void ClearListen()
	{
		context.unregisterReceiver(this);
	}
	
	/** 重写监听去电事件 */
	public void onReceive(Context context, Intent intent)
	{
		String phoneNumber = getResultData();
		OutCall(phoneNumber);
	}
	
	/** 去电时执行逻辑 */
	public abstract void OutCall(String phoneNumber);
}


