import { registerPaymentMethod } from '@woocommerce/blocks-registry';
import { decodeEntities } from '@wordpress/html-entities';
import { getSetting } from '@woocommerce/settings';
import { useState, useEffect } from '@wordpress/element';

const settings = getSetting('wc_auxpay_card_data', {});
const label = decodeEntities(settings.title || 'Credit Card');

const Content = (props) => {
    const { eventRegistration, emitResponse } = props;
    const { onPaymentSetup } = eventRegistration;
    const [ paymentToken, setPaymentToken ] = useState('');

    useEffect(() => {
        const interval = setInterval(() => {
            if (typeof WCAuxPaySDK !== 'undefined') {
                WCAuxPaySDK.mount();
                WCAuxPaySDK.onInputChange = function () {
                    setPaymentToken('');
                };
                clearInterval(interval); // stop immediately once mounted
                clearTimeout(timeout);   // cancel the stop timer
            }
        }, 500);

        const timeout = setTimeout(() => {
            clearInterval(interval);
        }, 3000);

        const unsubscribe = onPaymentSetup(async () => {
            if (paymentToken) {
                return {
                    type: emitResponse.responseTypes.SUCCESS,
                    meta: {
                        paymentMethodData: { 'payment_token': paymentToken }
                    },
                };
            }
            else if (typeof WCAuxPaySDK !== 'undefined') {
                try {
                    const tokenData = await WCAuxPaySDK.tokenize();
                    if (tokenData.token) {
                        setPaymentToken(tokenData.token);
                        return {
                            type: emitResponse.responseTypes.SUCCESS,
                            meta: {
                                paymentMethodData: { 'payment_token': tokenData.token }
                            },
                        };
                    }
                    else {
                        return {
                            type: emitResponse.responseTypes.ERROR,
                            message: tokenData.error || 'Tokenization Failed!',
                        };
                    }
                } catch (err) {
                    return {
                        type: emitResponse.responseTypes.ERROR,
                        message: err.message || err,
                    };
                }
            }
            else {
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Something went wrong, please try again.',
                };
            }
        });

        return () => {
            clearInterval(interval);
            clearTimeout(timeout);
            unsubscribe(); // Unsubscribes when this component is unmounted.
        };
    }, [
        emitResponse.responseTypes.ERROR,
        emitResponse.responseTypes.SUCCESS,
        onPaymentSetup,
        paymentToken
    ]);

    return (
        <>
            <style>
                {`
                    #auxpay_card_fields_container iframe {
                        height: 265px !important;
                        margin: 0 -10px;
                    }
                `}
            </style>
            <div dangerouslySetInnerHTML={{ __html: settings.description || '' }} />
            <div id="auxpay_card_fields_container"></div>
            <input type="hidden" name="payment_token" value={paymentToken} />
        </>
    );
};

const Label = ({ components }) => {
    const { PaymentMethodLabel } = components;
    return <PaymentMethodLabel text={ label } />;
};

const AuxPayCardBlocks = {
    name: 'wc_auxpay_card',
    label: <Label />,
    ariaLabel: label,
    content: <Content />,
    edit: <Content />,
    canMakePayment: () => true,
    supports: {
        features: settings.supports || [],
    },
};

registerPaymentMethod(AuxPayCardBlocks);
