<?php

if (!defined('ABSPATH')) {
    exit;
}

class WC_AuxPay_TextToPay extends WC_Payment_Gateway {

    private $api_key;
    private $api_url;
    public  $test_mode;

    public function __construct() {
        $this->id = 'wc_auxpay_texttopay';
        $this->method_title = __('AuxPay - Text to Pay', 'auxpay-payment-gateway');
        $this->method_description = 'Pay through SMS link.';

        $this->has_fields = false; // No fields to show
        $this->supports = ['products', 'subscriptions', 'refunds'];

        $this->init_form_fields();
        $this->init_settings();

        $this->test_mode    = get_option('wc_auxpay_test_enabled') === 'yes' ? ' - (Test Mode)' : '';
        $this->enabled      = $this->get_option('enabled');
        $this->title        = get_option('wc_auxpay_texttopay_title') . $this->test_mode;
        $this->description  = get_option('wc_auxpay_texttopay_description');
        $this->api_key      = get_option('wc_auxpay_api_key', '');
        $this->api_url      = get_option('wc_auxpay_api_url', '');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);

        // Webhook
        add_action('woocommerce_api_' . $this->id . '_webhook', [$this, 'webhook_handler']);
        
        // Blocks checkout hook
        add_action('woocommerce_rest_checkout_process_payment_with_context', [$this, 'process_payment_with_context'], 10, 2);
    }

    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title'       => 'Enable/Disable',
                'label'       => 'Text to Pay',
                'type'        => 'checkbox',
                'default'     => 'no',
            ),
        );
    }

    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        try {
            // Prepare payment data
            $payment_data = $this->prepare_payment_data($order);

            // Send the payment link
            $response = $this->send_payment_link($payment_data);

            if (in_array($response['status'], ['success', '201'])) {
                $order->add_order_note('Text-to-Pay link generated: ' . $response['data']['fullUrl']);
                $order->add_order_note('Payment ID: ' . $response['data']['paymentLinkId']);
                $order->add_order_note('Reference #: ' . $response['data']['refNo']);
                $order->update_meta_data('_auxpay_ref_no', $response['data']['refNo']);
                $order->save();

                return array(
                    'result'   => 'success',
                    'redirect' => $this->get_return_url($order),
                );
            } else {
                wc_add_notice('Text-to-Pay Failed: ' . $response['message'], 'error');
                $this->debug_log('Text-to-Pay Failed. order#' . $order_id . ': ' . $response['message'], 'error');
                $this->debug_log('Text-to-Pay Failed. payload: ' . json_encode($payment_data), 'error');
                $this->debug_log('Text-to-Pay Failed. response: ' . json_encode($response), 'error');
                return;
            }
        } catch (Exception $e) {
            wc_add_notice('Text-to-Pay Error: ' . $e->getMessage(), 'error');
            $this->debug_log('Text-to-Pay Error. order#' . $order_id . ': ' . $e->getMessage(), 'error');
            return;
        }
    }

    public function process_payment_with_context($context, $request) {
        if ($context->payment_method === $this->id) {
            $order = $context->order;
            try {
                // Prepare payment data
                $payment_data = $this->prepare_payment_data($order);

                // Send the payment link
                $response = $this->send_payment_link($payment_data);

                // Handle the response from the payment gateway
                if (in_array($response['status'], ['success', '201'])) {
                    $order->add_order_note('Text-to-Pay link generated: ' . $response['data']['fullUrl']);
                    $order->add_order_note('Payment ID: ' . $response['data']['paymentLinkId']);
                    $order->add_order_note('Reference #: ' . $response['data']['refNo']);
                    $order->update_meta_data('_auxpay_ref_no', $response['data']['refNo']);
                    $order->save();
                    
                    $request->set_status('success');
                    $request->set_redirect_url($this->get_return_url($order));
                } else {
                    wc_add_notice('Text-to-Pay Failed: ' . $response['message'], 'error');
                    $this->debug_log('Text-to-Pay Failed. order#' . $order->get_id() . ': ' . $response['message'], 'error');
                    $this->debug_log('Text-to-Pay Failed. payload: ' . json_encode($payment_data), 'error');
                    $this->debug_log('Text-to-Pay Failed. response: ' . json_encode($response), 'error');
                    return;
                }
            } catch (Exception $e) {
                wc_add_notice('Text-to-Pay Error: ' . $e->getMessage(), 'error');
                $this->debug_log('Text-to-Pay Error. order#' . $order->get_id() . ': ' . $e->getMessage(), 'error');
                return;
            }
        }
    }

    private function prepare_payment_data($order) {
        // Prepare the API request data
        $data = [
            "notifyUrl"            => add_query_arg('wc-api', $this->id.'_webhook', home_url('/')),
            "customerName"         => $order->get_formatted_billing_full_name(),
            "customerNumber"       => $order->get_billing_phone(),
            "countryCode"          => "+1",
            "IpAddress"            => WC_Geolocation::get_ip_address(),
            "amount"               => (float) wc_format_decimal($order->get_total(), 2),
            "ConvenienceFeeActive" => 'yes' === get_option('wc_auxpay_fee_enabled') ? true : false,
            "istextMessage"        => true,
            "type"                 => 1,
            "request-origin"       => 'woocommerce'
        ];

        return $data;
    }

    private function send_payment_link($payment_data) {

        $response = wp_remote_post($this->api_url . '/generate-url', [
            'headers' => [
                'Authorization' => $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payment_data),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        return array(
            'status' => $body['status'],
            'message' => $body['message'] ?? 'Unknown error',
            'data' => $body['data']
        );
    }

    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        try {
            if (!$order) return new WP_Error('invalid_order', 'Invalid order ID');
            if (!$amount || $amount <= 0) return new WP_Error('invalid_refund_amount', 'Refund amount must be greater than 0');
            
            // Prepare refund payload
            $payload = array(
                'Amount'         => $amount,
                'Reason'         => $reason,
                'TransactionFor' => "Refund",
                'TransactionId'  => $order->get_transaction_id(),
                'isWoocommerce'  => true
            );

            // Send refund request
            $response = $this->send_refund_request($payload);

            // Handle the response from the payment gateway
            if (in_array($response['status'], ['success', '201']) || $response['data']['Status'] == '1') {
                $order->add_order_note(sprintf('Refunded %1$s via AuxPay Gateway. Reason: %2$s', wc_price($amount), $reason ?? 'No reason provided.'));
                return true;
            } else {
                return new WP_Error('refund_failed', $response['message']);
            }
        } catch (Exception $e) {
            return new WP_Error('refund_error', $e->getMessage());
        }
    }

    private function send_refund_request($payload) {

        $response = wp_remote_post($this->api_url . '/refund', [
            'headers' => [
                'Authorization' => $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payload),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        return array(
            'status' => $body['status'],
            'message' => $body['message'] ?? 'Unknown error',
            'data' => $body['data']
        );
    }

    private function debug_log($message, $level = 'info') {
        if ('yes' === get_option('wc_auxpay_logs_enabled')) {
            $logger  = wc_get_logger();
            $context = array('source' => 'auxpay-payment-gateway');
            $logger->log($level, $message, $context);
        }
    }

    public function admin_options() {
        $settings_url = admin_url('admin.php?page=wc-settings&tab=wc_auxpay_settings');
        wp_safe_redirect($settings_url);
        exit;
    }

    public function webhook_handler() {
        $body = file_get_contents('php://input');
        $response = json_decode($body, true);
        $this->debug_log('Text-to-Pay Webhook: ' . json_encode($response));

        if (empty($response['data']['refNo']) || empty($response['status'])) {
            status_header(400);
            echo 'Invalid payload';
            exit;
        }

        // Find order by reference number
        $orders = wc_get_orders(['limit'=> 1, 'meta_key' => '_auxpay_ref_no', 'meta_value' => $response['data']['refNo'], 'meta_compare' => '=']);
        $order  = $orders ? $orders[0] : false;

        if (!$order) {
            status_header(404);
            echo 'Order not found';
            exit;
        }

        // Check if the order is already completed
        if ($order->get_status() === 'completed') {
            status_header(200);
            echo 'Order already completed';
            exit;
        }

        // Update order status
        if (in_array($response['status'], ['success', '201'])) {
            $order->payment_complete();
            $order->set_transaction_id($response['data']['TransactionId']);
            $order->save();

            $order->add_order_note('AuxPay payment completed. Ref#: ' . $response['data']['refNo']);
            $order->add_order_note('Transaction ID: ' . $response['data']['TransactionId']);

            wc_reduce_stock_levels($order->get_id());
        } else {
            $order->update_status('failed', 'AuxPay payment failed. Ref#: ' . $response['data']['refNo']);
        }

        status_header(200);
        echo 'OK';
        exit;
    }
}
