import { registerPaymentMethod } from '@woocommerce/blocks-registry';
import { decodeEntities } from '@wordpress/html-entities';
import { getSetting } from '@woocommerce/settings';
import { useState, useEffect } from '@wordpress/element';

const settings = getSetting('wc_auxpay_card_data', {});
const label = decodeEntities(settings.title || 'Credit Card');

const Content = (props) => {
    const { eventRegistration, emitResponse } = props;
    const { onPaymentSetup } = eventRegistration;
    const [ paymentToken, setPaymentToken ] = useState('');
    const [ payrixForm, setPayrixForm ] = useState('no');
    const [ hideLabels, setHideLabels ] = useState(true);

    useEffect(() => {
        const interval = setInterval(() => {
            if (typeof WCAuxPaySDK !== 'undefined') {
                WCAuxPaySDK.mount();
                WCAuxPaySDK.onInputChange = function () {
                    setPaymentToken('');
                };
                clearInterval(interval);
                clearTimeout(timeout);
            }
            else if (typeof PayFields !== 'undefined') {
                PayFields.ready();
                PayFields.reload();
                setPayrixForm('yes');
                setHideLabels(false);
                clearInterval(interval);
                clearTimeout(timeout);
            }
        }, 500);

        const timeout = setTimeout(() => {
            clearInterval(interval);
        }, 5000);

        const unsubscribe = onPaymentSetup(async () => {
            if (paymentToken) {
                return {
                    type: emitResponse.responseTypes.SUCCESS,
                    meta: {
                        paymentMethodData: { 'payment_token': paymentToken, 'payrix_form': payrixForm }
                    },
                };
            }
            else if (typeof WCAuxPaySDK !== 'undefined') {
                try {
                    const tokenData = await WCAuxPaySDK.tokenize();
                    if (tokenData.token) {
                        setPaymentToken(tokenData.token);
                        setPayrixForm('no');
                        return {
                            type: emitResponse.responseTypes.SUCCESS,
                            meta: {
                                paymentMethodData: { 'payment_token': tokenData.token, 'payrix_form': 'no' }
                            },
                        };
                    }
                    else {
                        return {
                            type: emitResponse.responseTypes.ERROR,
                            message: tokenData.error || 'Tokenization Failed!',
                        };
                    }
                } catch (err) {
                    return {
                        type: emitResponse.responseTypes.ERROR,
                        message: err.message || err,
                    };
                }
            }
            else if (typeof PayFields !== 'undefined') {
                return new Promise((resolve) => {
                    PayFields.onSuccess = function(response) {
                        setPaymentToken(response?.data?.[0]?.token);
                        setPayrixForm('yes');
                        resolve({
                            type: emitResponse.responseTypes.SUCCESS,
                            meta: {
                                paymentMethodData: { 'payment_token': response?.data?.[0]?.token, 'payrix_form': 'yes' }
                            },
                        });
                    };
                    PayFields.onFailure = function(error) {
                        resolve({
                            type: emitResponse.responseTypes.ERROR,
                            message: "Error! Please check your card details.",
                        });
                    };
                    PayFields.onValidationFailure = function() {
                        resolve({
                            type: emitResponse.responseTypes.ERROR,
                            message: "Validation error!",
                        });
                    };
                    PayFields.submit();
                });
            }
            else {
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Something went wrong, please try again.',
                };
            }
        });

        return () => {
            clearInterval(interval);
            clearTimeout(timeout);
            unsubscribe();
        };
    }, [
        emitResponse.responseTypes.ERROR,
        emitResponse.responseTypes.SUCCESS,
        onPaymentSetup,
        paymentToken
    ]);

    return (
        <>
            <style>
                {`
                    #auxpay_card_fields_container iframe {
                        height: 265px !important;
                        margin: 0 -10px;
                    }
                    .payfields {
                        padding: 10px;
                        overflow: auto;
                    }
                    .payfields #form-container {
                        width: 315px;
                    }
                    .payfields.hide,
                    .payfields .label.hide {
                        display: none;
                    }
                    .payfields .label {
                        font-size: 14px !important;
                        font-weight: 500 !important;
                        color: #616161 !important;
                    }
                    .payfields .form-row {
                        height: 60px;
                        width: 315px;
                    }
                    .payfields .card-input-wrapper {
                        display: flex;
                        gap: 12px;
                    }
                    .payfields .card-input-wrapper .form-row {
                        width: auto;
                    }
                    .payfields .card-input-wrapper .w-50 {
                        width: 50%;
                    }
                `}
            </style>
            <div dangerouslySetInnerHTML={{ __html: settings.description || '' }} />
            <div id="auxpay_card_fields_container"></div>
            <div className={`payfields ${hideLabels ? "hide" : ""}`}>
                <form>
                    <div id="form-container">
                        <label htmlFor="cc_number" className={`label ${hideLabels ? "hide" : ""}`}>Card Number:</label>
                        <div id="cc_number" className="form-row"></div>
                        <div className="card-input-wrapper">
                            <div className="w-50">
                                <label htmlFor="cc_expiry" className={`label ${hideLabels ? "hide" : ""}`}>Expiration:</label>
                                <div id="cc_expiry" className="form-row"></div>
                            </div>
                            <div className="w-50">
                                <label htmlFor="cc_cvv" className={`label ${hideLabels ? "hide" : ""}`}>CVV:</label>
                                <div id="cc_cvv" className="form-row"></div>
                            </div>
                        </div>
                        <label htmlFor="cc_name" className={`label ${hideLabels ? "hide" : ""}`}>Cardholder Name:</label>
                        <div id="cc_name" className="form-row"></div>
                    </div>
                </form>
            </div>
            <input type="hidden" name="payment_token" value={paymentToken} />
            <input type="hidden" name="payrix_form" value={payrixForm} />
        </>
    );
};

const Label = ({ components }) => {
    const { PaymentMethodLabel } = components;
    return <PaymentMethodLabel text={ label } />;
};

const AuxPayCardBlocks = {
    name: 'wc_auxpay_card',
    label: <Label />,
    ariaLabel: label,
    content: <Content />,
    edit: <Content />,
    canMakePayment: () => true,
    supports: {
        features: settings.supports || [],
    },
};

registerPaymentMethod(AuxPayCardBlocks);
