<?php

if (!defined('ABSPATH')) {
    exit;
}

class WC_AuxPay_ACH extends WC_Payment_Gateway {

    private $api_key;
    private $api_url;
    public  $test_mode;

    public function __construct() {
        $this->id = 'wc_auxpay_ach';
        $this->method_title = __('AuxPay - ACH', 'auxpay-payment-gateway');
        $this->method_description = 'Pay using ACH bank transfer.';

        $this->has_fields = true;
        $this->supports = ['products', 'subscriptions', 'refunds'];

        $this->init_form_fields();
        $this->init_settings();

        if ('no' === get_option('wc_auxpay_test_enabled')) {
            $this->test_mode   = '';
        }
        else {
            $this->test_mode   = ' - (Test Mode)';
        }

        $this->enabled      = $this->get_option('enabled');
        $this->title        = get_option('wc_auxpay_ach_title') . $this->test_mode;
        $this->description  = get_option('wc_auxpay_ach_description');
        $this->api_key      = get_option('wc_auxpay_api_key');
        $this->api_url      = get_option('wc_auxpay_api_url');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);

        // Blocks checkout hook
        add_action('woocommerce_rest_checkout_process_payment_with_context', [$this, 'process_payment_with_context'], 10, 2);
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'       => 'Enable/Disable',
                'label'       => 'ACH Payment',
                'type'        => 'checkbox',
                'default'     => 'no',
            ),
        );
    }

    public function payment_fields() {
        echo '<style>
            .wc-auxpay-ach-form {
                padding-top: 10px;
            }
            .wc-auxpay-ach-form input {
                padding: 10px;
                font-size: 16px;
                border-radius: 6px;
                margin-bottom: 10px;
                border: 1px solid #ccc;
            }
            </style>';

        // Display ach form
        echo '<p>' . esc_html($this->description) . '</p>';
        echo '<fieldset id="wc_auxpay_ach-form" class="wc-auxpay-ach-form">';
        echo '<div class="form-row">
                <input name="ach_account_number" type="text" autocomplete="off" placeholder="Account Number" maxlength="15" required>
            </div>
            <div class="form-row">
                <input name="ach_routing_number" type="text" autocomplete="off" placeholder="Routing Number" maxlength="9" required>
            </div>';
        echo '<div class="clear"></div></fieldset>';
        wp_nonce_field($this->id, 'wc_auxpay_ach_nonce');
    }

    public function validate_fields() {
        if (!isset($_POST['wc_auxpay_ach_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wc_auxpay_ach_nonce'])), $this->id)) {
            wc_add_notice('Security check failed. Please refresh and try again.', 'error');
            return false;
        }

        if (empty($_POST['ach_account_number']) || empty($_POST['ach_routing_number'])) {
            wc_add_notice(__('Please fill in all required ach fields.', 'auxpay-payment-gateway'), 'error');
            return false;
        }
        return true;
    }

    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        try {
            // Verify nonce
            if (!isset($_POST['wc_auxpay_ach_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wc_auxpay_ach_nonce'])), $this->id)) {
                wc_add_notice('Security check failed. Please refresh and try again.', 'error');
                return;
            }
            // Collect and sanitize input fields
            $account_number = sanitize_text_field(wp_unslash($_POST['ach_account_number'] ?? ''));
            $routing_number = sanitize_text_field(wp_unslash($_POST['ach_routing_number'] ?? ''));

            if (empty($account_number) || empty($routing_number)) {
                wc_add_notice('ACH fields missing, please try again.', 'error');
                return;
            }
            
            // Prepare payment data
            $payment_data = $this->prepare_payment_data($order, $account_number, $routing_number);

            // Make the payment
            $response = $this->make_payment($payment_data);

            // Handle the response from the payment gateway
            if (in_array($response['status'], ['success', '201']) || $response['data']['status'] == 'success') {
                $order->payment_complete();
                $order->set_transaction_id($response['data']['TransactionId']);
                $order->save();

                $order->add_order_note('Transaction ID: ' . $response['data']['TransactionId']);
                wc_reduce_stock_levels($order_id);

                return array(
                    'result'   => 'success',
                    'redirect' => $this->get_return_url($order),
                );
            } else {
                wc_add_notice('Payment Failed: ' . $response['message'], 'error');
                $this->debug_log('ACH Payment Failed. order#' . $order_id . ': ' . $response['message'], 'error');
                $this->debug_log('ACH Payment Failed. payload: ' . json_encode($payment_data), 'error');
                $this->debug_log('ACH Payment Failed. response: ' . json_encode($response), 'error');
                return;
            }
        } catch (Exception $e) {
            wc_add_notice('Payment Error: ' . $e->getMessage(), 'error');
            $this->debug_log('ACH Payment Error. order#' . $order_id . ': ' . $e->getMessage(), 'error');
            return;
        }
    }

    public function process_payment_with_context($context, $request) {
        if ($context->payment_method === $this->id) {
            $order = $context->order;
            try {
                // Collect and sanitize input fields
                $account_number = sanitize_text_field(wp_unslash($context->payment_data['ach_account_number'] ?? ''));
                $routing_number = sanitize_text_field(wp_unslash($context->payment_data['ach_routing_number'] ?? ''));

                if (empty($account_number) || empty($routing_number)) {
                    wc_add_notice('ACH fields missing, please try again.', 'error');
                    return;
                }

                // Prepare payment data
                $payment_data = $this->prepare_payment_data($order, $account_number, $routing_number);

                // Make the payment
                $response = $this->make_payment($payment_data);

                // Handle the response from the payment gateway
                if (in_array($response['status'], ['success', '201']) || $response['data']['status'] == 'success') {
                    $order->payment_complete();
                    $order->set_transaction_id($response['data']['TransactionId']);
                    $order->save();

                    $order->add_order_note('Transaction ID: ' . $response['data']['TransactionId']);
                    wc_reduce_stock_levels($order->get_id());
                    
                    $request->set_status('success');
                    $request->set_redirect_url($this->get_return_url($order));
                } else {
                    wc_add_notice('Payment Failed: ' . $response['message'], 'error');
                    $this->debug_log('ACH Payment Failed. order#' . $order->get_id() . ': ' . $response['message'], 'error');
                    $this->debug_log('ACH Payment Failed. payload: ' . json_encode($payment_data), 'error');
                    $this->debug_log('ACH Payment Failed. response: ' . json_encode($response), 'error');
                    return;
                }
            } catch (Exception $e) {
                wc_add_notice('Payment Error: ' . $e->getMessage(), 'error');
                $this->debug_log('ACH Payment Error. order#' . $order->get_id() . ': ' . $e->getMessage(), 'error');
                return;
            }
        }
    }

    private function prepare_payment_data($order, $account_number, $routing_number) {
        $fee_active = false;
        $base_amount = $order->get_total();
        $config = get_option('wc_auxpay_fee_config');
        if (is_array($config['ach']) && $config['ach']['active'] && 'yes' === get_option('wc_auxpay_fee_enabled')) {
            $fee_active = true;
            $processing_fee = 0;
            $title = get_option('wc_auxpay_fee_title', 'Processing Fee');
            $title = apply_filters('translate_text', $title, 'auxpay-payment-gateway');
            foreach ($order->get_fees() as $fee) {
                if ($fee->get_name() === $title) {
                    $processing_fee = $fee->get_total();
                }
            }
            $base_amount = $order->get_total() - $processing_fee;
        }

        // Extract billing information from the order
        $billing_address = array(
            "BillingCustomerName"   => $order->get_formatted_billing_full_name(),
            "BillingEmail"          => $order->get_billing_email(),
            "BillingPhoneNumber"    => $order->get_billing_phone(),
            "BillingAddress"        => $order->get_billing_address_1(),
            "BillingPostalCode"     => $order->get_billing_postcode(),
            "BillingCity"           => $order->get_billing_city(),
            "BillingState"          => $order->get_billing_state(),
            "BillingCountry"        => $order->get_billing_country(),
            "BillingCountryCode"    => "+1",
            "IpAddress"             => WC_Geolocation::get_ip_address(),
        );

        // Prepare the API request data
        $data = array_merge($billing_address, array(
            "Amount"               => (float) wc_format_decimal($base_amount, 2),
            "account_number"       => $account_number,
            "routing_number"       => $routing_number,
            "SuggestedMode"        => "ACH",
            "ConvenienceFeeActive" => $fee_active,
            "TransactionType"      => 1,
            "request-origin"       => 'woocommerce'
        ));

        return $data;
    }

    private function make_payment($payment_data) {

        $response = wp_remote_post($this->api_url . '/transaction', [
            'headers' => [
                'Authorization' => $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payment_data),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        return array(
            'status' => $body['status'],
            'message' => $body['message'] ?? 'Unknown error',
            'data' => $body['data']
        );
    }

    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        try {
            if (!$order) return new WP_Error('invalid_order', 'Invalid order ID');
            if (!$amount || $amount <= 0) return new WP_Error('invalid_refund_amount', 'Refund amount must be greater than 0');
            
            // Prepare refund payload
            $payload = array(
                'Amount'         => $amount,
                'Reason'         => $reason,
                'TransactionFor' => "Refund",
                'TransactionId'  => $order->get_transaction_id(),
                'isWoocommerce'  => true
            );

            // Send refund request
            $response = $this->send_refund_request($payload);

            // Handle the response from the payment gateway
            if (in_array($response['status'], ['success', '201']) || $response['data']['Status'] == '1') {
                $order->add_order_note(sprintf('Refunded %1$s via AuxPay Gateway. Reason: %2$s', wc_price($amount), $reason ?? 'No reason provided.'));
                return true;
            } else {
                return new WP_Error('refund_failed', $response['message']);
            }
        } catch (Exception $e) {
            return new WP_Error('refund_error', $e->getMessage());
        }
    }

    private function send_refund_request($payload) {

        $response = wp_remote_post($this->api_url . '/refund', [
            'headers' => [
                'Authorization' => $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payload),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        return array(
            'status' => $body['status'],
            'message' => $body['message'] ?? 'Unknown error',
            'data' => $body['data']
        );
    }

    private function debug_log($message, $level = 'info') {
        if ('yes' === get_option('wc_auxpay_logs_enabled')) {
            $logger  = wc_get_logger();
            $context = array('source' => 'auxpay-payment-gateway');
            $logger->log($level, $message, $context);
        }
    }

    public function admin_options() {
        $settings_url = admin_url('admin.php?page=wc-settings&tab=wc_auxpay_settings');
        wp_safe_redirect($settings_url);
        exit;
    }
}
