import { registerPaymentMethod } from '@woocommerce/blocks-registry';
import { decodeEntities } from '@wordpress/html-entities';
import { getSetting } from '@woocommerce/settings';
import { useState, useEffect } from '@wordpress/element';

const settings = getSetting('wc_auxpay_card_data', {});
const label = decodeEntities(settings.title || 'Credit Card');

function showOtpModal(phone, transactionId = null, orderId = null, amount = null) {
    const modalHtmlOtpFlow = `
        <div id="auxpay-otp-modal" class="auxpay-modal">
            <div class="auxpay-modal-content">
                <div id="auxpay-phone-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Confirm your approval</p>
                        <p class="desc text-grey">Enter your phone number to continue</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <h4 class="text-gray">Enter your mobile number</h4>
                        <p class="text-grey">We\'ll send a 4-digit code to confirm it\'s you.</p>
                        
                        <div class="phone-text">
                            <p><b><label>Mobile Number</label></b></p>
                            <p><input type="text" id="auxpay-otp-phone" value="${phone}" /></p>
                        </div>

                        <div class="form-group">
                            <button class="auxpay-cancel-otp btn btn-secondary">No thanks</button>
                            <button id="auxpay-send-otp" class="btn btn-success">Send code</button>
                        </div>
                    </div>
                </div>
                <div id="auxpay-otp-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Enter your 4-digit code</p>
                        <p class="desc text-grey">Verify your identity</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <h4 class="text-gray">Enter your 4-digit code</h4>
                        <p class="text-grey">Enter the code sent to your phone</p>

                        <div class="text-center phone-text">
                            <p><input type="text" id="auxpay-otp-code" value="" maxlength="4" /></p>
                            <p class="text-grey">Didn\'t get a code? &nbsp;&nbsp; <b>Resend</b> in (<span id="auxpay-countdown"></span><button id="auxpay-resend-otp" style="display:none;">Resend Code</button>)</p>
                        </div>

                        <div class="form-group">
                            <button class="auxpay-cancel-otp btn btn-secondary">No thanks</button>
                            <button id="auxpay-verify-otp" class="btn btn-success">Confirm & complete</button>
                        </div>
                    </div>
                </div>
                <div id="auxpay-otp-error" role="error" style="display:none;"></div>
                <div id="auxpay-otp-success" role="success" style="display:none;"></div>
            </div>
        </div>
    `;
    
    const modalHtmlDeclinedFlow = `
        <div id="auxpay-decline-modal" class="auxpay-modal">
            <div class="auxpay-modal-content">
                <div id="auxpay-agree-section">
                    <div class="modal-header">
                        <p class="title text-gray">Your card issuer declined this payment</p>
                        <p class="desc text-grey">Convert to gift card instead</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <h4 class="text-center text-red">Payment Declined</h4>
                        <p class="text-grey">We can still help you complete your order today. With your permission, we\'ll issue a <b>AuxCARD</b> e-gift card for <b class="text-green">$${amount}</b> and use it right away to pay for this order.</p>
                        
                        <div class="amount-text">
                            <h4 class="text-green">$${amount}</h4>
                            <label class="text-grey"><span>Transaction Amount</span></label>
                        </div>

                        <div class="form-group">
                            <p class="text-gray">You\'re in control. If you agree, we\'ll text you a 4-digit code to confirm.</p>
                        </div>

                        <div class="form-group cursor-pointer">
                            <input type="checkbox" id="auxpay-agree-terms" />
                            <label for="auxpay-agree-terms" class="text-gray">I agree to use a <b>AuxCARD</b> e-gift card to complete this order and I have read the <a class="text-blue" target="_blank" href="https://auxcard.net/program-terms">Program Terms</a> and <a class="text-blue" target="_blank" href="https://auxcard.net/privacy-policy">Privacy Notice</a>.</label>
                        </div>

                        <p class="text-grey"><b>Program Terms:</b> No extra fees. Card is single-use and valid only at AuxCARD for this order.</p>

                        <p class="text-grey"><b>Privacy Notice:</b> We use your number to send codes and receipts. See our privacy policy for details.</p>
                        
                        <div class="form-group">
                            <button class="auxpay-cancel-otp btn btn-secondary">No thanks</button>
                            <button id="auxpay-continue-otp" class="btn btn-success" disabled="true">Continue</button>
                        </div>
                    </div>
                </div>
                <div id="auxpay-phone-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Confirm your approval</p>
                        <p class="desc text-grey">Enter your phone number to continue</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <h4 class="text-gray">Enter your mobile number</h4>
                        <p class="text-grey">We\'ll send a 4-digit code to confirm it\'s you.</p>
                        
                        <div class="phone-text">
                            <p><b><label>Mobile Number</label></b></p>
                            <p><input type="text" id="auxpay-otp-phone" value="${phone}" /></p>
                        </div>

                        <div class="form-group">
                            <button class="auxpay-cancel-otp btn btn-secondary">No thanks</button>
                            <button id="auxpay-send-otp" class="btn btn-success">Send code</button>
                        </div>
                    </div>
                </div>
                <div id="auxpay-otp-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Enter your 4-digit code</p>
                        <p class="desc text-grey">Verify your identity</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <h4 class="text-gray">Enter your 4-digit code</h4>
                        <p class="text-grey">Enter the code sent to your phone</p>

                        <div class="text-center phone-text">
                            <p><input type="text" id="auxpay-otp-code" value="" maxlength="4" /></p>
                            <p class="text-grey">Didn\'t get a code? &nbsp;&nbsp; <b>Resend</b> in (<span id="auxpay-countdown"></span><button id="auxpay-resend-otp" style="display:none;">Resend Code</button>)</p>
                        </div>

                        <div class="form-group">
                            <button class="auxpay-cancel-otp btn btn-secondary">No thanks</button>
                            <button id="auxpay-verify-confirm-otp" class="btn btn-success">Confirm & complete</button>
                        </div>
                    </div>
                </div>
                <div id="auxpay-done-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Done!</p>
                        <p class="desc text-grey">Payment Successful</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <div class="text-center phone-text">
                            <h4 class="text-green">Payment Successful</h4>
                            <p class="text-grey">We issued a <b>AuxCARD</b> e-gift card and used it to pay for your order. You\'ll get a receipt for the gift card and your order.</p>
                        </div>
                    </div>
                </div>
                <div id="auxpay-fail-section" style="display:none;">
                    <div class="modal-header">
                        <p class="title text-gray">Declined!</p>
                        <p class="desc text-grey">Unable to complete payment</p>
                        <button class="auxpay-close-otp close">X</button>
                    </div>
                    <div class="modal-body">
                        <div class="text-center phone-text">
                            <h4 class="text-red">Payment Failed</h4>
                            <p class="text-grey">Please try again</p>
                        </div>
                    </div>
                </div>
                <div id="auxpay-otp-error" role="error" style="display:none;"></div>
                <div id="auxpay-otp-success" role="success" style="display:none;"></div>
            </div>
        </div>
    `;

    if (WCAuxPayCard.threeds === '1') {
        document.body.insertAdjacentHTML('beforeend', modalHtmlOtpFlow);
        document.getElementById('auxpay-otp-modal').style.display = 'block';
    } else {
        document.body.insertAdjacentHTML('beforeend', modalHtmlDeclinedFlow);
        document.getElementById('auxpay-decline-modal').style.display = 'block';
    }

    setupOtpModalEvents(transactionId, orderId);
}

function setupOtpModalEvents(transactionId, orderId) {
    document.addEventListener('change', (e) => {
        // Enable continue button only if terms are agreed
        if (e.target.id === 'auxpay-agree-terms') {
            const continueBtn = document.getElementById('auxpay-continue-otp');
            if (continueBtn) {
                continueBtn.disabled = !e.target.checked;
            }
        }
    });

    document.addEventListener('click', (e) => {
        // Close popup
        if (e.target.classList.contains('auxpay-close-otp')) {
            document.getElementById('auxpay-otp-modal')?.remove();
            document.getElementById('auxpay-decline-modal')?.remove();
        }

        // Continue to phone input
        if (e.target.id === 'auxpay-continue-otp') {
            document.getElementById('auxpay-agree-section').style.display = 'none';
            document.getElementById('auxpay-phone-section').style.display = 'block';
        }
        
        // Cancel OTP
        if (e.target.classList.contains('auxpay-cancel-otp')) {
            document.getElementById('auxpay-otp-error').style.display = 'none';
            document.getElementById('auxpay-otp-success').style.display = 'none';
            if (WCAuxPayCard.threeds === '1') {
                document.getElementById('auxpay-otp-modal')?.remove();
            } else {
                fetch(settings.ajaxUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'auxpay_cancel_otp',
                        nonce: settings.nonce,
                        transaction_id: transactionId,
                        order_id: orderId
                    })
                }).then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('auxpay-decline-modal')?.remove();
                    } else {
                        showOtpError(data.data?.message);
                    }
                }).catch(err => {
                    console.error(err);
                    showOtpError('Error cancelling OTP. Please try again.');
                });
            }
        }

        // Send or Resend OTP
        if (e.target.id === 'auxpay-send-otp' || e.target.id === 'auxpay-resend-otp') {
            document.getElementById('auxpay-otp-error').style.display = 'none';
            document.getElementById('auxpay-otp-success').style.display = 'none';
            document.getElementById('auxpay-otp-code').value = '';
            
            const sendBtn = document.getElementById('auxpay-send-otp');
            const resendBtn = document.getElementById('auxpay-resend-otp');
            sendBtn.disabled = true;
            resendBtn.disabled = true;
            
            fetch(settings.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'auxpay_generate_otp',
                    nonce: settings.nonce,
                    phone: document.getElementById('auxpay-otp-phone').value,
                    order_id: orderId
                })
            }).then(response => response.json())
            .then(data => {
                sendBtn.disabled = false;
                resendBtn.disabled = false;
                if (data.success) {
                    document.getElementById('auxpay-phone-section').style.display = 'none';
                    document.getElementById('auxpay-otp-section').style.display = 'block';
                    showOtpSuccess(data.data?.message);
                    startOtpCountdown(180); // 3 minutes countdown
                } else {
                    showOtpError(data.data?.message);
                }
            }).catch(err => {
                console.error(err);
                sendBtn.disabled = false;
                resendBtn.disabled = false;
                showOtpError('Error sending OTP. Please try again.');
            });
        }

        // Verify OTP and initiate transaction
        if (e.target.id === 'auxpay-verify-otp') {
            document.getElementById('auxpay-otp-error').style.display = 'none';
            document.getElementById('auxpay-otp-success').style.display = 'none';
            e.target.disabled = true;

            fetch(settings.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'auxpay_verify_otp',
                    nonce: settings.nonce,
                    phone: document.getElementById('auxpay-otp-phone').value,
                    otp: document.getElementById('auxpay-otp-code').value
                })
            })
            .then(response => response.json())
            .then(data => {
                e.target.disabled = false;
                if (data.success) {
                    // $('form[name="checkout"]').submit();
                    showOtpSuccess('Transaction in progress...');
                    setTimeout(() => {
                        document.getElementById('auxpay-otp-modal')?.remove();
                    }, 5000);
                    return {
                        type: emitResponse.responseTypes.SUCCESS,
                        meta: {
                            paymentMethodData: { 'payment_token': tokenData.token, 'payrix_form': 'no', 'auxpay_otp': document.getElementById('auxpay-otp-code').value }
                        },
                    };
                } else {
                    showOtpError(data.data?.message);
                }
            }).catch(err => {
                console.error(err);
                e.target.disabled = false;
                showOtpError('Error verifying OTP. Please try again.');
            });
        }
        
        // Verify OTP and complete pending transaction
        if (e.target.id === 'auxpay-verify-confirm-otp') {
            document.getElementById('auxpay-otp-error').style.display = 'none';
            document.getElementById('auxpay-otp-success').style.display = 'none';
            e.target.disabled = true;
            
            fetch(settings.ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'auxpay_verify_confirm_otp',
                    nonce: settings.nonce,
                    phone: document.getElementById('auxpay-otp-phone').value,
                    otp: document.getElementById('auxpay-otp-code').value,
                    transaction_id: transactionId,
                    order_id: orderId
                })
            }).then(response => response.json())
            .then(data => {
                e.target.disabled = false;
                if (data.success) {
                    document.getElementById('auxpay-otp-section').style.display = 'none';
                    document.getElementById('auxpay-done-section').style.display = 'block';
                    showOtpSuccess('Redirecting...');
                    setTimeout(() => {
                        window.location.href = data.data?.redirect;
                    }, 5000);
                } else {
                    if (data.data?.rerun) {
                        document.getElementById('auxpay-otp-section').style.display = 'none';
                        document.getElementById('auxpay-fail-section').style.display = 'block';
                        setTimeout(() => {
                            document.getElementById('auxpay-decline-modal')?.remove();
                        }, 3000);
                    }
                    showOtpError(data.data?.message);
                }
            }).catch(err => {
                console.error(err);
                e.target.disabled = false;
                showOtpError('Error verifying OTP. Please try again.');
            });
        }
    });
}

function startOtpCountdown(durationSeconds) {
    const countdownSpan = document.getElementById('auxpay-countdown');
    const resendBtn = document.getElementById('auxpay-resend-otp');
    let remaining = durationSeconds;
    
    countdownSpan.textContent = formatTime(remaining);
    resendBtn.style.display = 'none';

    const interval = setInterval(() => {
        remaining--;
        countdownSpan.textContent = formatTime(remaining);
        if (remaining <= 0) {
            clearInterval(interval);
            countdownSpan.textContent = '';
            resendBtn.style.display = 'inline';
        }
    }, 1000);

    function formatTime(sec) {
        const m = Math.floor(sec / 60);
        const s = sec % 60;
        return m + ':' + (s < 10 ? '0' : '') + s;
    }
}

function showOtpSuccess(message) {
    const successEl = document.getElementById('auxpay-otp-success');
    successEl.textContent = message;
    successEl.style.display = 'block';
    setTimeout(() => {
        successEl.style.display = 'none';
    }, 5000);
}

function showOtpError(message) {
    const errorEl = document.getElementById('auxpay-otp-error');
    errorEl.textContent = message;
    errorEl.style.display = 'block';
    setTimeout(() => {
        errorEl.style.display = 'none';
    }, 5000);
}

const Content = (props) => {
    const { eventRegistration, emitResponse } = props;
    const { onPaymentSetup, onCheckoutFail } = eventRegistration;
    const [ paymentToken, setPaymentToken ] = useState('');
    const [ payrixForm, setPayrixForm ] = useState('no');
    const [ hideLabels, setHideLabels ] = useState(true);
    const [ auxpaySuccess, setAuxpaySuccess ] = useState(false);

    useEffect(() => {
        const interval = setInterval(() => {
            if (typeof WCAuxPaySDK !== 'undefined') {
                WCAuxPaySDK.mount();
                WCAuxPaySDK.onInputChange = function () {
                    setPaymentToken('');
                };
                clearInterval(interval);
                clearTimeout(timeout);
            }
            else if (typeof PayFields !== 'undefined') {
                PayFields.ready();
                PayFields.reload();
                setPayrixForm('yes');
                setHideLabels(false);
                clearInterval(interval);
                clearTimeout(timeout);
            }
        }, 500);

        const timeout = setTimeout(() => {
            clearInterval(interval);
        }, 5000);

        const unsubscribe = onPaymentSetup(async () => {
            if (paymentToken) {
                return {
                    type: emitResponse.responseTypes.SUCCESS,
                    meta: {
                        paymentMethodData: { 'payment_token': paymentToken, 'payrix_form': payrixForm }
                    },
                };
            }
            else if (typeof WCAuxPaySDK !== 'undefined') {
                try {
                    const tokenData = await WCAuxPaySDK.tokenize();
                    if (tokenData.token) {
                        setPaymentToken(tokenData.token);
                        setPayrixForm('no');
                        setAuxpaySuccess(true);
                        if (WCAuxPayCard.threeds === '1') {
                            showOtpModal(document.getElementById('billing_phone').value);
                        } else {
                            return {
                                type: emitResponse.responseTypes.SUCCESS,
                                meta: {
                                    paymentMethodData: { 'payment_token': tokenData.token, 'payrix_form': 'no' }
                                },
                            };
                        }
                    }
                    else {
                        return {
                            type: emitResponse.responseTypes.ERROR,
                            message: tokenData.error || 'Tokenization Failed!',
                        };
                    }
                } catch (err) {
                    return {
                        type: emitResponse.responseTypes.ERROR,
                        message: err.message || err,
                    };
                }
            }
            else if (typeof PayFields !== 'undefined') {
                return new Promise((resolve) => {
                    PayFields.onSuccess = function(response) {
                        setPaymentToken(response?.data?.[0]?.token);
                        setPayrixForm('yes');
                        setAuxpaySuccess(true);
                        if (WCAuxPayCard.threeds === '1') {
                            showOtpModal(document.getElementById('billing_phone').value);
                        } else {
                            resolve({
                                type: emitResponse.responseTypes.SUCCESS,
                                meta: {
                                    paymentMethodData: { 'payment_token': response?.data?.[0]?.token, 'payrix_form': 'yes' }
                                },
                            });
                        }
                    };
                    PayFields.onFailure = function(error) {
                        resolve({
                            type: emitResponse.responseTypes.ERROR,
                            message: "Error! Please check your card details.",
                        });
                    };
                    PayFields.onValidationFailure = function() {
                        resolve({
                            type: emitResponse.responseTypes.ERROR,
                            message: "Validation error!",
                        });
                    };
                    PayFields.submit();
                });
            }
            else {
                return {
                    type: emitResponse.responseTypes.ERROR,
                    message: 'Something went wrong, please try again.',
                };
            }
        });

        // Listen for checkout errors to show OTP modal if required
        const unsubscribeFail = onCheckoutFail((error) => {
            if (error.processingResponse.paymentDetails?.['otp-required']) {
                try {
                    const otpData = JSON.parse(error.processingResponse.paymentDetails?.['otp-required']);
                    const { phone, order_id, transaction_id, amount } = otpData;
                    showOtpModal(phone, transaction_id, order_id, amount);
                } catch (e) {
                    console.error('Error parsing OTP data:', e);
                }
            }
        });

        return () => {
            clearInterval(interval);
            clearTimeout(timeout);
            unsubscribe();
            unsubscribeFail();
        };
    }, [
        emitResponse.responseTypes.ERROR,
        emitResponse.responseTypes.SUCCESS,
        onPaymentSetup,
        paymentToken
    ]);

    const cardLogos = [
        { key: 'isVisaEnabled', name: 'Visa', logo: 'assets/img/visa.svg' },
        { key: 'isMastercardEnabled', name: 'MC', logo: 'assets/img/mastercard.svg' },
        { key: 'isAmexEnabled', name: 'Amex', logo: 'assets/img/amex.svg' },
        { key: 'isDiscoverEnabled', name: 'Discover', logo: 'assets/img/discover.svg' },
    ];

    const acceptedCards = cardLogos.filter(card => 
        settings.cards && settings.cards[card.key] === true
    );

    return (
        <>
            <div dangerouslySetInnerHTML={{ __html: settings.description || '' }} />
            {acceptedCards.length > 0 && (
                <div className="auxpay-accepted-cards">
                    {acceptedCards.map((card, index) => (
                        <img 
                            key={index}
                            src={settings.pluginUrl + card.logo} 
                            alt={card.name} 
                            className="cards-img" 
                        />
                    ))}
                </div>
            )}
            <div className="auxpay-source-message" role="alert">
                {auxpaySuccess && (
                    <ul className="wc-auxpay-message wc-auxpay-success">
                        <li>
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false" style={{ fill: "#28a745" }}><path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM10 16l-4-4 1.4-1.4L10 13.2l6.6-6.6L18 8l-8 8z"></path></svg>
                            Card details tokenized successfully.
                        </li>
                    </ul>
                )}
            </div>
            <div id="auxpay_card_fields_container"></div>
            <div className={`payfields ${hideLabels ? "hide" : ""}`}>
                <form>
                    <div id="form-container">
                        <label htmlFor="cc_number" className={`label ${hideLabels ? "hide" : ""}`}>Card Number:</label>
                        <div id="cc_number" className="form-row"></div>
                        <div className="card-input-wrapper">
                            <div className="w-50">
                                <label htmlFor="cc_expiry" className={`label ${hideLabels ? "hide" : ""}`}>Expiration:</label>
                                <div id="cc_expiry" className="form-row"></div>
                            </div>
                            <div className="w-50">
                                <label htmlFor="cc_cvv" className={`label ${hideLabels ? "hide" : ""}`}>CVV:</label>
                                <div id="cc_cvv" className="form-row"></div>
                            </div>
                        </div>
                        <label htmlFor="cc_name" className={`label ${hideLabels ? "hide" : ""}`}>Cardholder Name:</label>
                        <div id="cc_name" className="form-row"></div>
                    </div>
                </form>
            </div>
            <input type="hidden" name="payment_token" value={paymentToken} />
            <input type="hidden" name="payrix_form" value={payrixForm} />
            <input type="hidden" name="auxpay_otp" value='' />
        </>
    );
};

const Label = ({ components }) => {
    const { PaymentMethodLabel } = components;
    return <PaymentMethodLabel text={ label } />;
};

const AuxPayCardBlocks = {
    name: 'wc_auxpay_card',
    label: <Label />,
    ariaLabel: label,
    content: <Content />,
    edit: <Content />,
    canMakePayment: () => true,
    supports: {
        features: settings.supports || [],
    },
};

registerPaymentMethod(AuxPayCardBlocks);
