<?php
/**
 * Plugin Name: AuxPay Payment Gateway
 * Description: A payment gateway for WooCommerce that allows users to pay via credit card, ACH, or text-to-pay.
 * Version: 3.0.5
 * Requires at least: 6.0
 * Requires PHP: 7.2
 * Author: AuxPay
 * Author URI: https://auxpay.net
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: auxpay-payment-gateway
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

define('WC_AUXPAY_VERSION', '3.0.5');
define('WC_AUXPAY_PLUGIN_DIR', plugin_dir_url(__FILE__));

/**
 * Main class for the AuxPay Payment Gateway.
 */
class WC_AuxPay_Payment_Gateway {

    /**
	 * Plugin bootstrapping.
	 */
    public function __construct() {
        // Load necessary files
        add_action('plugins_loaded', [$this, 'includes'], 0);

        // Initialize gateways
		add_filter( 'woocommerce_payment_gateways', [$this, 'add_gateway']);

        // Register block gateways
		add_action( 'woocommerce_blocks_loaded', [$this, 'register_blocks']);

        // Admin scripts
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);

        // Add settings page
        add_filter('woocommerce_get_settings_pages', [$this, 'add_settings_page']);

        // Plugin action links
        add_filter('plugin_action_links_' . plugin_basename( __FILE__ ), [$this, 'auxpay_plugin_action_links']);

        // AJAX handlers
        add_action('wp_ajax_wc_auxpay_test_connection', [$this, 'ajax_test_connection']);
        add_action('wp_ajax_wc_auxpay_fetch_config', [$this, 'ajax_fetch_config']);

        // Cart fee
        add_action('woocommerce_cart_calculate_fees', [$this, 'add_technology_fee']);

        // Webhook
        add_action('woocommerce_api_wc_auxpay_refund_webhook', [$this, 'refund_webhook_handler']);
        add_action('woocommerce_api_wc_auxpay_config_webhook', [$this, 'config_webhook_handler']);
    }

	public function includes() {
        // Load gateway classes
		if ( class_exists( 'WC_Payment_Gateway' ) ) {
            include_once 'includes/classic/class-auxpay-card.php';
            include_once 'includes/classic/class-auxpay-ach.php';
            include_once 'includes/classic/class-auxpay-texttopay.php';
		}
        // Load blocks
        if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
            include_once 'includes/blocks/class-auxpay-card-blocks.php';
            include_once 'includes/blocks/class-auxpay-ach-blocks.php';
            include_once 'includes/blocks/class-auxpay-texttopay-blocks.php';
        }
	}

	public function add_gateway( $gateways ) {
        $gateways[] = 'WC_AuxPay_Card';
        $gateways[] = 'WC_AuxPay_ACH';
        $gateways[] = 'WC_AuxPay_TextToPay';
		return $gateways;
	}

    public function register_blocks() {
        if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			add_action(
				'woocommerce_blocks_payment_method_type_registration',
				function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
                    $payment_method_registry->register(new WC_AuxPay_Card_Blocks());
                    $payment_method_registry->register(new WC_AuxPay_ACH_Blocks());
                    $payment_method_registry->register(new WC_AuxPay_TextToPay_Blocks());
				}
			);
		}
    }

    public function enqueue_admin_scripts() {
        wp_enqueue_script('wc-auxpay-admin', WC_AUXPAY_PLUGIN_DIR . 'assets/js/admin.js', ['jquery'], WC_AUXPAY_VERSION, true);
        wp_localize_script('wc-auxpay-admin', 'WCAuxPayGateway', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wc_auxpay_nonce'),
        ]);
    }

    public function add_settings_page($settings) {
        if (class_exists('WC_Settings_Page')) {
            // Load settings class
            require_once 'includes/class-auxpay-settings.php';
            $settings[] = new WC_AuxPay_Settings();
        }
        return $settings;
    }

    public function auxpay_plugin_action_links($links) {
        $settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=wc_auxpay_settings' ) ) . '">' . __( 'Settings', 'auxpay-payment-gateway' ) . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public function ajax_test_connection() {
        check_ajax_referer('wc_auxpay_nonce', 'nonce');
        $api_key = get_option('wc_auxpay_api_key');
        $api_url = get_option('wc_auxpay_api_url');
        if (!$api_key || !$api_url) {
            wp_send_json_error('Missing API credentials');
        }
        
        delete_option('wc_auxpay_merchant_config');
        $response = wp_remote_post($api_url . '/testConnection', [
            'headers' => [
                'Authorization' => $api_key,
                'Content-Type'  => 'application/json',
            ],
            'timeout' => 30,
        ]);
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!$body) {
            wp_send_json_error('Invalid JSON response');
        }

        // save merchant
        if ($body['status'] == 200) {
            $message = $body['message'] ? explode('merchant', $body['message']) : [];
            $message = isset($message[1]) ? trim($message[1]) : $body['message'];
            update_option('wc_auxpay_merchant_config', esc_html($message));
        }

        wp_send_json_success($body);
    }

    public function ajax_fetch_config() {
        check_ajax_referer('wc_auxpay_nonce', 'nonce');
        $api_key = get_option('wc_auxpay_api_key');
        $api_url = get_option('wc_auxpay_api_url');
        if (!$api_key || !$api_url) {
            wp_send_json_error('Missing API credentials');
        }
        $response = wp_remote_get($api_url . '/getConfig', [
            'headers' => [
                'Authorization' => $api_key,
                'Content-Type'  => 'application/json',
            ],
            'timeout' => 30,
        ]);
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!$body) {
            wp_send_json_error('Invalid JSON response');
        }

        // delete transients
        delete_transient('wc_auxpay_client_token');

        // save config
        update_option('wc_auxpay_3ds_config', $body['data']['aux3DsEnabled']);
        update_option('wc_auxpay_fee_config', $body['data']['fee']);
        update_option('wc_auxpay_payrix_config', $body['data']['payrix']);
        update_option('wc_auxpay_cards_config', $body['data']['cardBrandRestrictions']);

        wp_send_json_success($body);
    }

    public function add_technology_fee(WC_Cart $cart) {
        if (is_admin() && !defined('DOING_AJAX')) return;
        if ('yes' !== get_option('wc_auxpay_fee_enabled')) return;

        $chosen_gateway = WC()->session->get('chosen_payment_method');
        if (!in_array($chosen_gateway, ['wc_auxpay_card', 'wc_auxpay_ach'])) return;

        $config = get_option('wc_auxpay_fee_config');
        if (!is_array($config) || empty($config)) return;

        if ($chosen_gateway === 'wc_auxpay_card') {
            $config = $config['card'];
        } elseif ($chosen_gateway === 'wc_auxpay_ach') {
            $config = $config['ach'];
        }

        if (!is_array($config) || !$config['active']) return;

        $fee_value = floatval($config['value'] ?? 0);
        $fee_type  = $config['type'] ?? 'Percentage';
        $min_fee   = floatval($config['minimum'] ?? 0);
        $cart_total = $cart->get_subtotal() + $cart->get_shipping_total() + $cart->get_taxes_total();
        $cart_total = $cart_total - $cart->get_discount_total();

        $calculated_fee = 0;
        if ($fee_type == 'Percentage') {
            $calculated_fee = ($cart_total * $fee_value) / 100;
        } elseif ($fee_type == 'Fixed') {
            $calculated_fee = $fee_value;
        }

        $final_fee = max($calculated_fee, $min_fee);
        $final_fee = (float) wc_format_decimal($final_fee, 2);
        if ($final_fee > 0) {
            $title = get_option('wc_auxpay_fee_title', 'Processing Fee');
            $title = apply_filters('translate_text', $title, 'auxpay-payment-gateway');
            $cart->add_fee($title, $final_fee, false);
        }
    }


    public function refund_webhook_handler() {
        $body = file_get_contents('php://input');
        $response = json_decode($body, true);

        // Sanitize values
        $amount         = sanitize_text_field($response['data']['Amount']);
        $transaction_id = sanitize_text_field($response['data']['TransactionId']);

        if (empty($amount) || empty($transaction_id)) {
            status_header(400);
            echo 'Invalid payload';
            exit;
        }

        // Find order by transaction ID
        $orders = wc_get_orders(['limit'=> 1, 'meta_key' => '_transaction_id', 'meta_value' => $transaction_id, 'meta_compare' => '=']);
        $order  = $orders ? $orders[0] : false;

        if (!$order) {
            status_header(404);
            echo 'Order not found';
            exit;
        }

        // Prepare refund data
        $amount = wc_format_decimal($amount, 2);
        $reason = !empty($response['data']['Reason']) ? sanitize_text_field($response['data']['Reason']) : 'No reason provided.';

        // Create refund
        $refund = wc_create_refund([
            'amount'   => $amount,
            'reason'   => $reason,
            'order_id' => $order->get_id(),
        ]);

        if (is_wp_error($refund)) {
            $order->add_order_note('Refund webhook failed: ' . esc_html($refund->get_error_message()));
            status_header(500);
            echo esc_html($refund->get_error_message());
            exit;
        }
        else {
            $order->add_order_note(sprintf('Refund of %1$s processed via AuxPay Webhook. Reason: %2$s', wc_price($amount), $reason));
            status_header(200);
            echo 'Ok';
            exit;
        }
    }

    public function config_webhook_handler() {
        $body = file_get_contents('php://input');
        $response = json_decode($body, true);

        // Sanitize values
        $aux3Ds = sanitize_text_field($response['data']['aux3DsEnabled']);
        $fees   = sanitize_text_field($response['data']['fee']);
        $payrix = sanitize_text_field($response['data']['payrix']);
        $cards  = sanitize_text_field($response['data']['cardBrandRestrictions']);

        if (empty($fees) || empty($payrix)) {
            status_header(400);
            echo 'Invalid payload';
            exit;
        }

        // delete transients
        delete_transient('wc_auxpay_client_token');

        // save config
        update_option('wc_auxpay_3ds_config', $aux3Ds);
        update_option('wc_auxpay_fee_config', $fees);
        update_option('wc_auxpay_payrix_config', $payrix);
        update_option('wc_auxpay_cards_config', $cards);

        status_header(200);
        echo 'Ok';
        exit;
    }
}

// Initialize the plugin
new WC_AuxPay_Payment_Gateway();
