<?php

if (!defined('ABSPATH')) {
    exit;
}

class WC_AuxPay_Card extends WC_Payment_Gateway {

    private $api_key;
    private $api_url;
    private $client_token;
    private $environment;
    private $payrix_api;
    private $payrix_mid;
    private $is_payrix;
    private $threeds;
    public  $test_mode;

    public function __construct() {
        $this->id = 'wc_auxpay_card';
        $this->method_title = __('AuxPay - Credit Card', 'auxpay-payment-gateway');
        $this->method_description = 'Pay securely using credit card.';

        $this->has_fields = true;
        $this->supports = ['products', 'subscriptions', 'refunds'];

        $this->init_form_fields();
        $this->init_settings();

        if ('no' === get_option('wc_auxpay_test_enabled')) {
            $this->test_mode   = '';
            $this->environment = 'production';
        }
        else {
            $this->test_mode   = ' - (Test Mode)';
            $this->environment = 'sandbox';
        }

        $payrix = get_option('wc_auxpay_payrix_config');
        if (is_array($payrix) && $payrix['enabled']) {
            $this->payrix_api = $payrix['processorId'];
            $this->payrix_mid = $payrix['gMerchantId'];
            $this->is_payrix  = true;
        }
        else {
            $this->payrix_api = '';
            $this->payrix_mid = '';
            $this->is_payrix  = false;
        }

        $this->threeds     = get_option('wc_auxpay_3ds_config');
        $this->enabled     = $this->get_option('enabled');
        $this->title       = get_option('wc_auxpay_cc_title') . $this->test_mode;
        $this->description = get_option('wc_auxpay_cc_description');
        $this->api_key     = get_option('wc_auxpay_api_key');
        $this->api_url     = get_option('wc_auxpay_api_url');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
        
        // Blocks checkout hook
        add_action('woocommerce_rest_checkout_process_payment_with_context', [$this, 'process_payment_with_context'], 10, 2);

        // Frontend styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);

        // Payment scripts
        add_action('wp_enqueue_scripts', [$this, 'payment_scripts']);

        // AJAX for OTP generation
        add_action('wp_ajax_auxpay_generate_otp', [$this, 'ajax_generate_otp']);
        add_action('wp_ajax_nopriv_auxpay_generate_otp', [$this, 'ajax_generate_otp']);

        // AJAX for cancelling OTP
        add_action('wp_ajax_auxpay_cancel_otp', [$this, 'ajax_cancel_otp']);
        add_action('wp_ajax_nopriv_auxpay_cancel_otp', [$this, 'ajax_cancel_otp']);

        // AJAX for OTP verification
        add_action('wp_ajax_auxpay_verify_otp', [$this, 'ajax_verify_otp']);
        add_action('wp_ajax_nopriv_auxpay_verify_otp', [$this, 'ajax_verify_otp']);

        // AJAX for OTP verification and payment confirmation
        add_action('wp_ajax_auxpay_verify_confirm_otp', [$this, 'ajax_verify_confirm_otp']);
        add_action('wp_ajax_nopriv_auxpay_verify_confirm_otp', [$this, 'ajax_verify_confirm_otp']);
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title'       => 'Enable/Disable',
                'label'       => 'Credit Card Payment',
                'type'        => 'checkbox',
                'default'     => 'no',
            ),
        );
    }

    public function get_gateway() {
        try {
            $response = wp_remote_post($this->api_url . '/gateway-state', [
                'headers' => [
                    'Authorization' => $this->api_key,
                    'Content-Type'  => 'application/json',
                ],
                'timeout' => 10,
            ]);

            if (is_wp_error($response)) {
                throw new Exception(esc_html($response->get_error_message()));
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['success']) && $body['success'] === true && !empty($body['data'])) {
                $this->is_payrix = $this->payrix_api === '' ? false : $body['data']['payrixEnabled'];
                $expires_in = $body['data']['token_expires_in'];
                $this->client_token = $body['data']['client_token'];
                set_transient('wc_auxpay_client_token', $this->client_token, $expires_in);
            }
            else {
                throw new Exception(esc_html($body['message']));
            }
        } catch (Exception $e) {
            $this->debug_log('Gateway State Error: ' . $e->getMessage(), 'error');
            // Get Client Token for AuxPay SDK
            if ($this->is_payrix === false) {
                $this->get_client_token();
            }
        }
    }

    public function get_client_token() {
        try {
            $token = get_transient('wc_auxpay_client_token');

            if ($token) {
                $this->client_token = $token;
            }
            else {
                $response = $this->api_call(['api_key' => $this->api_key], 'client-token');

                if (in_array($response['status'], ['success', '201']) && !empty($response['data'])) {
                    $expires_in = $response['data']['expires_in'];
                    $this->client_token = $response['data']['client_token'];
                    set_transient('wc_auxpay_client_token', $this->client_token, $expires_in);
                }
                else {
                    throw new Exception(esc_html($response['message']));
                }
            }
        } catch (Exception $e) {
            wc_add_notice('Client Token Error: ' . $e->getMessage(), 'error');
            $this->debug_log('Client Token Error: ' . $e->getMessage(), 'error');
            return;
        }
    }

    public function payment_fields() {
        $acceptedCards = [];
        $cards = get_option('wc_auxpay_cards_config');
        $cardLogos = [
            ['key' => 'isVisaEnabled', 'name' => 'Visa', 'logo' => 'assets/img/visa.svg'],
            ['key' => 'isMastercardEnabled', 'name' => 'MC', 'logo' => 'assets/img/mastercard.svg'],
            ['key' => 'isAmexEnabled', 'name' => 'Amex', 'logo' => 'assets/img/amex.svg'],
            ['key' => 'isDiscoverEnabled', 'name' => 'Discover', 'logo' => 'assets/img/discover.svg'],
        ];

        if (is_array($cards) && !empty($cards)) {
            $acceptedCards = array_filter($cardLogos, function($info) use ($cards) {
                return isset($cards[$info['key']]) && $cards[$info['key']] == true;
            });
            $acceptedCards = array_map(function($info) {
                return "<img src='" . esc_url(WC_AUXPAY_PLUGIN_DIR . $info['logo']) . "' alt='" . esc_attr($info['name']) . "' class='cards-img' />";
            }, $acceptedCards);
        }

        echo '<p>' . esc_html($this->description) . '</p>';
        if (!empty($acceptedCards)) {
            echo '<div class="auxpay-accepted-cards">' . wp_kses_post(implode('', $acceptedCards)) . '</div>';
        }
        echo'
            <div class="auxpay-source-message" role="alert"></div>
            <div id="auxpay_card_fields_container"></div>
            <div class="payfields hide">
                <form>
                    <div id="form-container">
                        <label htmlFor="cc_number" class="label hide">Card Number:</label>
                        <div id="cc_number" class="form-row"></div>
                        <div class="card-input-wrapper">
                            <div class="w-50">
                                <label htmlFor="cc_expiry" class="label hide">Expiration:</label>
                                <div id="cc_expiry" class="form-row"></div>
                            </div>
                            <div class="w-50">
                                <label htmlFor="cc_cvv" class="label hide">CVV:</label>
                                <div id="cc_cvv" class="form-row"></div>
                            </div>
                        </div>
                        <label htmlFor="cc_name" class="label hide">Cardholder Name:</label>
                        <div id="cc_name" class="form-row"></div>
                    </div>
                </form>
            </div>';
        echo '<input type="hidden" id="payment_token" name="payment_token" />';
        echo '<input type="hidden" id="payrix_form" name="payrix_form" value="no" />';
        echo '<input type="hidden" id="auxpay_otp" name="auxpay_otp" value="" />';
        wp_nonce_field($this->id, 'wc_auxpay_card_nonce');
    }
    
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        try {
            // Verify nonce
            if (!isset($_POST['wc_auxpay_card_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wc_auxpay_card_nonce'])), $this->id)) {
                wc_add_notice('Security check failed. Please refresh and try again.', 'error');
                return;
            }
            // Collect payment token
            $payment_token = sanitize_text_field(wp_unslash($_POST['payment_token'] ?? ''));
            $payrix_form = sanitize_text_field(wp_unslash($_POST['payrix_form'] ?? ''));
            $auxpay_otp = sanitize_text_field(wp_unslash($_POST['auxpay_otp'] ?? ''));

            if (empty($payment_token)) {
                wc_add_notice('Payment token missing, please try again.', 'error');
                return;
            }

            // Prepare data to send to your API
            $payment_data = $this->prepare_payment_data($order, $payment_token, $payrix_form, $auxpay_otp);
            $total_amount = $order->get_total();
            $phone_number = $payment_data['BillingPhoneNumber'] ?? '';

            // Initiate 3ds flow
            if ($this->threeds && empty($auxpay_otp)) {
                $order->add_order_note('3Ds flow initiated.');
                wc_add_notice('<div class="3ds-required" data-order="'.$order_id.'" data-phone="'.$phone_number.'" data-amount="'.$total_amount.'">OTP verification required to continue.</div>', 'error');
                return;
            }

            // Make the payment
            $response = $this->api_call($payment_data, 'transaction');

            // Handle the response from the payment gateway
            if (in_array($response['status'], ['success', '201']) || $response['data']['Status'] == '1') {
                $transaction_id = sanitize_text_field($response['data']['TransactionId']);
                $order->payment_complete();
                $order->set_transaction_id($transaction_id);
                $order->save();
                $order->add_order_note('Transaction ID: ' . esc_html($transaction_id));
                wc_reduce_stock_levels($order_id);
                return array(
                    'result'   => 'success',
                    'redirect' => $this->get_return_url($order),
                );
            } elseif ($response['status'] == 'error' && ($response['data']['otpSent'] == true || $response['data']['watchdog']['otpSent'] == true)) {
                $transaction_id = sanitize_text_field($response['data']['TransactionId']);
                $order->set_transaction_id($transaction_id);
                $order->save();
                $order->add_order_note('Transaction ID: ' . esc_html($transaction_id));

                if ($this->threeds && !empty($auxpay_otp)) {
                    // OTP already verified rerun the transaction again
                    $order->add_order_note('Decline routing initiated.');
                    $confirm = $this->api_call(['originalTransactionId' => $transaction_id], 'rerun');

                    if (in_array($confirm['status'], ['success', '201'])) {
                        $order->payment_complete();
                        $order->add_order_note('Payment confirmed.');
                        wc_reduce_stock_levels($order_id);
                        return array(
                            'result'   => 'success',
                            'redirect' => $this->get_return_url($order),
                        );
                    } else {
                        $order->add_order_note('Payment failed.');
                        wc_add_notice('<div class="close-otp-model">Payment Declined: ' . $confirm['message'] . '</div>', 'error');
                        $this->debug_log('Payment Declined. order#' . $order_id . ': ' . $confirm['message'], 'error');
                        return;
                    }
                }
                else {
                    // OTP required
                    $order->add_order_note('OTP flow initiated.');
                    wc_add_notice('<div class="otp-required" data-order="'.$order_id.'" data-transaction="'.$transaction_id.'" data-phone="'.$phone_number.'" data-amount="'.$total_amount.'">OTP verification required to continue.</div>', 'error');
                    return;
                }
            } else {
                if ($this->threeds && !empty($auxpay_otp)) {
                    wc_add_notice('<div class="close-otp-model">Payment Failed: ' . $response['message'] . '</div>', 'error');
                } else {
                    wc_add_notice('Payment Failed: ' . $response['message'], 'error');
                }
                $this->debug_log('Card Payment Failed. order#' . $order_id . ': ' . $response['message'], 'error');
                $this->debug_log('Card Payment Failed. payload: ' . json_encode($payment_data), 'error');
                $this->debug_log('Card Payment Failed. response: ' . json_encode($response), 'error');
                return;
            }
        } catch (Exception $e) {
            wc_add_notice('Payment Error: ' . $e->getMessage(), 'error');
            $this->debug_log('Card Payment Error. order#' . $order_id . ': ' . $e->getMessage(), 'error');
            return;
        }
    }

    public function process_payment_with_context($context, $request) {
        if ($context->payment_method === $this->id) {
            $order = $context->order;
            try {
                // Collect payment token
                $payment_token = sanitize_text_field(wp_unslash($context->payment_data['payment_token'] ?? ''));
                $payrix_form = sanitize_text_field(wp_unslash($context->payment_data['payrix_form'] ?? ''));
                $auxpay_otp = sanitize_text_field(wp_unslash($context->payment_data['auxpay_otp'] ?? ''));

                if (empty($payment_token)) {
                    wc_add_notice('Payment token missing, please try again.', 'error');
                    return;
                }

                // Prepare payment data
                $payment_data = $this->prepare_payment_data($order, $payment_token, $payrix_form, $auxpay_otp);

                // Make the payment
                $response = $this->api_call($payment_data, 'transaction');

                // Handle the response from the payment gateway
                if (in_array($response['status'], ['success', '201']) || $response['data']['Status'] == '1') {
                    $transaction_id = sanitize_text_field($response['data']['TransactionId']);
                    $order->payment_complete();
                    $order->set_transaction_id($transaction_id);
                    $order->save();
                    $order->add_order_note('Transaction ID: ' . esc_html($transaction_id));
                    wc_reduce_stock_levels($order->get_id());

                    $request->set_status('success');
                    $request->set_redirect_url($this->get_return_url($order));
                } elseif ($response['status'] == 'error' && ($response['data']['otpSent'] == true || $response['data']['watchdog']['otpSent'] == true)) {
                    $total_amount = $order->get_total();
                    $phone_number = $payment_data['BillingPhoneNumber'] ?? '';
                    $transaction_id = sanitize_text_field($response['data']['TransactionId']);
                    $order->set_transaction_id($transaction_id);
                    $order->save();
                    $order->add_order_note('Transaction ID: ' . esc_html($transaction_id));

                    // OTP required
                    $order->add_order_note('OTP flow initiated.');
                    $request->set_status('failure');
                    $request->set_payment_details([
                        'otp-required' => json_encode([
                            'phone' => $phone_number,
                            'order_id' => $order->get_id(),
                            'transaction_id' => $transaction_id,
                            'amount' => $total_amount,
                        ])
                    ]);
                    return;
                } else {
                    wc_add_notice('Payment Failed: ' . $response['message'], 'error');
                    $this->debug_log('Card Payment Failed. order#' . $order->get_id() . ': ' . $response['message'], 'error');
                    $this->debug_log('Card Payment Failed. payload: ' . json_encode($payment_data), 'error');
                    $this->debug_log('Card Payment Failed. response: ' . json_encode($response), 'error');
                    return;
                }
            } catch (Exception $e) {
                wc_add_notice('Payment Error: ' . $e->getMessage(), 'error');
                $this->debug_log('Card Payment Error. order#' . $order->get_id() . ': ' . $e->getMessage(), 'error');
                return;
            }
        }
    }

    private function prepare_payment_data($order, $payment_token, $payrix_form, $auxpay_otp) {
        $fee_active = false;
        $base_amount = $order->get_total();
        $config = get_option('wc_auxpay_fee_config');
        if (is_array($config['card']) && $config['card']['active'] && 'yes' === get_option('wc_auxpay_fee_enabled')) {
            $fee_active = true;
            $processing_fee = 0;
            $title = get_option('wc_auxpay_fee_title', 'Processing Fee');
            $title = apply_filters('translate_text', $title, 'auxpay-payment-gateway');
            foreach ($order->get_fees() as $fee) {
                if ($fee->get_name() === $title) {
                    $processing_fee = $fee->get_total();
                }
            }
            $base_amount = $order->get_total() - $processing_fee;
        }

        // Extract billing information from the order
        $billing_address = array(
            "BillingCustomerName"   => $order->get_formatted_billing_full_name(),
            "BillingEmail"          => $order->get_billing_email(),
            "BillingPhoneNumber"    => $order->get_billing_phone(),
            "BillingAddress"        => $order->get_billing_address_1(),
            "BillingPostalCode"     => $order->get_billing_postcode(),
            "BillingCity"           => $order->get_billing_city(),
            "BillingState"          => $order->get_billing_state(),
            "BillingCountry"        => $order->get_billing_country(),
            "BillingCountryCode"    => "+1",
            "IpAddress"             => WC_Geolocation::get_ip_address(),
        );

        // Prepare the API request data
        $data = array_merge($billing_address, array(
            "Amount"               => (float) wc_format_decimal($base_amount, 2),
            "token"                => $payment_token,
            "SuggestedMode"        => "Card",
            "ConvenienceFeeActive" => $fee_active,
            "isPayrix"             => $payrix_form === 'yes' ? true : false,
            "isotpverified"        => $auxpay_otp !== '' ? true : false,
            "otp"                  => $auxpay_otp,
            "TransactionType"      => 1,
            "request-origin"       => 'woocommerce'
        ));

        return $data;
    }

    // Generate OTP
    public function ajax_generate_otp() {
        check_ajax_referer($this->id, 'nonce');

        $order_id = absint($_POST['order_id'] ?? 0);
        $order = wc_get_order($order_id);

        $phone = sanitize_text_field(wp_unslash($_POST['phone'] ?? ''));
        $response = $this->api_call(['phoneNumber' => $phone,'expiryMinutes' => 5], 'otp/generate');

        if (in_array($response['status'], ['success', '201'])) {
            if ($order) {
                $order->add_order_note('OTP sent to ' . esc_html($phone) . '.');
            }
            wp_send_json_success($response);
        }
        else {
            wp_send_json_error($response);
        }
    }

    // Cancel OTP
    public function ajax_cancel_otp() {
        check_ajax_referer($this->id, 'nonce');

        $order_id = absint($_POST['order_id'] ?? 0);
        $order = wc_get_order($order_id);

        $transaction_id = sanitize_text_field(wp_unslash($_POST['transaction_id'] ?? ''));
        $response = $this->api_call(['TransactionId' => $transaction_id], 'send-decline-notifications');

        if (in_array($response['status'], ['success', '201'])) {
            if ($order) {
                $order->add_order_note('OTP canceled.');
            }
            wp_send_json_success($response);
        }
        else {
            wp_send_json_error($response);
        }
    }

    // Verify OTP
    public function ajax_verify_otp() {
        check_ajax_referer($this->id, 'nonce');

        $phone = sanitize_text_field(wp_unslash($_POST['phone'] ?? ''));
        $otp = sanitize_text_field(wp_unslash($_POST['otp'] ?? ''));
        $response = $this->api_call(['phoneNumber' => $phone,'otpCode' => $otp], 'otp/verify');

        if (in_array($response['status'], ['success', '201'])) {
            wp_send_json_success($response);
        }
        else {
            wp_send_json_error($response);
        }
    }

    // Verify OTP and confirm payment
    public function ajax_verify_confirm_otp() {
        check_ajax_referer($this->id, 'nonce');

        $order_id = absint($_POST['order_id'] ?? 0);
        $order = wc_get_order($order_id);
        if (!$order) wp_send_json_error(['message' => 'Invalid order ID']);

        $phone = sanitize_text_field(wp_unslash($_POST['phone'] ?? ''));
        $otp = sanitize_text_field(wp_unslash($_POST['otp'] ?? ''));
        $verify = $this->api_call(['phoneNumber' => $phone,'otpCode' => $otp], 'otp/verify');

        if (in_array($verify['status'], ['success', '201'])) {
            $order->add_order_note('OTP verified.');
            $transaction_id = sanitize_text_field(wp_unslash($_POST['transaction_id'] ?? ''));
            $confirm = $this->api_call(['originalTransactionId' => $transaction_id], 'rerun');

            if (in_array($confirm['status'], ['success', '201'])) {
                $order->payment_complete();
                $order->add_order_note('Payment confirmed.');
                wc_reduce_stock_levels($order_id);
                wp_send_json_success(['success' => true, 'redirect' => $this->get_return_url($order)]);
            } else {
                $order->add_order_note('Payment failed.');
                wc_add_notice('Payment Declined: ' . $confirm['message'], 'error');
                $this->debug_log('Payment Declined. order#' . $order_id . ': ' . $confirm['message'], 'error');
                wp_send_json_error(array_merge($confirm, ['rerun' => true]));
            }
        }
        else {
            wp_send_json_error($verify, ['rerun' => true]);
        }
    }

    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        try {
            if (!$order) return new WP_Error('invalid_order', 'Invalid order ID');
            if (!$amount || $amount <= 0) return new WP_Error('invalid_refund_amount', 'Refund amount must be greater than 0');

            // Prepare refund payload
            $payload = array(
                'Amount'         => $amount,
                'Reason'         => $reason,
                'TransactionFor' => "Refund",
                'TransactionId'  => $order->get_transaction_id(),
                'isWoocommerce'  => true
            );

            // Send refund request
            $response = $this->api_call($payload, 'refund');

            // Handle the response from the payment gateway
            if (in_array($response['status'], ['success', '201']) || $response['data']['Status'] == '1') {
                $order->add_order_note(sprintf('Refunded %1$s via AuxPay Gateway. Reason: %2$s', wc_price($amount), $reason ?? 'No reason provided.'));
                return true;
            } else {
                return new WP_Error('refund_failed', $response['message']);
            }
        } catch (Exception $e) {
            return new WP_Error('refund_error', $e->getMessage());
        }
    }

    private function api_call($data, $endpoint) {

        $response = wp_remote_post($this->api_url . '/'. $endpoint, [
            'headers' => [
                'Authorization' => $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($data),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $success = $body['success'] ? 'success' : $body['status'];
        $message = $body['message'] ? $body['message'] : 'Unknown error';

        return array(
            'status' => $success,
            'message' => $message,
            'data' => $body['data'] ?? array()
        );
    }

    private function debug_log($message, $level = 'info') {
        if ('yes' === get_option('wc_auxpay_logs_enabled')) {
            $logger  = wc_get_logger();
            $context = array('source' => 'auxpay-payment-gateway');
            $logger->log($level, $message, $context);
        }
    }

    public function admin_options() {
        $settings_url = admin_url('admin.php?page=wc-settings&tab=wc_auxpay_settings');
        wp_safe_redirect($settings_url);
        exit;
    }

    public function enqueue_styles() {
        if ( !is_checkout() ) return;
        if ($this->enabled !== 'yes') return;
        wp_enqueue_style('wc-auxpay-card-checkout', WC_AUXPAY_PLUGIN_DIR . 'assets/css/auxpay-card-checkout.css', [], WC_AUXPAY_VERSION);
    }

    public function payment_scripts() {
        if ( !is_checkout() ) return;
        if ($this->enabled !== 'yes') return;

        // Get gateway
        $this->get_gateway();

        if ($this->is_payrix) {
            // Payrix script
            $sdk_url  = $this->environment === 'production' ? 'https://api.payrix.com/payFieldsScript' : 'https://test-api.payrix.com/payFieldsScript';
            // Payrix data
            $sdk_data = [
                'apiKey'     => $this->payrix_api,
                'merchantID' => $this->payrix_mid,
                'threeds'    => $this->threeds,
                'gatewayID'  => $this->id,
            ];
        }
        else {
            // AuxPay script
            $sdk_url  = $this->api_url . '/sdk/payment-sdk.js';
            // AuxPay data
            $sdk_data = [
                'environment' => $this->environment,
                'merchantURL' => get_site_url(),
                'clientToken' => $this->client_token,
                'hostedUrl'   => $this->api_url . '/sdk/',
                'threeds'    => $this->threeds,
                'gatewayID'   => $this->id,
            ];
        }

        // Checkout script
        $script_path = WC_AUXPAY_PLUGIN_DIR . 'assets/js/dist/auxpay-card-checkout';
        $script_url  = $script_path . '.js';
		$asset_path  = $script_path . '.asset.php';
		$asset_url   = file_exists($asset_path)
        ? require($asset_path)
        : array(
            'dependencies' => array(),
            'version' => WC_AUXPAY_VERSION
        );

        // Enqueue SDK
        wp_enqueue_script('wc-auxpay-sdk', $sdk_url, ['jquery'], WC_AUXPAY_VERSION, true);
        array_push($asset_url['dependencies'], 'jquery', 'wc-auxpay-sdk');
        // Enqueue Checkout
        wp_enqueue_script('wc-auxpay-card-checkout', $script_url, $asset_url['dependencies'], $asset_url['version'], true);
        // Localize Checkout with SDK data
        wp_localize_script('wc-auxpay-card-checkout', 'WCAuxPayCard', $sdk_data);
    }
}
