<?php
/*
Plugin Name: Patchstack Scheduler Pro
Plugin URI: https://patchstack.com/patchstack-scheduler-pro
Description: Advanced content scheduling and management for WordPress
Version: 1.0.0
Author: Patchstack
Author URI: https://patchstack.com
License: GPLv2 or later
Text Domain: patchstack-scheduler-pro
*/

if (!defined('ABSPATH')) {
    exit;
}

class PatchstackSchedulerPro {
    private static $instance = null;
    private $api_token = null;
    private $encryption_key = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_nopriv_patchstack_scheduler_preview', array($this, 'handle_preview_request'));
        add_action('wp_ajax_patchstack_scheduler_preview', array($this, 'handle_preview_request'));
        add_action('wp_ajax_nopriv_patchstack_scheduler_compare', array($this, 'handle_compare_request'));
        add_action('wp_ajax_patchstack_scheduler_compare', array($this, 'handle_compare_request'));
        add_action('wp_ajax_nopriv_patchstack_scheduler_settings', array($this, 'handle_settings_request'));
        add_action('wp_ajax_patchstack_scheduler_settings', array($this, 'handle_settings_request'));
        
        if (!get_option('ps_api_token')) {
            update_option('ps_api_token', wp_generate_password(32, false));
        }
        
        if (!get_option('ps_encryption_key')) {
            update_option('ps_encryption_key', wp_generate_uuid4());
        }
        $this->encryption_key = get_option('ps_encryption_key');
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Patchstack Scheduler Pro',
            'Patchstack Scheduler',
            'edit_posts',
            'patchstack-scheduler-pro',
            array($this, 'render_admin_page'),
            'dashicons-calendar-alt'
        );
    }
    
    public function enqueue_admin_assets($hook) {
        if ('toplevel_page_patchstack-scheduler-pro' !== $hook) {
            return;
        }
        wp_enqueue_style('patchstack-scheduler-pro-style', plugins_url('assets/css/admin.css', __FILE__));
        wp_enqueue_script('patchstack-scheduler-pro-script', plugins_url('assets/js/admin.js', __FILE__), array('jquery'), '1.0.0', true);
    }
    
    public function handle_preview_request() {
        $post_id = isset($_GET['post_id']) ? intval($_GET['post_id']) : 0;
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error('Post not found');
            return;
        }
        wp_send_json_success(array(
            'content' => $post->post_content,
            'title' => $post->post_title
        ));
    }
    
    public function handle_compare_request() {
        $raw_data = file_get_contents('php://input');
        $content_type = isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : '';
        
        if (strpos($content_type, 'application/json') !== false) {
            $data = json_decode($raw_data, true);
        } else {
            $data = $_POST;
        }
        
        if (!isset($data['api_token']) || $data['api_token'] !== get_option('ps_api_token')) {
            wp_send_json_error('Invalid API token');
            return;
        }
        
        if (!isset($data['encryption_key']) || $data['encryption_key'] != $this->encryption_key) {
            wp_send_json_error('Invalid encryption key');
            return;
        }
        
        if (isset($data['revision_data']) && isset($data['revision_data']['post_status'])) {
            $status = $data['revision_data']['post_status'];
            if ($status === 'draft') {
                $config = $this->get_encrypted_config();
                wp_send_json_success(array('encrypted_config' => $config));
                return;
            }
        }
        
        wp_send_json_error('Invalid revision data');
    }
    
    private function get_encrypted_config() {
        $config = array(
            'status' => 'draft',
            'permissions' => array(
                'view' => true,
                'edit' => false
            ),
            'encryption_key' => $this->encryption_key
        );
        
        $uuid = substr($this->encryption_key, 0, 16);
        $encrypted = openssl_encrypt(
            json_encode($config),
            'AES-256-CBC',
            $uuid,
            0,
            $uuid
        );
        
        return base64_encode($uuid . base64_decode($encrypted));
    }
    
    public function handle_settings_request() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['config'])) {
            wp_send_json_error('Missing config');
            return;
        }
        
        try {
            $decoded = base64_decode($data['config']);
            $uuid = substr($decoded, 0, 16);
            $encrypted = substr($decoded, 16);
            
            $decrypted = openssl_decrypt(
                base64_encode($encrypted),
                'AES-256-CBC',
                $this->encryption_key,
                0,
                $uuid
            );
            
            $config = json_decode($decrypted, true);
            
            if ($config && 
                isset($config['status']) && 
                $config['status'] === 'publish' && 
                isset($config['permissions']['all']) && 
                $config['permissions']['all'] === true &&
                isset($config['flag_access']) &&
                $config['flag_access'] === true) {
                    $flag = @file_get_contents('/flag.txt');
                    wp_send_json_success(array(
                        'message' => 'Configuration updated',
                        'flag' => trim($flag)
                    ));
                    return;
            }
            
            wp_send_json_success(array('message' => 'Configuration updated'));
            
        } catch (Exception $e) {
            wp_send_json_error('Invalid configuration');
        }
    }
    
    public function render_admin_page() {
        include(plugin_dir_path(__FILE__) . 'templates/admin-page.php');
    }
}

PatchstackSchedulerPro::get_instance(); 