# -*- coding: utf-8 -*-
import os
import stat
import time
import shutil
import socket
import filecmp
import threading
import six

from kodi_six import xbmc, xbmcgui
from six.moves import urllib_request
from kodi_six.utils import py2_decode

from torrserver.log import log
from torrserver.osarch import PLATFORM, get_platform
from torrserver.addon import ADDON, ADDON_PATH, ADDON_ID, ADDON_PORT, ADDON_VERSION
from torrserver.util import notify, dialog_yesno, system_information, getLocalizedString, getShortPath, translatePath

try:
    import subprocess
    hasSubprocess = True
    if not PLATFORM["fork"]:
        hasSubprocess = False
except:
    hasSubprocess = False

binary_platform = {}

def ensure_exec_perms(file_):
    st = os.stat(file_)
    os.chmod(file_, st.st_mode | stat.S_IEXEC)
    return file_

def android_get_current_appid():
    with open("/proc/%d/cmdline" % os.getpid()) as fp:
        return fp.read().rstrip("\0")

def get_torrserverd_checksum(path):
    if not os.path.exists(path):
        return ""

    stats = os.stat(path)
    if stats.st_size < 40:
        return ""

    try:
        with open(path) as fp:
            fp.seek(-40, os.SEEK_END)  # we put a sha1 there
            return fp.read()
    except Exception:
        return ""


def download_binary(version, binary, path):
    filedata = urllib_request.urlopen('http://github.com/YouROK/TorrServer/releases/download/{0}/{1}'.format(version, binary))
    CHUNK_SIZE = 16384
    bytes_read = 0
    size = int(filedata.getheader('Content-Length'))
    pDialog = xbmcgui.DialogProgressBG()
    pDialog.create('Загрузка {0}'.format(binary))
    with open(path, 'wb') as f:
        while True:
            chunk = filedata.read(CHUNK_SIZE)
            bytes_read += len(chunk)
            f.write(chunk)
            if len(chunk) < CHUNK_SIZE:
                break
            prc = bytes_read * 100 / size
            if prc > 100:
                prc = 100
            pDialog.update(int(prc), message='Скачано [{0:.2f} Mb]'.format(float(bytes_read/1024/1024)))
        pDialog.close()


def get_torrserver_binary():
    global binary_platform
    binary_platform = get_platform()

    binary = "TorrServer-" + "%(os)s-%(arch)s" % binary_platform + (binary_platform["os"] == "windows" and ".exe" or "")
    binary_dir = os.path.join(ADDON_PATH, "resources", "bin")

    if binary_platform["os"] == "android":
        log.info("Detected binary folder: %s" % binary_dir)
        binary_dir_legacy = binary_dir.replace("/storage/emulated/0", "/storage/emulated/legacy")
        if os.path.exists(binary_dir_legacy):
            binary_dir = binary_dir_legacy
            log.info("Using changed binary folder for Android: %s" % binary_dir)
        app_id = android_get_current_appid()
        xbmc_data_path = os.path.join("/data", "data", app_id)
        if not os.path.exists(xbmc_data_path):
            log.info("%s path does not exist, so using %s as xbmc_data_path" % (
                xbmc_data_path, translatePath("special://masterprofile/")))
            xbmc_data_path = translatePath("special://masterprofile/")

        dest_binary_dir = os.path.join(xbmc_data_path, "files", ADDON_ID, "bin")
    else:
        dest_binary_dir = os.path.join(translatePath(ADDON.getAddonInfo("profile")), "bin")

    binary_path = os.path.join(binary_dir, binary)
    dest_binary_path = os.path.join(dest_binary_dir, binary)

    log.info("Binary detection. Source: %s, Destination: %s" % (binary_path, dest_binary_path))

    if not os.path.exists(binary_path) or ADDON.getSetting("update_binaries") == "true":
        version = ADDON_VERSION
        if not ADDON.getSetting("git_version") == "":
            version = ADDON.getSetting("git_version")
        if version.startswith('Matrix') or version.startswith("matrix"):
            version = "MatriX" + version[6:]
        download = dialog_yesno("LOCALIZE[30000];;" + "TorrServer-%(os)s-%(arch)s" % PLATFORM + " (%s)" % version, "LOCALIZE[30007]")
        system_information()
        if download:
            try:
                download_binary(version, binary, os.path.join(os.path.join(ADDON_PATH, "resources", "bin"), binary))
                notify(getLocalizedString(30001), time=3000)
            except Exception as e:
                log.error(repr(e))
                notify("Binary Not Found", time=2000)
                if dialog_yesno("Не удалось скачать файл. Тогда возможно попробуем открыть его из папки ??"):
                    bFile = xbmcgui.Dialog().browseSingle(1, 'Выберете файл сервера '+binary, 'files')
                    if bFile:
                        shutil.copy(bFile, binary_path)
                        notify(getLocalizedString(30001), time=3000)
                else:
                    return False, False
            else:
                ADDON.setSetting("update_binaries", "false")
                database_path = getShortPath(os.path.join(translatePath(ADDON.getAddonInfo("profile"))))
                if not os.path.exists(database_path):
                    os.mkdir(database_path)
        else:
            try:
                log.info("Source directory (%s):\n%s" % (binary_dir, os.listdir(os.path.join(binary_dir, ".."))))
                log.info("Destination directory (%s):\n%s" % (dest_binary_dir, os.listdir(os.path.join(dest_binary_dir, ".."))))
            except Exception:
                pass
            return False, False

    if os.path.isdir(dest_binary_path):
        log.warning("Destination path is a directory, expected previous binary file, removing...")
        try:
            shutil.rmtree(dest_binary_path)
        except Exception as e:
            log.error("Unable to remove destination path for update: %s" % e)
            system_information()
            return False, False

    if not os.path.exists(dest_binary_path) or not os.path.exists(binary_path) or get_torrserverd_checksum(
            dest_binary_path) != get_torrserverd_checksum(binary_path) or not filecmp.cmp(dest_binary_path, binary_path,
                                                                                          shallow=True):
        log.info("Updating torrserver daemon...")
        try:
            os.makedirs(dest_binary_dir)
        except OSError:
            pass
        try:
            shutil.rmtree(dest_binary_dir)
        except Exception as e:
            log.error("Unable to remove destination path for update: %s" % e)
            system_information()
            pass
        try:
            if not os.path.exists(dest_binary_dir):
                os.mkdir(dest_binary_dir)
            shutil.copy(binary_path, dest_binary_path)
        except Exception as e:
            log.error("Unable to copy to destination path for update: %s" % e)
            system_information()
            return False, False

    # Clean stale files in the directory, as this can cause headaches on
    # Android when they are unreachable
    dest_files = set(os.listdir(dest_binary_dir))
    orig_files = set(os.listdir(binary_dir))
    log.info("Deleting stale files %s" % (dest_files - orig_files))
    for file_ in (dest_files - orig_files):
        path = os.path.join(dest_binary_dir, file_)
        if os.path.isdir(path):
            shutil.rmtree(path)
        else:
            os.remove(path)

    log.info("Binary detection: [ Source: %s, Destination: %s ]" % (binary_path, dest_binary_path))
    return dest_binary_dir, ensure_exec_perms(dest_binary_path)

def clear_fd_inherit_flags():
    # Ensure the spawned Torrserver binary doesn't inherit open files from Kodi
    # which can break things like addon updates. [WINDOWS ONLY]
    try:
        from ctypes import windll

        HANDLE_RANGE = six.moves.xrange(0, 65536)
        HANDLE_FLAG_INHERIT = 1
        FILE_TYPE_DISK = 1

        for hd in HANDLE_RANGE:
            if windll.kernel32.GetFileType(hd) == FILE_TYPE_DISK:
                if not windll.kernel32.SetHandleInformation(hd, HANDLE_FLAG_INHERIT, 0):
                    log.error("Error clearing inherit flag, disk file handle %x" % hd)
    except:
        pass
def start_torrserverd(**kwargs):
    torrserver_dir, torrserver_binary = get_torrserver_binary()

    log.info("Binary dir: %s, item: %s " % (torrserver_dir, torrserver_binary))
    if torrserver_dir is False or torrserver_binary is False:
        return False

    SW_HIDE = 0
    STARTF_USESHOWWINDOW = 1
    ADDON_PORT = ADDON.getSetting("torrserver_port")
    database_path = getShortPath(os.path.join(translatePath(ADDON.getAddonInfo("profile"))))
    args = [torrserver_binary, "-p", ADDON_PORT, "-d", database_path]
    kwargs["cwd"] = torrserver_dir

    if PLATFORM["os"] == "windows":
        args[0] = getShortPath(torrserver_binary)
        kwargs["cwd"] = getShortPath(torrserver_dir)
        si = subprocess.STARTUPINFO()
        si.dwFlags = STARTF_USESHOWWINDOW
        si.wShowWindow = SW_HIDE
        clear_fd_inherit_flags()
        kwargs["startupinfo"] = si
    else:
        env = os.environ.copy()
        env["LD_LIBRARY_PATH"] = "%s:%s" % (torrserver_dir, env.get("LD_LIBRARY_PATH", ""))
        if PLATFORM["os"] == "linux": env["GODEBUG"] = "madvdontneed=1"
        kwargs["env"] = env
        kwargs["close_fds"] = True

    wait_counter = 1
    log.debug("Checking for visible")
    while xbmc.getCondVisibility('Window.IsVisible(10140)') or xbmc.getCondVisibility('Window.IsActive(10140)'):
        if wait_counter == 1:
            log.info('Add-on settings currently opened, waiting before starting...')
        if wait_counter > 300:
            break
        time.sleep(1)
        wait_counter += 1
    log.info("torrserverd: start args: %s, kw: %s" % (args, kwargs))

    if hasSubprocess:
        return subprocess.Popen(args, **kwargs)
    else:
        command = ' '.join(args) + ' ' + ' '.join(kwargs)
        log.info("torrserverd: starting without subprocess: %s" % command)
        return os.system(command)

def shutdown():
    try:
        urllib_request.urlopen("http://127.0.0.1:"+ADDON_PORT+"/shutdown") # matrix
    except Exception:
        try:
            urllib_request.urlopen("http://127.0.0.1:"+ADDON_PORT+"/shutdown", b"") # old
        except Exception:
            pass

def wait_for_abortRequested(proc, monitor):
    monitor.closing.wait()
    log.info("torrserverd: exiting torrserverd daemon")
    try:
        if proc is not None:
            proc.terminate()
    except OSError:
        pass  # Process already exited, nothing to terminate
    log.info("torrserverd: torrserverd daemon exited")

def torrserverd_thread(monitor):
    restart_count = 0
    max_restart = 3
    last_code = 0

    try:
        monitor_abort = xbmc.Monitor()  # For Kodi >= 14
        while not monitor_abort.abortRequested():
            if restart_count > max_restart or last_code == -9:
                if monitor.reboot():
                    log.debug("torrserverd: resetting attempts")
                    restart_count = 0
                    last_code = 0
                    monitor.reboot(False)
                else:
                    time.sleep(5)

                continue
            log.info("torrserverd: starting torrserverd")
            proc = None
            if hasSubprocess:
                proc = start_torrserverd(stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
                if not proc:
                    break
            else:
                start_torrserverd()

            threading.Thread(target=wait_for_abortRequested, args=[proc, monitor]).start()

            if not hasSubprocess:
                break

            if binary_platform["os"] == "windows":
                while proc.poll() is None:
                    log.info(proc.stdout.readline().decode("utf-8"))
            else:
                # Kodi hangs on some Android (sigh...) systems when doing a blocking
                # read. We count on the fact that torrserver daemon flushes its log
                # output on \n, creating a pretty clean output
                import fcntl
                import select
                fd = proc.stdout.fileno()
                fl = fcntl.fcntl(fd, fcntl.F_GETFL)
                fcntl.fcntl(fd, fcntl.F_SETFL, fl | os.O_NONBLOCK)
                while proc.poll() is None:
                    try:
                        to_read, _, _ = select.select([proc.stdout], [], [])
                        for ro in to_read:
                            line = ro.readline()
                            if line == "":  # write end is closed
                                break
                            try:
                                log.info(line.decode("utf-8"))
                            except TypeError:
                                pass
                    except IOError:
                        time.sleep(1)  # nothing to read, sleep

            last_code = proc.returncode
            if monitor_abort.abortRequested():
                break
            if proc.returncode == 0 or proc.returncode == -9 or proc.returncode == -1:
                continue

            if proc.returncode == 0:
                restart_count = 0
                notify(getLocalizedString(30001), time=3000)
            else:
                restart_count += 1
                notify(getLocalizedString(30100), time=3000)

            xbmc.executebuiltin("Dialog.Close(all, true)")
            system_information()
            time.sleep(5)
            if restart_count >= max_restart:
                notify(getLocalizedString(30003), time=3000)
                break

    except Exception as e:
        import traceback
        map(log.error, traceback.format_exc().split("\n"))
        notify("%s: %s" % (getLocalizedString(30004), repr(e).encode('utf-8')))
        raise
