// dependency d3.js

// Normalize d, such that it falls inside [0,1], defined by high and low.
function norm(d, high, low) {
    return (d-low) / (high-low);
}

// Calculate the scalar needed to convert an XYZ color to Lab
function XYZtoLab(t) {
    if(t > Math.pow(6.0/29.0, 3)) {
	return Math.pow(t, 1/3);
    } else {
	return (1/3) * (29/6)*(29/6) * t + 4/29;
    }
}

// Calculate the scalar need to convert an Lab color to XYZ.
function LabToXYZ(t) {
    return t > 6/29 ? t*t*t : 3 * (6/29)*(6/29) * (t-4/29);
}

// Convert CIE Lab coordinates characterized with Illuminant D65 to CIE Lab
// characterized with Illuminant C. This conversion is required since our model
// relies on a D65 characterization, but Schloss and Palmer's pair preference
// function was defined in Illuminant C characterized CIE Lab space.
function IlluminantD65Lab_to_IlluminantCLab(c) {
  var ILLUMINANT_C_X = 98.074;
  var ILLUMINANT_C_Y = 100.0;
  var ILLUMINANT_C_Z = 118.232;

  var ILLUMINANT_D65_X = 95.0470;
  var ILLUMINANT_D65_Y = 100.0;
  var ILLUMINANT_D65_Z = 108.8830;

  // Derive X,Y,Z from Lab by inversing the transformation
  var LabToXYZ_f_x_input = (1/116)*(c.l + 16) + (1/500)*c.a;
  var LabToXYZ_f_y_input = (1/116)*(c.l + 16);
  var LabToXYZ_f_z_input = (1/116)*(c.l + 16) - (1/200)*c.b;

  var X = ILLUMINANT_D65_X * LabToXYZ(LabToXYZ_f_x_input);
  var Y = ILLUMINANT_D65_Y * LabToXYZ(LabToXYZ_f_y_input);
  var Z = ILLUMINANT_D65_Z * LabToXYZ(LabToXYZ_f_z_input);

  var deconstructY = XYZtoLab(Y / ILLUMINANT_C_Y);
  var L = 116 * deconstructY - 16;
  var a = 500 * ( XYZtoLab(X / ILLUMINANT_C_X) - deconstructY );
  var b = 200 * ( deconstructY - XYZtoLab(Z/ILLUMINANT_C_Z) );

  return d3.lab(L,a,b);
}

function pair_preference(c1, c2) {
  // See util/labToAchromaticPreference.h
  if(approximation_equal(c1.a,0) && approximation_equal(c1.b,0) && approximation_equal(c2.a,0) && approximation_equal(c2.b,0)) {
    return getAchromaticPreference(c1, c2);
  }
  if(approximation_equal(c1.a,0) && approximation_equal(c1.b,0)) {
    return getAchromaticPreference(c2, c1);
  } else if (approximation_equal(c2.a,0) && approximation_equal(c2.b,0)) {
    return getAchromaticPreference(c1, c2);
  }

  var labIdx1 = getLabIndex(c1);
  var labIdx2 = getLabIndex(c2);

  var coolness1 = COLORGORICAL_LAB_TO_COOLNESS[labIdx1];
  var coolness2 = COLORGORICAL_LAB_TO_COOLNESS[labIdx2];

  // Convert from D65 to Illuminant C Lab
  var Lab1 = IlluminantD65Lab_to_IlluminantCLab(c1);
  var Lab2 = IlluminantD65Lab_to_IlluminantCLab(c2);

  // Convert Lab Illuminant C to LCH
  var LCH1 = d3.hcl(Lab1);
  var LCH2 = d3.hcl(Lab2);

  var L1_IllC = LCH1.c;
  var L2_IllC = LCH2.c;
  var H1_IllC = LCH1.h;
  var H2_IllC = LCH2.h;

  // hue, lightness, and coolness weights taken from regression in Schloss &
  //  Palmer 2011 to estimate pairwise preference
  var wh = -46.4222; // hueAngleDiff weight
  var wl = 47.6133; // lightnessDiff weight
  var wc = 75.1481; // coolness weight
  // We can drop the constant k = -39.3888

  // These min and max values are taken from the regression normalization
  // Need to normalize based on these to verify the variables are on same range
  var cMax = 36.0;
  var cMin = 4.0;
  var hMax = 179.266981384;
  var hMin = 0.033547949;
  var lMax = 63.3673; // difference in L_max = 112 (IllC), NOT D65 white
  var lMin = 0.0;

  var diffL = Math.abs(L1_IllC - L2_IllC);
  var diffH = Math.abs(H1_IllC - H2_IllC);
  var sumC = coolness1 + coolness2;

  var pp = wl*norm(diffL, lMax, lMin)
              + wh*norm(diffH, hMax, hMin) + wc*norm(sumC, cMax, cMin);

  return pp;
}

// perceptual distance of color1 and color2
function ciede2000(c1, c2)
{
    var kl = 1.0, kc = 1.0, kh = 1.0;

    var C1ab = Math.sqrt(c1.a*c1.a + c1.b*c1.b),
	C2ab = Math.sqrt(c2.a*c2.a + c2.b*c2.b),
	Cabarithmean = (C1ab + C2ab)/2;

    var G = 0.5 * (1 - Math.sqrt(Math.pow(Cabarithmean, 7)/
	(Math.pow(Cabarithmean, 7) + Math.pow(25, 7))));

    // calculate a'
    var a1p = (1+G)*c1.a,
	a2p = (1+G)*c2.a,
	C1p = Math.sqrt(a1p*a1p + c1.b*c1.b),
	C2p = Math.sqrt(a2p*a2p + c2.b*c2.b);

    // Compute the product of chromas and locations at which it is 0
    var Cpprod = C1p * C2p;

    // Make sure that hue is between 0 and 2pi
    var h1p = Math.atan2(c1.b, a1p);
    if(h1p < 0) h1p += 2*Math.PI;
    var h2p = Math.atan2(c2.b, a2p);
    if(h2p < 0) h2p += 2*Math.PI;

    var dL = c2.l - c1.l,
	dC = C2p - C1p;
    // Compute hue distance
    var dhp = h2p - h1p;
    if(dhp > Math.PI) dhp -= 2*Math.PI;
    else if(dhp < -Math.PI) dhp += 2*Math.PI;
    // Set chroma difference to zero if the product of chromas is zero
    if(Cpprod == 0) dhp = 0;

    // CIEDE2000 requires signed hue and chroma differences, differing from older color difference formula
    var dH = 2*Math.sqrt(Cpprod)*Math.sin(dhp/2);

    // Weighting functions
    var Lp = (c1.l + c2.l)/2,
	Cp = (C1p + C2p)/2;
    // Compute average hue
    // avg hue is computed in radians and converted to degrees only where needed
    var hp = (h1p + h2p)/2;
    // Identify positions for which abs hue diff > 180 degrees
    if(Math.abs(h1p - h2p) > Math.PI) hp -= Math.PI;
    // rollover those that are under
    if(hp < 0) hp += 2 * Math.PI;
    // if one of the chroma values = 0, set mean hue to the sum of two chromas
    if(Cpprod == 0) hp = h1p + h2p;

    var Lpm502 = (Lp-50)*(Lp-50), //Lp minus 50 pow 2
	Sl = 1 + 0.015*Lpm502 / Math.sqrt(20+Lpm502),
	Sc = 1 + 0.045*Cp,
	T = 1 - 0.17*Math.cos(hp - Math.PI/6.0)
              + 0.24*Math.cos(2.0*hp)
              + 0.32*Math.cos(3.0*hp + Math.PI/30.0)
              - 0.20*Math.cos(4.0*hp - 63.0*Math.PI/180.0),
	Sh = 1.0 + 0.015*Cp*T,
	delthetarad = (30*Math.PI/180) *
                       Math.exp(-1* (Math.pow((180/Math.PI*hp - 275.0)/25.0, 2))),
	Rc = 2.0 * Math.sqrt(Math.pow(Cp, 7.0)/(Math.pow(Cp, 7.0) + Math.pow(25.0, 7.0))),
	RT = -1.0 * Math.sin(2.0*delthetarad)*Rc;

    var klSl = kl*Sl,
	kcSc = kc*Sc,
	khSh = kh*Sh;

    var de = Math.sqrt(Math.pow(dL/klSl, 2.0) + Math.pow(dC/kcSc, 2.0) + Math.pow(dH/khSh, 2.0) +
                    RT*(dC/kcSc)*(dH/khSh));

    return de;
}

function cie76(c1, c2)
{
    var dL = c2.l - c1.l,
	da = c2.a - c1.a,
	db = c2.a - c1.a;
    var de = Math.sqrt(dL*dL + da*da + db*db);

    return de;
}

function euclidean_of_rgb(c1, c2)
{
    var r_avg = (c1.r + c2.r)/2,
	dR = c1.r - c2.r,
	dG = c1.g - c2.g,
	dB = c1.b - c2.b;
    var de = Math.sqrt(2*dR*dR + 4*dG*dG + 3*dB*dB + r_avg*(dR*dR - dB*dB)/256);

    return de;
}

function euclidean_of_CIELuv(c1, c2)
{
    var dL = c1.l - c2.l,
        du = c1.u - c2.u,
        dv = c1.v - c2.v;
    var de = Math.sqrt(dL*dL + du*du + dv*dv);

    return de;
}
