
% Author: Xavier Bresson (xbresson at math.ucla.edu)
% Last version: Oct 06 2008
% Name: createTrainingSetOfEllipses
% Description: This function computes the signed distance functions of 30
% random ellipses.


function createTrainingSetOfEllipses

close all;


% parameters
Nb_ellipses = 30; % number of ellipses

N = 3000; % number of points for the parametric representation of ellipses

Nx = 128; Ny = Nx; % size of image domain

a_mean = 27; 
b_mean = 22;
a_scale = 2.3; 
b_scale = 0;





%%%%%%%%% generate parametric representation of ellipses %%%%%%%%%
disp('Generate parametric representation of ellipses');
X = zeros(4*N,Nb_ellipses);
Y = zeros(4*N,Nb_ellipses);
Nc = Nx/2;
for e=1:Nb_ellipses

  a = a_mean+ a_scale^2* randn(1); % 
  b = b_mean+ b_scale^2* randn(1);
  
  for cpt=N:-1:1
    X(N-cpt+1,e) = Nc + (cpt-1)* a /(N-1);
    Y(N-cpt+1,e) = sqrt( a^2- ( X(N-cpt+1,e)-Nc )^2 )* b/a + Nc;
    if (cpt==N)
      Y(N-cpt+1,e) = real(Y(N-cpt+1,e));
    end
  end
  
  for cpt=1:N
    X(N+cpt,e) = Nc - (cpt-1)* a/ (N-1);
    Y(N+cpt,e) = sqrt( a^2- ( X(N+cpt,e)-Nc )^2 )* b/a + Nc;
    if (cpt==N)
      Y(N+cpt,e) = real(Y(N+cpt,e));
    end
  end
  
  for cpt=N:-1:1
    X(3*N-cpt+1,e) = Nc - (cpt-1)* a/ (N-1);
    Y(3*N-cpt+1,e) = -sqrt( a^2- ( X(3*N-cpt+1,e)-Nc )^2 )* b/a + Nc;
    if (cpt==N)
      Y(3*N-cpt+1,e) = real(Y(3*N-cpt+1,e));
    end
  end
  
  for cpt=1:N
    X(3*N+cpt,e) = Nc + (cpt-1)* a/ (N-1);
    Y(3*N+cpt,e) = -sqrt( a^2- ( X(3*N+cpt,e)-Nc )^2 )* b/a + Nc;
    if (cpt==N)
      Y(3*N+cpt,e) = real(Y(3*N+cpt,e));
    end
  end
  
end
%%%%%%%%% END generate parametric representation of ellipses %%%%%%%%%




%%%%%%%%% generate non-parametric (signed distance function) representation of ellipses %%%%%%%%%
disp('Generate signed distance functions (it may take time...)');
training_set_curves = zeros(Ny,Nx,Nb_ellipses);
for e=1:Nb_ellipses
  for cpt=1:4*N
    training_set_curves(round(Y(cpt,e)),round(X(cpt,e)),e) = 1;
  end
  training_set_curves(:,:,e) = double(bwmorph(training_set_curves(:,:,e),'ske',Inf));
end

training_set_SDF = zeros(Ny,Nx,Nb_ellipses);
fMaxDist = Nx*Ny;
for e=1:Nb_ellipses
    %e
    SDF = double(bwfill(training_set_curves(:,:,e),'holes')); SDF(SDF<0.5)=-1;
    scale = 2; Ng = ceil(6*scale); Gaussian = fspecial('gaussian',[Ng Ng],scale); SDF = conv2(SDF,Gaussian,'same');
    SDF = sdf_mex(single(SDF),single(fMaxDist),int32(1),int32(1)); SDF=double(SDF);
    %figure(14); imagesc(SDF); hold on; contour(SDF, [0 0],'m');
    training_set_SDF(:,:,e) = SDF;
    %pause
end
training_set = training_set_SDF;

disp('Save training set of ellipses');
savefile = '../data/trainingSetEllipses.mat';
save(savefile,'training_set');
%%%%%%%%% END generate non-parametric (signed distance function) representation of ellipses %%%%%%%%%




%%%%%%%%% compute mean shape %%%%%%%%%
Sum = zeros(Ny,Nx);
for cpt = 1:Nb_ellipses
  Sum = Sum + training_set_SDF(:,:,cpt);
end
MeanData = Sum/ Nb_ellipses;
%%%%%%%%% END compute mean shape %%%%%%%%%


%%%%%%%%% display results %%%%%%%%%
cpt_fig = 1;
figure(cpt_fig); imagesc(zeros(Ny,Nx),[-10 0]); colormap(gray); hold on;
for cpt=1:Nb_ellipses
    contour(training_set_SDF(:,:,cpt),[0 0],'m');
end
contour(MeanData,[0 0],'k');
f=3; truesize(cpt_fig,[round(f*Ny) round(f*Nx)]);
title('Aligned contours');


disp('Push any button to display all ellipses');
cpt_fig = 2;
figure(cpt_fig);
for cpt=1:Nb_ellipses
    imagesc(training_set_SDF(:,:,cpt)); hold on; axis off;
    contour(training_set_SDF(:,:,cpt), [0 0],'m'); colorbar; contour(MeanData,[0 0],'k');
    title( [ 'Ellipse nb = ',num2str( double(cpt) ) ] );
    f=3; truesize(cpt_fig,[round(f*Ny) round(f*Nx)]);
    pause
end
%%%%%%%%% END display results %%%%%%%%%





