
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acsp.c
// Description: General function that handles the boundary, shape and region-based terms
// Read [X. Bresson, P. Vandergheynst and J.-P. Thiran, A Variational Model for Object Segmentation
// Using Boundary Information and Shape Prior Driven by the Mumford-Shah Functional,
// IJCV 68(2), 2006, p145-162.]. See specifically Eqs (4) and (29-33).


#include "acspCommon.h"



///////////////////////////////////////////////////////
// Sub-functions (Allocation and free memory)
///////////////////////////////////////////////////////


/****************************************/
int iArrayMemoryAllocation
(
float ***pppfArray,
int   iNbCols, /* = Nx */
int   iNbRows  /* = Ny */
)

{
    int i;
    
    (*pppfArray) = (float **) calloc( (unsigned)(iNbCols), sizeof(float *) );
    if (!(*pppfArray))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    for (i=0; i<iNbCols; i++)
    {
        (*pppfArray)[i] = (float *) calloc( (unsigned)(iNbRows), sizeof(float)  );
        if (!(*pppfArray)[i])
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
    }
    
    
    return(1);
}
/****************************************/

/****************************************/
void vArrayMemoryFree
(
float ***pppfArray,
int   iNbCols,
int   iNbRows
)

{
    int i;
    for (i=0; i<iNbCols; i++)
        free( (float *) ((*pppfArray)[i]) );
    free( (float **) (*pppfArray) );
}
/****************************************/

/****************************************/
int iArray3DMemoryAllocation
(
float ****ppppfArray,
int   iNbArray,
int   iNbCols, /* = Nx */
int   iNbRows  /* = Ny */
)

{
    int  i,j;
    
    (*ppppfArray) = (float ***) calloc( (unsigned)(iNbArray), sizeof(float **) );
    if (!(*ppppfArray))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    for (i=0; i<iNbArray; i++)
    {
        (*ppppfArray)[i] = (float **) calloc( (unsigned)(iNbCols), sizeof(float *)  );
        if (!(*ppppfArray)[i])
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        for (j=0; j<iNbCols; j++)
        {
            (*ppppfArray)[i][j] = (float *) calloc( (unsigned)(iNbRows), sizeof(float)  );
            if (!(*ppppfArray)[i][j])
            {
                mexPrintf("Memory allocation failure\n");
                return(0);
            }
        }
    }
    
    
    return(1);
}
/****************************************/

/****************************************/
void vArray3DMemoryFree
(
float ****ppppfArray,
int   iNbArray,
int   iNbCols,
int   iNbRows
)

{
    int i,j;
    
    for (i=0; i<iNbArray; i++)
    {
        for (j=0; j<iNbCols; j++)
            free( (float *) ((*ppppfArray)[i][j]) );
        free( (float **) ((*ppppfArray)[i]) );
    }
    free( (float ***) (*ppppfArray) );
}
/****************************************/

/****************************************/
int iArray1DMemoryAllocation
(
float **ppfArray,
int   iNbElts
)

{
    
    (*ppfArray) = (float *) calloc( (unsigned)(iNbElts), sizeof(float) );
    if (!(*ppfArray))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    return(1);
}
/****************************************/

/****************************************/
void vArray1DMemoryFree
(
float **ppfArray,
int   iNbElts
)

{
    free( (float *) (*ppfArray) );
}
/****************************************/


/****************************************/
int iStructureAcspMemoryAllocation
(
struct sAcspTag *psAcsp,
int    iDimIm[],
int    iDimPca[],
float  *pfVecBoundaryTerm,
float  *pfVecShapeTerm,
float  *pfVecRegionTerm
)

{
    
    int    iNx, iNy, iNxPca, iNyPca, ip;
    float  fWeightBoundaryTerm, fWeightShapeTerm, fWeightRegionTerm;
    
    
    
    iNy = iDimIm[0];
    iNx = iDimIm[1];
    
    iNyPca = iNy;
    iNxPca = iNx;
    ip  = iDimPca[2];
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0];
    fWeightShapeTerm = pfVecShapeTerm[0];
    fWeightRegionTerm = pfVecRegionTerm[0];
    
    
    if (!iArrayMemoryAllocation(&(psAcsp->ppfPhi),iNx,iNy))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    if (!iArrayMemoryAllocation(&(psAcsp->ppfDiracPhi),iNx,iNy))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
        
    if (!iArrayMemoryAllocation(&(psAcsp->ppfBoundaryTerm),iNx,iNy))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    if (!iArrayMemoryAllocation(&(psAcsp->ppfShapeTerm),iNx,iNy))
    {
        mexPrintf("Memory allocation failure\n");
        return(0);
    }
    
    
    if ( fWeightBoundaryTerm >0.0 || fWeightShapeTerm> 0.0 )
    {
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDxp),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDxm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDyp),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDym),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfNormGradPhiPlus),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfNormGradPhiMinus),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
    }
    
    
    if ( fWeightBoundaryTerm >0.0 )
    {
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfCurvatureACTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfAttractionValleyACTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
    }
    
    
    
    
    if ( fWeightShapeTerm >0.0 )
    {
        if (!iArrayMemoryAllocation(&(psAcsp->ppfAttractionValleyShapeTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfSumPcaShapeTerm),ip))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
    }
    
    
    if ( fWeightShapeTerm >0.0 || fWeightRegionTerm >0.0 )
    {
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfPCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if ( (iNyPca != iNy) || (iNxPca != iNx) )
            if (!iArrayMemoryAllocation(&(psAcsp->ppfPCAFunctionTemp),iNxPca,iNyPca))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
            }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDxPCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDyPCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray3DMemoryAllocation(&(psAcsp->pppfPcaEigenVectors),ip,iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfSumPca),ip))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfMaxSumPcaFar),ip))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfMaxSumPcaClose),ip))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfVectorSpatialTransformation),7))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
    }
    
    if ( fWeightRegionTerm >0.0 )
    {
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfDiracPCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfHeavisidePCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfNegHeavisidePCAFunction),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfUinRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfUoutRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfUinTempRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfUoutTempRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfMSRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfFidelityRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfGradRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArray1DMemoryAllocation(&(psAcsp->pfSumPcaRegionTerm),ip))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfG2UinRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfG2UoutRegionTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
        if (!iArrayMemoryAllocation(&(psAcsp->ppfPCACurvatureShapeTerm),iNx,iNy))
        {
            mexPrintf("Memory allocation failure\n");
            return(0);
        }
        
    }
    
    return(1);
    
}
/****************************************/



/****************************************/
void vStructureAcspMemoryFree
(
struct sAcspTag sAcsp,
int    iDimIm[],
int    iDimPca[],
float  *pfVecBoundaryTerm,
float  *pfVecShapeTerm,
float  *pfVecRegionTerm
)

{
    
    int    iNx, iNy, iNxPca, iNyPca, ip;
    float  fWeightBoundaryTerm, fWeightShapeTerm, fWeightRegionTerm;
    
    iNy = iDimIm[0];
    iNx = iDimIm[1];
    
    iNyPca = iNy;
    iNxPca = iNx;
    ip  = iDimPca[2];
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0];
    fWeightShapeTerm = pfVecShapeTerm[0];
    fWeightRegionTerm = pfVecRegionTerm[0];
    
    vArrayMemoryFree(&(sAcsp.ppfPhi),iNx,iNy);
    vArrayMemoryFree(&(sAcsp.ppfDiracPhi),iNx,iNy);
    vArrayMemoryFree(&(sAcsp.ppfBoundaryTerm),iNx,iNy);
    vArrayMemoryFree(&(sAcsp.ppfShapeTerm),iNx,iNy);
    
    
    if ( fWeightBoundaryTerm >0.0 || fWeightShapeTerm> 0.0 )
    {
        vArrayMemoryFree(&(sAcsp.ppfDxp),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfDxm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfDyp),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfDym),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfNormGradPhiPlus),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfNormGradPhiMinus),iNx,iNy);
    }
    
    if ( fWeightBoundaryTerm >0.0 )
    {
        vArrayMemoryFree(&(sAcsp.ppfCurvatureACTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfAttractionValleyACTerm),iNx,iNy);
    }
    
    
    if ( fWeightShapeTerm >0.0 )
    {
        vArrayMemoryFree(&(sAcsp.ppfAttractionValleyShapeTerm),iNx,iNy);
        vArray1DMemoryFree(&(sAcsp.pfSumPcaShapeTerm),ip);
    }
    
    if ( fWeightShapeTerm >0.0 || fWeightRegionTerm >0.0 )
    {
        vArrayMemoryFree(&(sAcsp.ppfPCAFunction),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfDxPCAFunction),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfDyPCAFunction),iNx,iNy);
        vArray3DMemoryFree(&(sAcsp.pppfPcaEigenVectors),ip,iNx,iNy);
        vArray1DMemoryFree(&(sAcsp.pfSumPca),ip);
        vArray1DMemoryFree(&(sAcsp.pfMaxSumPcaFar),ip);
        vArray1DMemoryFree(&(sAcsp.pfMaxSumPcaClose),ip);
        vArray1DMemoryFree(&(sAcsp.pfVectorSpatialTransformation),7);
    }
    
    
    if ( fWeightRegionTerm >0.0 )
    {
        vArrayMemoryFree(&(sAcsp.ppfDiracPCAFunction),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfHeavisidePCAFunction),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfNegHeavisidePCAFunction),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfUinRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfUoutRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfUinTempRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfUoutTempRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfMSRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfFidelityRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfGradRegionTerm),iNx,iNy);
        vArray1DMemoryFree(&(sAcsp.pfSumPcaRegionTerm),ip);
        vArrayMemoryFree(&(sAcsp.ppfG2UinRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfG2UoutRegionTerm),iNx,iNy);
        vArrayMemoryFree(&(sAcsp.ppfPCACurvatureShapeTerm),iNx,iNy);
    }
    
}
/****************************************/









///////////////////////////////////////////////////////
// Main function
///////////////////////////////////////////////////////

int iAcShapePrior // extern procedure
(
int    iDimIm[],
int    iDimPca[],
float  *pfVecGeneralParameters,
float  *pfPhiAcIn, // input varphi in Eqs (29-33)
float  *pfPhiAcOut, // output varphi in Eqs (29-33)
float  *pfIm, // I in Eqs (29-33)
float  *pfVecBoundaryTerm,
float  *pfEdgeMap, // g in Eq (29)
float  *pfVecShapeTerm,
float  *pfPCAFunctionOut, // hat{phi} in Eqs (4)
float  *pfVecRigidTransformationsOut, // x_T in Eqs (29-33)
float  *pfVpcaOut, // x_pca in Eqs (29-33)
float  *pfEVec, // W_p in Eqs (4)
float  *pfEVal, // lambda_k
float  *pfPCAMeanValue, // bar{phi} in Eqs (4)
float  *pfVecRegionTerm,
float  *pfMSRegionTerm, // s = u_in*H(\hat{phi}) + u_out*(1-H(\hat{phi}))
float  *pfUinRegionTerm, // u_in in Eqs (29-33)
float  *pfUoutRegionTerm, // u_out in Eqs (29-33)
float  *pfTemp
)

{
    
    float  fDistNarrowBand, fWeightShapeTerm, fWeightRegionTerm, fWeightBoundaryTerm, fDeltaT;
    float  fDistanceMaxComputations;
    int    iDisplayResults, iNbTemporalSteps;
    int    iNbIterSDF, ip, iTypeSpatialTransformation, iNx, iNy, iNxPca, iNyPca, ix, iy, iNbIter, i;
    struct sAcspTag sAcsp;
    
    
    
    // values of parameters
    iDisplayResults = (int) pfVecGeneralParameters[0]; // Display intermediate results = YES or NO
    
    iNy = iDimIm[0];
    iNx = iDimIm[1];
    
    iNyPca = iNy;
    iNxPca = iNx;
    
    fDistNarrowBand = 2.0; // distance narrowband around zero level set. width of narrowband is 2.0
    fDistanceMaxComputations = fDistNarrowBand; // computations are done in a narrowband around zero level set
    
    sAcsp.iSplineDegree = 3; // spatial transformations are done with bsplines of degree 3
    
    sAcsp.fDiracEps = 1.5; // value of epsilon used in the regularization/approximation of the Dirac and Heaviside functions
    sAcsp.fHeaviside0 = HEAVISIDE(0.0,sAcsp.fDiracEps);
    
    fWeightRegionTerm = pfVecRegionTerm[0]; // weight of the region-based term, beta_r in Eqs (29-33)
    
    fWeightShapeTerm = pfVecShapeTerm[0]; // weight of shape-based term, beta_s in Eqs (29-33)
    
    iNbTemporalSteps = (int) pfVecGeneralParameters[1]; // number of iterations
    iNbIterSDF = (int) pfVecGeneralParameters[2]; // iNbTemporalSteps%iNbIterSDF = number of re-distancing of level set function
    fDeltaT = pfVecGeneralParameters[3]; // temporal step
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0]; // weight of boundary-based term, beta_b in Eqs (29-33)
    
    ip = iDimPca[2]; // number of eigenvectors in PCA
    iTypeSpatialTransformation = (int) pfVecShapeTerm[2]; // type of spatial transformations = RIGID or AFFINE
    
    
    
    
    // memory allocation
    if (!iStructureAcspMemoryAllocation(&sAcsp,iDimIm,iDimPca,pfVecBoundaryTerm,pfVecShapeTerm,pfVecRegionTerm))
    {
        mexPrintf("Memory allocation failure for the structure sAcsp\n");
        return(0);
    }
    
    
    // init u_in and u_out
    if ( fWeightRegionTerm > 0.0 )
    {
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
        {
            sAcsp.ppfUinRegionTerm[ix][iy] = pfUinRegionTerm[ix*iNy+ iy];
            sAcsp.ppfUoutRegionTerm[ix][iy] = pfUoutRegionTerm[ix*iNy+ iy];
            
            sAcsp.ppfUinTempRegionTerm[ix][iy] = pfUinRegionTerm[ix*iNy+ iy];
            sAcsp.ppfUoutTempRegionTerm[ix][iy] = pfUoutRegionTerm[ix*iNy+ iy];
            }
    }
    
    
    // find max value of mean PCA and save center of PCA shapes
    if ( fWeightShapeTerm>0.0 || fWeightRegionTerm> 0.0 )
    {
        pfVecRigidTransformationsOut[0] -= 1.0; // From matlab -> C 
        pfVecRigidTransformationsOut[1] -= 1.0;
        
        sAcsp.fMaxPCAMeanValue = 0.0;
        for (ix=0; ix< iNxPca; ix++)
            for (iy=0; iy< iNyPca; iy++)
                if (ABS(pfPCAMeanValue[ix*iNyPca+ iy])>ABS(sAcsp.fMaxPCAMeanValue))
                    sAcsp.fMaxPCAMeanValue = pfPCAMeanValue[ix*iNyPca+ iy];
        
        sAcsp.fPhiCx = pfVecRigidTransformationsOut[0];
        sAcsp.fPhiCy = pfVecRigidTransformationsOut[1];
    }
    
    
    // init active contour
    for (ix=0; ix< iNx; ix++)
        for (iy=0; iy< iNy; iy++)
            sAcsp.ppfPhi[ix][iy] = pfPhiAcIn[ix*iNy+ iy]; // varphi in Eqs (29-33)
    
    
    
    // main loop
    for (iNbIter=0; iNbIter< iNbTemporalSteps; iNbIter++)
    {
        
        sAcsp.iNbIter = iNbIter;
        
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
        {
            if ( ABS(sAcsp.ppfPhi[ix][iy]) < 1.5* sAcsp.fDiracEps )
                sAcsp.ppfDiracPhi[ix][iy] = DIRAC(sAcsp.ppfPhi[ix][iy],sAcsp.fDiracEps); // compute Dirac(varphi) in Eqs (29-33)
            }
        
        
        if ( fWeightBoundaryTerm> 0.0 || fWeightShapeTerm> 0.0 )
            vComputeGrad(iNx,iNy,sAcsp,pfTemp); // compute upwind gradient of level set function varphi, |nabla varphi|
        
        
        if ( fWeightBoundaryTerm> 0.0 )
        {
            // compute attraction term in boundary-based term, <nabla g,nabla varphi/|nabla varphi|>*dirac(varphi)
            vComputeAttractionValleyTerm(iNx,iNy,sAcsp,pfEdgeMap,pfVecBoundaryTerm,iDisplayResults,pfTemp); 
            
            // compute curvature term in boundary-based term, kappa*g*dirac(varphi)
            vComputeCurvatureTerm(iNx,iNy,sAcsp,pfEdgeMap,pfVecBoundaryTerm,iDisplayResults,pfTemp); 
            
            // compute boundary-based term
            vComputeBoundaryTerm(iNx,iNy,sAcsp,pfEdgeMap,pfVecBoundaryTerm,iDisplayResults,pfTemp); 
        }
       
        
        if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm< 1e-6 )
        {
            vComputeCentroidAC(iNx,iNy,sAcsp,&sAcsp.fPhiCx,&sAcsp.fPhiCy,&sAcsp.fPhiLength,iDisplayResults); // compute center of current active contour
            
            vComputeSimilarityFunction(iNx,iNy,sAcsp,iDimPca,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,pfVecRegionTerm,pfVecGeneralParameters,pfTemp); // update PCA function, hat{phi} in eq 29.
            
            vComputeRegistrationParameters(iNx,iNy,sAcsp,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,fDeltaT,0.0,pfVecRegionTerm,pfIm,iNbIter,
            iNbTemporalSteps,pfVecGeneralParameters,pfTemp); // update rigid and pca parameters, x_T and x_pca in eqs 29-32
            
            vComputeAttractionValleyShapeTerm(iNx,iNy,sAcsp,pfVecShapeTerm,iDisplayResults,pfTemp); // <nabla hat{phi}^2,nabla varphi>*dirac(varphi)
            
            vComputeShapeTerm(iNx,iNy,sAcsp,pfVecShapeTerm,iDisplayResults,pfTemp); // compute shape-based term
        }
        
        if ( fWeightRegionTerm> 0.0 && fWeightShapeTerm< 1e-6 )
        {
            vComputeSimilarityFunction(iNx,iNy,sAcsp,iDimPca,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,pfVecRegionTerm,pfVecGeneralParameters,pfTemp); // update PCA function, hat{phi} in eq 29.
            
            vComputeRegistrationParameters(iNx,iNy,sAcsp,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,fDeltaT,0.0,pfVecRegionTerm,pfIm,iNbIter,
            iNbTemporalSteps,pfVecGeneralParameters,pfTemp); // update rigid and pca parameters, x_T and x_pca in eqs 29-32
        }
        
        if ( fWeightRegionTerm> 0.0 && fWeightShapeTerm> 0.0 )
        {
            vComputeCentroidAC(iNx,iNy,sAcsp,&sAcsp.fPhiCx,&sAcsp.fPhiCy,&sAcsp.fPhiLength,iDisplayResults); // compute center of current active contour
            
            vComputeSimilarityFunction(iNx,iNy,sAcsp,iDimPca,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,pfVecRegionTerm,pfVecGeneralParameters,pfTemp);  // update PCA function, hat{phi} in eq 29.
            
            vComputeRegistrationParameters(iNx,iNy,sAcsp,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
            pfVecRigidTransformationsOut,pfVecShapeTerm,fDeltaT,0.0,pfVecRegionTerm,pfIm,iNbIter,
            iNbTemporalSteps,pfVecGeneralParameters,pfTemp); // update rigid and pca parameters, x_T and x_pca in eqs 29-32
            
            vComputeAttractionValleyShapeTerm(iNx,iNy,sAcsp,pfVecShapeTerm,iDisplayResults,pfTemp); // <nabla hat{phi}^2,nabla varphi>*dirac(varphi)
            
            vComputeShapeTerm(iNx,iNy,sAcsp,pfVecShapeTerm,iDisplayResults,pfTemp); // compute shape-based term
        }
        
        
        // update level set function (active contour), varphi in eq 30.
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                    sAcsp.ppfPhi[ix][iy] += fDeltaT* ( fWeightBoundaryTerm* sAcsp.ppfBoundaryTerm[ix][iy] + fWeightShapeTerm* sAcsp.ppfShapeTerm[ix][iy] );
        
        
        
        // Re-distance the level set function as a signed distance function
        if ( iNbIterSDF>0 )
            if ( iNbIter%iNbIterSDF==0 )
                if ( !iFM2D_SDF(sAcsp.ppfPhi,iDimIm,fDistanceMaxComputations,iDisplayResults,pfTemp) )
        {
            mexPrintf("\n\nError in iFM2D_SDF: return an array with null elements\n");
            for (i=0; i< iDimIm[0]* iDimIm[1]; i++)  pfPhiAcOut[i] = 0.0;
            return(0);
                }
        
    }
    
    
    if ( fWeightBoundaryTerm> 0.0 || fWeightShapeTerm> 0.0 )
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
                pfPhiAcOut[ix*iNy+ iy] = sAcsp.ppfPhi[ix][iy]; // output active contour
    
    
    if ( fWeightShapeTerm>0.0 || fWeightRegionTerm> 0.0 )
    {
        vComputeFinalPcaFunction(iNx,iNy,sAcsp,iDimPca,pfPCAFunctionOut,pfEVec,pfEVal,pfPCAMeanValue,ip,pfVpcaOut,iTypeSpatialTransformation,
        pfVecRigidTransformationsOut,pfVecShapeTerm,pfVecGeneralParameters,pfTemp); // compute final PCA function, hat{phi} in 29.
        
        pfVecRigidTransformationsOut[0] += 1.0; // From C -> matlab
        pfVecRigidTransformationsOut[1] += 1.0;
    }
    
    
    
    // output functions u_in and u_out and piecewise-smooth approximation of the image
    if ( fWeightRegionTerm > 0.0 )
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
    {
        pfUinRegionTerm[ix*iNy+ iy] = sAcsp.ppfUinRegionTerm[ix][iy]; // u_in
        pfUoutRegionTerm[ix*iNy+ iy] = sAcsp.ppfUoutRegionTerm[ix][iy]; // u_out 
        pfMSRegionTerm[ix*iNy+ iy] = sAcsp.ppfUinRegionTerm[ix][iy]* sAcsp.ppfHeavisidePCAFunction[ix][iy] + 
        sAcsp.ppfUoutRegionTerm[ix][iy]* sAcsp.ppfNegHeavisidePCAFunction[ix][iy]; // s = u_in*H(\hat{phi}) + u_out*(1-H(\hat{phi}))
            }
    
    
    // free memory
    vStructureAcspMemoryFree(sAcsp,iDimIm,iDimPca,pfVecBoundaryTerm,pfVecShapeTerm,pfVecRegionTerm);
    
    
    return(1);
    
}




/****************** End of file *******************/
