
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspCommon.h
// Description: This file holds common functions, definitions and
// structures.


#ifndef COMMON_H
#define COMMON_H


#include <mex.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <stddef.h>
#include <float.h>
#include <string.h>


#define PI 3.1415926
#define EPS 0.0001
#define INF 1024.0
#define EPS_FM 0.001
#define DBL_EPSILON2 0

#define MODE_PCA_DECOMPOSITION 0
#define MODE_PCA_RECONSTRUCTION 1
#define MODE_PCA_PROJECTION 2

#define RIGID 0
#define AFFINE 1

#define YES 0
#define NO 1



#define SWAP(a,b,tmp) (tmp)=(a); (a)=(b); (b)=(tmp)

#define MAX(a,b) ( a > b ? a : b )

#define MIN(a,b) ( a < b ? a : b )

#define ABS(x) ( x >= 0.0 ? x : -(x) )

#define DIRAC(x,eps) (fabs(x) > eps ? 0.0 : 0.5/ eps* ( 1.0 + cos(PI*x/eps) ))

#define HEAVISIDE(x,eps) ( x> eps ? 1.0 : ( x< -eps ? 0.0 : 0.5*( 1.0 + x/eps + sin(PI*x/eps)/PI ) ) )

#define SIGN(x) ( x > 0.0 ? 1.0 : -1.0 )

#define SQR(x) (x)*(x)

#define BOUNDEDNESS(Value,Max) ( fabs(Value)>Max ? ( Value>0.0 ? Max : -Max ) : Value )

#define MAX4(a,b,c,d) MAX(MAX(a,b), MAX(c,d))

#define HEAVISIDE_OFFSET(x,eps,offset) ( x> offset+eps ? 1.0 : ( x< offset-eps ? 0.0 : 0.5*( 1.0 + (x-offset)/eps + sin(PI*(x-offset)/eps)/PI ) ) )

#define MAX5(a,b,c,d,e) MAX( MAX(MAX(a,b), MAX(c,d)), e )
#define MIN5(a,b,c,d,e) MIN( MIN(MIN(a,b), MIN(c,d)), e )

#define SWAP_FM(a,b,tmp) tmp=a; a=b; b=tmp

#ifndef HAVE_RINT 
#define rint(A) floor((A)+(((A) < 0)? -0.5 : 0.5)) 
#endif



// general structure
struct sAcspTag {
  float  **ppfPhi;
  float  **ppfDiracPhi;
  float  **ppfDxp;
  float  **ppfDxm;
  float  **ppfDyp;
  float  **ppfDym;
  float  **ppfNormGradPhiPlus;
  float  **ppfNormGradPhiMinus;

  float  **ppfCurvatureACTerm;
  float  **ppfAttractionValleyACTerm;
  float  **ppfBoundaryTerm;

  float  **ppfAttractionValleyShapeTerm;
  float  **ppfShapeTerm;
  float  **ppfPCAFunction;
  float  **ppfPCAFunctionTemp;
  float  **ppfDxPCAFunction;
  float  **ppfDyPCAFunction;
  float  ***pppfPcaEigenVectors;
  float  *pfSumPca;
  float  *pfMaxSumPcaFar;
  float  *pfMaxSumPcaClose;
  float  *pfVectorSpatialTransformation;

  int    iSplineDegree;
  float  fDiracEps;
  float  fHeaviside0;
  float  fPhiCx;
  float  fPhiCy;
  float  fPhiLength;
  float  fMaxPCAMeanValue;

  int    iNGaussianRegionTerm;
  float  **ppfGaussianRegionTerm;
  float  fDeltatU;
  int    iIterComputeUi;

  float  **ppfDiracPCAFunction;
  float  **ppfHeavisidePCAFunction;
  float  **ppfNegHeavisidePCAFunction;
  float  **ppfUinRegionTerm;
  float  **ppfUoutRegionTerm;  
  float  **ppfUinTempRegionTerm;
  float  **ppfUoutTempRegionTerm;  
  float  **ppfFidelityRegionTerm;    
  float  **ppfGradRegionTerm;  
  float  **ppfMSRegionTerm;
  float  *pfSumPcaShapeTerm;
  float  *pfSumPcaRegionTerm;

  int    iNbIter;

  float  **ppfG2UinRegionTerm;
  float  **ppfG2UoutRegionTerm;
  float  **ppfPCACurvatureShapeTerm;

};


// general function (acsp_mex.c)
int iAcShapePrior
(
  int    iDimIm[],
  int    iDimPca[],
  float  *pfVecGeneralParameters,
  float  *pfPhiAcIn,
  float  *pfPhiAcOut,
  float  *pfIm,
  float  *pfVecBoundaryTerm,
  float  *pfEdgeMap,
  float  *pfVecShapeTerm,
  float  *pfPCAFunctionOut,
  float  *pfVecRigidTransformationsOut,
  float  *pfVpcaOut,
  float  *pfEVec,
  float  *pfEVal,
  float  *pfPCAMeanValue,
  float  *pfVecRegionTerm,
  float  *pfMSRegionTerm,
  float  *pfUinRegionTerm,
  float  *pfUoutRegionTerm,
  float  *pfTemp
   );


// functions for boundary-based term (acspBoundaryTerm.c)
void vComputeGrad
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfTemp
  );

void vComputeCurvatureTerm
(
 int    iNx,
 int    iNy,
 struct sAcspTag sAcsp,
 float  *pfEdgeDetector,
 float  *pfVecBoundaryTerm,
 int    iDisplay,
 float  *pfTemp
 );

void vComputeAttractionValleyTerm
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfEdgeDetector,
  float  *pfVecBoundaryTerm,
  int    iDisplay,
  float  *pfTemp
  );

void vComputeBoundaryTerm
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfEdgeDetector,
  float  *pfVecBoundaryTerm,
  int    iDisplay,
  float  *pfTemp
  );


// functions for shape-based term (acspShapeTerms.c)
void vComputeCentroidAC
(
 int    iNx,
 int    iNy,
 struct sAcspTag sAcsp,
 float  *pfPhiCx,
 float  *pfPhiCy,
 float  *pfPhiLength,
 int    iDisplay
 );

void vComputeSimilarityFunction
(
 int    iNx,
 int    iNy,
 struct sAcspTag sAcsp,
 int    iDimPca[], 
 float  *pfEVec,
 float  *pfEVal,
 float  *pfPCAMeanValue,
 int    ip,
 float  *pfVpcaOut,
 int    iTypeSpatialTransformation,
 float  *pfOutputVecRigidTransformations,
 float  *pfVecShapeTerm,
 float  *pfVecRegionTerm,
 float  *pfVecGeneralParameters,
 float  *pfTemp
);

void vComputeAttractionValleyShapeTerm
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfVecShapeTerm,
  int    iDisplay,
  float  *pfTemp
  );

void vComputeShapeTerm
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfVecShapeTerm,
  int    iDisplay,
  float  *pfTemp
  );

void vComputeRegistrationParameters
(
  int    iNx,
  int    iNy,
  struct sAcspTag sAcsp,
  float  *pfEVec,
  float  *pfEVal,
  float  *pfPCAMeanValue,
  int    ip,
  float  *pfVpcaOut,
  int    iTypeSpatialTransformation,
  float  *pfOutputVecRigidTransformations,
  float  *pfVecShapeTerm,
  float  fDeltaT,
  float  fSign,
  float  *pfVecRegionTerm,
  float  *pfIm,
  int    iNbIter,
  int    iNbTemporalSteps,
  float  *pfVecGeneralParameters,
  float  *pfTemp
);

void vComputeFinalPcaFunction
(
 int    iNx,
 int    iNy,
 struct sAcspTag sAcsp,
 int    iDimPca[], 
 float  *pfPCAFunctionOut,
 float  *pfEVec,
 float  *pfEVal,
 float  *pfPCAMeanValue,
 int    ip,
 float  *pfVpcaOut,
 int    iTypeSpatialTransformation,
 float  *pfOutputVecRigidTransformations,
 float  *pfVecShapeTerm,
 float  *pfVecGeneralParameters,
 float  *pfTemp
 );

extern void vComputePCAReconstruction
(
 float    **ppfImRecons,
 float    *pfEVec,
 float    *pfMeanValue,
 float    *pfVcoef,
 float    *pfEVal,
 int      iDim[],
 float    *pfPu
  );


// functions for spatial transformations (bspline_acShapeTerm.c)
extern int iBSplineImage
(
 float   **ppfImage,
 float   **ppfDiracPhi,
 int     iDim[],
 int     iSplineDegree,
 float   *pfVectorSpatialTransformation,
 int     iTypeSpatialTransformation,
 float   fValueNotInterpolated,
 int     iDisplay
 );


// functions for region-based term (extensionU.c)
extern int iExtensionFMDataUi
(
 int    iDim[2],
 float  **ppfImToExtend,
 float  *pfVecGeneralParameters,
 float  *pfTemp
 );

extern int iExtensionFMDataG2Ui
(
 int    iDim[2],
 float  **ppfImToExtend,
 float  *pfVecGeneralParameters,
 float  *pfTemp
 );


// function for re-distancing the level set function (FM_ReDistSDF.c)
extern int iFM2D_SDF
(
 float  **ppfSDF,
 int    iDim[2],
 float  fDistanceMaxComputations,
 int    iDisplay,
 float  *pfTemp
 );




#endif

