
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspReDistLSF.c
// Description: This function computes the signed distance function of the zero level set
// of the input function "pfDistSDF". The re-distancing the level set function is done with
// the Fast Marching Method: [D. Adalsteinsson and J. Sethian, "A Fast Level Set Method
// for Propagating Interfaces," Journal of Computational Physics, vol. 118, pp. 269-277,1995.]



#include "acspCommon.h"




/****************************************/
/**  Declaration of structures         **/
/****************************************/

typedef struct{
  float *pfPhiValue;
  int   *pix;
  int   *piy;
  int   *piFlagNarrowBand;
  int   iLength;
} sNarrowBand;


/****************************************/


/****************************************/
/**  Declaration of static procedures  **/
/****************************************/

/****************************************/
static void vAddElement2NarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
);
/****************************************/

/****************************************/
static void vRemoveFirstElementOfNarrowBand
(
 sNarrowBand  *psNarBand
);
/****************************************/

/****************************************/
static void vUpdatePixelFromNarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
);
/****************************************/


/****************************************/





/****************************************/
/**  Definition of static procedures   **/
/****************************************/


/****************************************/
static void vAddElement2NarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
)

{

  int    current, parent, iTmp;
  float  fTmp;

  psNarBand->pfPhiValue[psNarBand->iLength] = fPhiValue;
  psNarBand->pix[psNarBand->iLength] = ix;
  psNarBand->piy[psNarBand->iLength] = iy;

  current = psNarBand->iLength;
  psNarBand->iLength++;

  parent = (current - 1)/2;
  while (current != 0)
    {
      if (ABS(psNarBand->pfPhiValue[current]) < ABS(psNarBand->pfPhiValue[parent]))
        {
          SWAP_FM(psNarBand->pfPhiValue[current],psNarBand->pfPhiValue[parent],fTmp);
	  SWAP_FM(psNarBand->pix[current],psNarBand->pix[parent],iTmp);
	  SWAP_FM(psNarBand->piy[current],psNarBand->piy[parent],iTmp);
          current = parent;
          parent = (current-1)/2;
        }
      else
        break;
    }

}
/****************************************/


/****************************************/
static void vRemoveFirstElementOfNarrowBand
(
 sNarrowBand  *psNarBand
)

{

  int    best, current, c1, c2, iTmp;
  float  fTmp;

  psNarBand->pfPhiValue[0] = psNarBand->pfPhiValue[psNarBand->iLength-1];
  psNarBand->pix[0] = psNarBand->pix[psNarBand->iLength-1];
  psNarBand->piy[0] = psNarBand->piy[psNarBand->iLength-1];
  psNarBand->iLength--;
 
  current = 0;
  c1 = 1;
  c2 = 2;
  while (c1 < psNarBand->iLength)
    {
      if (c2 >= psNarBand->iLength)
	best = c1;
      else
	{
	  if (ABS(psNarBand->pfPhiValue[c1]) < ABS(psNarBand->pfPhiValue[c2]))
	    best = c1;
	  else
	    best = c2;
	}
      if (ABS(psNarBand->pfPhiValue[best]) < ABS(psNarBand->pfPhiValue[current]))
	{
          SWAP_FM(psNarBand->pfPhiValue[best],psNarBand->pfPhiValue[current],fTmp);
	  SWAP_FM(psNarBand->pix[best],psNarBand->pix[current],iTmp);
	  SWAP_FM(psNarBand->piy[best],psNarBand->piy[current],iTmp);
	  current = best;
	  c1 = 2*current + 1;
	  c2 = c1 + 1;
	}
      else
	break;
    }
  
}
/****************************************/


/****************************************/
static void vUpdatePixelFromNarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
)

{

  int    current, parent, iTmp, i;
  float  fTmp;


  for(i=0; i< psNarBand->iLength; i++)
    if ( (psNarBand->pix[i] == ix) && (psNarBand->piy[i] == iy) )
      current = i;

  psNarBand->pfPhiValue[current] = fPhiValue;

  parent = (current - 1)/2;
  while (current != 0)
    {
      if (ABS(psNarBand->pfPhiValue[current]) < ABS(psNarBand->pfPhiValue[parent]))
        {
          SWAP_FM(psNarBand->pfPhiValue[current],psNarBand->pfPhiValue[parent],fTmp);
	  SWAP_FM(psNarBand->pix[current],psNarBand->pix[parent],iTmp);
	  SWAP_FM(psNarBand->piy[current],psNarBand->piy[parent],iTmp);
          current = parent;
          parent = (current-1)/2;
        }
      else
        break;
    }

}
/****************************************/


/****************************************/
extern int iFM2D_SDF
(
 float  **ppfSDF,
 int    iDim[2],
 float  fMaxDistanceComputations,
 int    iDisplay,
 float  *pfTemp
 )

{

  float   fPhi00, fPhi0p, fPhi0m, fPhip0, fPhim0, fd, fa, fb, fPhi1, fPhi2;
  float   fd1, fd2, fd3, fd4, *pfDistSDF, *pfNewSDF;
  int     iNy, iNx, iy, ix, iy2, ix2, iCpt;
  int     *piBurntPixels, iNbInter;
  sNarrowBand   sNarBand;


  /* Get the size of the image */
  iNy = iDim[0]; 
  iNx = iDim[1]; 


  /*****************************************************************/
  /* Memory allocations */

  pfDistSDF = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfDistSDF)
    {
      mexPrintf("Memory allocation failure for pfDistSDF\n");
      return(0);
    }

  pfNewSDF = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfNewSDF)
    {
      mexPrintf("Memory allocation failure for pfNewSDF\n");
      return(0);
    }


  sNarBand.pfPhiValue = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!sNarBand.pfPhiValue)
    {
      mexPrintf("Memory allocation failure for sNarBand.pfPhiValue\n");
      return(0);
    }

  sNarBand.pix = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.pix)
    {
      mexPrintf("Memory allocation failure for sNarBand.pix\n");
      return(0);
    }

  sNarBand.piy = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piy)
    {
      mexPrintf("Memory allocation failure for sNarBand.piy\n");
      return(0);
    }

  sNarBand.piFlagNarrowBand = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piFlagNarrowBand)
    {
      mexPrintf("Memory allocation failure for sNarBand.piFlagNarrowBand\n");
      return(0);
    }

  sNarBand.iLength = 0;

  piBurntPixels = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!piBurntPixels)
    {
      mexPrintf("Memory allocation failure for piBurntPixels\n");
      return(0);
    }

  /*****************************************************************/



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++) 
      {   
	piBurntPixels[ix*iNy+ iy] = 0;
	sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;
      }


  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      pfDistSDF[ix*iNy + iy] = ppfSDF[ix][iy];


  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++) 
      if ( pfDistSDF[ix*iNy + iy] >= 0.0 )
	pfNewSDF[ix*iNy + iy] = INF;
      else
	pfNewSDF[ix*iNy + iy] = -INF;
  

  for(ix=1; ix< iNx-1; ix++)
    for(iy=1; iy< iNy-1; iy++) 
      {   

	fPhi00 = pfDistSDF[ix*iNy + iy];
	fPhip0 = pfDistSDF[(ix+1)*iNy + iy];
	fPhim0 = pfDistSDF[(ix-1)*iNy + iy];
	fPhi0p = pfDistSDF[ix*iNy + iy+1];
	fPhi0m = pfDistSDF[ix*iNy + iy-1];

	if ( ABS(fPhi00) < EPS_FM )  pfNewSDF[ix*iNy + iy] = 0.0;

	else if ( MAX5(fPhi00,fPhip0,fPhim0,fPhi0p,fPhi0m)>= 0.0  &&  MIN5(fPhi00,fPhip0,fPhim0,fPhi0p,fPhi0m)<= 0.0 )
	  {

	    iNbInter = 0;

	    if ( ABS(fPhi00-fPhip0) > EPS_FM )  fd = fPhi00/ (fPhi00-fPhip0);
	    else  fd = -INF;
	    if ( fd>= 0.0 && fd<= 1.0 )  { fd1 = ABS(fd); iNbInter++; }
	    else  fd1 = -INF;

	    if ( ABS(fPhi00-fPhi0p) > EPS_FM )  fd = fPhi00/ (fPhi00-fPhi0p);
	    else  fd = -INF;
	    if ( fd>= 0.0 && fd<= 1.0 )  { fd2 = ABS(fd); iNbInter++; }
	    else  fd2 = -INF;
	    
	    if ( ABS(fPhim0-fPhi00) > EPS_FM )  fd = fPhi00/ (fPhim0-fPhi00);
	    else  fd = INF;
	    if ( fd>= -1.0 && fd<= 0.0 )  { fd3 = ABS(fd); iNbInter++; }
	    else  fd3 = -INF;
	    
	    if ( ABS(fPhi0m-fPhi00) > EPS_FM )  fd = fPhi00/ (fPhi0m-fPhi00);
	    else  fd = INF;
	    if ( fd>= -1.0 && fd<= 0.0 )  { fd4 = ABS(fd); iNbInter++; }
	    else  fd4 = -INF;
	  
	 
	    if ( iNbInter == 1 )
	      {
		if ( fd1>= 0.0 )       pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fd1;
		else if ( fd2>= 0.0 )  pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fd2;
		else if ( fd3>= 0.0 )  pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fd3;
		else if ( fd4>= 0.0 )  pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fd4;
	      }


	    else if ( iNbInter == 2 )
	      {
		if ( fd1>= 0.0  &&  fd3>= 0.0 )
		  pfNewSDF[ix*iNy + iy] = MIN(fd1,fd3);
		else if ( fd2>= 0.0  &&  fd4>= 0.0 )
		  pfNewSDF[ix*iNy + iy] = MIN(fd2,fd4);
		else 
		  {
		    iCpt = 0;
		    if ( fd1>= 0.0 ) { fa = fd1; iCpt++; }
		    if ( fd2>= 0.0 )
		      {
			if (iCpt==0)  fa = fd2;
			else  fb = fd2;
			iCpt++;
		      }
		    if ( fd3>= 0.0 )
		      {
			if (iCpt==0)  fa = fd3;
			else  fb = fd3;
		      }
		    if ( fd4>= 0.0 )  fb = fd4;
		     
		    pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fa* fb/ sqrt(SQR(fa)+ SQR(fb));
		  }
	      }


	    else if ( iNbInter == 3 )
	      {
		if ( fd1>= 0.0  &&  fd3>= 0.0 )
		  {
		    fa = MIN(fd1,fd3);
		    if ( fd2>= 0.0 )  fb = fd2;
		    else  fb = fd4;
		  }
		else 
		  {
		    fa = MIN(fd2,fd4);
		    if ( fd1>= 0.0 )  fb = fd1;
		    else  fb = fd3;
		  }
		pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fa* fb/ sqrt(SQR(fa)+ SQR(fb));
	      }


	    else if ( iNbInter == 4 )
	      {
		fa = MIN(fd1,fd3);
		fb = MIN(fd2,fd4);
		pfNewSDF[ix*iNy + iy] = SIGN(fPhi00)* fa* fb/ sqrt(SQR(fa)+ SQR(fb));
	      }

	  }

      }


  for(ix=1; ix< iNx-1; ix++)
    for(iy=1; iy< iNy-1; iy++) 
      {   
	if ( ABS(pfNewSDF[ix*iNy+ iy])< INF )
	  {

	    sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 1;
	    vAddElement2NarrowBand(&sNarBand,pfNewSDF[ix*iNy+ iy],ix,iy);
	    piBurntPixels[ix*iNy+ iy] = 1;

	  }

      }


  while ( sNarBand.iLength > 0 )
    {

      /* Take the point in "narrowband" that has the smallest value */
      ix = sNarBand.pix[0]; 
      iy = sNarBand.piy[0];
     
      /* Add the point to "burnt" and remove it from "narrowband" */
      piBurntPixels[ix*iNy+ iy] = 1;
      vRemoveFirstElementOfNarrowBand(&sNarBand);
      sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;

      for(ix2=ix-1; ix2<= ix+1; ix2++)
	for(iy2=iy-1; iy2<= iy+1; iy2++)
	  if ( ix2>0  &&  ix2<iNx-1  &&  iy2>0  &&  iy2<iNy-1 )
	    if ( piBurntPixels[ix2*iNy+ iy2] == 0 )
	      {

		if ( pfNewSDF[ix2*iNy+ iy2]> 0.0 )
		  {
		    fPhi1 = MIN(pfNewSDF[(ix2+1)*iNy+ iy2],pfNewSDF[(ix2-1)*iNy+ iy2]);
		    fPhi2 = MIN(pfNewSDF[ix2*iNy+ iy2+1],pfNewSDF[ix2*iNy+ iy2-1]);
		  }
		else
		  {
		    fPhi1 = MIN(ABS(pfNewSDF[(ix2+1)*iNy+ iy2]),ABS(pfNewSDF[(ix2-1)*iNy+ iy2]));
		    fPhi2 = MIN(ABS(pfNewSDF[ix2*iNy+ iy2+1]),ABS(pfNewSDF[ix2*iNy+ iy2-1]));
		  }

		if ( ABS(fPhi1- fPhi2) <= 1.0 )
		  if ( pfNewSDF[ix2*iNy+ iy2]> 0.0 )
		    pfNewSDF[ix2*iNy+ iy2] = 0.5* ( fPhi1+ fPhi2+ sqrt(2.0- SQR(fPhi1- fPhi2)) );
		  else
		    pfNewSDF[ix2*iNy+ iy2] = -0.5* ( fPhi1+ fPhi2+ sqrt(2.0- SQR(fPhi1- fPhi2)) );
		else 
		  if ( pfNewSDF[ix2*iNy+ iy2]> 0.0 )
		    pfNewSDF[ix2*iNy+ iy2] = MIN(fPhi1,fPhi2) + 1.0;
		  else
		    pfNewSDF[ix2*iNy+ iy2] = -MIN(ABS(fPhi1),ABS(fPhi2)) - 1.0;

		if ( ABS(pfNewSDF[ix2*iNy+ iy2]) <= fMaxDistanceComputations )
		  {

		    if ( sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] == 0 )
		      {
			sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] = 1;
			vAddElement2NarrowBand(&sNarBand,pfNewSDF[ix2*iNy+ iy2],ix2,iy2);
		      }
		    else
		      vUpdatePixelFromNarrowBand(&sNarBand,pfNewSDF[ix2*iNy+ iy2],ix2,iy2);

		  }


	      }
    
    }


  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++) 
      if ( ABS(pfNewSDF[ix*iNy+ iy]) == INF )
	pfNewSDF[ix*iNy+ iy] = pfDistSDF[ix*iNy + iy];


  /* Neumann boundary condition */
  for (ix=0; ix< iNx; ix++)
    {
      pfNewSDF[ix*iNy] = pfNewSDF[ix*iNy+ 1];
      pfNewSDF[ix*iNy+ iNy-1] = pfNewSDF[ix*iNy+ iNy-2];
    }
  for (iy=0; iy< iNy; iy++)
    {
      pfNewSDF[iy] = pfNewSDF[iNy+ iy];
      pfNewSDF[(iNx-1)*iNy+ iy] = pfNewSDF[(iNx-2)*iNy+ iy];
    }
	   


  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      ppfSDF[ix][iy] = pfNewSDF[ix*iNy + iy];



  /*****************************************************************/
  /* Free memory  */

  free((float*) pfDistSDF);
  free((float*) pfNewSDF);
  free((float*) sNarBand.pfPhiValue);
  free((int*)   sNarBand.pix); 
  free((int*)   sNarBand.piy); 
  free((int*)   sNarBand.piFlagNarrowBand); 
  free((int*)   piBurntPixels);

  /*****************************************************************/

  return(1);
 
}
/****************************************/

/****************************************/

/**************************************** End of file ****************************************/
