
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspShapeTerm.c
// Description: this function computes the shape-based term
// See "acsp.c"


#include "acspCommon.h"



// compute length of zero level set of shape prior/PCA function, hat{phi}
void vComputeLengthShapePrior
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfLengthShapePrior,
float  *pfTemp
)

{
    
    int    ix, iy, iCpt;
    float  fsum_plus;
    
    
    iCpt = 0;
    for (ix=1; ix< iNx-1; ix++)
        for (iy=1; iy< iNy-1; iy++)
            if ( sAcsp.ppfDiracPCAFunction[ix][iy] > EPS )
    {
        fsum_plus = (sAcsp.ppfPCAFunction[ix][iy]>0) + (sAcsp.ppfPCAFunction[ix][iy-1]>0) + (sAcsp.ppfPCAFunction[ix][iy+1]>0) + (sAcsp.ppfPCAFunction[ix-1][iy]>0) + (sAcsp.ppfPCAFunction[ix+1][iy]>0);
        if ( fsum_plus>=1  &&  sAcsp.ppfPCAFunction[ix][iy]<=0 )
        {
            iCpt++;
        }
            }
    *pfLengthShapePrior = (float) iCpt; // length of shape prior
    
}



// compute u_in in {hat{phi}>=0} and u_out in {hat{phi}<0}
void vComputeUinUoutRegionTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfIm0,
float  *pfVecRegionTerm,
float  *pfTemp
)

{
    
    int    ix, iy;
    float  f1, f2, f3, f4, fDeltaTu, fMu, fLaplacianIn, fLaplacianOut;
    
    
    fMu = pfVecRegionTerm[1];  // mu
    fDeltaTu = pfVecRegionTerm[3]; // temporal step to update u_in and u_out
    
    
    for (ix=1; ix< iNx-1; ix++)
        for  (iy=1; iy< iNy-1; iy++)
            if ( sAcsp.ppfHeavisidePCAFunction[ix][iy] >= sAcsp.fHeaviside0 ) /* INSIDE */
    {
        
        if ( (sAcsp.ppfHeavisidePCAFunction[ix+1][iy]>=sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix-1][iy]>=sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix][iy+1]>=sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix][iy-1]>=sAcsp.fHeaviside0) )
        {
            f1 = ( sAcsp.ppfHeavisidePCAFunction[ix+1][iy] + sAcsp.ppfHeavisidePCAFunction[ix][iy] )* ( sAcsp.ppfUinTempRegionTerm[ix+1][iy] - sAcsp.ppfUinTempRegionTerm[ix][iy] )/ 2.0;
            f2 = ( sAcsp.ppfHeavisidePCAFunction[ix][iy] + sAcsp.ppfHeavisidePCAFunction[ix-1][iy] )* ( sAcsp.ppfUinTempRegionTerm[ix][iy] - sAcsp.ppfUinTempRegionTerm[ix-1][iy] )/ 2.0;
            f3 = ( sAcsp.ppfHeavisidePCAFunction[ix][iy+1] + sAcsp.ppfHeavisidePCAFunction[ix][iy] )* ( sAcsp.ppfUinTempRegionTerm[ix][iy+1] - sAcsp.ppfUinTempRegionTerm[ix][iy] )/ 2.0;
            f4 = ( sAcsp.ppfHeavisidePCAFunction[ix][iy] + sAcsp.ppfHeavisidePCAFunction[ix][iy-1] )* ( sAcsp.ppfUinTempRegionTerm[ix][iy] - sAcsp.ppfUinTempRegionTerm[ix][iy-1] )/ 2.0;
            fLaplacianIn = f1 - f2 + f3 - f4;
            sAcsp.ppfUinRegionTerm[ix][iy] = sAcsp.ppfUinTempRegionTerm[ix][iy] - fDeltaTu* ( (sAcsp.ppfUinTempRegionTerm[ix][iy] - pfIm0[ix*iNy+ iy])* sAcsp.ppfHeavisidePCAFunction[ix][iy] - fMu* fLaplacianIn );
        }
        else
            sAcsp.ppfUinRegionTerm[ix][iy] = -INF;
        
        sAcsp.ppfUoutRegionTerm[ix][iy] = -INF;
        
            }
            else /* OUTSIDE */
    {
        
        if ( (sAcsp.ppfHeavisidePCAFunction[ix+1][iy]<sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix-1][iy]<sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix][iy+1]<sAcsp.fHeaviside0) && (sAcsp.ppfHeavisidePCAFunction[ix][iy-1]<sAcsp.fHeaviside0) )
        {
            f1 = ( sAcsp.ppfNegHeavisidePCAFunction[ix+1][iy] + sAcsp.ppfNegHeavisidePCAFunction[ix][iy] )* ( sAcsp.ppfUoutTempRegionTerm[ix+1][iy] - sAcsp.ppfUoutTempRegionTerm[ix][iy] )/ 2.0;
            f2 = ( sAcsp.ppfNegHeavisidePCAFunction[ix][iy] + sAcsp.ppfNegHeavisidePCAFunction[ix-1][iy] )* ( sAcsp.ppfUoutTempRegionTerm[ix][iy] - sAcsp.ppfUoutTempRegionTerm[ix-1][iy] )/ 2.0;
            f3 = ( sAcsp.ppfNegHeavisidePCAFunction[ix][iy+1] + sAcsp.ppfNegHeavisidePCAFunction[ix][iy] )* ( sAcsp.ppfUoutTempRegionTerm[ix][iy+1] - sAcsp.ppfUoutTempRegionTerm[ix][iy] )/ 2.0;
            f4 = ( sAcsp.ppfNegHeavisidePCAFunction[ix][iy] + sAcsp.ppfNegHeavisidePCAFunction[ix][iy-1] )* ( sAcsp.ppfUoutTempRegionTerm[ix][iy] - sAcsp.ppfUoutTempRegionTerm[ix][iy-1] )/ 2.0;
            fLaplacianOut = f1 - f2 + f3 - f4;
            sAcsp.ppfUoutRegionTerm[ix][iy] = sAcsp.ppfUoutTempRegionTerm[ix][iy] - fDeltaTu* ( (sAcsp.ppfUoutTempRegionTerm[ix][iy] - pfIm0[ix*iNy+ iy])* sAcsp.ppfNegHeavisidePCAFunction[ix][iy] - fMu* fLaplacianOut );
        }
        else
            sAcsp.ppfUoutRegionTerm[ix][iy] = -INF;
        
        sAcsp.ppfUinRegionTerm[ix][iy] = -INF;
        
            }
    
    
    
  /* Neumann boundary condition */
    for (ix=0; ix< iNx; ix++)
    {
        sAcsp.ppfUinRegionTerm[ix][0] = sAcsp.ppfUinRegionTerm[ix][1];
        sAcsp.ppfUinRegionTerm[ix][iNy-1] = sAcsp.ppfUinRegionTerm[ix][iNy-2];
    }
    for (iy=0; iy< iNy; iy++)
    {
        sAcsp.ppfUinRegionTerm[0][iy] = sAcsp.ppfUinRegionTerm[1][iy];
        sAcsp.ppfUinRegionTerm[iNx-1][iy] = sAcsp.ppfUinRegionTerm[iNx-2][iy];
    }
    
    
  /* Neumann boundary condition */
    for (ix=0; ix< iNx; ix++)
    {
        sAcsp.ppfUoutRegionTerm[ix][0] = sAcsp.ppfUoutRegionTerm[ix][1];
        sAcsp.ppfUoutRegionTerm[ix][iNy-1] = sAcsp.ppfUoutRegionTerm[ix][iNy-2];
    }
    for (iy=0; iy< iNy; iy++)
    {
        sAcsp.ppfUoutRegionTerm[0][iy] = sAcsp.ppfUoutRegionTerm[1][iy];
        sAcsp.ppfUoutRegionTerm[iNx-1][iy] = sAcsp.ppfUoutRegionTerm[iNx-2][iy];
    }
    
    
    
}



// extend u_in in {hat{phi}<0} and u_out in {hat{phi}>=0} using the Fast Marching Method
void vComputeExtensionUinUoutRegionTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfVecGeneralParameters,
float  *pfVecRegionTerm,
float  *pfTemp
)

{
    
    int     iDim[2], ix, iy, iDisplay;
    float   pfVecGenParExtension[2], fDx, fDy, fDistExt;
    time_t  start_time, end_time;
    
    iDisplay = (int) pfVecGeneralParameters[0];
    
    
    
    
    for (ix=1; ix< iNx-1; ix++)
        for  (iy=1; iy< iNy-1; iy++)
            if ( sAcsp.ppfHeavisidePCAFunction[ix][iy] >= sAcsp.fHeaviside0 ) /* INSIDE */
    {
        if ( (ABS(sAcsp.ppfUinRegionTerm[ix+1][iy])<INF) && (ABS(sAcsp.ppfUinRegionTerm[ix-1][iy])<INF) )
        {
            fDx = (sAcsp.ppfUinRegionTerm[ix+1][iy] -  sAcsp.ppfUinRegionTerm[ix-1][iy])/ 2.0;
        }
        else if ( ABS(sAcsp.ppfUinRegionTerm[ix+1][iy])<INF )
        {
            fDx = sAcsp.ppfUinRegionTerm[ix+1][iy] -  sAcsp.ppfUinRegionTerm[ix][iy];
        }
        else
        {
            fDx = sAcsp.ppfUinRegionTerm[ix][iy] -  sAcsp.ppfUinRegionTerm[ix-1][iy];
        }
        
        if ( (ABS(sAcsp.ppfUinRegionTerm[ix][iy+1])<INF) && (ABS(sAcsp.ppfUinRegionTerm[ix][iy-1])<INF) )
        {
            fDy = (sAcsp.ppfUinRegionTerm[ix][iy+1] -  sAcsp.ppfUinRegionTerm[ix][iy-1])/ 2.0;
        }
        else if ( ABS(sAcsp.ppfUinRegionTerm[ix][iy+1])<INF )
        {
            fDy = sAcsp.ppfUinRegionTerm[ix][iy+1] -  sAcsp.ppfUinRegionTerm[ix][iy];
        }
        else
        {
            fDy = sAcsp.ppfUinRegionTerm[ix][iy] -  sAcsp.ppfUinRegionTerm[ix][iy-1];
        }
        
        sAcsp.ppfG2UinRegionTerm[ix][iy] = SQR(fDx) + SQR(fDy);
        sAcsp.ppfG2UoutRegionTerm[ix][iy] = -INF*1.;
            }
            else /* OUTSIDE */
    {
        if ( (ABS(sAcsp.ppfUoutRegionTerm[ix+1][iy])<INF) && (ABS(sAcsp.ppfUoutRegionTerm[ix-1][iy])<INF) )
        {
            fDx = (sAcsp.ppfUoutRegionTerm[ix+1][iy] -  sAcsp.ppfUoutRegionTerm[ix-1][iy])/ 2.0;
        }
        else if ( ABS(sAcsp.ppfUoutRegionTerm[ix+1][iy])<INF )
        {
            fDx = sAcsp.ppfUoutRegionTerm[ix+1][iy] -  sAcsp.ppfUoutRegionTerm[ix][iy];
        }
        else
        {
            fDx = sAcsp.ppfUoutRegionTerm[ix][iy] -  sAcsp.ppfUoutRegionTerm[ix-1][iy];
        }
        
        if ( (ABS(sAcsp.ppfUoutRegionTerm[ix][iy+1])<INF) && (ABS(sAcsp.ppfUoutRegionTerm[ix][iy-1])<INF) )
        {
            fDy = (sAcsp.ppfUoutRegionTerm[ix][iy+1] -  sAcsp.ppfUoutRegionTerm[ix][iy-1])/ 2.0;
        }
        else if ( ABS(sAcsp.ppfUoutRegionTerm[ix][iy+1])<INF )
        {
            fDy = sAcsp.ppfUoutRegionTerm[ix][iy+1] -  sAcsp.ppfUoutRegionTerm[ix][iy];
        }
        else
        {
            fDy = sAcsp.ppfUoutRegionTerm[ix][iy] -  sAcsp.ppfUoutRegionTerm[ix][iy-1];
        }
        
        sAcsp.ppfG2UoutRegionTerm[ix][iy] = SQR(fDx) + SQR(fDy);
        sAcsp.ppfG2UinRegionTerm[ix][iy] = -INF*1.;
            }
    
    
    
    // Extension u_in and u_out
    fDistExt = 4.0; // distance of extension
    
    pfVecGenParExtension[0] = iDisplay;
    pfVecGenParExtension[1] = fDistExt;
    
    iDim[0] = iNy;
    iDim[1] = iNx;
    
    start_time = clock();
    if ( !iExtensionFMDataUi(iDim,sAcsp.ppfUinRegionTerm,pfVecGenParExtension,pfTemp) )
    {
        mexPrintf("\n\nError in extensionFMnHnData\n");
    }
    
    end_time = clock();
    if (iDisplay == YES)  mexPrintf("\nComputing Time for Uin extension = %.2f sec\n",difftime(end_time,start_time)/1000);
    
    start_time = clock();
    if ( !iExtensionFMDataUi(iDim,sAcsp.ppfUoutRegionTerm,pfVecGenParExtension,pfTemp) )
    {
        mexPrintf("\n\nError in extensionFMnHnData\n");
    }
    
    end_time = clock();
    if (iDisplay == YES)  mexPrintf("Computing Time for Uout extension = %.2f sec\n",difftime(end_time,start_time)/1000);
    
    
    
    
    for (ix=0; ix< iNx; ix++)
        for  (iy=0; iy< iNy; iy++)
    {
        sAcsp.ppfUinTempRegionTerm[ix][iy] = sAcsp.ppfUinRegionTerm[ix][iy];
        sAcsp.ppfUoutTempRegionTerm[ix][iy] = sAcsp.ppfUoutRegionTerm[ix][iy];
        }
    
    
    
    
    // Extension  of square of |grad u_in| and  |grad u_out|
    pfVecGenParExtension[0] = iDisplay;
    pfVecGenParExtension[1] = fDistExt;
    
    iDim[0] = iNy;
    iDim[1] = iNx;
    
    
    start_time = clock();
    if ( !iExtensionFMDataG2Ui(iDim,sAcsp.ppfG2UinRegionTerm,pfVecGenParExtension,pfTemp) )
    {
        mexPrintf("\n\nError in extensionFMnHnData\n");
    }
    
    end_time = clock();
    if (iDisplay == YES)  mexPrintf("Computing Time for G2Uin extension = %.2f sec\n",difftime(end_time,start_time)/1000);
    
    
    
    start_time = clock();
    if ( !iExtensionFMDataG2Ui(iDim,sAcsp.ppfG2UoutRegionTerm,pfVecGenParExtension,pfTemp) )
    {
        mexPrintf("\n\nError in extensionFMnHnData\n");
    }
    
    end_time = clock();
    if (iDisplay == YES)  mexPrintf("Computing Time for G2Uout extension = %.2f sec\n\n",difftime(end_time,start_time)/1000);
    
    
    
}



// Compute the Mumford-Shah/region-based term, Theta_in and Theta_out in eq 31-32
void vComputeMSTermRegionTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfIm0,
float  *pfVecGeneralParameters,
float  *pfVecRegionTerm,
int    iDisplay,
float  *pfTemp
)
{
    
    int    ix, iy, iIterComputeUi;
    float  fMaxFidelityTerm, fWeightRegionTerm;
    float  fMu, fMaxData, fMaxFidelityTermMS, fMaxGradTermMS, fI, fFidelityTerm, fGradTerm, fData;
    
    
    iDisplay = (int) pfVecGeneralParameters[0];
    fWeightRegionTerm = pfVecRegionTerm[0];
    fMu = pfVecRegionTerm[1];
    iIterComputeUi = (int) pfVecRegionTerm[2]; // number of updates of u_in and u_out
    
    
    
    fMaxData = 0.0;
    fMaxFidelityTermMS = 0.0;
    fMaxGradTermMS = 0.0;
    fMaxData = 0.0;
    fMaxFidelityTerm = 0.0;
    
    
    
    
    if ( fWeightRegionTerm> 0.0 )
    {
        
        if ( (sAcsp.iNbIter==1) || (sAcsp.iNbIter%iIterComputeUi==0) )
        {            
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
            {
                
                fI = pfIm0[ix*iNy+ iy];
                fFidelityTerm = SQR(sAcsp.ppfUinRegionTerm[ix][iy]- fI) - SQR(sAcsp.ppfUoutRegionTerm[ix][iy]- fI);
                fGradTerm = sAcsp.ppfG2UinRegionTerm[ix][iy] - sAcsp.ppfG2UoutRegionTerm[ix][iy];
                fData = fFidelityTerm + fMu* fGradTerm;
                sAcsp.ppfMSRegionTerm[ix][iy] = fData;
                
                if (iDisplay == YES)
                {
                    if ( ABS(fFidelityTerm) > ABS(fMaxFidelityTermMS) ) fMaxFidelityTermMS = ABS(fFidelityTerm);
                    if ( ABS(fGradTerm) > ABS(fMaxGradTermMS) ) fMaxGradTermMS = ABS(fGradTerm);
                    if ( ABS(fData) > ABS(fMaxData) ) fMaxData = ABS(fData);
                    if ( ABS(sAcsp.ppfMSRegionTerm[ix][iy]) > ABS(fMaxFidelityTerm) ) fMaxFidelityTerm = ABS(sAcsp.ppfMSRegionTerm[ix][iy]);
                }
                
                }
            
            if (iDisplay == YES)
            {
                mexPrintf("\nfMaxFidelityTermMS*fWeight (RegionTerm)= %.3f\n",fWeightRegionTerm* fMaxFidelityTermMS);
                mexPrintf("fMaxGradTerm*fWeight (RegionTerm)= %.3f\n",fWeightRegionTerm* fMu* fMaxGradTermMS);
                mexPrintf("fMaxMSTerm (RegionTerm)= %.3f\n\n",fWeightRegionTerm* fMaxFidelityTerm);
            }
            
        }
        
    }

    
}








// compute center of current active contour
void vComputeCentroidAC
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfPhiCx,
float  *pfPhiCy,
float  *pfPhiLength,
int    iDisplay
)
{
    int    ix, iy;
    float  fLenghPhi;
    
    fLenghPhi = 0.0;
    *pfPhiCx = 0.0;
    *pfPhiCy = 0.0;
    
    
    
  /* Centroid computed with the regularized Dirac function of the level set function */
    for (ix=1; ix< iNx-1; ix++)
        for (iy=1; iy< iNy-1; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
    {
        fLenghPhi +=  sAcsp.ppfDiracPhi[ix][iy];
        *pfPhiCx += sAcsp.ppfDiracPhi[ix][iy]* ((float)ix);
        *pfPhiCy += sAcsp.ppfDiracPhi[ix][iy]* ((float)iy);
            }
    
    
    *pfPhiCx /= fLenghPhi;
    *pfPhiCy /= fLenghPhi;
    *pfPhiLength = fLenghPhi;
    
    
}



// compute new PCA function, hat{phi} in eq 29.
void vComputeSimilarityFunction
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
int    iDimPca[],
float  *pfEVec,
float  *pfEVal,
float  *pfPCAMeanValue,
int    ip,
float  *pfVpcaOut,
int    iTypeSpatialTransformation,
float  *pfOutputVecRigidTransformations,
float  *pfVecShapeTerm,
float  *pfVecRegionTerm,
float  *pfVecGeneralParameters,
float  *pfTemp
)
{
    
    int    ix, iy, i, iDim[3];
    int    iNyPca, iNxPca, iDisplay;
    float  fXCenter, fYCenter, fScale, fAngle, fTx, fTy, fPu, fShearing;
    float  **ppfComputationInAllImageDomain, **ppfPCAFunctionTemp, fScaleMinBeforeReDist;
    float  fWeightShapeTerm, fWeightRegionTerm, fDiracEpsScale;
    time_t start_time, end_time;
    
    
    iDisplay = (int) pfVecGeneralParameters[0];
    
    
    iDim[0] = iNy;
    iDim[1] = iNx;
    iDim[2] = ip;
    iNyPca = iDimPca[0];
    iNxPca = iDimPca[1];
    iDimPca[2] = ip;
    
    
    fWeightShapeTerm = pfVecShapeTerm[0];
    fWeightRegionTerm = pfVecRegionTerm[0];
    
    
    ppfPCAFunctionTemp = sAcsp.ppfPCAFunction;
    
    
    // Compute of a new PCA function
    if (*pfEVec!=0.0)
    {
        vComputePCAReconstruction(ppfPCAFunctionTemp,pfEVec,pfPCAMeanValue,pfVpcaOut,pfEVal,iDimPca,&fPu);
        if (iDisplay == YES)
        {
            mexPrintf("PCA reconstruction for an image %ix%i and %i principal component(s)\nprobability of the reconstructed data = %.4f\n",iNy,iNx,ip,fPu);
            for (i=0; i< ip; i++)  mexPrintf("eigenmode[%i] = %.3f  ",i,pfVpcaOut[i]);
            mexPrintf(" \n");
        }
    }
    else
        for (ix=0; ix< iNxPca; ix++)
            for (iy=0; iy< iNyPca; iy++)
                sAcsp.ppfPCAFunction[ix][iy] = pfPCAMeanValue[ix*iNyPca + iy];
    
    
    
    
    // apply spatial transformation
    fXCenter = pfOutputVecRigidTransformations[0];
    fYCenter = pfOutputVecRigidTransformations[1];
    fScale = pfOutputVecRigidTransformations[2];
    fAngle = pfOutputVecRigidTransformations[3];
    fTx = pfOutputVecRigidTransformations[4];
    fTy = pfOutputVecRigidTransformations[5];
    if ( iTypeSpatialTransformation == AFFINE )
        fShearing = pfOutputVecRigidTransformations[6];
    
    
  /**************************/
  /* ONLY SHAPE TERM */
    if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm <1e-6 )
    {
        
        fScaleMinBeforeReDist = 0.8;
        if ( 1./fScale <= fScaleMinBeforeReDist )
        {
            
            sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
            sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
            sAcsp.pfVectorSpatialTransformation[2] = fScale;
            sAcsp.pfVectorSpatialTransformation[3] = fAngle;
            sAcsp.pfVectorSpatialTransformation[4] = fTx;
            sAcsp.pfVectorSpatialTransformation[5] = fTy;
            if ( iTypeSpatialTransformation == AFFINE )
                sAcsp.pfVectorSpatialTransformation[6] = 0.0;
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
            {
                mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            start_time = clock();
            if ( !iFM2D_SDF(sAcsp.ppfPCAFunction,iDim,(float)(3*iNy),iDisplay,pfTemp) )
            {
                mexPrintf("\n\nError in SDF_FM\n");
                for (ix=0; ix< iNx; ix++)
                    for (iy=0; iy< iNy; iy++)
                        sAcsp.ppfPCAFunction[ix][iy] = 0.0;
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)  mexPrintf("Computing Time Fast Marching=  %.3f sec\n",difftime(end_time,start_time)/1000);
            }
            
            for (ix=0; ix< iNx; ix++)
                for (iy=0; iy< iNy; iy++)
                    sAcsp.ppfPCAFunction[ix][iy] *= fScale;
            
            if ( iTypeSpatialTransformation == AFFINE )
            {
                
                sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
                sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
                sAcsp.pfVectorSpatialTransformation[2] = 1.0;
                sAcsp.pfVectorSpatialTransformation[3] = 0.0;
                sAcsp.pfVectorSpatialTransformation[4] = 0.0;
                sAcsp.pfVectorSpatialTransformation[5] = 0.0;
                if ( iTypeSpatialTransformation == AFFINE )
                    sAcsp.pfVectorSpatialTransformation[6] = fShearing;
                
                
                start_time = clock();
                if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
                {
                    mexPrintf("\n\nError in bspline\n \n");
                }
                else
                {
                    end_time = clock();
                    if (iDisplay == YES)
                    {
                        mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                    }
                }
                
            }
            
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
            {
                sAcsp.ppfDxPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix+1][iy]- sAcsp.ppfPCAFunction[ix-1][iy])/ 2. /fScale;
                sAcsp.ppfDyPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix][iy+1]- sAcsp.ppfPCAFunction[ix][iy-1])/ 2. /fScale;
                }
            
        }
        else /* if ( 1./fScale > fScaleMinBeforeReDist ) */
        {
            
            sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
            sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
            sAcsp.pfVectorSpatialTransformation[2] = fScale;
            sAcsp.pfVectorSpatialTransformation[3] = fAngle;
            sAcsp.pfVectorSpatialTransformation[4] = fTx;
            sAcsp.pfVectorSpatialTransformation[5] = fTy;
            if ( iTypeSpatialTransformation == AFFINE )
                sAcsp.pfVectorSpatialTransformation[6] = fShearing;
            
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
            {
                sAcsp.ppfDxPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix+1][iy]- sAcsp.ppfPCAFunction[ix-1][iy])/ 2.;
                sAcsp.ppfDyPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix][iy+1]- sAcsp.ppfPCAFunction[ix][iy-1])/ 2.;
                }
            
            ppfComputationInAllImageDomain = sAcsp.ppfDiracPhi;
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfPCAFunction,ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
            {
                if (iDisplay == YES)  mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfDxPCAFunction,ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
            {
                if (iDisplay == YES)  mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfDyPCAFunction,ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
            {
                mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
        } /* end of  if ( 1./fScale <= fScaleMinBeforeReDist ) */
        
        
        
        sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
        sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
        sAcsp.pfVectorSpatialTransformation[2] = fScale;
        sAcsp.pfVectorSpatialTransformation[3] = fAngle;
        sAcsp.pfVectorSpatialTransformation[4] = fTx;
        sAcsp.pfVectorSpatialTransformation[5] = fTy;
        if ( iTypeSpatialTransformation == AFFINE )
            sAcsp.pfVectorSpatialTransformation[6] = fShearing;
        
        
        ppfComputationInAllImageDomain = sAcsp.ppfDiracPhi;
        
        
        if (*pfEVec!=0.0)
        {
            
            for (i=0; i< ip; i++)
            {
                
                    for (ix=0; ix< iNx; ix++)
                        for (iy=0; iy< iNy; iy++)
                            sAcsp.pppfPcaEigenVectors[i][ix][iy] = pfEVec[i*iNx*iNy + ix*iNy+ iy];
                
                
                start_time = clock();
                if ( !iBSplineImage(sAcsp.pppfPcaEigenVectors[i],ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
                {
                    mexPrintf("\n\nError in bspline\n \n");
                }
                else
                {
                    end_time = clock();
                    if (iDisplay == YES)
                    {
                        mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                    }
                }
                
            }
            
        } /* end of if (*pfEVec!=0.0) */
        
    }
  /* END OF ONLY SHAPE TERM */
  /**************************/
    
    
  /**************************/
  /* ONLY REGION SHAPE TERM */
    if ( fWeightRegionTerm> 0.0 && fWeightShapeTerm <1e-6 )
    {
        
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
        {
            sAcsp.ppfDxPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix+1][iy]- sAcsp.ppfPCAFunction[ix-1][iy])/ 2.;
            sAcsp.ppfDyPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix][iy+1]- sAcsp.ppfPCAFunction[ix][iy-1])/ 2.;
            }
        
        sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
        sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
        sAcsp.pfVectorSpatialTransformation[2] = fScale;
        sAcsp.pfVectorSpatialTransformation[3] = fAngle;
        sAcsp.pfVectorSpatialTransformation[4] = fTx;
        sAcsp.pfVectorSpatialTransformation[5] = fTy;
        if ( iTypeSpatialTransformation == AFFINE )
            sAcsp.pfVectorSpatialTransformation[6] = fShearing;
        
        start_time = clock();
        if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
        {
            mexPrintf("\n\nError in bspline\n \n");
        }
        else
        {
            end_time = clock();
            if (iDisplay == YES)
            {
                mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
            }
        }
        
        
        
        fDiracEpsScale = sAcsp.fDiracEps*fScale;
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
        {
            if ( ABS(sAcsp.ppfPCAFunction[ix][iy]) < 2.0* fDiracEpsScale )
                sAcsp.ppfDiracPCAFunction[ix][iy] = DIRAC(sAcsp.ppfPCAFunction[ix][iy],fDiracEpsScale);
            sAcsp.ppfHeavisidePCAFunction[ix][iy] = HEAVISIDE(sAcsp.ppfPCAFunction[ix][iy],fDiracEpsScale);
            sAcsp.ppfNegHeavisidePCAFunction[ix][iy] = 1.- sAcsp.ppfHeavisidePCAFunction[ix][iy];
            }
        
        
        ppfComputationInAllImageDomain = sAcsp.ppfDiracPCAFunction;
      /*   ppfComputationInAllImageDomain = NULL; */
        
        start_time = clock();
        if ( !iBSplineImage(sAcsp.ppfDxPCAFunction,ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
        {
            mexPrintf("\n\nError in bspline\n \n");
        }
        else
        {
            end_time = clock();
            if (iDisplay == YES)
            {
                mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
            }
        }
        
        start_time = clock();
        if ( !iBSplineImage(sAcsp.ppfDyPCAFunction,ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
        {
            mexPrintf("\n\nError in bspline\n \n");
        }
        else
        {
            end_time = clock();
            if (iDisplay == YES)
            {
                mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
            }
        }
        
        
        
        if (*pfEVec!=0.0)
        {
            
            for (i=0; i< ip; i++)
            {
                
                    for (ix=0; ix< iNx; ix++)
                        for (iy=0; iy< iNy; iy++)
                            sAcsp.pppfPcaEigenVectors[i][ix][iy] = pfEVec[i*iNx*iNy + ix*iNy+ iy];
                
                
                start_time = clock();
                if ( !iBSplineImage(sAcsp.pppfPcaEigenVectors[i],ppfComputationInAllImageDomain,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
                {
                    mexPrintf("\n\nError in bspline\n \n");
                }
                else
                {
                    end_time = clock();
                    if (iDisplay == YES)
                    {
                        mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                    }
                }
                
            }
            
            
        } /* end of if (*pfEVec!=0.0) */
        
    }
  /* END OF  ONLY REGION SHAPE TERM */
  /**************************/
    
    
    
  /**************************/
  /* SHAPE TERM AND REGION SHAPE TERM */
    if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm> 0.0 )
    {
        
        fScaleMinBeforeReDist = 0.8;
        if ( 1./fScale <= fScaleMinBeforeReDist )
        {
            
            sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
            sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
            sAcsp.pfVectorSpatialTransformation[2] = fScale;
            sAcsp.pfVectorSpatialTransformation[3] = fAngle;
            sAcsp.pfVectorSpatialTransformation[4] = fTx;
            sAcsp.pfVectorSpatialTransformation[5] = fTy;
            if ( iTypeSpatialTransformation == AFFINE )
                sAcsp.pfVectorSpatialTransformation[6] = 0.0;
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
            {
                mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            start_time = clock();
            if ( !iFM2D_SDF(sAcsp.ppfPCAFunction,iDim,(float)(3*iNy),iDisplay,pfTemp) )
            {
                mexPrintf("\n\nError in SDF_FM\n");
                for (ix=0; ix< iNx; ix++)
                    for (iy=0; iy< iNy; iy++)
                        sAcsp.ppfPCAFunction[ix][iy] = 0.0;
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)  mexPrintf("Computing Time Fast Marching=  %.3f sec\n",difftime(end_time,start_time)/1000);
            }
            
            for (ix=0; ix< iNx; ix++)
                for (iy=0; iy< iNy; iy++)
                    sAcsp.ppfPCAFunction[ix][iy] *= fScale;
            
            if ( iTypeSpatialTransformation == AFFINE )
            {
                
                sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
                sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
                sAcsp.pfVectorSpatialTransformation[2] = 1.0;
                sAcsp.pfVectorSpatialTransformation[3] = 0.0;
                sAcsp.pfVectorSpatialTransformation[4] = 0.0;
                sAcsp.pfVectorSpatialTransformation[5] = 0.0;
                if ( iTypeSpatialTransformation == AFFINE )
                    sAcsp.pfVectorSpatialTransformation[6] = fShearing;
                
                
                start_time = clock();
                if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
                {
                    mexPrintf("\n\nError in bspline\n \n");
                }
                else
                {
                    end_time = clock();
                    if (iDisplay == YES)
                    {
                        mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                    }
                }
                
            }
            
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
            {
                sAcsp.ppfDxPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix+1][iy]- sAcsp.ppfPCAFunction[ix-1][iy])/ 2. /fScale;
                sAcsp.ppfDyPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix][iy+1]- sAcsp.ppfPCAFunction[ix][iy-1])/ 2. /fScale;
                }
            
        }
        else /* if ( 1./fScale > fScaleMinBeforeReDist ) */
        {
            
            sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
            sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
            sAcsp.pfVectorSpatialTransformation[2] = fScale;
            sAcsp.pfVectorSpatialTransformation[3] = fAngle;
            sAcsp.pfVectorSpatialTransformation[4] = fTx;
            sAcsp.pfVectorSpatialTransformation[5] = fTy;
            if ( iTypeSpatialTransformation == AFFINE )
                sAcsp.pfVectorSpatialTransformation[6] = fShearing;
            
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
            {
                sAcsp.ppfDxPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix+1][iy]- sAcsp.ppfPCAFunction[ix-1][iy])/ 2.;
                sAcsp.ppfDyPCAFunction[ix][iy] = (sAcsp.ppfPCAFunction[ix][iy+1]- sAcsp.ppfPCAFunction[ix][iy-1])/ 2.;
                }
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
            {
                if (iDisplay == YES)  mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfDxPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
            {
                if (iDisplay == YES)  mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
            start_time = clock();
            if ( !iBSplineImage(sAcsp.ppfDyPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
            {
                mexPrintf("\n\nError in bspline\n \n");
            }
            else
            {
                end_time = clock();
                if (iDisplay == YES)
                {
                    mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                    if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                }
            }
            
        } /* end of  if ( 1./fScale <= fScaleMinBeforeReDist ) */
        
        
        fDiracEpsScale = sAcsp.fDiracEps*fScale;
        for (ix=0; ix< iNx; ix++)
            for (iy=0; iy< iNy; iy++)
        {
            if ( ABS(sAcsp.ppfPCAFunction[ix][iy]) < 2.0* fDiracEpsScale )
                sAcsp.ppfDiracPCAFunction[ix][iy] = DIRAC(sAcsp.ppfPCAFunction[ix][iy],fDiracEpsScale);
            sAcsp.ppfHeavisidePCAFunction[ix][iy] = HEAVISIDE(sAcsp.ppfPCAFunction[ix][iy],fDiracEpsScale);
            sAcsp.ppfNegHeavisidePCAFunction[ix][iy] = 1.- sAcsp.ppfHeavisidePCAFunction[ix][iy];
            }
        
        
        
        
        sAcsp.pfVectorSpatialTransformation[0] = fXCenter;
        sAcsp.pfVectorSpatialTransformation[1] = fYCenter;
        sAcsp.pfVectorSpatialTransformation[2] = fScale;
        sAcsp.pfVectorSpatialTransformation[3] = fAngle;
        sAcsp.pfVectorSpatialTransformation[4] = fTx;
        sAcsp.pfVectorSpatialTransformation[5] = fTy;
        if ( iTypeSpatialTransformation == AFFINE )
            sAcsp.pfVectorSpatialTransformation[6] = fShearing;
        
        
        if (*pfEVec!=0.0)
        {
            
            for (i=0; i< ip; i++)
            {
                
                    for (ix=0; ix< iNx; ix++)
                        for (iy=0; iy< iNy; iy++)
                            sAcsp.pppfPcaEigenVectors[i][ix][iy] = pfEVec[i*iNx*iNy + ix*iNy+ iy];
                
                
                start_time = clock();
                if ( !iBSplineImage(sAcsp.pppfPcaEigenVectors[i],NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,0.,iDisplay) )
                {
                    mexPrintf("\n\nError in bspline\n \n");
                }
                else
                {
                    end_time = clock();
                    if (iDisplay == YES)
                    {
                        mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.pfVectorSpatialTransformation[0]+1.0,sAcsp.pfVectorSpatialTransformation[1]+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
                        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
                    }
                }
                
            }
            
            
        } /* end of if (*pfEVec!=0.0) */
        
    }
  /* END OF  SHAPE TERM AND REGION SHAPE TERM */
  /**************************/
    
    
    
}




// update x_T in 32 and x_pca in 31
void vComputeRegistrationParameters
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfEVec,
float  *pfEVal,
float  *pfPCAMeanValue,
int    ip,
float  *pfVpcaOut,
int    iTypeSpatialTransformation,
float  *pfOutputVecRigidTransformations,
float  *pfVecShapeTerm,
float  fDeltaT,
float  fSign,
float  *pfVecRegionTerm,
float  *pfIm,
int    iNbIter,
int    iNbTemporalSteps,
float  *pfVecGeneralParameters,
float  *pfTemp
)
{
  /* the normal N is inward */
    
    int    ix, iy, i;
    float  fScale, fAngle, fTx, fTy;
    float  fSumScale, fSumAngle, fSumTx, fSumTy, fShearing;
    float  fXCenter, fYCenter, fx, fy, fXnew, fYnew, fTempX, fTempY;
    float  fTempSumTx, fTempSumTy, fTempSumScale, fTempSumAngle, fThresholdPca;
    float  fSumShearing, fTempSumShearing;
    float  fFactorMaxEigenValue;
    float  fRs11, fRs12, fRs21, fRs22, fRa11, fRa12, fRa21, fRa22;
    float  fAs11, fAs12, fAs21, fAs22, fAa11, fAa12, fAa21, fAa22;
    float  fAsh11, fAsh12, fAsh21, fAsh22;
    float  fFunction, fWeightShapeTerm;
    float  fMaxSumTxFar, fMaxSumTyFar, fThresholdSumTxFar, fThresholdSumTyFar;
    float  fThresholdSumScaleFar, fThresholdSumAngleFar, fThresholdSumShearingFar;
    float  fThresholdSumTxClose, fThresholdSumTyClose, fMaxSumScaleClose, fMaxSumAngleClose;
    float  fMaxSumShearingClose, fMaxSumScaleFar, fMaxSumAngleFar, fMaxSumShearingFar;
    float  fLambdaTx, fLambdaTy, fLambdaScale, fLambdaAngle, fLambdaShearing, fMaxShearing;
    float  fThresholdSumTxPca, fThresholdSumTyPca, fThresholdSumScalePca, fThresholdSumAnglePca;
    float  fThresholdSumShearingPca, fPcaPercentageClose, fPcaPercentageFar;
    float  fWeightRegionTerm, fLengthShapePrior, fHomogeneityDescriptor;
    float  fSumScaleShapeTerm, fSumAngleShapeTerm, fSumTxShapeTerm, fSumTyShapeTerm;
    float  fSumShearingShapeTerm, fSumScaleRegionTerm, fSumAngleRegionTerm;
    float  fSumTxRegionTerm, fSumTyRegionTerm, fSumShearingRegionTerm;
    int    iDisplay;
    int    iIterComputeUi;
    
    
    iDisplay = (int) pfVecGeneralParameters[0];
    
    
    fWeightShapeTerm = pfVecShapeTerm[0];
    fWeightRegionTerm = pfVecRegionTerm[0];
    
    
    fXCenter = sAcsp.fPhiCx;
    fYCenter = sAcsp.fPhiCy;
    fScale = pfOutputVecRigidTransformations[2];
    fAngle = pfOutputVecRigidTransformations[3];
    fTx = pfOutputVecRigidTransformations[4];
    fTy = pfOutputVecRigidTransformations[5];
    if ( iTypeSpatialTransformation == AFFINE )
        fShearing = pfOutputVecRigidTransformations[6];
    
    
    if ( iTypeSpatialTransformation == RIGID )
    {
        fRs11 = cos(fAngle); /* scale */
        fRs12 = sin(fAngle);
        fRs21 = -sin(fAngle);
        fRs22 = cos(fAngle);
        
        fRa11 = fScale* sin(fAngle); /* angle */
        fRa12 = -fScale* cos(fAngle);
        fRa21 = fScale* cos(fAngle);
        fRa22 = fScale* sin(fAngle);
    }
    else if ( iTypeSpatialTransformation == AFFINE )
    {
        fAs11 = cos(fAngle); /* scale */
        fAs12 = fShearing* cos(fAngle) + sin(fAngle);
        fAs21 = -sin(fAngle);
        fAs22 = -fShearing* sin(fAngle) + cos(fAngle);
        
        fAa11 = fScale* sin(fAngle); /* angle */
        fAa12 = fScale* fShearing* sin(fAngle) - fScale* cos(fAngle);
        fAa21 = fScale* cos(fAngle);
        fAa22 = fScale* fShearing* cos(fAngle) + fScale* sin(fAngle);
        
        fAsh11 = 0.0; /* shearing */
        fAsh12 = fScale* cos(fAngle);
        fAsh21 = 0.0;
        fAsh22 = -fScale* sin(fAngle);
    }
    
    
    fSumScaleShapeTerm = 0.0;
    fSumAngleShapeTerm = 0.0;
    fSumTxShapeTerm = 0.0;
    fSumTyShapeTerm = 0.0;
    fSumShearingShapeTerm = 0.0;
    
    if ( fWeightShapeTerm> 0.0 )
    {
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
        {
            
            fFunction = sAcsp.ppfPCAFunction[ix][iy];
            
            fTempX = sAcsp.ppfDiracPhi[ix][iy]* fFunction* sAcsp.ppfDxPCAFunction[ix][iy];
            fTempY = sAcsp.ppfDiracPhi[ix][iy]* fFunction* sAcsp.ppfDyPCAFunction[ix][iy];
            
            fSumTxShapeTerm += fTempX;
            fSumTyShapeTerm += fTempY;
            
            fx = (float)(ix);
            fy = (float)(iy);
            
            if ( iTypeSpatialTransformation == RIGID )
            {
                fXnew = fRs11* (fx - fXCenter) + fRs12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fRs21* (fx - fXCenter) + fRs22* (fy - fYCenter)/*  + fYCenter */;
            }
            else if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAs11* (fx - fXCenter) + fAs12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAs21* (fx - fXCenter) + fAs22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumScaleShapeTerm += fTempX* fXnew + fTempY* fYnew;
            
            if ( iTypeSpatialTransformation == RIGID )
            {
                fXnew = fRa11* (fx - fXCenter) + fRa12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fRa21* (fx - fXCenter) + fRa22* (fy - fYCenter)/*  + fYCenter */;
            }
            else if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAa11* (fx - fXCenter) + fAa12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAa21* (fx - fXCenter) + fAa22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumAngleShapeTerm += fTempX* fXnew + fTempY* fYnew;
            
            if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAsh11* (fx - fXCenter) + fAsh12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAsh21* (fx - fXCenter) + fAsh22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumShearingShapeTerm += fTempX* fXnew + fTempY* fYnew;
            
                }
        
    }
    
    
    fSumScaleRegionTerm = 0.0;
    fSumAngleRegionTerm = 0.0;
    fSumTxRegionTerm = 0.0;
    fSumTyRegionTerm = 0.0;
    fSumShearingRegionTerm = 0.0;
    
    if ( fWeightRegionTerm> 0.0 )
    {
        
        iIterComputeUi = (int) pfVecRegionTerm[2];
        if ( (sAcsp.iNbIter==1) || (sAcsp.iNbIter%iIterComputeUi==0) )
        {
            if (iDisplay == YES)  mexPrintf("Compute New Uin and Uout in RegionTerm.\n");
            vComputeUinUoutRegionTerm(iNx,iNy,sAcsp,pfIm,pfVecRegionTerm,pfTemp);
            vComputeExtensionUinUoutRegionTerm(iNx,iNy,sAcsp,pfVecGeneralParameters,pfVecRegionTerm,pfTemp);
        }
        
        vComputeMSTermRegionTerm(iNx,iNy,sAcsp,pfIm,pfVecGeneralParameters,pfVecRegionTerm,iDisplay,pfTemp);
        
        vComputeLengthShapePrior(iNx,iNy,sAcsp,&fLengthShapePrior,pfTemp);
        if (iDisplay == YES)  mexPrintf("fLengthShapePrior = %.2f\n",fLengthShapePrior);
        
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPCAFunction[ix][iy] > EPS )
        {
            
            fHomogeneityDescriptor = sAcsp.ppfMSRegionTerm[ix][iy];
            
            fTempX = sAcsp.ppfDiracPCAFunction[ix][iy]* fHomogeneityDescriptor* sAcsp.ppfDxPCAFunction[ix][iy];
            fTempY = sAcsp.ppfDiracPCAFunction[ix][iy]* fHomogeneityDescriptor* sAcsp.ppfDyPCAFunction[ix][iy];
            
            fSumTxRegionTerm += fTempX;
            fSumTyRegionTerm += fTempY;
            
            fx = (float)(ix);
            fy = (float)(iy);
            
            if ( iTypeSpatialTransformation == RIGID )
            {
                fXnew = fRs11* (fx - fXCenter) + fRs12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fRs21* (fx - fXCenter) + fRs22* (fy - fYCenter)/*  + fYCenter */;
            }
            else if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAs11* (fx - fXCenter) + fAs12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAs21* (fx - fXCenter) + fAs22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumScaleRegionTerm += fTempX* fXnew + fTempY* fYnew;
            
            
            if ( iTypeSpatialTransformation == RIGID )
            {
                fXnew = fRa11* (fx - fXCenter) + fRa12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fRa21* (fx - fXCenter) + fRa22* (fy - fYCenter)/*  + fYCenter */;
            }
            else if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAa11* (fx - fXCenter) + fAa12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAa21* (fx - fXCenter) + fAa22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumAngleRegionTerm += fTempX* fXnew + fTempY* fYnew;
            
            if ( iTypeSpatialTransformation == AFFINE )
            {
                fXnew = fAsh11* (fx - fXCenter) + fAsh12* (fy - fYCenter)/*  + fXCenter */;
                fYnew = fAsh21* (fx - fXCenter) + fAsh22* (fy - fYCenter)/*  + fYCenter */;
            }
            fSumShearingRegionTerm += fTempX* fXnew + fTempY* fYnew;
            
                }
        
        
    }
    
    
    if ( fWeightShapeTerm> 0.0 )
    {
      /* Normalizations */
        fSumTxShapeTerm /= sAcsp.fPhiLength;
        fSumTyShapeTerm /= sAcsp.fPhiLength;
        fSumScaleShapeTerm /= ((float)iNx)* sAcsp.fPhiLength;
        fSumAngleShapeTerm /= ((float)iNx)* sAcsp.fPhiLength;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingShapeTerm /= ((float)iNx)* sAcsp.fPhiLength;
        
        fWeightShapeTerm = pfVecShapeTerm[0];
        
        fSumTxShapeTerm *= fDeltaT* fWeightShapeTerm;
        fSumTyShapeTerm *= fDeltaT* fWeightShapeTerm;
        fSumScaleShapeTerm *= fDeltaT* fWeightShapeTerm;
        fSumAngleShapeTerm *= fDeltaT* fWeightShapeTerm;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingShapeTerm *= fDeltaT* fWeightShapeTerm;
        
        
      /************** SSD *************/
        fSumTxShapeTerm *= 30.;
        fSumTyShapeTerm *= 30.;
        fSumScaleShapeTerm *= 100.;
        fSumAngleShapeTerm *= 250.;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingShapeTerm *= 250.;
      /************** END SSD *************/
        
        
    }
    
    
    if ( fWeightRegionTerm> 0.0 )
    {
      /* Normalizations */
        fSumTxRegionTerm /= fLengthShapePrior;
        fSumTyRegionTerm /= fLengthShapePrior;
        fSumScaleRegionTerm /= ((float)iNx)* fLengthShapePrior;
        fSumAngleRegionTerm /= ((float)iNx)* fLengthShapePrior;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingRegionTerm /= ((float)iNx)* fLengthShapePrior;
        
        fSumTxRegionTerm *= fDeltaT* fWeightRegionTerm;
        fSumTyRegionTerm *= fDeltaT* fWeightRegionTerm;
        fSumScaleRegionTerm *= fDeltaT* fWeightRegionTerm;
        fSumAngleRegionTerm *= fDeltaT* fWeightRegionTerm;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingRegionTerm *= fDeltaT* fWeightRegionTerm;
        
      /**********************/
        fSumTxRegionTerm *= 50.;
        fSumTyRegionTerm *= 50.;
        fSumScaleRegionTerm *= 300.;
        fSumAngleRegionTerm *= 300.;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearingRegionTerm *= 1000.;
      /**********************/
    }
    
    if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm<1e-6 )
    {
        fSumTx = fSumTxShapeTerm;
        fSumTy = fSumTyShapeTerm;
        fSumScale = fSumScaleShapeTerm;
        fSumAngle = fSumAngleShapeTerm;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearing = fSumShearingShapeTerm;
        
      /* Max values when C* far away from the solution */
        fMaxSumTxFar = 0.4;
        fMaxSumTyFar = 0.4;
        fThresholdSumTxFar = 30.;
        fThresholdSumTyFar = 30.;
        fThresholdSumScaleFar = 30.;
        fThresholdSumAngleFar = 30.;
        if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingFar = 30.;
        
      /* Max values when C* close to the solution */
        fThresholdSumTxClose = fThresholdSumTxFar/ 2.;
        fThresholdSumTyClose = fThresholdSumTyFar/ 2.;
        fMaxSumScaleClose = 0.01;
        fMaxSumAngleClose = 0.5* PI/180.;
        if ( iTypeSpatialTransformation == AFFINE )  fMaxSumShearingClose = 0.01;
    }
    
    if ( fWeightRegionTerm> 0.0 && fWeightShapeTerm<1e-6 )
    {
        fSumTx = fSumTxRegionTerm;
        fSumTy = fSumTyRegionTerm;
        fSumScale = fSumScaleRegionTerm;
        fSumAngle = fSumAngleRegionTerm;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearing = fSumShearingRegionTerm;
        
      /* Max values when C* far away from the solution */
        fMaxSumTxFar = 0.2;
        fMaxSumTyFar = 0.2;
        fThresholdSumTxFar = 100.;
        fThresholdSumTyFar = 100.;
        fThresholdSumScaleFar = 100.;
        fThresholdSumAngleFar = 100.;
        if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingFar = 100.;
        
      /* Max values when C* close to the solution */
        fThresholdSumTxClose = fThresholdSumTxFar/ 3.;
        fThresholdSumTyClose = fThresholdSumTyFar/ 3.;
        fMaxSumScaleClose = 0.005;
        fMaxSumAngleClose = 0.4* PI/180.;
        if ( iTypeSpatialTransformation == AFFINE )  fMaxSumShearingClose = 0.005;
    }
    
    if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm> 0.0 )
    {
        fSumTx = fWeightShapeTerm* fSumTxShapeTerm + fWeightRegionTerm* fSumTxRegionTerm;
        fSumTy = fWeightShapeTerm* fSumTyShapeTerm + fWeightRegionTerm* fSumTyRegionTerm;
        fSumScale = fWeightShapeTerm* fSumScaleShapeTerm + fWeightRegionTerm* fSumScaleRegionTerm;
        fSumAngle = fWeightShapeTerm* fSumAngleShapeTerm + fWeightRegionTerm* fSumAngleRegionTerm;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearing = fWeightShapeTerm* fSumShearingShapeTerm + fWeightRegionTerm* fSumShearingRegionTerm;
        
      /* Max values when C* far away from the solution */
        fMaxSumTxFar = 0.3;
        fMaxSumTyFar = 0.3;
        fThresholdSumTxFar = 50.;
        fThresholdSumTyFar = 50.;
        fThresholdSumScaleFar = 50.;
        fThresholdSumAngleFar = 50.;
        if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingFar = 50.;
        
      /* Max values when C* close to the solution */
        fThresholdSumTxClose = fThresholdSumTxFar/ 2.;
        fThresholdSumTyClose = fThresholdSumTyFar/ 2.;
        fMaxSumScaleClose = 0.008;
        fMaxSumAngleClose = 0.4* PI/180.;
        if ( iTypeSpatialTransformation == AFFINE )  fMaxSumShearingClose = 0.008;
        
        
        
      /* Max values when C* far away from the solution */
        fMaxSumTxFar = 0.2;
        fMaxSumTyFar = 0.2;
        fThresholdSumTxFar = 100.;
        fThresholdSumTyFar = 100.;
        fThresholdSumScaleFar = 100.;
        fThresholdSumAngleFar = 100.;
        if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingFar = 100.;
        
      /* Max values when C* close to the solution */
        fThresholdSumTxClose = fThresholdSumTxFar/ 3.;
        fThresholdSumTyClose = fThresholdSumTyFar/ 3.;
        fMaxSumScaleClose = 0.005;
        fMaxSumAngleClose = 0.4* PI/180.;
        if ( iTypeSpatialTransformation == AFFINE )  fMaxSumShearingClose = 0.005;
        
        
        
    }
    
    
    fMaxSumScaleFar = fMaxSumScaleClose/ 10.;
    fMaxSumAngleFar = fMaxSumAngleClose/ 10.;
    if ( iTypeSpatialTransformation == AFFINE )  fMaxSumShearingFar = fMaxSumShearingClose/ 10.;
    
    
    fLambdaTx = fMaxSumTxFar/ fThresholdSumTxFar;
    fLambdaTy = fMaxSumTyFar/ fThresholdSumTyFar;
    fLambdaScale = fMaxSumScaleClose/ fThresholdSumScaleFar;
    fLambdaAngle = fMaxSumAngleClose/ fThresholdSumAngleFar;
    if ( iTypeSpatialTransformation == AFFINE )  fLambdaShearing = fMaxSumShearingClose/ fThresholdSumShearingFar;
    
    
    if (iDisplay == YES)
    {
        mexPrintf("\nfSumTx = %.5f\n",fSumTx);
        mexPrintf("fSumTy = %.5f\n",fSumTy);
        mexPrintf("fSumScale = %.5f\n",fSumScale);
        mexPrintf("fSumAngle = %.5f\n",fSumAngle);
        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("fSumShearing = %.5f\n",fSumShearing);
    }
    
    
    fTempSumTx = fSumTx;
    fTempSumTy = fSumTy;
    fTempSumScale = fSumScale;
    fTempSumAngle = fSumAngle;
    if ( iTypeSpatialTransformation == AFFINE )  fTempSumShearing = fSumShearing;
    
    
    if ( ABS(fSumTx)>fThresholdSumTxFar )  fSumTx = SIGN(fSumTx)* fMaxSumTxFar;
    else  fSumTx *= fLambdaTx;
    
    if ( ABS(fSumTy)>fThresholdSumTyFar )  fSumTy = SIGN(fSumTy)* fMaxSumTyFar;
    else  fSumTy *= fLambdaTy;
    
    
    if ( (ABS(fTempSumTx)<fThresholdSumTxClose) && (ABS(fTempSumTy)<fThresholdSumTyClose) )
    {
        if ( ABS(fSumScale)>fThresholdSumScaleFar )  fSumScale = SIGN(fSumScale)* fMaxSumScaleClose;
        else  fSumScale *= fLambdaScale;
        if ( ABS(fSumAngle)>fThresholdSumAngleFar )  fSumAngle = SIGN(fSumAngle)* fMaxSumAngleClose;
        else  fSumAngle *= fLambdaAngle;
        if ( iTypeSpatialTransformation == AFFINE )
            if ( ABS(fSumShearing)>fThresholdSumShearingFar )  fSumShearing = SIGN(fSumShearing)* fMaxSumShearingClose;
            else  fSumShearing *= fLambdaShearing;
    }
    else
    {
        fSumScale = SIGN(fSumScale)* fMaxSumScaleFar;
        fSumAngle = SIGN(fSumAngle)* fMaxSumAngleFar;
        if ( iTypeSpatialTransformation == AFFINE )  fSumShearing = SIGN(fSumShearing)* fMaxSumShearingFar;
    }
    
    
    
    if (iDisplay == YES)
    {
        mexPrintf("\nfSumTx after = %.5f\n",fSumTx);
        mexPrintf("fSumTy after = %.5f\n",fSumTy);
        mexPrintf("fSumScale after = %.5f\n",fSumScale);
        mexPrintf("fSumAngle after = %.5f, fSumAngle(deg) = %.5f\n",fSumAngle,fSumAngle*180/PI);
        if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("fSumShearing after = %.5f\n",fSumShearing);
        mexPrintf("\n",fSumAngle);
    }
    
    fTx -= fSumTx;
    fTy -= fSumTy;
    fScale = 1./(1./fScale + fSumScale);
    fAngle += fSumAngle;
    if ( iTypeSpatialTransformation == AFFINE )  fShearing -= fSumShearing;
    
    fMaxShearing = 0.5;
    if ( ABS(fShearing) > fMaxShearing )  fShearing = SIGN(fShearing)* fMaxShearing;
    
    
  /* Comment here to keep the same rigid parameters */
    pfOutputVecRigidTransformations[2] = fScale;
    pfOutputVecRigidTransformations[3] = fAngle;
    pfOutputVecRigidTransformations[4] = fTx;
    pfOutputVecRigidTransformations[5] = fTy;
    if ( iTypeSpatialTransformation == AFFINE )  pfOutputVecRigidTransformations[6] = fShearing;
    
    
    
    if (*pfEVec!=0.0)
    {
        
        if ( fWeightShapeTerm> 0.0 )
        {
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaShapeTerm[i] = 0.0;
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
                    if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
            {
                fFunction = sAcsp.ppfPCAFunction[ix][iy];
                for (i=0; i< ip; i++)
                    sAcsp.pfSumPcaShapeTerm[i] += sAcsp.ppfDiracPhi[ix][iy]* fFunction* sAcsp.pppfPcaEigenVectors[i][ix][iy];
                    }
            
            
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaShapeTerm[i] *= 500.;
            
            
      /* Parameters of the pca function */
            fFactorMaxEigenValue = 2.5;
            
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaShapeTerm[i] *= fDeltaT* fWeightShapeTerm/ sAcsp.fPhiLength;
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaShapeTerm[i] *= pfEVal[0];
            
        }
        
        
        if ( fWeightRegionTerm> 0.0 )
        {
            
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaRegionTerm[i] = 0.0;
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
                    if ( sAcsp.ppfDiracPCAFunction[ix][iy] > EPS )
            {
                fHomogeneityDescriptor = sAcsp.ppfMSRegionTerm[ix][iy];
                for (i=0; i< ip; i++)
                    sAcsp.pfSumPcaRegionTerm[i] += sAcsp.ppfDiracPCAFunction[ix][iy]* fHomogeneityDescriptor* sAcsp.pppfPcaEigenVectors[i][ix][iy];
                    }
            
            
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaRegionTerm[i] *= 100.;
            
            
            fFactorMaxEigenValue = 2.5;
            
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaRegionTerm[i] *= fDeltaT* fWeightRegionTerm/ fLengthShapePrior;
            for (i=0; i< ip; i++)  sAcsp.pfSumPcaRegionTerm[i] *= pfEVal[0];
            
        }
        
        
        if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm<1e-6 )
        {
            for (i=0; i< ip; i++)  sAcsp.pfSumPca[i] = sAcsp.pfSumPcaShapeTerm[i];
            
            fThresholdPca = 40.0;
            fThresholdSumTxPca = fThresholdSumTxFar/ 2.;
            fThresholdSumTyPca = fThresholdSumTyFar/ 2.;
            fThresholdSumScalePca = fThresholdSumScaleFar/ 2.;
            fThresholdSumAnglePca = fThresholdSumAngleFar/ 2.;
            if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingPca = fThresholdSumShearingFar/ 4.;
            fPcaPercentageClose = 0.05*100./pfEVal[0];
            fPcaPercentageFar = fPcaPercentageClose/ 10.;
        }
        
        
        if ( fWeightRegionTerm> 0.0 && fWeightShapeTerm<1e-6 )
        {
            for (i=0; i< ip; i++)  sAcsp.pfSumPca[i] = sAcsp.pfSumPcaRegionTerm[i];
            
            fThresholdPca = 100.0;
            fThresholdSumTxPca = fThresholdSumTxFar/ 2.;
            fThresholdSumTyPca = fThresholdSumTyFar/ 2.;
            fThresholdSumScalePca = fThresholdSumScaleFar/ 2.;
            fThresholdSumAnglePca = fThresholdSumAngleFar/ 2.;
            if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingPca = fThresholdSumShearingFar/ 4.;
            fPcaPercentageClose = 0.05*100./pfEVal[0];/* 0.05 */;
            fPcaPercentageFar = fPcaPercentageClose/ 10.;
            
        }
        
        if ( fWeightShapeTerm> 0.0 && fWeightRegionTerm> 0.0 )
        {
            for (i=0; i< ip; i++)  sAcsp.pfSumPca[i] = fWeightShapeTerm* sAcsp.pfSumPcaShapeTerm[i] + fWeightRegionTerm* sAcsp.pfSumPcaRegionTerm[i];
            
            fThresholdPca = 60.0;
            fThresholdSumTxPca = fThresholdSumTxFar/ 2.;
            fThresholdSumTyPca = fThresholdSumTyFar/ 2.;
            fThresholdSumScalePca = fThresholdSumScaleFar/ 2.;
            fThresholdSumAnglePca = fThresholdSumAngleFar/ 2.;
            if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingPca = fThresholdSumShearingFar/ 4.;
            fPcaPercentageClose = 0.05*100./pfEVal[0];
            fPcaPercentageFar = fPcaPercentageClose/ 10.;
            
            
            
            fThresholdPca = 100.0;
            fThresholdSumTxPca = fThresholdSumTxFar/ 2.;
            fThresholdSumTyPca = fThresholdSumTyFar/ 2.;
            fThresholdSumScalePca = fThresholdSumScaleFar/ 2.;
            fThresholdSumAnglePca = fThresholdSumAngleFar/ 2.;
            if ( iTypeSpatialTransformation == AFFINE )  fThresholdSumShearingPca = fThresholdSumShearingFar/ 4.;
            fPcaPercentageClose = 0.05*100./pfEVal[0];/* 0.05 */;
            fPcaPercentageFar = fPcaPercentageClose/ 10.;
            
        }
        
        if (iDisplay == YES)  for (i=0; i< ip; i++)  mexPrintf("sAcsp.pfSumPca[%i] = %.5f\n",i,sAcsp.pfSumPca[i]);
        
        
        
        
        for (i=0; i< ip; i++)  sAcsp.pfMaxSumPcaFar[i] = fPcaPercentageFar* pfEVal[i];
        for (i=0; i< ip; i++)  sAcsp.pfMaxSumPcaClose[i] = fPcaPercentageClose* pfEVal[i];
        
        if ( iTypeSpatialTransformation == RIGID )
        {
            
            
            
            for (i=0; i< ip; i++)
                if (ABS(sAcsp.pfSumPca[i])>fThresholdPca)  sAcsp.pfSumPca[i] = SIGN(sAcsp.pfSumPca[i])* sAcsp.pfMaxSumPcaClose[i];
                else  sAcsp.pfSumPca[i] *= sAcsp.pfMaxSumPcaClose[i]/ fThresholdPca;
            
            
        }
        else if ( iTypeSpatialTransformation == AFFINE )
        {
            
            
            for (i=0; i< ip; i++)
                if (ABS(sAcsp.pfSumPca[i])>fThresholdPca)  sAcsp.pfSumPca[i] = SIGN(sAcsp.pfSumPca[i])* sAcsp.pfMaxSumPcaClose[i];
                else  sAcsp.pfSumPca[i] *= sAcsp.pfMaxSumPcaClose[i]/ fThresholdPca;
            
            
        }
        
        if (iDisplay == YES)  mexPrintf("\n");
        if (iDisplay == YES)  for (i=0; i< ip; i++)  mexPrintf("sAcsp.pfSumPca[%i] after = %.5f\n",i,sAcsp.pfSumPca[i]);
        
        
        // Comment here to keep same pca parameters
        for (i=0; i< ip; i++)  pfVpcaOut[i] -= sAcsp.pfSumPca[i];
        for (i=0; i< ip; i++)  if (ABS(pfVpcaOut[i])>fFactorMaxEigenValue* pfEVal[i]) pfVpcaOut[i] = SIGN(pfVpcaOut[i])* fFactorMaxEigenValue* pfEVal[i];
        

    }
    
}



// compute final PCA function, hat{phi} in eq 29.
void vComputeFinalPcaFunction
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
int    iDimPca[],
float  *pfPCAFunctionOut,
float  *pfEVec,
float  *pfEVal,
float  *pfPCAMeanValue,
int    ip,
float  *pfVpcaOut,
int    iTypeSpatialTransformation,
float  *pfOutputVecRigidTransformations,
float  *pfVecShapeTerm,
float  *pfVecGeneralParameters,
float  *pfTemp
)

{
    
    int    ix, iy, i, iDim[3];
    float  **ppfPCAFunctionTemp, fPu;
    int    iNyPca, iNxPca, iDisplay;
    time_t start_time, end_time;
    
    
    iDisplay = (int) pfVecGeneralParameters[0];
    
    iDim[0] = iNy;
    iDim[1] = iNx;
    iDim[2] = ip;
    
    iNyPca = iDimPca[0];
    iNxPca = iDimPca[1];
    iDimPca[2] = ip;
    
    
    ppfPCAFunctionTemp = sAcsp.ppfPCAFunction;

    
  /* Computation of a new data */
    if (*pfEVec!=0.0)
    {
        vComputePCAReconstruction(ppfPCAFunctionTemp,pfEVec,pfPCAMeanValue,pfVpcaOut,pfEVal,iDimPca,&fPu);
        if (iDisplay == YES)
        {
            mexPrintf("\nPCA reconstruction for an image %ix%i and %i principal component(s)\nprobability of the reconstructed data = %.4f\n",iNy,iNx,ip,fPu);
            for (i=0; i< ip; i++)  mexPrintf("eigenmode[%i] = %.3f  ",i,pfVpcaOut[i]);
            mexPrintf(" \n");
        }
    }
    else
        for (ix=0; ix< iNxPca; ix++)
            for (iy=0; iy< iNyPca; iy++)
                sAcsp.ppfPCAFunction[ix][iy] = pfPCAMeanValue[ix*iNyPca + iy];
    
    
    
    sAcsp.pfVectorSpatialTransformation[0] = pfOutputVecRigidTransformations[0]; /* fXCenter */
    sAcsp.pfVectorSpatialTransformation[1] = pfOutputVecRigidTransformations[1]; /* fYCenter */
    sAcsp.pfVectorSpatialTransformation[2] = pfOutputVecRigidTransformations[2]; /* fScale */
    sAcsp.pfVectorSpatialTransformation[3] = pfOutputVecRigidTransformations[3]; /* fAngle */
    sAcsp.pfVectorSpatialTransformation[4] = pfOutputVecRigidTransformations[4]; /* fTx */
    sAcsp.pfVectorSpatialTransformation[5] = pfOutputVecRigidTransformations[5]; /* fTy */
    if ( iTypeSpatialTransformation == AFFINE )
        sAcsp.pfVectorSpatialTransformation[6] = pfOutputVecRigidTransformations[6]; /* fShearing */
    
    start_time = clock();
    if ( !iBSplineImage(sAcsp.ppfPCAFunction,NULL,iDim,sAcsp.iSplineDegree,sAcsp.pfVectorSpatialTransformation,iTypeSpatialTransformation,sAcsp.fMaxPCAMeanValue,iDisplay) )
    {
        mexPrintf("\n\nError in bspline\n \n");
    }
    else
    {
        end_time = clock();
        if (iDisplay == YES)
        {
            mexPrintf("\nComputing time for the B-Splines: %.3f sec for an image %ix%i\nCenter: XCenter = %.3f, YCenter = %.3f\nSpatial transformations: Tx = %.3f, Ty = %.3f, Angle = %.3f, Scale = %.3f\n",difftime(end_time,start_time)/1000,iDim[0],iDim[1],sAcsp.fPhiCx+1.0,sAcsp.fPhiCy+1.0,sAcsp.pfVectorSpatialTransformation[4],sAcsp.pfVectorSpatialTransformation[5],sAcsp.pfVectorSpatialTransformation[3]*180/PI,sAcsp.pfVectorSpatialTransformation[2]);
            if ( iTypeSpatialTransformation == AFFINE )  mexPrintf("Shearing = %.3f\n",sAcsp.pfVectorSpatialTransformation[6]);
        }
    }
    
    for (ix=0; ix< iNx; ix++)
        for (iy=0; iy< iNy; iy++)
            pfPCAFunctionOut[ix*iNy + iy] = sAcsp.ppfPCAFunction[ix][iy];
    
    
    
}




// compute attraction term = <grad hat{phi}^2,grad varphi>
void vComputeAttractionValleyShapeTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfVecShapeTerm,
int    iDisplay,
float  *pfTemp
)
{
    int    ix, iy;
    float  fDxShape, fDyShape, fDxPlusAcsp, fDxMinusAcsp;
    float  fDyPlusAcsp, fDyMinusAcsp, fMaxAttractionValleyShapeTerm;
    float  ABSPCAFunction, fMaxPCAFunction;
    float  fMaxPCAFunctionDiv2, fPCAFunction, fWeightAttractionValleyShapeTerm, fWeightShapeTerm;
    
    
    fMaxPCAFunction = 10.;
    fMaxPCAFunctionDiv2 = fMaxPCAFunction/2.0;
    
    
    fWeightAttractionValleyShapeTerm = pfVecShapeTerm[1];
    fWeightShapeTerm = pfVecShapeTerm[0];
    
    if ( fWeightAttractionValleyShapeTerm> 0.0 )
    {
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
        {
            
            ABSPCAFunction = HEAVISIDE_OFFSET(ABS(sAcsp.ppfPCAFunction[ix][iy]),fMaxPCAFunctionDiv2,fMaxPCAFunctionDiv2);
            fPCAFunction = SIGN(sAcsp.ppfPCAFunction[ix][iy])* ABSPCAFunction;
            
            fDxShape = 2.0* fPCAFunction* sAcsp.ppfDxPCAFunction[ix][iy];
            fDyShape = 2.0* fPCAFunction* sAcsp.ppfDyPCAFunction[ix][iy];
            
            fDxPlusAcsp = sAcsp.ppfDxp[ix][iy];
            fDxMinusAcsp = sAcsp.ppfDxm[ix][iy];
            
            fDyPlusAcsp = sAcsp.ppfDyp[ix][iy];
            fDyMinusAcsp = sAcsp.ppfDym[ix][iy];
            
            sAcsp.ppfAttractionValleyShapeTerm[ix][iy] = ( MAX(fDxShape,0.0)* fDxPlusAcsp +  MIN(fDxShape,0.0)* fDxMinusAcsp ) + 
            ( MAX(fDyShape,0.0)* fDyPlusAcsp +  MIN(fDyShape,0.0)* fDyMinusAcsp );
            
            sAcsp.ppfAttractionValleyShapeTerm[ix][iy] *= sAcsp.ppfDiracPhi[ix][iy];
            
                }
        
        
        if (iDisplay == YES)
        {
            fMaxAttractionValleyShapeTerm = 0.0;
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
                    if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                        if ( ABS(sAcsp.ppfAttractionValleyShapeTerm[ix][iy])> ABS(fMaxAttractionValleyShapeTerm) )
                            fMaxAttractionValleyShapeTerm = sAcsp.ppfAttractionValleyShapeTerm[ix][iy];
            mexPrintf("fMaxAttractionValleyShapeTerm*fWeight (ShapeTerm)= %.3f\n",fWeightShapeTerm* fWeightAttractionValleyShapeTerm* fMaxAttractionValleyShapeTerm);
        }
        
    }
    
}





// compute shape-based term
void vComputeShapeTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfVecShapeTerm,
int    iDisplay,
float  *pfTemp
)
{
    int    ix, iy;
    float  fMaxShapeTerm, fWeightAttractionValleyShapeTerm;
    
    
    fWeightAttractionValleyShapeTerm = pfVecShapeTerm[1];
    
    
    for (ix=1; ix< iNx-1; ix++)
        for (iy=1; iy< iNy-1; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfShapeTerm[ix][iy] = fWeightAttractionValleyShapeTerm* sAcsp.ppfAttractionValleyShapeTerm[ix][iy];
    
    
    if (iDisplay == YES)
    {
        fMaxShapeTerm = 0.0;
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                    if ( ABS(sAcsp.ppfShapeTerm[ix][iy])> ABS(fMaxShapeTerm) )
                        fMaxShapeTerm = sAcsp.ppfShapeTerm[ix][iy];
        mexPrintf("fMaxShapeTerm = %.3f\n",fMaxShapeTerm);
    }
    
    
}





// This function computes a new PCA function hat{phi} WITHOUT any spatial transformation, eq 4.
void vComputePCAReconstruction
(
 float    **ppfImRecons,
 float    *pfEVec,
 float    *pfMeanValue,
 float    *pfVcoef,
 float    *pfEVal,
 int      iDim[],
 float    *pfPu
)
{
  int     iNx, iNy, ip, ix, iy, n, d, i, j;
  float   fSum;
 

  iNy = iDim[0];
  iNx = iDim[1];
  ip  = iDim[2];
  
  d = iNy* iNx; /* number of rows in data matrix */
  n = ip;      /* number of columns in data matrix */

/* Computation of a new data */
  for (ix=0; ix< iNx; ix++)	
    for (iy=0; iy< iNy; iy++)
      {
	j = ix*iNy+ iy+ 1;
	ppfImRecons[ix][iy] = 0.0;
	for (i = 1; i <= n; i++) 
	  ppfImRecons[ix][iy] += pfEVec[(i-1)*iNy*iNx + j-1]* pfVcoef[i-1];
	ppfImRecons[ix][iy] += pfMeanValue[j-1];
      }

  fSum = 0.0;
  for (i=0; i< ip; i++)
    if ( SQR(pfEVal[i]) > 0.001 )
      fSum += pfVcoef[i]* pfVcoef[i]/ SQR(pfEVal[i]);
  fSum = -fSum/ 2.;
  *pfPu = (float) exp(fSum);


}



/****************** End of file *******************/
