
% Author: Xavier Bresson (xbresson at math.ucla.edu)
% Last version: Oct 06 2008
% Name: test_activeContourShapePrior
% Description: Active contour with shape prior, see 
% [X. Bresson, P. Vandergheynst and J.-P. Thiran, A Variational Model for Object
% Segmentation Using Boundary Information and Shape Prior Driven by the 
% Mumford-Shah Functional, IJCV 68(2), 2006, p145-162.] 
% and more specifically Eqs (4) and (29-33).



%%%%%%%%%%% Main function %%%%%%%%%%%%%%%%%%%%%

function runActiveContourShapePrior

close all;


%%%%%%%%% definition of constants %%%%%%%%%
YES = 0;
NO = 1;

RIGID = 0;
AFFINE = 1;
%%%%%%%%% END definition of constants %%%%%%%%%




%%%%%%%%% EXAMPLE #1 (1000iter 65sec) %%%%%%%%%
load_image = '../images/ellipse_cut.gif'; f=3.5;
dist_to_borders = 25; % distance from the border for initial active contour
stdUinit=20; % standard deviation for Gaussian blurring for init u_in and u_out

% Parameters used in the active contour program 
% General
displayData = YES;
displayData = NO;
nb_temporal_steps = 60; % 60
nb_iter_reinitSDF = 5; % 5
deltat = 0.4; % 0.4
% Boundary
weightBoundaryTerm = 1; % 1
weightCurvatureTerm = 0.1; % 0.1
weightAttractionValleyTerm = 0.1; % 0.1
% Shape
weightShapeTerm = 1/3;  % 1/3
weightAttractionValleyShapeTerm = 1.7; % 1.7
choice_spatial_transformations = RIGID;
%choice_spatial_transformations = AFFINE;
% Region
weightRegionTerm = 10;  % 10
Mu = 50; % 50
iterComputeUi = 5; % 5
deltatU = 0.05/ Mu;
% spatial transformations (mu=scale, theta=rotation, T=translation, sh=shearing)
mu0 = 1/1.8; theta_rd0 = (0)* pi/180; Tx0 = -0; Ty0 = 0; sh0 = 0; 
% eigencoefficients
Vpca(1) = 200;
%%%%%%%%% END EXAMPLE #1 %%%%%%%%%



% %%%%%%%%% EXAMPLE #2 (450iter 30sec) %%%%%%%%%
% load_image = '../images/ellipse_occlusion.gif'; f=3.5;
% dist_to_borders = 50; % distance from the border for initial active contour
% stdUinit=8; % standard deviation for Gaussian blurring for init u_in and u_out
% 
% % Parameters used in the active contour program 
% % General
% displayData = YES;
% displayData = NO;
% nb_temporal_steps = 20; % 20
% nb_iter_reinitSDF = 5; % 5
% deltat = 0.4; % 0.4
% % Boundary
% weightBoundaryTerm = 1; % 1
% weightCurvatureTerm = 1; % 1
% weightAttractionValleyTerm = 5; % 5
% % Shape
% weightShapeTerm = 1/3; % 1/3
% weightAttractionValleyShapeTerm = 1.5; % 1.5
% choice_spatial_transformations = RIGID;
% %choice_spatial_transformations = AFFINE;
% % Region
% weightRegionTerm = 10; % 10
% Mu = 50; % 50
% iterComputeUi = 5; % 5
% deltatU = 0.05/ Mu;
% % spatial transformations (mu=scale, theta=rotation, T=translation, sh=shearing)
% mu0 = 1/0.8; theta_rd0 = (0)* pi/180; Tx0 = 0; Ty0 = 0; sh0 = 0; 
% % eigencoefficients
% Vpca(1) = 200;
% %%%%%%%%% END EXAMPLE #2 %%%%%%%%%






%%%%%%%%% shape prior %%%%%%%%%
disp('load training set of ellipses');
loadfile = '../data/pcaEllipses.mat';
load(loadfile,'p','EVec','EVal','MeanValue');
[Ny,Nx,p] = size(EVec);
EVal = sqrt(double(EVal)); % sqrt of eigenvalues
%%%%%%%%% END shape prior %%%%%%%%%


%%%%%%%%% Vectors of parameters %%%%%%%%%
VecGeneralParameters = [displayData; nb_temporal_steps; nb_iter_reinitSDF; deltat;];

VecBoundaryTerm = [weightBoundaryTerm; weightCurvatureTerm; weightAttractionValleyTerm;];

VecShapeTerm = [weightShapeTerm; weightAttractionValleyShapeTerm; choice_spatial_transformations;];

VecRegionTerm = [weightRegionTerm; Mu; iterComputeUi; deltatU;];

XCenterPCA = compute_center_sdf(double(MeanValue));
xOrigin = XCenterPCA(1); yOrigin = XCenterPCA(2);
RTs = [xOrigin; yOrigin; mu0; theta_rd0; Tx0; Ty0; sh0;];
%%%%%%%%% Vectors of parameters %%%%%%%%%


%%%%%%%%% load image to be segmented %%%%%%%%%
disp('load image to be segmented');
[ImToSegment,map] = imread(load_image);

ImToSegment = double(ImToSegment);
ImToSegment = ImToSegment/ max(ImToSegment(:));
[Ny,Nx] = size(ImToSegment);
%%%%%%%%% load image to be segmented %%%%%%%%%


%%%%%%%%% compute edge detector function %%%%%%%%%
disp('compute edge detector function');
std = 0.5; % standard deviation of the Gaussian function
beta = 4; % control the depth of the edge detector function

Ng = ceil(6*std); Gaussian = fspecial('gaussian',[Ny Nx],std);
ImToSegmentSmooth = ifftshift( ifft2( fft2(ImToSegment) .* fft2(Gaussian) ) );
[Gx,Gy] = gradient(ImToSegmentSmooth);
GradImToSegmentSmooth = sqrt(Gx.^2 + Gy.^2); 
grad = GradImToSegmentSmooth./ max(GradImToSegmentSmooth(:));
edgeDetector = 1./ (1 + beta* grad.^2);

figure(1); clf; imagesc(edgeDetector); colormap(gray); colorbar; hold on;
truesize(1,[round(f*Ny) round(f*Nx)]);
title('edge detector function');
%pause;
%%%%%%%%% compute edge detector function %%%%%%%%%


%%%%%%%%% initial level set function %%%%%%%%%
disp('compute initial level set function');

SDF = zeros(Ny,Nx);
SDF(dist_to_borders:Ny-dist_to_borders,dist_to_borders:Nx-dist_to_borders)=1; SDF(SDF<0.5)=-1;
SDF = sdf_mex(single(SDF),single(Ny*Nx),int32(1),int32(1)); SDF=double(SDF);

figure(2); clf; imagesc(SDF); colormap(gray); colorbar; hold on; contour(SDF,[0 0],'m');
[c0,h0] = contour(SDF,[0 0],'r'); set(h0,'LineWidth',2);
truesize(2,[round(f*Ny) round(f*Nx)]);
title('Initial level set function');
%pause

SDF_initial_ac = SDF;
%%%%%%%%% initial level set function %%%%%%%%%


%%%%%%%%% display initial active contour %%%%%%%%%
disp('display initial active contour');
figure(3); clf; imagesc(ImToSegment); hold on; colormap(gray); %axis off;
[c0,h0] = contour(SDF_initial_ac,[0 0],'r'); set(h0,'LineWidth',2);
title('Initial active contour');
truesize(3,[round(f*Ny) round(f*Nx)]);
%pause;
%%%%%%%%% display initial active contour and shape prior %%%%%%%%%


%%%%%%%%% smooth initial uin and uout %%%%%%%%%
disp('smooth initial uin and uout');
Ng = ceil(6*stdUinit); Gaussian = fspecial('gaussian',[Ny Nx],stdUinit);
uin = ifftshift( ifft2( fft2(ImToSegment) .* fft2(Gaussian) ) );
uout = uin;
figure(4); imagesc(uin); colorbar; hold on; colormap(gray);
truesize(4,[round(f*Ny) round(f*Nx)]);
title('Initial u_{in} and u_{out}');
%pause;
%%%%%%%%% smoothe initial uin and uout %%%%%%%%%



%%%%%%%%% active contour with shape prior %%%%%%%%%
disp('enter loop for active contour with shape prior');
ac = SDF_initial_ac;
max_nb_iters = 100000;
nb_iters = 1;
while ( nb_iters <= max_nb_iters )
 
  nb_iters
  
  [ac_new,PcaFunction_new,RTs_new,Vpca_new,MSRegionS_new,uin_new,uout_new,Temp] = ...
      acsp_mex(single(VecGeneralParameters),...
    single(ac),single(ImToSegment),single(VecBoundaryTerm),single(edgeDetector),...
    single(VecShapeTerm),single(RTs),single(Vpca),single(EVec),single(EVal),single(MeanValue),...
    single(VecRegionTerm),single(uin),single(uout));
  ac_new = double(ac_new);
  RTs_new = double(RTs_new);
  Vpca_new = double(Vpca_new);
  MSRegionS_new = double(MSRegionS_new);
  uin_new = double(uin_new);
  uout_new = double(uout_new);
  Temp = double(Temp);  
    
  
  RTs_display = RTs_new;
  RTs_display(4) = RTs_display(4)*180/pi;
  RTs_display = RTs_display(3:7);
  
  
  % display spatial transformation and eigencoefficients
  RTs_display
  Vpca_new
  

  % Display the piecewise-smooth approximation defined by the shape prior
  figure(5); clf;
  imagesc(MSRegionS_new); hold on; colormap(gray); axis off; %colorbar;
  [c0,h0] = contour(PcaFunction_new,[0 0],'g'); set(h0,'LineWidth',2);
  [c0,h0] = contour(ac_new,[0 0],'r'); set(h0,'LineWidth',2.5);
  title('Piecewise-smooth approximation. Active contour (red) and Shape prior (green).');
  truesize(5,[round(f*Ny) round(f*Nx)]);
  


  % Display active contour and shape prior
  figure(6); clf; imagesc(ImToSegment,[0 1]); hold on; colormap(gray); axis off;
  [c0,h0] = contour(PcaFunction_new,[0 0],'g'); set(h0,'LineWidth',2);
  [c0,h0] = contour(ac_new,[0 0],'r'); set(h0,'LineWidth',2.5);
  title('Active contour (red) and Shape prior (green).');
  truesize(6,[round(f*Ny) round(f*Nx)]);
  %pause;

  
  ac = ac_new;  
  RTs = RTs_new;
  Vpca = Vpca_new;
  uin = uin_new;
  uout = uout_new;
  nb_iters = nb_iters + 1;

end
%%%%%%%%% active contour with shape prior %%%%%%%%%

%%%%%%%%%%% END Main function %%%%%%%%%%%%%%%%%%%%%












%%%%%%%%%%% Sub-function %%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%% compute center of level set function %%%%%%%%%%%%%%%%%%%%%
function XCenterSDF = compute_center_sdf(phi)

[Ny,Nx] = size(phi);

Cx = 0;
Cy = 0;
cpt = 0;
for x=1:Nx
  for y=1:Ny
    if round(phi(y,x)) == 0
      Cx = Cx + x;
      Cy = Cy + y;
      cpt = cpt + 1;
    end
  end
end

XCenterSDF(1) = Cx/ cpt;
XCenterSDF(2) = Cy/ cpt;
%%%%%%%%%%% compute center of level set function %%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%% END Sub-function %%%%%%%%%%%%%%%%%%%%%

