
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: pca_mex.c
// Description: This function perfoms the principal components analysis (PCA) on a 
// training set of signed distance functions. This code uses Numerical Recipies and 
// the paper of [M. Leventon, W. Grimson, and O. Faugeras, "Statistical Shape Inuence 
// in Geodesic Active Contours," IEEE CVPR, 2000, p316-323].

// Compilation (run in matlab):
// mex -v -g pca_mex.c nrutil.c 
// mex pca_mex.c nrutil.c 




#include <stdio.h>
#include <string.h>
#include <math.h>
#include <mex.h>
#include <time.h>

#include "nrutil.h"



#define SIGN2(a, b) ( (b) < 0 ? -fabs(a) : fabs(a) )

#define SWAP(a,b,tmp) (tmp)=(a); (a)=(b); (b)=(tmp)

#define MODE_PCA_DECOMPOSITION 0
#define MODE_PCA_RECONSTRUCTION 1
#define MODE_PCA_PROJECTION 2

#define PI 3.1415926

#define YES 0
#define NO 1

struct sTemp {
   float  **ppfEVecTemp;
   float  **ppfEValTemp;
   float  **ppfMeanValueTemp;
   float  **ppfuTemp;
   float  **ppfVcoefTemp;
};





/********************************************************************************************/
/**************************  Declaration of static procedures  ******************************/
/********************************************************************************************/

/********************************************************************************************/
static int iArrayMemoryAllocation
(
   float ***pppfArray,
   int   iNbRows,
   int   iNbCols
);
/*********************************************************************************************/

/*********************************************************************************************/
static void vArrayMemoryFree
(
   float ***pppfArray,
   int   iNbRows,
   int   iNbCols
);
/*********************************************************************************************/

/*********************************************************************************************/
static int iComputeDecomposition
(
   float    *pfImageIn,
   int      iDim[],
   float    fFittingGoodness,
   int      *pip,
   struct   sTemp *psDataTemp,
   int      iDisplay
);
/*********************************************************************************************/

/*********************************************************************************************/
static int iComputeReconstruction
(
  float    *pfEVec,
  float    *pfMeanValue,
  float    *pfVcoef,
  int      iDim[],
  struct   sTemp *psDataTemp,
  int      iDisplay
);
/*********************************************************************************************/

/*********************************************************************************************/
static int iComputeProjection
(
  float    *pfEVec,
  float    *pfEVal,
  float    *pfMeanValue,
  float    *pfuNew,
  int      iDim[],
  struct   sTemp *psDataTemp,
  int      iDisplay
);
/*********************************************************************************************/

/*********************************************************************************************/
static void vComputeDualCovarianceMatrix
(
  float  **ppfX,
  float  **ppfT,
  struct sTemp  *psDataTemp,
  int    d,
  int    n
);
/*********************************************************************************************/

/*********************************************************************************************/
static void svd
(
  float   **a, 
  int     m, 
  int     n, 
  float   **w, 
  float   **v
);
/*********************************************************************************************/



/*********************************************************************************************/
/*****************************  Definition of static procedures   ****************************/
/*********************************************************************************************/

/*********************************************************************************************/
static int iArrayMemoryAllocation
(
   float ***pppfArray,
   int   iNbRows,
   int   iNbCols
)

{
   int i;

   (*pppfArray) = (float **) calloc( (unsigned)(iNbRows), sizeof(float *) );
   if (!(*pppfArray))
   {
     mexPrintf("Memory allocation failure 1 in iArrayMemoryAllocation\n");
     return(0);
   }

   for (i=0; i<iNbRows; i++)
   {
     (*pppfArray)[i] = (float *) calloc( (unsigned)(iNbCols), sizeof(float)  );
     if (!(*pppfArray)[i])
     {
       mexPrintf("Memory allocation failure 2 in iArrayMemoryAllocation\n");
       return(0);     
     }
   }

   return(1);
}
/*********************************************************************************************/


/*********************************************************************************************/
static void vArrayMemoryFree
(
   float ***pppfArray,
   int   iNbRows,
   int   iNbCols
)

{
   int i;
   for (i=0; i<iNbRows; i++)
     free( (float *) ((*pppfArray)[i]) );
   free( (float **) (*pppfArray) );
}
/*********************************************************************************************/


/*********************************************************************************************/
static void svd(float **a, int m, int n, float **w, float **v)
/* This function comes from Numerical Recipies in C                                          */
/* (http://lib-www.lanl.gov/numerical/bookcpdf.html):                                        */ 
/* Given a matrix a[1..m][1..n], this routine computes its                                   */
/* Singular Value Decomposition (SVD) A = U  W  V. The matrix U replaces a on output.      */
/* The diagonal matrix of singular values W is out- put as a vector w[1..n].                 */
/* The matrix V (not the transpose V' ) is output as v[1..n][1..n].                          */

{ 
  float   pythag(float a, float b); 
  int     flag,i,its,j,jj,k,l,nm;
  float   anorm,c,f,g,h,s,scale,x,y,z,*rv1;

  rv1=vector(1,n);

  g=scale=anorm=0.0; /* Householder reduction to bidiagonal form.*/ 
  for (i=1;i<=n;i++) 
  { 
    l=i+1; 
    rv1[i]=scale*g; 
    g=s=scale=0.0; 
    if (i <= m) 
    { 
      for (k=i;k<=m;k++) scale += fabs(a[k-1][i-1]); 
      if (scale) 
      { 
        for (k=i;k<=m;k++) 
	{ 
	  a[k-1][i-1] /= scale; 
	  s += a[k-1][i-1]*a[k-1][i-1]; 
	} 
	f=a[i-1][i-1]; 
	g = -SIGN(sqrt(s),f); 
	h=f*g-s; 
	a[i-1][i-1]=f-g; 
	for (j=l;j<=n;j++) 
	{ 
	  for (s=0.0,k=i;k<=m;k++) s += a[k-1][i-1]*a[k-1][j-1]; 
	  f=s/h; 
	  for (k=i;k<=m;k++) a[k-1][j-1] += f*a[k-1][i-1]; 
	} 
	for (k=i;k<=m;k++) a[k-1][i-1] *= scale; 
      }
    } 
    w[i-1][0]=scale *g; 
    g=s=scale=0.0; 
    if (i <= m && i != n) 
    { 
      for (k=l;k<=n;k++) scale += fabs(a[i-1][k-1]); 
      if (scale) 
      {
        for (k=l;k<=n;k++) 
	{ 
	  a[i-1][k-1] /= scale; 
	  s += a[i-1][k-1]*a[i-1][k-1]; 
	} 
	f=a[i-1][l-1]; 
	g = -SIGN(sqrt(s),f); 
	h=f*g-s; 
	a[i-1][l-1]=f-g; 
	for (k=l;k<=n;k++) rv1[k]=a[i-1][k-1]/h; 
	for (j=l;j<=m;j++) 
	{ 
	  for (s=0.0,k=l;k<=n;k++) s += a[j-1][k-1]*a[i-1][k-1]; 
	  for (k=l;k<=n;k++) a[j-1][k-1] += s*rv1[k]; 
	} 
	for (k=l;k<=n;k++) a[i-1][k-1] *= scale; 
      } 
    } 
    anorm=FMAX(anorm,(fabs(w[i-1][0])+fabs(rv1[i]))); 
  } 
  for (i=n;i>=1;i--) 
  { /* Accumulation of right-hand transformations. */ 
    if (i < n) 
    {
      if (g) 
      { 
        for (j=l;j<=n;j++) /* Double division to avoid possible under  o . */  
	  v[j-1][i-1]=(a[i-1][j-1]/a[i-1][l-1])/g; 
	for (j=l;j<=n;j++) 
	{ 
	for (s=0.0,k=l;k<=n;k++) s += a[i-1][k-1]*v[k-1][j-1]; 
	for (k=l;k<=n;k++) v[k-1][j-1] += s*v[k-1][i-1]; 
	} 
      } 
      for (j=l;j<=n;j++) v[i-1][j-1]=v[j-1][i-1]=0.0; 
    } 
    v[i-1][i-1]=1.0;
    g=rv1[i]; 
    l=i; 
  } 
  for (i=IMIN(m,n);i>=1;i--) 
  { /* Accumulation of left-hand transformations. */ 
    l=i+1; 
    g=w[i-1][0]; 
    for (j=l;j<=n;j++) a[i-1][j-1]=0.0; 
    if (g) 
    { 
      g=1.0/g; 
      for (j=l;j<=n;j++) 
      { 
        for (s=0.0,k=l;k<=m;k++) s += a[k-1][i-1]*a[k-1][j-1]; 
	f=(s/a[i-1][i-1])*g;
	for (k=i;k<=m;k++) a[k-1][j-1] += f*a[k-1][i-1]; 
      } 
      for (j=i;j<=m;j++) a[j-1][i-1] *= g; 
    } 
    else for (j=i;j<=m;j++) a[j-1][i-1]=0.0;
    ++a[i-1][i-1]; 
  } 
  for (k=n;k>=1;k--) 
  {  /* Diagonalization of the bidiagonal form:Loop over singular values,and over allo ed iterations. */
    for (its=1;its<=30;its++) 
    {
      flag=1; 
      for (l=k;l>=1;l--) 
      { /* Test for splitting. */
        nm=l-1; /*  Note that rv1[1] is always zero.*/
	if ((float)(fabs(rv1[l])+anorm) == anorm) 
	{ 
	  flag=0; 
	  break; 
	} 
	if ((float)(fabs(w[nm-1][0])+anorm) == anorm) break; 
      } 
      if (flag) 
      { 
        c=0.0;/* Cancellation of rv1[l], if l > 1 */
	s=1.0;
	for (i=l;i<=k;i++) 
	{
	  f=s*rv1[i]; 
	  rv1[i]=c*rv1[i]; 
	  if ((float)(fabs(f)+anorm) == anorm) break; 
	  g=w[i-1][0]; 
	  h=pythag(f,g); 
	  w[i-1][0]=h; 
	  h=1.0/h; 
	  c=g*h; 
	  s = -f*h; 
	  for (j=1;j<=m;j++) 
	  { 
	    y=a[j-1][nm-1]; 
	    z=a[j-1][i-1]; 
	    a[j-1][nm-1]=y*c+z*s; 
	    a[j-1][i-1]=z*c-y*s; 
	  } 
	} 
      } 
      z=w[k-1][0]; 
      if (l == k) 
      {/*  Convergence. */ 
        if (z < 0.0) 
	{ /* Singular value is made nonnegative. */ 
	  w[k-1][0] = -z; 
	  for (j=1;j<=n;j++) v[j-1][k-1] = -v[j-1][k-1]; 
	} 
	break; 
      } 
      if (its == 30) nrerror("no convergence in 30 svdcmp iterations"); 
      x=w[l-1][0];/*  Shift from bottom 2-by-2 minor. */
      nm=k-1; 
      y=w[nm-1][0]; 
      g=rv1[nm]; 
      h=rv1[k]; 
      f=((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y); 
      g=pythag(f,1.0);
      f=((x-z)*(x+z)+h*((y/(f+SIGN(g,f)))-h))/x; 
      c=s=1.0;/*  Next QR transformation:  */
      for (j=l;j<=nm;j++) 
      { 
        i=j+1; 
	g=rv1[i]; 
	y=w[i-1][0]; 
	h=s*g; 
	g=c*g; 
	z=pythag(f,h); 
	rv1[j]=z; 
	c=f/z; 
	s=h/z; 
	f=x*c+g*s; 
	g = g*c-x*s; 
	h=y*s; 
	y *=c; 
	for (jj=1;jj<=n;jj++) 
	{ 
	  x=v[jj-1][j-1];
	  z=v[jj-1][i-1]; 
	  v[jj-1][j-1]=x*c+z*s;
	  v[jj-1][i-1]=z*c-x*s; 
	} 
	z=pythag(f,h); 
	w[j-1][0]=z; /* Rotation can be arbitrary if z = 0 */
	if (z)
	{ 
	  z=1.0/z; 
	  c=f*z; 
	  s=h*z; 
	} 
	f=c*g+s*y;
	x=c*y-s*g;
	for (jj=1;jj<=m;jj++) 
	{ 
	  y=a[jj-1][j-1]; 
	  z=a[jj-1][i-1]; 
	  a[jj-1][j-1]=y*c+z*s;
	  a[jj-1][i-1]=z*c-y*s;
	} 
      }
      rv1[l]=0.0; 
      rv1[k]=f; 
      w[k-1][0]=x; 
    } 
  } 

  free_vector(rv1,1,n);

}
/*********************************************************************************************/


/*********************************************************************************************/
float pythag(float a, float b) 
 /* Computes (a 2 + b 2 ) 1/2 without destructive under  o or over  ow.  */
{ 
  float absa,absb; 
  absa=fabs(a); 
  absb=fabs(b); 
  if (absa > absb) return absa*sqrt(1.0+SQR(absb/absa)); 
  else return (absb == 0.0 ? 0.0 : absb*sqrt(1.0+SQR(absa/absb))); 
}
/*********************************************************************************************/


/*********************************************************************************************/
static void vComputeDualCovarianceMatrix
(
  float  **ppfX,
  float  **ppfT,
  struct sTemp  *psDataTemp,
  int    d,
  int    n
)

{
  int i, j, j1, j2;

  /* Computation of the mean of column vectors (i.e. SDFs) of data matrix ppfX */
  for (j = 1; j <= d; j++)
  {
    psDataTemp->ppfMeanValueTemp[j-1][0] = 0.;
    for (i = 1; i <= n; i++) 
      psDataTemp->ppfMeanValueTemp[j-1][0] += ppfX[j-1][i-1];   
    psDataTemp->ppfMeanValueTemp[j-1][0] /= (float)n;
  }

  /* Center the column vectors */
  for (j = 1; j <= d; j++)
    for (i = 1; i <= n; i++)
      ppfX[j-1][i-1] -= psDataTemp->ppfMeanValueTemp[j-1][0];
        
  /* Computation of n x n covariance matrix */
  for (j1 = 1; j1 <= n; j1++) 
    for (j2 = j1; j2 <= n; j2++)
    {
      ppfT[j1-1][j2-1] = 0.0;
      for (j = 1; j <= d; j++)
        ppfT[j1-1][j2-1] += ppfX[j-1][j1-1]* ppfX[j-1][j2-1];
      ppfT[j1-1][j2-1] /= (float)n;
      ppfT[j2-1][j1-1] = ppfT[j1-1][j2-1]; /* a covariance matrix is always symmetric */
    }
    
 /*  for (j1 = 1; j1 <= n; j1++) 
    for (j2 = 1; j2 <= n; j2++) 
      mexPrintf("T[%i][%i]= %.1f\n",j2-1,j1-1,ppfT[j2-1][j1-1]);  */
  
}
/*********************************************************************************************/


/*********************************************************************************************/
static int iComputeDecomposition
(
   float    *pfImageIn,
   int      iDim[],
   float    fFittingGoodness,
   int      *pip,
   struct   sTemp *psDataTemp,
   int      iDisplay
)

{

  float   **ppfX, **ppfXbis, **ppfT, **ppfEVal, **ppfEVec, **ppfV, **ppfU;
  float   ftmp, fTotEval, fPercentage, fSum;
  int     iNx, iNy, iNd, id, ix, iy, ip;
  int     n, d,  i, j, i2;


  iNy = iDim[0];
  iNx = iDim[1];
  iNd = iDim[2];
  
  d = iNy* iNx; /* number of rows in data matrix */
  n = iNd;      /* number of columns in data matrix */

  if (!iArrayMemoryAllocation(&ppfX,d,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfX\n"); 
    return(0);
  }


  if (!iArrayMemoryAllocation(&ppfXbis,d,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfXbis\n"); 
    return(0);
  }

  for (id=0; id< iNd; id++)
    for (ix=0; ix< iNx; ix++) 
      for (iy=0; iy< iNy; iy++)
      {
	ppfX[ix*iNy+ iy][id] = pfImageIn[id*iNy*iNx+ ix*iNy+ iy];
	ppfXbis[ix*iNy+ iy][id] = pfImageIn[id*iNy*iNx+ ix*iNy+ iy];
      }

  if (!iArrayMemoryAllocation(&psDataTemp->ppfEVecTemp,d,n)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfEVecTemp\n"); 
    return(0);
  }   
  
  if (!iArrayMemoryAllocation(&psDataTemp->ppfEValTemp,n,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfEValTem\n"); 
    return(0);
  }   
  
  if (!iArrayMemoryAllocation(&psDataTemp->ppfMeanValueTemp,d,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfMeanValueTemp\n");
    return(0);
  }  

  if (iDisplay == YES)  mexPrintf("Computation of Dual Covariance Matrix\n");

  if (!iArrayMemoryAllocation(&ppfT,n,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfT\n");
    return(0);
  }  

  /* Computation of the dual covariance matrix */
  vComputeDualCovarianceMatrix(ppfXbis,ppfT,psDataTemp,d,n);

  if (iDisplay == YES)  mexPrintf("Computation of Eigenvectors and Eigenvalues\n");

  if (!iArrayMemoryAllocation(&ppfEVec,n,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfT\n");
    return(0);
  } 

  if (!iArrayMemoryAllocation(&ppfEVal,n,1)) 
  { 
    mexPrintf("Memory allocation failure for ppfT\n");
    return(0);
  } 

  if (!iArrayMemoryAllocation(&ppfV,n,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfT\n");
    return(0);
  } 

  for (j = 1; j <= n; j++) 
    for (i = 1; i <= n; i++)
      ppfEVec[j-1][i-1] = ppfT[j-1][i-1];

  /* Decomposition in singular values of the dual covariance matrix */
  svd(ppfEVec,n,n,ppfEVal,ppfV);

  /* Sort eigenvalues and eigenvectors in decreasing order of eigenvalues */
  for (i = 1; i <= n; i++)
    for (i2 = i+1; i2 <= n; i2++)
      if ( ppfEVal[i-1][0] < ppfEVal[i2-1][0] )
      {
	SWAP(ppfEVal[i-1][0],ppfEVal[i2-1][0],ftmp);
        for (j = 1; j <= n; j++)
	{
	  SWAP(ppfEVec[j-1][i-1],ppfEVec[j-1][i2-1],ftmp);    
	} 
      } 

  for (i = 1; i <= n; i++)
    psDataTemp->ppfEValTemp[i-1][0] = ppfEVal[i-1][0];

/*
  mexPrintf("\nEigenvalues:\n");
  for (i = 1; i <= n; i++) 
    mexPrintf("EVal[%i]= %.4f\n",i,ppfEVal[i-1][0]);

  mexPrintf("\nEigenvectors:\n"); 
  for (i = 1; i <= n; i++) 
  { 
    for (j = 1; j <= n; j++) 
      mexPrintf("EVec[%i]= %.4f\n",i,ppfEVec[j-1][i-1]); 
    mexPrintf("\n"); 
  }
*/


  if (iDisplay == YES)  mexPrintf("Computation of the matrix containing the basis of SDF- principal components\n");

  /* U is the matrix whose columns are SDF principal components */
  if (!iArrayMemoryAllocation(&ppfU,d,n)) 
  { 
    mexPrintf("Memory allocation failure for ppfU\n");
    return(0);
  } 

  /* Computation of U  */
  for (i = 1; i <= n; i++)
    for (j = 1; j <= d; j++) 
    {
      ppfU[j-1][i-1] = 0.;
      for (i2 = 1; i2 <= n; i2++) 
	ppfU[j-1][i-1] += ppfX[j-1][i2-1]* ppfEVec[i2-1][i-1];
    }
  

  /* Computation of the p principal components */
  if ( *pip > 0 ) {
    if (iDisplay == YES)  mexPrintf("number of p principal components: %i\n",*pip); }
  else
  {
    if (iDisplay == YES)  mexPrintf("Computation of p principal modes\n");

    fTotEval = 0;
    for (i = 1; i <= n; i++)
      fTotEval += ppfEVal[i-1][0];

    ip = 1;
    fSum = 0.;
    for (i = 1; i <= n; i++)
    {
      fSum += ppfEVal[i-1][0];
      fPercentage = fSum/ fTotEval;
      if ( fPercentage >= fFittingGoodness )
        i = n+1;
      else
        ip++;
    }

    *pip = ip;
    if (iDisplay == YES)  mexPrintf("number of p principal components: %i for a fitting goodness= %.4f\n",*pip,fPercentage);
  }

  /* Normalize principal component vectors to 1 */
  for (i = 1; i <= n; i++)
  {
    fSum = 0.;
    for (j = 1; j <= d; j++)
      fSum += ppfU[j-1][i-1]* ppfU[j-1][i-1];
    for (j = 1; j <= d; j++)
      ppfU[j-1][i-1] /= sqrt(fSum);
  }

  for (i = 1; i <= n; i++)
    for (j = 1; j <= d; j++)
      psDataTemp->ppfEVecTemp[j-1][i-1] = ppfU[j-1][i-1];

  vArrayMemoryFree(&ppfX,d,n);
  vArrayMemoryFree(&ppfXbis,d,n);
  vArrayMemoryFree(&ppfT,n,n);
  vArrayMemoryFree(&ppfEVec,n,n);
  vArrayMemoryFree(&ppfEVal,n,1);
  vArrayMemoryFree(&ppfV,n,n);
  vArrayMemoryFree(&ppfU,d,n);

  return(1);

}
/*********************************************************************************************/


/*********************************************************************************************/
static int iComputeReconstruction
(
  float    *pfEVec,
  float    *pfMeanValue,
  float    *pfVcoef,
  int      iDim[],
  struct   sTemp *psDataTemp,
  int      iDisplay
)
{
  int     iNx, iNy, ip, ix, iy, ik, n, d, i, j;

  if (iDisplay == YES)  mexPrintf("Construction of a new data using the basis of the principal components\n");

  iNy = iDim[0];
  iNx = iDim[1];
  ip = iDim[2];
  
  d = iNy* iNx; /* number of rows in data matrix */
  n = ip;      /* number of columns in data matrix */

  if (!iArrayMemoryAllocation(&psDataTemp->ppfEVecTemp,d,n)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfEVecTemp\n"); 
    return(0);
  }   
  
  for (ik=0; ik< ip; ik++)
    for (ix=0; ix< iNx; ix++) 
      for (iy=0; iy< iNy; iy++)
        psDataTemp->ppfEVecTemp[ix*iNy+ iy][ik] = pfEVec[ik*iNy*iNx+ ix*iNy+ iy];

  if (!iArrayMemoryAllocation(&psDataTemp->ppfMeanValueTemp,d,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfMeanValueTemp\n");
    return(0);
  }  

  for (ix=0; ix< iNx; ix++) 
    for (iy=0; iy< iNy; iy++)
      psDataTemp->ppfMeanValueTemp[ix*iNy+ iy][0] = pfMeanValue[ix*iNy+ iy];

  if (!iArrayMemoryAllocation(&psDataTemp->ppfuTemp,d,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfuTemp\n");
    return(0);
  }  


 /* for (i = 1; i <= n; i++)
    mexPrintf("pfVcoef[%i]= %f\n",i-1,pfVcoef[i-1]);*/

  /* Computation of a new data */
  for (j = 1; j <= d; j++) 
  {
    psDataTemp->ppfuTemp[j-1][0] = 0.0;
    for (i = 1; i <= n; i++) 
      psDataTemp->ppfuTemp[j-1][0] += psDataTemp->ppfEVecTemp[j-1][i-1]* pfVcoef[i-1];
    psDataTemp->ppfuTemp[j-1][0] += psDataTemp->ppfMeanValueTemp[j-1][0];
  }

  return(1);

}
/*********************************************************************************************/


/*********************************************************************************************/
static int iComputeProjection
(
  float    *pfEVec,
  float    *pfEVal,
  float    *pfMeanValue,
  float    *pfuNew,
  int      iDim[],
  struct   sTemp *psDataTemp,
  int      iDisplay
)
{

  int     iNx, iNy, ip, ix, iy, ik, n, d, i, j;

  if (iDisplay == YES)  mexPrintf("Projection of a data\n");

  iNy = iDim[0];
  iNx = iDim[1];
  ip = iDim[2];

  d = iNy* iNx; /* number of rows in data matrix */
  n = ip;      /* number of columns in data matrix */

  if (!iArrayMemoryAllocation(&psDataTemp->ppfEVecTemp,d,n)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfEVecTemp\n"); 
    return(0);
  }   
  
  for (ik=0; ik< ip; ik++)
    for (ix=0; ix< iNx; ix++) 
      for (iy=0; iy< iNy; iy++)
        psDataTemp->ppfEVecTemp[ix*iNy+ iy][ik] = pfEVec[ik*iNy*iNx+ ix*iNy+ iy];

  if (!iArrayMemoryAllocation(&psDataTemp->ppfMeanValueTemp,d,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfMeanValueTemp\n");
    return(0);
  }  

  for (ix=0; ix< iNx; ix++) 
    for (iy=0; iy< iNy; iy++)
      psDataTemp->ppfMeanValueTemp[ix*iNy+ iy][0] = pfMeanValue[ix*iNy+ iy];

  if (!iArrayMemoryAllocation(&psDataTemp->ppfuTemp,d,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfuTemp\n");
    return(0);
  }  

  for (ix=0; ix< iNx; ix++) 
    for (iy=0; iy< iNy; iy++)
      psDataTemp->ppfMeanValueTemp[ix*iNy+ iy][0] = pfMeanValue[ix*iNy+ iy];

  if (!iArrayMemoryAllocation(&psDataTemp->ppfVcoefTemp,n,1)) 
  { 
    mexPrintf("Memory allocation failure for psDataTemp->ppfVcoefTemp\n");
    return(0);
  }  


  /* Center the new data */
  for (j = 1; j <= d; j++) 
    psDataTemp->ppfuTemp[j-1][0] = pfuNew[j-1] - psDataTemp->ppfMeanValueTemp[j-1][0];

  /* Computation of eigen coefficients  */
  for (i = 1; i <= n; i++)
  {
    psDataTemp->ppfVcoefTemp[i-1][0] = 0.;
    for (j = 1; j <= d; j++) 
    psDataTemp->ppfVcoefTemp[i-1][0] += psDataTemp->ppfEVecTemp[j-1][i-1]* psDataTemp->ppfuTemp[j-1][0];
  }


/*
  for (i = 1; i <= n; i++)
    mexPrintf("Vcoef[%i]= %f\n",i-1,psDataTemp->ppfVcoefTemp[i-1][0]);
*/

  /* Computation of the projection of the new data in the principal components basis */	
  for (j = 1; j <= d; j++) 
  {
    psDataTemp->ppfuTemp[j-1][0] = 0.0;
    for (i = 1; i <= n; i++) 
      psDataTemp->ppfuTemp[j-1][0] += psDataTemp->ppfEVecTemp[j-1][i-1]* psDataTemp->ppfVcoefTemp[i-1][0];
    psDataTemp->ppfuTemp[j-1][0] += psDataTemp->ppfMeanValueTemp[j-1][0];
  }


  return(1);

}
/*********************************************************************************************/


/*********************************************************************************************/
extern void mexFunction(int iNbOut, mxArray *pmxOut[],
		 int iNbIn, const mxArray *pmxIn[])
{

  /* iNbOut: number of outputs
     pmxOut: array of pointers to output arguments */

  /* iNbIn: number of inputs
     pmxIn: array of pointers to input arguments */

  
  float   *pfImageIn, *pfFittingGoodness, fFittingGoodness, *pfEVec, *pfMeanValue, *pfu;
  float   *pfVcoef, *pfPu, *pfEVal, fSum, *pfPuProjected, *pfuProjected, *pfuNew;
  struct  sTemp sDataTemp;
  int     *piNx, *piNy, *piNd, iNx, iNy, iNd, ik, ix, iy, d, n, *pip, ip, i;
  int     *piPCAMode, iPCAMode, *piDisplay, iDisplay, iNdim, iDim[3], iDim2[3];
  time_t  start_time;
  time_t  end_time;


  start_time = clock();

  piPCAMode = mxGetData(pmxIn[0]);
  iPCAMode = *piPCAMode;

  if ( iPCAMode == MODE_PCA_DECOMPOSITION )
  {

    pfImageIn = mxGetData(pmxIn[1]);

    piNx = mxGetData(pmxIn[2]);
    iNx = *piNx;

    piNy = mxGetData(pmxIn[3]);
    iNy = *piNy;

    piNd = mxGetData(pmxIn[4]);
    iNd = *piNd;

    iDim[0] = iNy;
    iDim[1] = iNx;
    iDim[2] = iNd;

    d = iNy* iNx; /* number of rows in data matrix */
    n = iNd;      /* number of columns in data matrix */

  /*  mexPrintf("Ny= %i, Nx= %i, Nd= %i\n",iNy,iNx,iNd); */

    pfFittingGoodness = mxGetData(pmxIn[5]);
    fFittingGoodness = *pfFittingGoodness;

    pip = mxGetData(pmxIn[6]);
    ip = *pip;

    piDisplay = mxGetData(pmxIn[7]);
    iDisplay = *piDisplay;

    if (iDisplay == YES)  mexPrintf("\nStart PCA program\n");
    if (iDisplay == YES)  mexPrintf("MODE_PCA_DECOMPOSITION\n");

    if (!iComputeDecomposition(pfImageIn,iDim,fFittingGoodness,&ip,&sDataTemp,iDisplay))
    {
      mexPrintf("Error in PCA: MODE_PCA_DECOMPOSITION\n");
    }
    else
    {
      iNdim = 3;
      iDim2[0] = iNy;
      iDim2[1] = iNx;
      iDim2[2] = ip;
    /*  mexPrintf("iDim2[0]= %i, iDim2[1]= %i\n",iDim2[0],iDim2[0]);*/
      pmxOut[0] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfEVec = mxGetData(pmxOut[0]);

      for (ik=0; ik< ip; ik++)
        for (ix=0; ix< iNx; ix++) 
	  for (iy=0; iy< iNy; iy++)
	    pfEVec[ik*iNy*iNx+ ix*iNy+ iy] = sDataTemp.ppfEVecTemp[ix*iNy+ iy][ik];


      iNdim = 2;
      iDim2[0] = ip;
      iDim2[1] = 1;
      pmxOut[1] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfEVal = mxGetData(pmxOut[1]);

      for (ik=0; ik< ip; ik++)
	pfEVal[ik] = sDataTemp.ppfEValTemp[ik][0];


      iNdim = 2;
      iDim2[0] = 1;
      iDim2[1] = 1;
      pmxOut[2] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxINT32_CLASS,mxREAL);
      pip = mxGetData(pmxOut[2]);

      *pip = ip;


      iNdim = 2;
      iDim2[0] = iNy;
      iDim2[1] = iNx;
      pmxOut[3] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfMeanValue = mxGetData(pmxOut[3]);

      for (ix=0; ix< iNx; ix++) 
	for (iy=0; iy< iNy; iy++)
	  pfMeanValue[ix*iNy+ iy] = sDataTemp.ppfMeanValueTemp[ix*iNy+ iy][0];

      vArrayMemoryFree(&sDataTemp.ppfEVecTemp,d,n);
      vArrayMemoryFree(&sDataTemp.ppfEValTemp,n,1);
      vArrayMemoryFree(&sDataTemp.ppfMeanValueTemp,d,1);
      
      iDisplay = *piDisplay;
      
      end_time = clock();
      if (iDisplay == YES)  
          mexPrintf("Computing Time for PCA program: %.5f sec \n   for an image %ix%i and %i training data\n",difftime(end_time,start_time)/1000,iNy,iNx,iNd);
    }

  } /* end of if ( iPCAMode == MODE_PCA_DECOMPOSITION ) */


  else if( iPCAMode == MODE_PCA_RECONSTRUCTION )
  {

    pfEVec = mxGetData(pmxIn[1]);

    pfEVal = mxGetData(pmxIn[2]);

    pfMeanValue = mxGetData(pmxIn[3]);

    pfVcoef = mxGetData(pmxIn[4]);

    pip = mxGetData(pmxIn[5]);
    ip = *pip;

    piNx = mxGetData(pmxIn[6]);
    iNx = *piNx;

    piNy = mxGetData(pmxIn[7]);
    iNy = *piNy;

    piDisplay = mxGetData(pmxIn[8]);
    iDisplay = *piDisplay;

    if (iDisplay == YES)  mexPrintf("\nStart PCA program\n");
    if (iDisplay == YES)  mexPrintf("MODE_PCA_RECONSTRUCTION\n");


    iDim[0] = iNy;
    iDim[1] = iNx;
    iDim[2] = ip;

    d = iNy* iNx; /* number of rows in data matrix */
    n = ip;       /* number of columns in data matrix */

    if (!iComputeReconstruction(pfEVec,pfMeanValue,pfVcoef,iDim,&sDataTemp,iDisplay))
    {
      mexPrintf("Error in PCA: MODE_PCA_RECONSTRUCTION\n");
    }
    else
    {
      iNdim = 2;
      iDim2[0] = iNy;
      iDim2[1] = iNx;
      pmxOut[0] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfu = mxGetData(pmxOut[0]);

      for (ix=0; ix< iNx; ix++) 
	for (iy=0; iy< iNy; iy++)
	  pfu[ix*iNy+ iy] = sDataTemp.ppfuTemp[ix*iNy+ iy][0];


      iNdim = 2;
      iDim2[0] = 1;
      iDim2[1] = 1;
      pmxOut[1] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfPu = mxGetData(pmxOut[1]);

      /* Computation of the probability that the reconstructed data belongs to the class */
      /* of the training set */
      *pfPu = 0.0;

      fSum = 0.;
      for (i=0; i< ip; i++)
	if ( pfEVal[i] > 0.001 )
	  fSum += pfVcoef[i]* pfVcoef[i]/ pfEVal[i];
      fSum = -fSum/ 2.;
      *pfPu = (float) exp(fSum);
      fSum = 1.;
      for (i=0; i< ip; i++)
        if ( pfEVal[i] > 0.001 )	 
	  fSum *= pfEVal[i];
      fSum = (float) sqrt(fSum);
      fSum /= pow(2*PI,(double)(ip/2.));

      /*  *pfPu /= fSum; */ /* Normalize probability */
      if (iDisplay == YES)  mexPrintf("Probability of the reconstructed data: %.4f\n",*pfPu);
    }

    vArrayMemoryFree(&sDataTemp.ppfEVecTemp,d,n);
    vArrayMemoryFree(&sDataTemp.ppfMeanValueTemp,d,1);
    vArrayMemoryFree(&sDataTemp.ppfuTemp,d,1);

    end_time = clock();
    if (iDisplay == YES)  mexPrintf("Computing Time for PCA program: %.5f sec \n   for an image %ix%i and %i principal components\n",difftime(end_time,start_time)/1000,iNy,iNx,ip);

  } /* end of if ( iPCAMode == MODE_PCA_RECONSTRUCTION ) */


  else if( iPCAMode == MODE_PCA_PROJECTION )
  {

    pfEVec = mxGetData(pmxIn[1]);

    pfEVal = mxGetData(pmxIn[2]);

    pfMeanValue = mxGetData(pmxIn[3]);

    pfuNew = mxGetData(pmxIn[4]);

    pip = mxGetData(pmxIn[5]);
    ip = *pip;

    piNx = mxGetData(pmxIn[6]);
    iNx = *piNx;

    piNy = mxGetData(pmxIn[7]);
    iNy = *piNy;

    piDisplay = mxGetData(pmxIn[8]);
    iDisplay = *piDisplay;

    if (iDisplay == YES)  mexPrintf("\nStart PCA program\n");
    if (iDisplay == YES)  mexPrintf("MODE_PCA_PROJECTION\n\n");

    iDim[0] = iNy;
    iDim[1] = iNx;
    iDim[2] = ip;

    d = iNy* iNx; /* number of rows in data matrix */
    n = ip;      /* number of columns in data matrix */

    if (!iComputeProjection(pfEVec,pfEVal,pfMeanValue,pfuNew,iDim,&sDataTemp,iDisplay))
    {
      mexPrintf("Error in PCA: MODE_PCA_PROJECTION\n");
    }
    else
    {
      iNdim = 2;
      iDim2[0] = iNy;
      iDim2[1] = iNx;
      pmxOut[0] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfuProjected = mxGetData(pmxOut[0]);

      for (ix=0; ix< iNx; ix++) 
	for (iy=0; iy< iNy; iy++)
	  pfuProjected[ix*iNy+ iy] = sDataTemp.ppfuTemp[ix*iNy+ iy][0];


      iNdim = 2;
      iDim2[0] = ip;
      iDim2[1] = 1;
      pmxOut[1] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfVcoef = mxGetData(pmxOut[1]);

      for (i=0; i< ip; i++) 
	pfVcoef[i] = sDataTemp.ppfVcoefTemp[i][0];


      iNdim = 2;
      iDim2[0] = 1;
      iDim2[1] = 1;
      pmxOut[2] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
      pfPuProjected = mxGetData(pmxOut[2]);

      /* Computation of the probability that the projected data belongs to the class */
      /* of the training set */
      *pfPuProjected = 0.0;

      fSum = 0.;
      for (i=0; i< ip; i++)
	if ( pfEVal[i] > 0.001 )
	  fSum += pfVcoef[i]* pfVcoef[i]/ pfEVal[i];
      fSum = -fSum/ 2.;
      *pfPuProjected = (float) exp(fSum);
      fSum = 1.;
      for (i=0; i< ip; i++)
        if ( pfEVal[i] > 0.001 )	 
	  fSum *= pfEVal[i];
      fSum = (float) sqrt(fSum);
      fSum /= pow(2*PI,(double)(ip/2.));

    /*  *pfPuProjected /= fSum; */ /* Normalize probability */
      if (iDisplay == YES)  mexPrintf("Probability of the projected data: %.4f\n",*pfPuProjected);

    }

    vArrayMemoryFree(&sDataTemp.ppfEVecTemp,d,n);
    vArrayMemoryFree(&sDataTemp.ppfMeanValueTemp,d,1);
    vArrayMemoryFree(&sDataTemp.ppfuTemp,d,1);
    vArrayMemoryFree(&sDataTemp.ppfVcoefTemp,n,1);

    end_time = clock();
    if (iDisplay == YES)  mexPrintf("Computing Time for PCA program: %.2f sec \n   for an image %ix%i and %i principal components\n\n",difftime(end_time,start_time)/1000000,iNy,iNx,ip);


  } /* end of if ( iPCAMode == MODE_PCA_PROJECTION ) */


  if (iDisplay == YES)  mexPrintf("End PCA program\n\n");

}
/*********************************************************************************************/

/**************************************** End of file ****************************************/
