
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspBoundaryTerm.c
// Description: this function computes the boundary-based term
// See "acsp.c"



#include "acspCommon.h"



// Forward and backward approximations of the first and second spatial derivates
float fDXM ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix][iy] - ppfPhi[ix-1][iy]); }

float fDXP ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix+1][iy] - ppfPhi[ix][iy]); }

float fDYM ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix][iy] - ppfPhi[ix][iy-1]); }

float fDYP ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix][iy+1] - ppfPhi[ix][iy]); }


// Approximations of second spatial partial derivates
float fDXMXP ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix+1][iy] - 2.* ppfPhi[ix][iy] + ppfPhi[ix-1][iy]); }

float fDYMYP ( float **ppfPhi, int ix, int iy )
{ return (ppfPhi[ix][iy+1] - 2.* ppfPhi[ix][iy] + ppfPhi[ix][iy-1]); }


// Compute upwind gradient of level set function
void vComputeGrad
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfTemp
)

{
    
    int ix, iy;
    
    
    for (iy = 0; iy<iNy; iy++)
    {
        for (ix = 1; ix<iNx; ix++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfDxm[ix][iy] = fDXM(sAcsp.ppfPhi,ix,iy);
        if ( sAcsp.ppfDiracPhi[0][iy] > EPS )
            sAcsp.ppfDxm[0][iy] = 0.0;
    }
    
    for (iy = 0; iy<iNy; iy++)
    {
        for (ix = 0; ix<iNx-1; ix++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfDxp[ix][iy] = fDXP(sAcsp.ppfPhi,ix,iy);
        if ( sAcsp.ppfDiracPhi[iNx-1][iy] > EPS )
            sAcsp.ppfDxp[iNx-1][iy] = 0.0;
    }
    
    for (ix = 0; ix<iNx; ix++)
    {
        for (iy = 1; iy<iNy; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfDym[ix][iy] = fDYM(sAcsp.ppfPhi,ix,iy);
        if ( sAcsp.ppfDiracPhi[ix][0] > EPS )
            sAcsp.ppfDym[ix][0] = 0.0;
    }
    
    for (ix = 0; ix<iNx; ix++)
    {
        for (iy = 0; iy<iNy-1; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfDyp[ix][iy] = fDYP(sAcsp.ppfPhi,ix,iy);
        if ( sAcsp.ppfDiracPhi[ix][iNy-1] > EPS )
            sAcsp.ppfDyp[ix][iNy-1] = 0.0;
    }
    
    
    
    // norm of the upwind gradient of the level-set function, |nabla varphi|
    for (ix=0; ix< iNx; ix++)
        for (iy=0; iy< iNy; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
    {
        sAcsp.ppfNormGradPhiPlus[ix][iy] = sqrt( SQR( MAX(-MIN(sAcsp.ppfDxp[ix][iy],0.0), MAX(sAcsp.ppfDxm[ix][iy],0.0)) ) +
        SQR( MAX(-MIN(sAcsp.ppfDyp[ix][iy],0.0), MAX(sAcsp.ppfDym[ix][iy],0.0)) ) );
        sAcsp.ppfNormGradPhiMinus[ix][iy] = sqrt( SQR( MAX(MAX(sAcsp.ppfDxp[ix][iy],0.0), -MIN(sAcsp.ppfDxm[ix][iy],0.0)) ) +
        SQR( MAX(MAX(sAcsp.ppfDyp[ix][iy],0.0), -MIN(sAcsp.ppfDym[ix][iy],0.0)) ) );
            }
    
}



// compute curvature of level set function, kappa*g*dirac(varphi)
void vComputeCurvatureTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfEdgeDetector,
float  *pfVecBoundaryTerm,
int    iDisplay,
float  *pfTemp
)

{
    int    ix, iy;
    float  fDx, fDy, fDxx, fDyy, fDxy, fCurvature, fNorm, fMaxCurvature;
    float  fMaxCurvatureTerm, fWeightCurvatureTerm, fWeightBoundaryTerm;
    
    
    fMaxCurvature = 2.0;
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0];
    fWeightCurvatureTerm = pfVecBoundaryTerm[1];
    
    if ( fWeightCurvatureTerm> 0.0 )
    {
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
        {
            
            fDx = (sAcsp.ppfPhi[ix+1][iy] - sAcsp.ppfPhi[ix-1][iy])/ 2;
            fDy = (sAcsp.ppfPhi[ix][iy+1] - sAcsp.ppfPhi[ix][iy-1])/ 2;
            fDxx = sAcsp.ppfPhi[ix+1][iy] - 2.* sAcsp.ppfPhi[ix][iy] + sAcsp.ppfPhi[ix-1][iy];
            fDyy = sAcsp.ppfPhi[ix][iy+1] - 2.* sAcsp.ppfPhi[ix][iy] + sAcsp.ppfPhi[ix][iy-1];
            fDxy = (sAcsp.ppfPhi[ix+1][iy+1] + sAcsp.ppfPhi[ix-1][iy-1] - sAcsp.ppfPhi[ix+1][iy-1] - sAcsp.ppfPhi[ix-1][iy+1])/ 4;
            fNorm = sqrt( fDx*fDx+ fDy*fDy );
            if ( ABS(fNorm)> EPS )
                fCurvature = (fDxx* fDy* fDy - 2.* fDx* fDy* fDxy + fDyy* fDx* fDx)/ pow(fNorm, 3.0);
            else
                fCurvature = 0.0;
            
            sAcsp.ppfCurvatureACTerm[ix][iy] = fCurvature;
            
            if ( ABS(sAcsp.ppfCurvatureACTerm[ix][iy]) > fMaxCurvature )
                sAcsp.ppfCurvatureACTerm[ix][iy] = SIGN(sAcsp.ppfCurvatureACTerm[ix][iy])* fMaxCurvature;
            
            sAcsp.ppfCurvatureACTerm[ix][iy] *= sAcsp.ppfDiracPhi[ix][iy]* pfEdgeDetector[ix*iNy+ iy];
            
                }
        
        
        
        if (iDisplay == YES)
        {
            fMaxCurvatureTerm = 0.0;
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
                    if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                        if (ABS(sAcsp.ppfCurvatureACTerm[ix][iy])>ABS(fMaxCurvatureTerm))
                            fMaxCurvatureTerm = sAcsp.ppfCurvatureACTerm[ix][iy];
            mexPrintf("MaxCurvatureTerm*Weight (BoundaryTerm)= %.3f\n",fWeightBoundaryTerm*fWeightCurvatureTerm*fMaxCurvatureTerm);
        }
    }
    
}




// compute attraction term, <nabla g,nabla varphi/|nabla varphi|>*dirac(varphi)
void vComputeAttractionValleyTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfEdgeDetector,
float  *pfVecBoundaryTerm,
int    iDisplay,
float  *pfTemp
)
{
    int    ix, iy;
    float  fDxBoundary, fDyBoundary, fDxPlusAcsp, fDxMinusAcsp;
    float  fDyPlusAcsp, fDyMinusAcsp, fMaxAttractionValleyTerm, fWeightAttractionValleyTerm, fWeightBoundaryTerm;
    
    
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0];
    fWeightAttractionValleyTerm = pfVecBoundaryTerm[2];
    
    
    if ( fWeightAttractionValleyTerm> 0.0 )
    {
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
        {
            
            fDxBoundary = (pfEdgeDetector[(ix+1)*iNy+ iy]- pfEdgeDetector[(ix-1)*iNy+ iy])/ 2.;
            fDyBoundary = (pfEdgeDetector[ix*iNy+ iy+1]- pfEdgeDetector[ix*iNy+ iy-1])/ 2.;
            
            fDxPlusAcsp = sAcsp.ppfDxp[ix][iy];
            fDxMinusAcsp = sAcsp.ppfDxm[ix][iy];
            
            fDyPlusAcsp = sAcsp.ppfDyp[ix][iy];
            fDyMinusAcsp = sAcsp.ppfDym[ix][iy];
            
            sAcsp.ppfAttractionValleyACTerm[ix][iy] = ( MAX(fDxBoundary,0.0)* fDxPlusAcsp +  MIN(fDxBoundary,0.0)* fDxMinusAcsp ) + 
            ( MAX(fDyBoundary,0.0)* fDyPlusAcsp +  MIN(fDyBoundary,0.0)* fDyMinusAcsp );
            
            sAcsp.ppfAttractionValleyACTerm[ix][iy] *= sAcsp.ppfDiracPhi[ix][iy]/ sAcsp.ppfNormGradPhiPlus[ix][iy];
            
                }
        
        
        if (iDisplay == YES)
        {
            fMaxAttractionValleyTerm = 0.0;
            for (ix=1; ix< iNx-1; ix++)
                for (iy=1; iy< iNy-1; iy++)
                    if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                        if ( ABS(sAcsp.ppfAttractionValleyACTerm[ix][iy])> ABS(fMaxAttractionValleyTerm) )
                            fMaxAttractionValleyTerm = sAcsp.ppfAttractionValleyACTerm[ix][iy];
            if (iDisplay == YES)  mexPrintf("MaxAttractionValleyTerm*Weight (BoundaryTerm)= %.3f\n",fWeightBoundaryTerm*fWeightAttractionValleyTerm*fMaxAttractionValleyTerm);
        }
    }
    
}



// compute boundary-based term
void vComputeBoundaryTerm
(
int    iNx,
int    iNy,
struct sAcspTag sAcsp,
float  *pfEdgeDetector,
float  *pfVecBoundaryTerm,
int    iDisplay,
float  *pfTemp
)
{
    int    ix, iy;
    float  fMaxBoundaryTerm, fWeightCurvatureTerm;
    float  fWeightAttractionValleyTerm, fWeightBoundaryTerm;
    
    
    fWeightBoundaryTerm = pfVecBoundaryTerm[0];
    fWeightCurvatureTerm = pfVecBoundaryTerm[1];
    fWeightAttractionValleyTerm = pfVecBoundaryTerm[2];
    
    
    for (ix=0; ix< iNx; ix++)
        for (iy=0; iy< iNy; iy++)
            if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                sAcsp.ppfBoundaryTerm[ix][iy] = fWeightCurvatureTerm* sAcsp.ppfCurvatureACTerm[ix][iy]+ 
                fWeightAttractionValleyTerm* sAcsp.ppfAttractionValleyACTerm[ix][iy];
    
    
    if (iDisplay == YES)
    {
        fMaxBoundaryTerm = 0.0;
        for (ix=1; ix< iNx-1; ix++)
            for (iy=1; iy< iNy-1; iy++)
                if ( sAcsp.ppfDiracPhi[ix][iy] > EPS )
                    if ( ABS(sAcsp.ppfBoundaryTerm[ix][iy])> ABS(fMaxBoundaryTerm) )
                        fMaxBoundaryTerm = sAcsp.ppfBoundaryTerm[ix][iy];
        mexPrintf("MaxBoundaryTerm*Weight= %.3f\n",fWeightBoundaryTerm*fMaxBoundaryTerm);
    }
    
}



/****************** End of file *******************/
