
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspBsplines.c
// Description: This function performs spatial transformations based on bsplines, see 
// [M. Unser, "Splines: A Perfect Fit for Signal and Image Processing," 
// IEEE Signal Processing Magazine, vol. 16(6), pp. 22-38, 1999]




#include "acspCommon.h" 



/*****************************************************************************
 *	Declaration of static procedures
 ****************************************************************************/


static int SamplesToCoefficients
(
 float	*Image,		            /* in-place processing */
 int	Width,		            /* width of the image */
 int	Height,		            /* height of the image */
 int	SplineDegree                /* degree of the spline model */
 );

/*--------------------------------------------------------------------------*/
static void ConvertToInterpolationCoefficients
(
  float c[],		/* input samples --> output coefficients */
  int	 DataLength,	/* number of samples or coefficients */
  float z[],		/* poles */
  int	 NbPoles,	/* number of poles */
  float Tolerance	/* admissible relative error */
);

/*--------------------------------------------------------------------------*/
static void GetColumn
(
  float *Image,	/* input image array */
  int	Width,		/* width of the image */
  int	x,		/* x coordinate of the selected line */
  float Line[],	/* output linear array */
  int	Height		/* length of the line and height of the image */
);

/*--------------------------------------------------------------------------*/
static void GetRow
(
  float  *Image,	 /* input image array */
  int	  y,		 /* y coordinate of the selected line */
  float  Line[],	 /* output linear array */
  int	  Width,       	 /* length of the line and width of the image */
  int	  Height
);

/*--------------------------------------------------------------------------*/
static float InitialCausalCoefficientMirror
(
  float  c[],		/* coefficients */
  int	  DataLength,	/* number of coefficients */
  float  z,		/* actual pole */
  float  Tolerance	/* admissible relative error */
);

/*--------------------------------------------------------------------------*/
static float InitialAntiCausalCoefficientMirror
(
  float  c[],		/* coefficients */
  int	  DataLength,	/* number of samples or coefficients */
  float  z	       	/* actual pole */
);

/*--------------------------------------------------------------------------*/
static void PutColumn
(
  float  *Image,	/* output image array */
  int	 Width,		/* width of the image */
  int	 x,		/* x coordinate of the selected line */
  float Line[],	/* input linear array */
  int	 Height       	/* length of the line and height of the image */
);

/*--------------------------------------------------------------------------*/
static void PutRow
(
  float *Image,	/* output image array */
  int	 y,    		/* y coordinate of the selected line */
  float Line[],       	/* input linear array */
  int	 Width,		/* length of the line and width of the image */
  int	 Height
);







static float InterpolatedValue
(
 float	*Bcoeff,                     /* input B-spline array of coefficients */
 int	Width,                       /* width of the image */
 int	Height,	                     /* height of the image */
 float	x,	                     /* x coordinate where to interpolate */
 float	y,                           /* y coordinate where to interpolate */
 int	SplineDegree,                /* degree of the spline model */
 float  fValueNotInterpolated
);






/*****************************************************************************
 *	Definition of static procedures
 ****************************************************************************/
/*--------------------------------------------------------------------------*/
static void ConvertToInterpolationCoefficients
(
  float  c[],		/* input samples --> output coefficients */
  int	  DataLength,	/* number of samples or coefficients */
  float  z[],		/* poles */
  int	  NbPoles,	/* number of poles */
  float  Tolerance	/* admissible relative error */
)

{ /* begin ConvertToInterpolationCoefficients */
	
  float Lambda = 1.0;
  int	n, k;
	
  /* special case required by mirror boundaries */
  if (DataLength == 1)
    return;
  
  /* compute the overall gain */
  for (k = 0L; k < NbPoles; k++)
    Lambda = Lambda * (1.0 - z[k]) * (1.0 - 1.0 / z[k]);
  
  /* apply the gain */
  for (n = 0L; n < DataLength; n++)
    c[n] *= Lambda;

  /* loop over all poles */
  for (k = 0L; k < NbPoles; k++) {
 
  /* causal initialization with mirror boundaries for images */
  c[0] = InitialCausalCoefficientMirror(c, DataLength, z[k], Tolerance);
  /* causal recursion */
  for (n = 1; n < DataLength; n++)
    c[n] += z[k] * c[n - 1];	
  /* anticausal initialization with mirror boundaries for images */
  c[DataLength - 1] = InitialAntiCausalCoefficientMirror(c, DataLength, z[k]);
  /* anticausal recursion */
  for (n = DataLength - 2; 0 <= n; n--)
    c[n] = z[k] * (c[n + 1] - c[n]);
  }
 
} /* end ConvertToInterpolationCoefficients */

/*--------------------------------------------------------------------------*/
static float	InitialCausalCoefficientMirror
(
  float  c[],		/* coefficients */
  int	  DataLength,	/* number of coefficients */
  float  z,			/* actual pole */
  float  Tolerance	/* admissible relative error */
)
	
{ /* begin InitialCausalCoefficient */

  float  Sum, zn, z2n, iz;
  int	  n, Horizon;
	
  /* this initialization corresponds to mirror boundaries */
  Horizon = DataLength;
  if (Tolerance > 0.0)
     Horizon = (int)ceil(log(Tolerance) / log(fabs(z)));
  if (Horizon < DataLength) {
     /* accelerated loop */
     zn = z;
     Sum = c[0];
     for (n = 1; n < Horizon; n++) {
	     Sum += zn * c[n];
	     zn *= z;
     }
     return(Sum);
  }
  else {
     /* full loop */
     zn = z;
     iz = 1.0 / z;
     z2n = pow(z, (double)(DataLength - 1));
     Sum = c[0] + z2n * c[DataLength - 1];
     z2n *= z2n * iz;
     for (n = 1; n <= DataLength - 2; n++) {
	     Sum += (zn + z2n) * c[n];
	     zn *= z;
	     z2n *= iz;
     }
     return(Sum / (1.0 - zn * zn));
  }
} /* end InitialCausalCoefficient */

/*--------------------------------------------------------------------------*/
static void		GetColumn
(
  float *Image,	/* input image array */
  int	 Width,		/* width of the image */
  int	 x,		/* x coordinate of the selected line */
  float Line[],	/* output linear array */
  int	 Height       	/* length of the line */
)
	
{ /* begin GetColumn */
 
  int	y; 
  Image = Image + (ptrdiff_t)(x * Height);
  for (y = 0L; y < Height; y++)
    Line[y] = *Image++;

} /* end GetColumn */

/*--------------------------------------------------------------------------*/
static void		GetRow
(
  float  *Image,		/* input image array */
  int	  y,			/* y coordinate of the selected line */
  float  Line[],		/* output linear array */
  int	  Width,		/* length of the line */
  int	  Height
)
	
{ /* begin GetRow */

  int  x;

  Image = Image + (ptrdiff_t)y; 
  for (x = 0L; x < Width; x++) { 
    Line[x] = *Image; 
    Image += (ptrdiff_t)Height; 
  }
 

} /* end GetRow */

/*--------------------------------------------------------------------------*/
static float	InitialAntiCausalCoefficientMirror
(
  float  c[],		/* coefficients */
  int    DataLength,	/* number of samples or coefficients */
  float  z		/* actual pole */
	)
	
{ /* begin InitialAntiCausalCoefficient */
  /* this initialization corresponds to mirror boundaries */
	return((z / (z * z - 1.0)) * (z * c[DataLength - 2] + c[DataLength - 1]));
} /* end InitialAntiCausalCoefficient */

/*--------------------------------------------------------------------------*/
static void		PutColumn
(
  float  *Image,     	/* output image array */
  int	  Width,      	/* width of the image */
  int	  x,	      	/* x coordinate of the selected line */
  float  Line[],      	/* input linear array */
  int	  Height      	/* length of the line and height of the image */
)
	
{ /* begin PutColumn */

  int	y;
  Image = Image + (ptrdiff_t)(x * Height);
  for (y = 0L; y < Height; y++)
    *Image++ = Line[y];
 
} /* end PutColumn */

/*--------------------------------------------------------------------------*/
static void		PutRow
(
  float  *Image,      	/* output image array */
  int    y,	       	/* y coordinate of the selected line */
  float  Line[],      	/* input linear array */
  int	  Width,       	/* length of the line and width of the image */
  int	  Height
)
	
{ /* begin PutRow */

int	x;

  Image = Image + (ptrdiff_t)y;
  for (x = 0L; x < Width; x++) {
    *Image = Line[x];
    Image += (ptrdiff_t)Height;
  }

} /* end PutRow */



/*--------------------------------------------------------------------------*/
static int		SamplesToCoefficients
(
  float *Image,       	/* in-place processing */
  int	 Width,		/* width of the image */
  int	 Height,       	/* height of the image */
  int	 SplineDegree   /* degree of the spline model */
)

{ /* begin SamplesToCoefficients */

  float  *Line, *Col;
  float  Pole[2];
  int	  NbPoles;
  int	  x, y;

  /* recover the poles from a lookup table */
  switch (SplineDegree) {
  case 2:
    NbPoles = 1;
    Pole[0] = sqrt(8.0) - 3.0;
    break;
  case 3:
    NbPoles = 1;
    Pole[0] = sqrt(3.0) - 2.0;
    break;
  case 4:
    NbPoles = 2;
    Pole[0] = sqrt(664.0 - sqrt(438976.0)) + sqrt(304.0) - 19.0;
    Pole[1] = sqrt(664.0 + sqrt(438976.0)) - sqrt(304.0) - 19.0;
    break;
  case 5:
    NbPoles = 2;
    Pole[0] = sqrt(135.0 / 2.0 - sqrt(17745.0 / 4.0)) + sqrt(105.0 / 4.0) - 13.0 / 2.0;
    Pole[1] = sqrt(135.0 / 2.0 + sqrt(17745.0 / 4.0)) - sqrt(105.0 / 4.0) - 13.0 / 2.0;
    break;
  default:
    mexPrintf("\nInvalid spline degree");
    return(0);
  }

  /* convert the image samples into B-spline coefficients */
  /* in-place separable process, along x */
  Line = (float *) calloc(Width, sizeof(float));
  if (Line == (float *)NULL) {
    mexPrintf("\nRow allocation failed 1\n");
    return(0);
  }

  for (y = 0L; y < Height; y++) {
    GetRow(Image, y, Line, Width, Height);
    ConvertToInterpolationCoefficients(Line, Width, Pole, NbPoles, DBL_EPSILON2);
    PutRow(Image, y, Line, Width, Height);

 /*   mexPrintf("\nGetRow nb=%i",y); 
     for (i=0L; i<Width; i++)   
      mexPrintf("\nLine[%i] = %f",i,Line[i]); */
  }
  
  free((float *) Line);

  /* in-place separable process, along y */
  Col = (float *) calloc(Height, sizeof(float));
  if (Col == (float *)NULL) {
    mexPrintf("\nColumn allocation failed 1\n");
    return(0);
  }

  for (x = 0L; x < Width; x++) {
    GetColumn(Image, Width, x, Col, Height);
    ConvertToInterpolationCoefficients(Col, Height, Pole, NbPoles, DBL_EPSILON2);
    PutColumn(Image, Width, x, Col, Height);
  }

  free((float *) Col);

  return(1);


} /* end SamplesToCoefficients */








/*--------------------------------------------------------------------------*/
static float	InterpolatedValue
(
  float  *Bcoeff,	             /* input B-spline array of coefficients */
  int	  Width,       	             /* width of the image */
  int	  Height,      	             /* height of the image */
  float   x,	       	             /* x coordinate where to interpolate */
  float   y,	      	             /* y coordinate where to interpolate */
  int	  SplineDegree,              /* degree of the spline model */
  float   fValueNotInterpolated
)
	
{ /* begin InterpolatedValue */

  float  *p;
  float  xWeight[6], yWeight[6];
  float  interpolated;
  float  w, w2, w4, t, t0, t1;
  int	  xIndex[6], yIndex[6];
  int	  Width2, Height2;
  int	  i, j, k;


/*   mexPrintf("x= %f, y= %f\n",x,y); */

  Width2 = 2L * Width - 2L;
  Height2 = 2L * Height - 2L;

  /* compute the interpolation indexes */
  if (SplineDegree & 1L) {
     i = (int)floor(x) - SplineDegree / 2L;
     j = (int)floor(y) - SplineDegree / 2L;
     for (k = 0L; k <= SplineDegree; k++) {
       xIndex[k] = i++;
       yIndex[k] = j++;
     }
  }
  else {
     i = (int)floor(x + 0.5) - SplineDegree / 2L;
     j = (int)floor(y + 0.5) - SplineDegree / 2L;
     for (k = 0L; k <= SplineDegree; k++) {
       xIndex[k] = i++;
       yIndex[k] = j++;
     }
  }
  
  /* compute the values of B-splines */
  switch (SplineDegree) {
  case 2L:
    /* x */
    w = x - (float)xIndex[1];
    xWeight[1] = 3.0 / 4.0 - w * w;
    xWeight[2] = (1.0 / 2.0) * (w - xWeight[1] + 1.0);
    xWeight[0] = 1.0 - xWeight[1] - xWeight[2];
    /* y */
    w = y - (float)yIndex[1];
    yWeight[1] = 3.0 / 4.0 - w * w;
    yWeight[2] = (1.0 / 2.0) * (w - yWeight[1] + 1.0);
    yWeight[0] = 1.0 - yWeight[1] - yWeight[2];
    break;
  case 3L:
    /* x */
    w = x - (float)xIndex[1];
    xWeight[3] = (1.0 / 6.0) * w * w * w;
    xWeight[0] = (1.0 / 6.0) + (1.0 / 2.0) * w * (w - 1.0) - xWeight[3];
    xWeight[2] = w + xWeight[0] - 2.0 * xWeight[3];
    xWeight[1] = 1.0 - xWeight[0] - xWeight[2] - xWeight[3];
    /* y */
    w = y - (float)yIndex[1];
    yWeight[3] = (1.0 / 6.0) * w * w * w;
    yWeight[0] = (1.0 / 6.0) + (1.0 / 2.0) * w * (w - 1.0) - yWeight[3];
    yWeight[2] = w + yWeight[0] - 2.0 * yWeight[3];
    yWeight[1] = 1.0 - yWeight[0] - yWeight[2] - yWeight[3];
    break;
  case 4L:
    /* x */
    w = x - (float)xIndex[2];
    w2 = w * w;
    t = (1.0 / 6.0) * w2;
    xWeight[0] = 1.0 / 2.0 - w;
    xWeight[0] *= xWeight[0];
    xWeight[0] *= (1.0 / 24.0) * xWeight[0];
    t0 = w * (t - 11.0 / 24.0);
    t1 = 19.0 / 96.0 + w2 * (1.0 / 4.0 - t);
    xWeight[1] = t1 + t0;
    xWeight[3] = t1 - t0;
    xWeight[4] = xWeight[0] + t0 + (1.0 / 2.0) * w;
    xWeight[2] = 1.0 - xWeight[0] - xWeight[1] - xWeight[3] - xWeight[4];
    /* y */
    w = y - (float)yIndex[2];
    w2 = w * w;
    t = (1.0 / 6.0) * w2;
    yWeight[0] = 1.0 / 2.0 - w;
    yWeight[0] *= yWeight[0];
    yWeight[0] *= (1.0 / 24.0) * yWeight[0];
    t0 = w * (t - 11.0 / 24.0);
    t1 = 19.0 / 96.0 + w2 * (1.0 / 4.0 - t);
    yWeight[1] = t1 + t0;
    yWeight[3] = t1 - t0;
    yWeight[4] = yWeight[0] + t0 + (1.0 / 2.0) * w;
    yWeight[2] = 1.0 - yWeight[0] - yWeight[1] - yWeight[3] - yWeight[4];
    break;
  case 5L:
    /* x */
    w = x - (float)xIndex[2];
    w2 = w * w;
    xWeight[5] = (1.0 / 120.0) * w * w2 * w2;
    w2 -= w;
    w4 = w2 * w2;
    w -= 1.0 / 2.0;
    t = w2 * (w2 - 3.0);
    xWeight[0] = (1.0 / 24.0) * (1.0 / 5.0 + w2 + w4) - xWeight[5];
    t0 = (1.0 / 24.0) * (w2 * (w2 - 5.0) + 46.0 / 5.0);
    t1 = (-1.0 / 12.0) * w * (t + 4.0);
    xWeight[2] = t0 + t1;
    xWeight[3] = t0 - t1;
    t0 = (1.0 / 16.0) * (9.0 / 5.0 - t);
    t1 = (1.0 / 24.0) * w * (w4 - w2 - 5.0);
    xWeight[1] = t0 + t1;
    xWeight[4] = t0 - t1;
    /* y */
    w = y - (float)yIndex[2];
    w2 = w * w;
    yWeight[5] = (1.0 / 120.0) * w * w2 * w2;
    w2 -= w;
    w4 = w2 * w2;
    w -= 1.0 / 2.0;
    t = w2 * (w2 - 3.0);
    yWeight[0] = (1.0 / 24.0) * (1.0 / 5.0 + w2 + w4) - yWeight[5];
    t0 = (1.0 / 24.0) * (w2 * (w2 - 5.0) + 46.0 / 5.0);
    t1 = (-1.0 / 12.0) * w * (t + 4.0);
    yWeight[2] = t0 + t1;
    yWeight[3] = t0 - t1;
    t0 = (1.0 / 16.0) * (9.0 / 5.0 - t);
    t1 = (1.0 / 24.0) * w * (w4 - w2 - 5.0);
    yWeight[1] = t0 + t1;
    yWeight[4] = t0 - t1;
    break;
  default:
    mexPrintf("Invalid spline degree in bspline_interpol 2\n");
    return(0.0);

  }

  /* apply the mirror boundary conditions */
 /*  for (k = 0L; k <= SplineDegree; k++) { */
/*      xIndex[k] = (Width == 1L) ? (0L) : ((xIndex[k] < 0L) ? (-xIndex[k] - Width2 * ((-xIndex[k]) / Width2)) : (xIndex[k] - Width2 * (xIndex[k] / Width2))); */
/*      if (Width <= xIndex[k]) { */
/* 	     xIndex[k] = Width2 - xIndex[k]; */
/*      } */
/*      yIndex[k] = (Height == 1L) ? (0L) : ((yIndex[k] < 0L) ? (-yIndex[k] - Height2 * ((-yIndex[k]) / Height2)) : (yIndex[k] - Height2 * (yIndex[k] / Height2))); */
/*      if (Height <= yIndex[k]) { */
/* 	     yIndex[k] = Height2 - yIndex[k]; */
/*      } */
/*   } */

 
  /* no boundary conditions */
/*   for (k = 0L; k <= SplineDegree; k++)  */
/*     { */

/*       if ( (Width == 1) || (xIndex[k] < 0) || (xIndex[k] >= Width) ) */
/* 	return(0.0); */

/*       if ( (Height == 1) || (yIndex[k] < 0) || (yIndex[k] >= Height) ) */
/* 	return(0.0); */
/*     } */
 

  /* boundary conditions */
 /*  for (k = 0L; k <= SplineDegree; k++)  */
/*     { */

/*       if ( (Width == 1) || (xIndex[k] < 0) || (xIndex[k] >= Width) ) */
/* 	return( sqrt(Width*Width + Height*Height) ); */

/*       if ( (Height == 1) || (yIndex[k] < 0) || (yIndex[k] >= Height) ) */
/* 	return( sqrt(Width*Width + Height*Height) ); */
/*     } */

  for (k = 0L; k <= SplineDegree; k++) 
    {

      if ( (Width == 1) || (xIndex[k] < 0) || (xIndex[k] >= Width) )
	return(fValueNotInterpolated);

      if ( (Height == 1) || (yIndex[k] < 0) || (yIndex[k] >= Height) )
	return(fValueNotInterpolated);
    }


 /* perform interpolation */
  interpolated = 0.0; 
  for (j = 0L; j <= SplineDegree; j++) { 
    p = Bcoeff + (ptrdiff_t)(xIndex[j] * Height); 
    w = 0.0; 
    for (i = 0L; i <= SplineDegree; i++)
      w += yWeight[i] * p[yIndex[i]];
    interpolated += xWeight[j] * w;
  }

  return(interpolated);


} /* end InterpolatedValue */



















/*****************************************************************************
 *	Definition of extern procedures
 ****************************************************************************/


/* main function */
extern int iBSplineImage
(
 float   **ppfImage,
 float   **ppfDiracPhi,
 int     iDim[],
 int     iSplineDegree,
 float   *pfVectorSpatialTransformation,
 int     iTypeSpatialTransformation,
 float   fValueNotInterpolated,
 int     iDisplay
 )

{

  int    iNy, iNx, ix, iy, i;
  float  *pfVcoeff, *pfCoord;
  float  fx, fy, fx0, fy0;
  float  fxOrigin, fyOrigin, fScale, fShearing, fAngle, fTx, fTy;
  float  fR11, fR12, fR21, fR22, fA11, fA12, fA21, fA22;

  iNy = iDim[0];
  iNx = iDim[1];
/*   mexPrintf("BSplines  iDim[0] = %i, iDim[1] = %i\n",iDim[0],iDim[1]); */

  fxOrigin = pfVectorSpatialTransformation[0];
  fyOrigin = pfVectorSpatialTransformation[1];
  fScale = pfVectorSpatialTransformation[2];
  fAngle = pfVectorSpatialTransformation[3];
  fTx = pfVectorSpatialTransformation[4];
  fTy = pfVectorSpatialTransformation[5];
  if ( iTypeSpatialTransformation == AFFINE )
    fShearing = pfVectorSpatialTransformation[6];

  pfVcoeff = (float *) calloc(iNy* iNx, sizeof(float));
  if ( !pfVcoeff )
    {
      mexPrintf("Memory allocation failure 3 in bspline_images.c\n"); 
      return(0);
    }

  for (ix=0; ix<iNx; ix++)
    for (iy=0; iy<iNy; iy++) 
      {
	i = ix*iNy+ iy;
	pfVcoeff[i] = ppfImage[ix][iy];
      }

  // Compute B-spline coefficients from image samples 
  if (!SamplesToCoefficients(pfVcoeff,iNx,iNy,iSplineDegree) )
    {
      mexPrintf("Memory allocation failure 4 in bspline_images.c\n"); 
      return(0);
    }

  pfCoord = (float *) calloc(2* iNy* iNx, sizeof(float));
  if ( !pfCoord )
    {
      mexPrintf("Memory allocation failure 5 in bspline_images.c\n"); 
      return(0);
    }


  if ( iTypeSpatialTransformation == RIGID )
    {
      fR11 = fScale* cos(fAngle);
      fR12 = fScale* sin(fAngle);
      fR21 = -fScale* sin(fAngle);
      fR22 = fScale* cos(fAngle);

      fx0 = fR11* fxOrigin + fR12* fyOrigin;
      fy0 = fR21* fxOrigin + fR22* fyOrigin;
    }
  else if ( iTypeSpatialTransformation == AFFINE )
    {
      fA11 = fScale* cos(fAngle);
      fA12 = fScale* fShearing* cos(fAngle)+ fScale* sin(fAngle);
      fA21 = -fScale* sin(fAngle);
      fA22 = -fScale* fShearing* sin(fAngle)+ fScale* cos(fAngle);

      fx0 = fA11* fxOrigin + fA12* fyOrigin;
      fy0 = fA21* fxOrigin + fA22* fyOrigin;
    }

  fTx = fxOrigin + fTx - fx0;
  fTy = fyOrigin + fTy - fy0;


  if (ppfDiracPhi!=NULL)
    {
 
      i = 0;
      for (ix=0; ix<iNx; ix++)
	for (iy=0; iy<iNy; iy++)
	  if ( ppfDiracPhi[ix][iy] > EPS )
	    {
	      fx = (float)(ix);
	      fy = (float)(iy);
	      if ( iTypeSpatialTransformation == RIGID )
		{
		  pfCoord[i] = fR11* fx + fR12* fy + fTx;  i++;
		  pfCoord[i] = fR21* fx + fR22* fy + fTy;  i++;
		}
	      else if ( iTypeSpatialTransformation == AFFINE )
		{
		  pfCoord[i] = fA11* fx + fA12* fy + fTx;  i++;
		  pfCoord[i] = fA21* fx + fA22* fy + fTy;  i++;
		}
	    }
	  else
	    i+=2;

      
      // Interpolate
      for (ix=0; ix<iNx; ix++)
	for (iy=0; iy<iNy; iy++)
	  if ( ppfDiracPhi[ix][iy] > EPS )
	    {
	      i = 2*( ix*iNy+ iy);
	      ppfImage[ix][iy] = InterpolatedValue(pfVcoeff,iNx,iNy,pfCoord[i],pfCoord[i+1],iSplineDegree,fValueNotInterpolated);
	    }
	  else
	    ppfImage[ix][iy] = 0.0;

    }
  else
    {

      i = 0;
      for (ix=0; ix<iNx; ix++)
	for (iy=0; iy<iNy; iy++)
	  {
	    fx = (float)(ix);
	    fy = (float)(iy);
	    if ( iTypeSpatialTransformation == RIGID )
	      {
		pfCoord[i] = fR11* fx + fR12* fy + fTx;  i++;
		pfCoord[i] = fR21* fx + fR22* fy + fTy;  i++;
	      }
	    else if ( iTypeSpatialTransformation == AFFINE )
	      {
		pfCoord[i] = fA11* fx + fA12* fy + fTx;  i++;
		pfCoord[i] = fA21* fx + fA22* fy + fTy;  i++;
	      }
	  }


      // Interpolate
      for (ix=0; ix<iNx; ix++)
	for (iy=0; iy<iNy; iy++)
	  {
	    i = 2*( ix*iNy+ iy);
	    ppfImage[ix][iy] = InterpolatedValue(pfVcoeff,iNx,iNy,pfCoord[i],pfCoord[i+1],iSplineDegree,fValueNotInterpolated);
	  }

    }

  free((float *) pfVcoeff);
  free((float *) pfCoord);

  return(1);
  
}
