
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acspExtensionU.c
// Description: This function extends functions u_in, u_out, |nabla u_in|^2, |nabla u_out|^2
// in the normal direction of boundary using the Fast Marching Method, see
// [D. Adalsteinsson, D. and J.A. Sethian, The Fast Construction of Extension Velocities in Level Set Methods,
// Journal of Computational Physics, vol. 148, 2-22, 1999.]



#include "acspCommon.h"



/****************************************/
/**  Declaration of structures         **/
/****************************************/
typedef struct{
  float *pfPhiValue;
  int   *pix;
  int   *piy;
  int   *piFlagNarrowBand;
  int   iLength;
} sNarrowBand;
/****************************************/



/****************************************/
static void vAddElement2NarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
)

{
  int    current, parent, iTmp;
  float  fTmp;

  psNarBand->pfPhiValue[psNarBand->iLength] = fPhiValue;
  psNarBand->pix[psNarBand->iLength] = ix;
  psNarBand->piy[psNarBand->iLength] = iy;

  current = psNarBand->iLength;
  psNarBand->iLength++;

  parent = (current - 1)/2;
  while (current != 0)
    {
      if (ABS(psNarBand->pfPhiValue[current]) < ABS(psNarBand->pfPhiValue[parent]))
        {
          SWAP(psNarBand->pfPhiValue[current],psNarBand->pfPhiValue[parent],fTmp);
	  SWAP(psNarBand->pix[current],psNarBand->pix[parent],iTmp);
	  SWAP(psNarBand->piy[current],psNarBand->piy[parent],iTmp);
          current = parent;
          parent = (current-1)/2;
        }
      else
        break;
    }

}
/****************************************/


/****************************************/
static void vRemoveFirstElementOfNarrowBand
(
 sNarrowBand  *psNarBand
)

{
  int    best, current, c1, c2, iTmp;
  float  fTmp;


  psNarBand->pfPhiValue[0] = psNarBand->pfPhiValue[psNarBand->iLength-1];
  psNarBand->pix[0] = psNarBand->pix[psNarBand->iLength-1];
  psNarBand->piy[0] = psNarBand->piy[psNarBand->iLength-1];
  psNarBand->iLength--;
 
  current = 0;
  c1 = 1;
  c2 = 2;
  while (c1 < psNarBand->iLength)
    {
      if (c2 >= psNarBand->iLength)
	best = c1;
      else
	{
	  if (ABS(psNarBand->pfPhiValue[c1]) < ABS(psNarBand->pfPhiValue[c2]))
	    best = c1;
	  else
	    best = c2;
	}
      if (ABS(psNarBand->pfPhiValue[best]) < ABS(psNarBand->pfPhiValue[current]))
	{
          SWAP(psNarBand->pfPhiValue[best],psNarBand->pfPhiValue[current],fTmp);
	  SWAP(psNarBand->pix[best],psNarBand->pix[current],iTmp);
	  SWAP(psNarBand->piy[best],psNarBand->piy[current],iTmp);
	  current = best;
	  c1 = 2*current + 1;
	  c2 = c1 + 1;
	}
      else
	break;
    }
  
}
/****************************************/


/****************************************/
static void vUpdatePixelFromNarrowBand
(
 sNarrowBand  *psNarBand,
 float        fPhiValue,
 int          ix,
 int          iy
)

{

  int    current, parent, iTmp, i;
  float  fTmp;


  for(i=0; i< psNarBand->iLength; i++)
    if ( (psNarBand->pix[i] == ix) && (psNarBand->piy[i] == iy) )
      current = i;

  psNarBand->pfPhiValue[current] = fPhiValue;

  parent = (current - 1)/2;
  while (current != 0)
    {
      if (ABS(psNarBand->pfPhiValue[current]) < ABS(psNarBand->pfPhiValue[parent]))
        {
          SWAP(psNarBand->pfPhiValue[current],psNarBand->pfPhiValue[parent],fTmp);
	  SWAP(psNarBand->pix[current],psNarBand->pix[parent],iTmp);
	  SWAP(psNarBand->piy[current],psNarBand->piy[parent],iTmp);
          current = parent;
          parent = (current-1)/2;
        }
      else
        break;
    }

}
/****************************************/

















// MAIN FUNCTION
extern int iExtensionFMDataUi
(
 int    iDim[2],
 float  **ppfImToExtend,
 float  *pfVecGeneralParameters,
 float  *pfTemp
 )

{
  float    fMaxDistExtension, *pfDM, *pfImExtended, *pfImToExtend;
  float    fPhi1, fF1, fPhi2, fF2;
  float    fThreshold, fx1;
  int      iNy, iNx, iy, ix, iDisplay;
  int      *piBurntPixels, iCptNarrowBand;
  int      iTest, ix2, iy2;
  sNarrowBand   sNarBand;


  /* Get the size of the image */
  iNy = iDim[0]; 
  iNx = iDim[1]; 


  iDisplay = (int) pfVecGeneralParameters[0];
  fMaxDistExtension = 2.5;


  /*****************************************************************/
  /* Memory allocations */

  sNarBand.pfPhiValue = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!sNarBand.pfPhiValue)
    {
      mexPrintf("Memory allocation failure for sNarBand.pfPhiValue\n");
      return(0);
    }

  sNarBand.pix = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.pix)
    {
      mexPrintf("Memory allocation failure for sNarBand.pix\n");
      return(0);
    }

  sNarBand.piy = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piy)
    {
      mexPrintf("Memory allocation failure for sNarBand.piy\n");
      return(0);
    }

  sNarBand.piFlagNarrowBand = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piFlagNarrowBand)
    {
      mexPrintf("Memory allocation failure for sNarBand.piFlagNarrowBand\n");
      return(0);
    }

  sNarBand.iLength = 0;

  piBurntPixels = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!piBurntPixels)
    {
      mexPrintf("Memory allocation failure for piBurntPixels\n");
      return(0);
    }

  /*****************************************************************/


  /*****************************************************************/
  /* Memory allocation */

  pfDM = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfDM)
    {
      mexPrintf("Memory allocation failure for pfDM\n");
      return(0);
    }

  pfImExtended = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfImExtended)
    {
      mexPrintf("Memory allocation failure for pfImExtended\n");
      return(0);
    }

  pfImToExtend = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfImToExtend)
    {
      mexPrintf("Memory allocation failure for pfImExtended\n");
      return(0);
    }

  /*****************************************************************/



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      pfImToExtend[ix*iNy+ iy] = ppfImToExtend[ix][iy];



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      {
	piBurntPixels[ix*iNy+ iy] = 0;
	sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;
      }


  for(ix=0; ix< iNx; ix++)
    {
      pfImToExtend[ix*iNy] = INF;
      pfImToExtend[ix*iNy+ iNy-1] = INF;
    }
  for(iy=0; iy< iNy; iy++)
    {
      pfImToExtend[iy] = INF;
      pfImToExtend[(iNx-1)*iNy+ iy] = INF;
    }

  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(pfImToExtend[ix*iNy+ iy])< INF )
	pfImExtended[ix*iNy+ iy] = pfImToExtend[ix*iNy+ iy];
      else	
	pfImExtended[ix*iNy+ iy] = INF;





  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(pfImToExtend[ix*iNy+ iy])< INF )
	{
	  piBurntPixels[ix*iNy+ iy] = 1;
	  pfDM[ix*iNy+ iy] = -1.0;
	}
      else
	pfDM[ix*iNy+ iy] = INF;



  /* Initialization of the narrow band */
  for(ix=1; ix< iNx-1; ix++)
    for(iy=1; iy< iNy-1; iy++)
      {
	if ( ABS(pfImToExtend[ix*iNy+ iy])<INF  &&  (pfImExtended[(ix+1)*iNy+ iy]==INF || pfImExtended[(ix-1)*iNy+ iy]==INF || pfImExtended[ix*iNy+ iy+1]==INF || pfImExtended[ix*iNy+ iy-1]==INF)  )
	  {
	    pfDM[ix*iNy+ iy] = 0.0;
	    piBurntPixels[ix*iNy+ iy] = 1;
	    sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 1;
	    vAddElement2NarrowBand(&sNarBand,pfDM[ix*iNy+ iy],ix,iy);
	  }
      }
 


  fThreshold = 2.0;
  while ( sNarBand.iLength > 0 )
    {

      /* Take the point in "narrowband" that has the smallest value */
      ix = sNarBand.pix[0];
      iy = sNarBand.piy[0];
     
      /* Add the point to "burnt" and remove it from "narrowband" */
      piBurntPixels[ix*iNy+ iy] = 1;
      vRemoveFirstElementOfNarrowBand(&sNarBand);
      sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;

      /* Neighborhood for computing Distance Map */
      for (iCptNarrowBand=0; iCptNarrowBand<4; iCptNarrowBand++)
	{

	  iTest = 0;
	  if (iCptNarrowBand==0)
	    {
	      ix2=ix-1;  if (ix2<=0)  iTest = 1;
	      iy2=iy;
	    }
	  else if (iCptNarrowBand==1)
	    {
	      ix2=ix+1;  if (ix2>=iNx-1)  iTest = 1;
	      iy2=iy;
	    }
	  else if (iCptNarrowBand==2)
	    {
	      iy2=iy-1;  if (iy2<=0)  iTest = 1;
	      ix2=ix;
	    }
	  else if (iCptNarrowBand==3)
	    {
	      iy2=iy+1;  if (iy2>=iNy-1)  iTest = 1;
	      ix2=ix;
	    }

	  if ( iTest == 0 )
	    if ( piBurntPixels[ix2*iNy+ iy2] == 0 )
	      {
		    
		if ( pfDM[(ix2+1)*iNy+ iy2] < pfDM[(ix2-1)*iNy+ iy2] )
		  {
		    fPhi1 = pfDM[(ix2+1)*iNy+ iy2];
		    fF1 = pfImExtended[(ix2+1)*iNy+ iy2];
		  }
		else
		  {
		    fPhi1 = pfDM[(ix2-1)*iNy+ iy2];
		    fF1 = pfImExtended[(ix2-1)*iNy+ iy2];
		  }
	    
		if ( pfDM[ix2*iNy+ iy2+1] < pfDM[ix2*iNy+ iy2-1] )
		  {
		    fPhi2 = pfDM[ix2*iNy+ iy2+1];
		    fF2 = pfImExtended[ix2*iNy+ iy2+1];
		  }
		else
		  {
		    fPhi2 = pfDM[ix2*iNy+ iy2-1];
		    fF2 = pfImExtended[ix2*iNy+ iy2-1];
		  }


		if ( fPhi1==INF || fPhi2==INF )
		  {
		    pfDM[ix2*iNy+ iy2] = MIN(fPhi1,fPhi2) + 1.0;
		    pfImExtended[ix2*iNy+ iy2] = MIN(fF1,fF2);
		  }
		else
		  {
		    fx1 = SQR(fPhi1- fPhi2);
		    if( fx1 < fThreshold )
		      {
			pfDM[ix2*iNy+ iy2] = 0.5* ( fPhi1+ fPhi2+ sqrt(2.0- fx1) );
			pfImExtended[ix2*iNy+ iy2] = ( fF1* (pfDM[ix2*iNy+ iy2]- fPhi1) + fF2* (pfDM[ix2*iNy+ iy2]- fPhi2) )/ ( 2.0* pfDM[ix2*iNy+ iy2]- fPhi1- fPhi2 );
		      }
		    else
		      {
			pfDM[ix2*iNy+ iy2] = MIN(fPhi1,fPhi2) + 1.0;
			if ( fPhi1 < fPhi2 )
			  pfImExtended[ix2*iNy+ iy2] = fF1;
			else
			  pfImExtended[ix2*iNy+ iy2] = fF2;
		      }
		  }

		if ( ABS(pfDM[ix2*iNy+ iy2]) <= fMaxDistExtension )
		  {
		    if ( sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] == 0 )
		      {
			sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] = 1;
			vAddElement2NarrowBand(&sNarBand,pfDM[ix2*iNy+ iy2],ix2,iy2);
		      }
		    else
		      vUpdatePixelFromNarrowBand(&sNarBand,pfDM[ix2*iNy+ iy2],ix2,iy2);
		  }
	      }

	}

    }
    


  /* Neumann bondaries condition */
  for (ix=0; ix< iNx; ix++)
    {
      pfDM[ix*iNy] = pfDM[ix*iNy+ 1];
      pfDM[ix*iNy+ iNy-1] = pfDM[ix*iNy+ iNy-2];
    }
  for (iy=0; iy< iNy; iy++)
    {
      pfDM[iy] = pfDM[iNy+ iy];
      pfDM[(iNx-1)*iNy+ iy] = pfDM[(iNx-2)*iNy+ iy];
    }

  for (ix=0; ix< iNx; ix++)
    {
      pfImExtended[ix*iNy] = pfImExtended[ix*iNy+ 1];
      pfImExtended[ix*iNy+ iNy-1] = pfImExtended[ix*iNy+ iNy-2];
    }
  for (iy=0; iy< iNy; iy++)
    {
      pfImExtended[iy] = pfImExtended[iNy+ iy];
      pfImExtended[(iNx-1)*iNy+ iy] = pfImExtended[(iNx-2)*iNy+ iy];
    }



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(pfImExtended[ix*iNy+ iy])< INF )
	ppfImToExtend[ix][iy] = pfImExtended[ix*iNy+ iy];
      else
	ppfImToExtend[ix][iy] = 0.0;



  /*****************************************************************/
  /* Free memory  */

  free((float*) sNarBand.pfPhiValue);
  free((int*)   sNarBand.pix); 
  free((int*)   sNarBand.piy); 
  free((int*)   sNarBand.piFlagNarrowBand); 
  free((int*)   piBurntPixels);

  free((float*) pfDM);
  free((float*) pfImExtended);
  free((float*) pfImToExtend);
  /*****************************************************************/




  return(1);
 
}
/****************************************/






/****************************************/
extern int iExtensionFMDataG2Ui
(
 int    iDim[2],
 float  **ppfImToExtend,
 float  *pfVecGeneralParameters,
 float  *pfTemp
 )

{
  float    fMaxDistExtension, *pfDM, *pfImExtended;
  float    fPhi1, fF1, fPhi2, fF2, fThreshold, fx1;
  int      iNy, iNx, iy, ix, iDisplay, *piBurntPixels, iCptNarrowBand;
  int      iTest, ix2, iy2;
  sNarrowBand   sNarBand;


  /* Get the size of the image */
  iNy = iDim[0]; 
  iNx = iDim[1]; 


  iDisplay = (int) pfVecGeneralParameters[0];
  fMaxDistExtension = 2.5;



  /*****************************************************************/
  /* Memory allocations */

  sNarBand.pfPhiValue = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!sNarBand.pfPhiValue)
    {
      mexPrintf("Memory allocation failure for sNarBand.pfPhiValue\n");
      return(0);
    }

  sNarBand.pix = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.pix)
    {
      mexPrintf("Memory allocation failure for sNarBand.pix\n");
      return(0);
    }

  sNarBand.piy = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piy)
    {
      mexPrintf("Memory allocation failure for sNarBand.piy\n");
      return(0);
    }

  sNarBand.piFlagNarrowBand = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!sNarBand.piFlagNarrowBand)
    {
      mexPrintf("Memory allocation failure for sNarBand.piFlagNarrowBand\n");
      return(0);
    }

  sNarBand.iLength = 0;

  piBurntPixels = (int *) calloc( iNy* iNx,sizeof(int) );
  if (!piBurntPixels)
    {
      mexPrintf("Memory allocation failure for piBurntPixels\n");
      return(0);
    }

  /*****************************************************************/


  /*****************************************************************/
  /* Memory allocation */

  pfDM = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfDM)
    {
      mexPrintf("Memory allocation failure for pfDM\n");
      return(0);
    }

  pfImExtended = (float *) calloc( iNy* iNx,sizeof(float) );
  if (!pfImExtended)
    {
      mexPrintf("Memory allocation failure for pfImExtended\n");
      return(0);
    }

  /*****************************************************************/



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      {
	piBurntPixels[ix*iNy+ iy] = 0;
	sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;
      }


  for(ix=0; ix< iNx; ix++)
    {
      ppfImToExtend[ix][0] = INF;
      ppfImToExtend[ix][iNy-1] = INF;
    }
  for(iy=0; iy< iNy; iy++)
    {
      ppfImToExtend[0][iy] = INF;
      ppfImToExtend[iNx-1][iy] = INF;
    }

  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(ppfImToExtend[ix][iy])< INF )
	pfImExtended[ix*iNy+ iy] = ppfImToExtend[ix][iy];
      else	
	pfImExtended[ix*iNy+ iy] = INF;





  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(ppfImToExtend[ix][iy])< INF )
	{
	  piBurntPixels[ix*iNy+ iy] = 1;
	  pfDM[ix*iNy+ iy] = -1.0;
	}
      else
	pfDM[ix*iNy+ iy] = INF;



  /* Initialization of the narrow band */
  for(ix=1; ix< iNx-1; ix++)
    for(iy=1; iy< iNy-1; iy++)
      {
	if ( ABS(ppfImToExtend[ix][iy])<INF  &&  (pfImExtended[(ix+1)*iNy+ iy]==INF || pfImExtended[(ix-1)*iNy+ iy]==INF || pfImExtended[ix*iNy+ iy+1]==INF || pfImExtended[ix*iNy+ iy-1]==INF)  )
	  {
	    pfDM[ix*iNy+ iy] = 0.0;
	    piBurntPixels[ix*iNy+ iy] = 1;
	    sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 1;
	    vAddElement2NarrowBand(&sNarBand,pfDM[ix*iNy+ iy],ix,iy);
	  }
      }
 


  fThreshold = 2.0;

  while ( sNarBand.iLength > 0 )
    {

      /* Take the point in "narrowband" that has the smallest value */
      ix = sNarBand.pix[0];
      iy = sNarBand.piy[0];
     
      /* Add the point to "burnt" and remove it from "narrowband" */
      piBurntPixels[ix*iNy+ iy] = 1;
      vRemoveFirstElementOfNarrowBand(&sNarBand);
      sNarBand.piFlagNarrowBand[ix*iNy+ iy] = 0;

      /* Neighborhood for computing Distance Map */
      for (iCptNarrowBand=0; iCptNarrowBand<4; iCptNarrowBand++)
	{

	  iTest = 0;
	  if (iCptNarrowBand==0)
	    {
	      ix2=ix-1;  if (ix2<=0)  iTest = 1;
	      iy2=iy;
	    }
	  else if (iCptNarrowBand==1)
	    {
	      ix2=ix+1;  if (ix2>=iNx-1)  iTest = 1;
	      iy2=iy;
	    }
	  else if (iCptNarrowBand==2)
	    {
	      iy2=iy-1;  if (iy2<=0)  iTest = 1;
	      ix2=ix;
	    }
	  else if (iCptNarrowBand==3)
	    {
	      iy2=iy+1;  if (iy2>=iNy-1)  iTest = 1;
	      ix2=ix;
	    }

	  if ( iTest == 0 )
	    if ( piBurntPixels[ix2*iNy+ iy2] == 0 )
	      {
		    
		if ( pfDM[(ix2+1)*iNy+ iy2] < pfDM[(ix2-1)*iNy+ iy2] )
		  {
		    fPhi1 = pfDM[(ix2+1)*iNy+ iy2];
		    fF1 = pfImExtended[(ix2+1)*iNy+ iy2];
		  }
		else
		  {
		    fPhi1 = pfDM[(ix2-1)*iNy+ iy2];
		    fF1 = pfImExtended[(ix2-1)*iNy+ iy2];
		  }
	    
		if ( pfDM[ix2*iNy+ iy2+1] < pfDM[ix2*iNy+ iy2-1] )
		  {
		    fPhi2 = pfDM[ix2*iNy+ iy2+1];
		    fF2 = pfImExtended[ix2*iNy+ iy2+1];
		  }
		else
		  {
		    fPhi2 = pfDM[ix2*iNy+ iy2-1];
		    fF2 = pfImExtended[ix2*iNy+ iy2-1];
		  }


		if ( fPhi1==INF || fPhi2==INF )
		  {
		    pfDM[ix2*iNy+ iy2] = MIN(fPhi1,fPhi2) + 1.0;
		    pfImExtended[ix2*iNy+ iy2] = MIN(fF1,fF2);
		  }
		else
		  {
		    fx1 = SQR(fPhi1- fPhi2);
		    if( fx1 < fThreshold )
		      {
			pfDM[ix2*iNy+ iy2] = 0.5* ( fPhi1+ fPhi2+ sqrt(2.0- fx1) );
			pfImExtended[ix2*iNy+ iy2] = ( fF1* (pfDM[ix2*iNy+ iy2]- fPhi1) + fF2* (pfDM[ix2*iNy+ iy2]- fPhi2) )/ ( 2.0* pfDM[ix2*iNy+ iy2]- fPhi1- fPhi2 );
		      }
		    else
		      {
			pfDM[ix2*iNy+ iy2] = MIN(fPhi1,fPhi2) + 1.0;
			if ( fPhi1 < fPhi2 )
			  pfImExtended[ix2*iNy+ iy2] = fF1;
			else
			  pfImExtended[ix2*iNy+ iy2] = fF2;
		      }
		  }

		if ( ABS(pfDM[ix2*iNy+ iy2]) <= fMaxDistExtension )
		  {
		    if ( sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] == 0 )
		      {
			sNarBand.piFlagNarrowBand[ix2*iNy+ iy2] = 1;
			vAddElement2NarrowBand(&sNarBand,pfDM[ix2*iNy+ iy2],ix2,iy2);
		      }
		    else
		      vUpdatePixelFromNarrowBand(&sNarBand,pfDM[ix2*iNy+ iy2],ix2,iy2);
		  }

	      }

	}

    }
    


  /* Neumann bondaries condition */
  for (ix=0; ix< iNx; ix++)
    {
      pfDM[ix*iNy] = pfDM[ix*iNy+ 1];
      pfDM[ix*iNy+ iNy-1] = pfDM[ix*iNy+ iNy-2];
    }
  for (iy=0; iy< iNy; iy++)
    {
      pfDM[iy] = pfDM[iNy+ iy];
      pfDM[(iNx-1)*iNy+ iy] = pfDM[(iNx-2)*iNy+ iy];
    }

  for (ix=0; ix< iNx; ix++)
    {
      pfImExtended[ix*iNy] = pfImExtended[ix*iNy+ 1];
      pfImExtended[ix*iNy+ iNy-1] = pfImExtended[ix*iNy+ iNy-2];
    }
  for (iy=0; iy< iNy; iy++)
    {
      pfImExtended[iy] = pfImExtended[iNy+ iy];
      pfImExtended[(iNx-1)*iNy+ iy] = pfImExtended[(iNx-2)*iNy+ iy];
    }



  for(ix=0; ix< iNx; ix++)
    for(iy=0; iy< iNy; iy++)
      if ( ABS(pfImExtended[ix*iNy+ iy])< INF )
	ppfImToExtend[ix][iy] = pfImExtended[ix*iNy+ iy];
  else
    ppfImToExtend[ix][iy] = 0.0;




  /*****************************************************************/
  /* Free memory  */

  free((float*) sNarBand.pfPhiValue);
  free((int*)   sNarBand.pix); 
  free((int*)   sNarBand.piy); 
  free((int*)   sNarBand.piFlagNarrowBand); 
  free((int*)   piBurntPixels);

  free((float*) pfDM);
  free((float*) pfImExtended);
  /*****************************************************************/




  return(1);
 
}
/****************************************/













/**************************************** End of file ****************************************/
