
// Author: Xavier Bresson (xbresson at math.ucla.edu)
// Last version: Oct 06 2008
// Name: acsp_mex.c
// Description: Active contour with shape prior, proposed in
// [X. Bresson, P. Vandergheynst and J.-P. Thiran, A Variational Model for Object Segmentation
// Using Boundary Information and Shape Prior Driven by the Mumford-Shah Functional,
// IJCV 68(2), 2006, p145-162.]. See specifically Eqs (4) and (29-33).

// Compilation (run in matlab):
// mex -v -g acsp_mex.c acsp.c acspReDistLSF.c acspBoundaryTerm.c acspShapeTerms.c acspBsplines.c acspExtensionU.c
// mex acsp_mex.c acsp.c acspReDistLSF.c acspBoundaryTerm.c acspShapeTerms.c acspBsplines.c acspExtensionU.c


#include "acspCommon.h"


extern void mexFunction(int iNbOut, mxArray *pmxOut[],
int iNbIn, const mxArray *pmxIn[])
{
    
  /* iNbOut: number of outputs
     pmxOut: array of pointers to output arguments */
    
  /* iNbIn: number of inputs
     pmxIn: array of pointers to input arguments */
    
    float  *pfVecGeneralParameters, *pfPhiAcIn, *pfIm, *pfVecBoundaryTerm, *pfEdgeMap;
    float  *pfVecShapeTerm, *pfVecRigidTransformationsIn, *pfVpcaIn;
    float  *pfEVec, *pfEVal, *pfPCAMeanValue, *pfVecRegionTerm;
    float  *pfPhiAcOut, *pfPCAFunctionOut, *pfVecRigidTransformationsOut, *pfVpcaOut, *pfTemp, *pfMSRegionTerm;
    float  *pfUinRegionTermInput, *pfUoutRegionTermInput, *pfUinRegionTerm, *pfUoutRegionTerm;
    int    iDim[3], iDim2[3], iNbRows, iNbCols, iNx, iNy, ip, iDisplayResults, i, iNdim;
    int    iNbTemporalSteps;
    int    iDimIm[3], iDimPca[3], ix, iy;
    time_t start_time, end_time;
    
    start_time = clock();
    
    
  /* Vector of general parameters */
    pfVecGeneralParameters = mxGetData(pmxIn[0]);
    
  /* Input active Contour */
    pfPhiAcIn = mxGetData(pmxIn[1]); // varphi in Eqs (29-33)
    
  /* Given image */
    pfIm = mxGetData(pmxIn[2]); // I in Eqs (29-33)
    iNbRows = mxGetM(pmxIn[2]);     /* number of rows */
    iNbCols = mxGetN(pmxIn[2]);     /* number of columns */
    iNy = iNbRows;
    iNx = iNbCols;
    iDimIm[0] = iNy;
    iDimIm[1] = iNx;
    
  /* Vector of boundary term */
    pfVecBoundaryTerm = mxGetData(pmxIn[3]);
    
  /* Edge detector function */
    pfEdgeMap = mxGetData(pmxIn[4]); // g in Eq (29)
    
  /* Vector of shape term */
    pfVecShapeTerm = mxGetData(pmxIn[5]);
    
  /* Vector of rigid transformations */
    pfVecRigidTransformationsIn = mxGetData(pmxIn[6]); // x_T in Eqs (29-33)
    
 /* Vector of eigencoefficients */
    pfVpcaIn = mxGetData(pmxIn[7]); // x_{pca} in Eqs (29-33)
    
 /* Vector of eigenvectors */
    pfEVec = mxGetData(pmxIn[8]); // W_p in Eqs (4)
    
 /* Vector of eigenvalues */
    pfEVal = mxGetData(pmxIn[9]); // lambda_k
    ip = mxGetM(pmxIn[9]);
    iDimPca[0] = iNy;
    iDimPca[1] = iNx;
    iDimPca[2] = ip;
    
 /* Vector of mean shape value */
    pfPCAMeanValue = mxGetData(pmxIn[10]); // bar{phi} in Eq (4)
    
  /* Vector of region-based shape */
    pfVecRegionTerm = mxGetData(pmxIn[11]);
    
  /* Uin and Uout */
    pfUinRegionTermInput = mxGetData(pmxIn[12]); // u_{in} in Eqs (29-33)
    pfUoutRegionTermInput = mxGetData(pmxIn[13]); // u_{out} in Eqs (29-33)
    
    
    
    iDisplayResults = (int) pfVecGeneralParameters[0];
    
    
    iNdim = 2;
    iDim2[0] = iNy;
    iDim2[1] = iNx;
    pmxOut[0] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfPhiAcOut = mxGetData(pmxOut[0]);  // output active contour
    
    iNdim = 2;
    iDim2[0] = iNy;
    iDim2[1] = iNx;
    pmxOut[1] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfPCAFunctionOut = mxGetData(pmxOut[1]); // output pca function
    
    iNdim = 2;
    iDim2[0] = 7+2;
    iDim2[1] = 1;
    pmxOut[2] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfVecRigidTransformationsOut = mxGetData(pmxOut[2]); // output spatial transformations
    for (i=0; i< 7; i++)  pfVecRigidTransformationsOut[i] = pfVecRigidTransformationsIn[i];
    
    iNdim = 2;
    iDim2[0] = ip;
    iDim2[1] = 1;
    pmxOut[3] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfVpcaOut = mxGetData(pmxOut[3]); // output vector of pca eigenvector
    for (i=0; i< ip; i++)  pfVpcaOut[i] = pfVpcaIn[i];
    
    
    iNdim = 2;
    iDim2[0] = iNy;
    iDim2[1] = iNx;
    
    pmxOut[4] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfMSRegionTerm = mxGetData(pmxOut[4]); // output of piecewise-smooth function
    
    pmxOut[5] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfUinRegionTerm = mxGetData(pmxOut[5]); // output of u_in
    
    pmxOut[6] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfUoutRegionTerm = mxGetData(pmxOut[6]); // output of u_out
    
    
    for (ix=0; ix< iNx; ix++)
        for (iy=0; iy< iNy; iy++)
    {
        pfUinRegionTerm[ix*iNy+ iy] = pfUinRegionTermInput[ix*iNy+ iy];
        pfUoutRegionTerm[ix*iNy+ iy] = pfUoutRegionTermInput[ix*iNy+ iy];
        }
    
    
    iNdim = 3;
    iDim2[0] = iNy;
    iDim2[1] = iNx;
    iDim2[2] = 15;
    pmxOut[7] = mxCreateNumericArray(iNdim,(const int*)iDim2,mxSINGLE_CLASS,mxREAL);
    pfTemp = mxGetData(pmxOut[7]);
    
    
    if ( !iAcShapePrior(iDimIm,iDimPca,pfVecGeneralParameters,pfPhiAcIn,pfPhiAcOut,pfIm,pfVecBoundaryTerm,pfEdgeMap,pfVecShapeTerm,pfPCAFunctionOut,pfVecRigidTransformationsOut,pfVpcaOut,pfEVec,pfEVal,pfPCAMeanValue,pfVecRegionTerm,pfMSRegionTerm,pfUinRegionTerm,pfUoutRegionTerm,pfTemp) )
    {
        mexPrintf("Error in acsp_mex: return an array with null elements\n");
        for (i=0; i< iDim[0]* iDim[1]; i++)  pfPhiAcOut[i] = 0.0;
    }
    else
    {
        end_time = clock();
        iNbTemporalSteps = (int) pfVecGeneralParameters[1];
        mexPrintf("\nComputing Time Active Contour with Shape Prior= %.5f sec\n",difftime(end_time,start_time)/1000);
        mexPrintf("# iterations= %i, Image size = %i x %i.\n\n",iNbTemporalSteps,iNx,iNy);
    }
    
}


/****************** End of file *******************/
