'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

exports.buildPredicate = buildPredicate;
exports.reduceTreeBySelector = reduceTreeBySelector;
exports.reduceTreesBySelector = reduceTreesBySelector;

var _rstSelectorParser = require('rst-selector-parser');

var _object = require('object.values');

var _object2 = _interopRequireDefault(_object);

var _arrayPrototype = require('array.prototype.flat');

var _arrayPrototype2 = _interopRequireDefault(_arrayPrototype);

var _objectIs = require('object-is');

var _objectIs2 = _interopRequireDefault(_objectIs);

var _has = require('has');

var _has2 = _interopRequireDefault(_has);

var _byConstructor = require('html-element-map/byConstructor');

var _byConstructor2 = _interopRequireDefault(_byConstructor);

var _RSTTraversal = require('./RSTTraversal');

var _Utils = require('./Utils');

var _getAdapter = require('./getAdapter');

var _getAdapter2 = _interopRequireDefault(_getAdapter);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

function _toConsumableArray(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

// our CSS selector parser instance
var parser = (0, _rstSelectorParser.createParser)();

// Combinators that allow you to chance selectors
var CHILD = 'childCombinator';
var ADJACENT_SIBLING = 'adjacentSiblingCombinator';
var GENERAL_SIBLING = 'generalSiblingCombinator';
var DESCENDANT = 'descendantCombinator';

// Selectors for targeting elements
var SELECTOR = 'selector';
var TYPE_SELECTOR = 'typeSelector';
var CLASS_SELECTOR = 'classSelector';
var ID_SELECTOR = 'idSelector';
var UNIVERSAL_SELECTOR = 'universalSelector';
var ATTRIBUTE_PRESENCE = 'attributePresenceSelector';
var ATTRIBUTE_VALUE = 'attributeValueSelector';
// @TODO we dont support these, throw if they are used
var PSEUDO_CLASS = 'pseudoClassSelector';
var PSEUDO_ELEMENT = 'pseudoElementSelector';

var EXACT_ATTRIBUTE_OPERATOR = '=';
var WHITELIST_ATTRIBUTE_OPERATOR = '~=';
var HYPHENATED_ATTRIBUTE_OPERATOR = '|=';
var PREFIX_ATTRIBUTE_OPERATOR = '^=';
var SUFFIX_ATTRIBUTE_OPERATOR = '$=';
var SUBSTRING_ATTRIBUTE_OPERATOR = '*=';

function unique(arr) {
  return [].concat(_toConsumableArray(new Set(arr)));
}

/**
 * Calls reduce on a array of nodes with the passed
 * function, returning only unique results.
 * @param {Function} fn
 * @param {Array<Node>} nodes
 */
function uniqueReduce(fn, nodes) {
  return unique(nodes.reduce(fn, []));
}

/**
 * Takes a CSS selector and returns a set of tokens parsed
 * by scalpel.
 * @param {String} selector
 */
function safelyGenerateTokens(selector) {
  try {
    return parser.parse(selector);
  } catch (err) {
    throw new Error('Failed to parse selector: ' + String(selector));
  }
}

function matchAttributeSelector(node, token) {
  var operator = token.operator,
      value = token.value,
      name = token.name;

  var nodeProps = (0, _Utils.propsOfNode)(node);
  var descriptor = Object.getOwnPropertyDescriptor(nodeProps, name);
  if (descriptor && descriptor.get) {
    return false;
  }
  var nodePropValue = nodeProps[name];
  if (typeof nodePropValue === 'undefined') {
    return false;
  }
  if (token.type === ATTRIBUTE_PRESENCE) {
    return (0, _has2['default'])(nodeProps, token.name);
  }
  // Only the exact value operator ("=") can match non-strings
  if (typeof nodePropValue !== 'string' || typeof value !== 'string') {
    if (operator !== EXACT_ATTRIBUTE_OPERATOR) {
      return false;
    }
  }
  switch (operator) {
    /**
     * Represents an element with the att attribute whose value is exactly "val".
     * @example
     * [attr="val"] matches attr="val"
     */
    case EXACT_ATTRIBUTE_OPERATOR:
      return (0, _objectIs2['default'])(nodePropValue, value);
    /**
     * Represents an element with the att attribute whose value is a whitespace-separated
     * list of words, one of which is exactly
     * @example
     *  [rel~="copyright"] matches rel="copyright other"
     */
    case WHITELIST_ATTRIBUTE_OPERATOR:
      return nodePropValue.split(' ').indexOf(value) !== -1;
    /**
     * Represents an element with the att attribute, its value either being exactly the
     * value or beginning with the value immediately followed by "-"
     * @example
     * [hreflang|="en"] matches hreflang="en-US"
     */
    case HYPHENATED_ATTRIBUTE_OPERATOR:
      return nodePropValue === value || nodePropValue.startsWith(String(value) + '-');
    /**
     * Represents an element with the att attribute whose value begins with the prefix value.
     * If the value is the empty string then the selector does not represent anything.
     * @example
     * [type^="image"] matches type="imageobject"
     */
    case PREFIX_ATTRIBUTE_OPERATOR:
      return value === '' ? false : nodePropValue.slice(0, value.length) === value;
    /**
     * Represents an element with the att attribute whose value ends with the suffix value.
     * If the value is the empty string then the selector does not represent anything.
     * @example
     * [type$="image"] matches type="imageobject"
     */
    case SUFFIX_ATTRIBUTE_OPERATOR:
      return value === '' ? false : nodePropValue.slice(-value.length) === value;
    /**
     * Represents an element with the att attribute whose value contains at least one
     * instance of the value. If value is the empty string then the
     * selector does not represent anything.
     * @example
     * [title*="hello"] matches title="well hello there"
     */
    case SUBSTRING_ATTRIBUTE_OPERATOR:
      return value === '' ? false : nodePropValue.indexOf(value) !== -1;
    default:
      throw new Error('Enzyme::Selector: Unknown attribute selector operator "' + String(operator) + '"');
  }
}

function matchPseudoSelector(node, token, root) {
  var name = token.name,
      parameters = token.parameters;

  if (name === 'not') {
    // eslint-disable-next-line no-use-before-define
    return parameters.every(function (selector) {
      return reduceTreeBySelector(selector, node).length === 0;
    });
  }
  if (name === 'empty') {
    return (0, _RSTTraversal.treeFilter)(node, function (n) {
      return n !== node;
    }).length === 0;
  }
  if (name === 'first-child') {
    var _findParentNode = (0, _RSTTraversal.findParentNode)(root, node),
        rendered = _findParentNode.rendered;

    var _rendered = _slicedToArray(rendered, 1),
        firstChild = _rendered[0];

    return firstChild === node;
  }
  if (name === 'last-child') {
    var _findParentNode2 = (0, _RSTTraversal.findParentNode)(root, node),
        _rendered2 = _findParentNode2.rendered;

    return _rendered2[_rendered2.length - 1] === node;
  }
  if (name === 'focus') {
    if (typeof document === 'undefined') {
      throw new Error('Enzyme::Selector does not support the ":focus" pseudo-element without a global `document`.');
    }
    var adapter = (0, _getAdapter2['default'])();
    /* eslint-env browser */
    return document.activeElement && adapter.nodeToHostNode(node) === document.activeElement;
  }

  throw new TypeError('Enzyme::Selector does not support the "' + String(token.name) + '" pseudo-element or pseudo-class selectors.');
}

/**
 * Takes a node and a token and determines if the node
 * matches the predicate defined by the token.
 * @param {Node} node
 * @param {Token} token
 */
function nodeMatchesToken(node, token, root) {
  if (node === null || typeof node === 'string') {
    return false;
  }
  switch (token.type) {
    /**
     * Match every node
     * @example '*' matches every node
     */
    case UNIVERSAL_SELECTOR:
      return true;
    /**
     * Match against the className prop
     * @example '.active' matches <div className='active' />
     */
    case CLASS_SELECTOR:
      return (0, _RSTTraversal.hasClassName)(node, token.name);
    /**
     * Simple type matching
     * @example 'div' matches <div />
     */
    case TYPE_SELECTOR:
      return (0, _Utils.nodeHasType)(node, token.name);
    /**
     * Match against the `id` prop
     * @example '#nav' matches <ul id="nav" />
     */
    case ID_SELECTOR:
      return (0, _RSTTraversal.nodeHasId)(node, token.name);
    /**
     * Matches if an attribute is present, regardless
     * of its value
     * @example '[disabled]' matches <a disabled />
     */
    case ATTRIBUTE_PRESENCE:
      return matchAttributeSelector(node, token);
    /**
     * Matches if an attribute is present with the
     * provided value
     * @example '[data-foo=foo]' matches <div data-foo="foo" />
     */
    case ATTRIBUTE_VALUE:
      return matchAttributeSelector(node, token);
    case PSEUDO_ELEMENT:
    case PSEUDO_CLASS:
      return matchPseudoSelector(node, token, root);
    default:
      throw new Error('Unknown token type: ' + String(token.type));
  }
}

/**
 * Returns a predicate function that checks if a
 * node matches every token in the body of a selector
 * token.
 * @param {Token} token
 */
function buildPredicateFromToken(token, root) {
  return function (node) {
    return token.body.every(function (bodyToken) {
      return nodeMatchesToken(node, bodyToken, root);
    });
  };
}

/**
 * Returns whether a parsed selector is a complex selector, which
 * is defined as a selector that contains combinators.
 * @param {Array<Token>} tokens
 */
function isComplexSelector(tokens) {
  return tokens.some(function (token) {
    return token.type !== SELECTOR;
  });
}

/**
 * Takes a component constructor, object, or string representing
 * a simple selector and returns a predicate function that can
 * be applied to a single node.
 * @param {EnzymeSelector} selector
 */
function buildPredicate(selector) {
  // If the selector is a string, parse it as a simple CSS selector
  if (typeof selector === 'string') {
    var tokens = safelyGenerateTokens(selector);
    if (isComplexSelector(tokens)) {
      throw new TypeError('This method does not support complex CSS selectors');
    }
    // Simple selectors only have a single selector token
    return buildPredicateFromToken(tokens[0]);
  }

  // If the selector is an element type, check if the node's type matches
  var adapter = (0, _getAdapter2['default'])();
  var isElementType = adapter.isValidElementType ? adapter.isValidElementType(selector) : typeof selector === 'function';
  if (isElementType) {
    return function (node) {
      return adapter.matchesElementType(node, selector);
    };
  }
  // If the selector is an non-empty object, treat the keys/values as props
  if ((typeof selector === 'undefined' ? 'undefined' : _typeof(selector)) === 'object') {
    if (!Array.isArray(selector) && selector !== null && Object.keys(selector).length > 0) {
      var hasUndefinedValues = (0, _object2['default'])(selector).some(function (value) {
        return typeof value === 'undefined';
      });
      if (hasUndefinedValues) {
        throw new TypeError('Enzyme::Props can’t have `undefined` values. Try using ‘findWhere()’ instead.');
      }
      return function (node) {
        return (0, _RSTTraversal.nodeMatchesObjectProps)(node, selector);
      };
    }
    throw new TypeError('Enzyme::Selector does not support an array, null, or empty object as a selector');
  }

  throw new TypeError('Enzyme::Selector expects a string, object, or valid element type (Component Constructor)');
}

/**
 * Matches only nodes which are adjacent siblings (direct next sibling)
 * against a predicate, returning those that match.
 * @param {Array<Node>} nodes
 * @param {Function} predicate
 * @param {Node} root
 */
function matchAdjacentSiblings(nodes, predicate, root) {
  return nodes.reduce(function (matches, node) {
    var parent = (0, _RSTTraversal.findParentNode)(root, node);
    // If there's no parent, there's no siblings
    if (!parent) {
      return matches;
    }
    var parentChildren = (0, _RSTTraversal.childrenOfNode)(parent);
    var nodeIndex = parentChildren.indexOf(node);
    var adjacentSibling = parentChildren[nodeIndex + 1];
    // No sibling
    if (!adjacentSibling) {
      return matches;
    }
    if (predicate(adjacentSibling)) {
      matches.push(adjacentSibling);
    }
    return matches;
  }, []);
}

/**
 * Matches only nodes which are general siblings (any sibling *after*)
 * against a predicate, returning those that match.
 * @param {Array<Node>} nodes
 * @param {Function} predicate
 * @param {Node} root
 */
function matchGeneralSibling(nodes, predicate, root) {
  return uniqueReduce(function (matches, node) {
    var parent = (0, _RSTTraversal.findParentNode)(root, node);
    if (!parent) {
      return matches;
    }
    var parentChildren = (0, _RSTTraversal.childrenOfNode)(parent);
    var nodeIndex = parentChildren.indexOf(node);
    var youngerSiblings = parentChildren.slice(nodeIndex + 1);
    return matches.concat(youngerSiblings.filter(predicate));
  }, nodes);
}

/**
 * Matches only nodes which are direct children (not grandchildren, etc.)
 * against a predicate, returning those that match.
 * @param {Array<Node>} nodes
 * @param {Function} predicate
 */
function matchDirectChild(nodes, predicate) {
  return uniqueReduce(function (matches, node) {
    return matches.concat((0, _RSTTraversal.childrenOfNode)(node).filter(predicate));
  }, nodes);
}

/**
 * Matches all descendant nodes against a predicate,
 * returning those that match.
 * @param {Array<Node>} nodes
 * @param {Function} predicate
 */
function matchDescendant(nodes, predicate) {
  return uniqueReduce(function (matches, node) {
    return matches.concat((0, _RSTTraversal.treeFilter)(node, predicate));
  }, (0, _arrayPrototype2['default'])(nodes.map(_RSTTraversal.childrenOfNode)));
}

/**
 * Takes an RST and reduces it to a set of nodes matching
 * the selector. The selector can be a simple selector, which
 * is handled by `buildPredicate`, or a complex CSS selector which
 * reduceTreeBySelector parses and reduces the tree based on the combinators.
 *
 * @param {EnzymeSelector} selector
 * @param {RSTNode} root
 */
function reduceTreeBySelector(selector, root) {
  if (typeof selector !== 'string') {
    var elements = (0, _byConstructor2['default'])(selector);
    if (elements.length > 0) {
      return (0, _arrayPrototype2['default'])(elements.map(function (x) {
        return reduceTreeBySelector(x.tag, root);
      }));

      // when https://github.com/aweary/rst-selector-parser/issues/15 is resolved
      // const htmlTagNames = elements.map(x => x.tag).join(', ');
      // return reduceTreeBySelector(htmlTagNames, root);
    }
  }

  if (typeof selector === 'function' || (typeof selector === 'undefined' ? 'undefined' : _typeof(selector)) === 'object') {
    return (0, _RSTTraversal.treeFilter)(root, buildPredicate(selector));
  }

  var results = [];
  if (typeof selector === 'string') {
    var tokens = safelyGenerateTokens(selector);
    var index = 0;
    while (index < tokens.length) {
      var token = tokens[index];
      /**
       * There are two types of tokens in a CSS selector:
       *
       * 1. Selector tokens. These target nodes directly, like
       *    type or attribute selectors. These are easy to apply
       *    because we can traverse the tree and return only
       *    the nodes that match the predicate.
       *
       * 2. Combinator tokens. These tokens chain together
       *    selector nodes. For example > for children, or +
       *    for adjacent siblings. These are harder to match
       *    as we have to track where in the tree we are
       *    to determine if a selector node applies or not.
       */
      if (token.type === SELECTOR) {
        var predicate = buildPredicateFromToken(token, root);
        results = results.concat((0, _RSTTraversal.treeFilter)(root, predicate));
      } else {
        // We can assume there always all previously matched tokens since selectors
        // cannot start with combinators.
        var type = token.type;
        // We assume the next token is a selector, so move the index
        // forward and build the predicate.

        index += 1;
        var _predicate = buildPredicateFromToken(tokens[index], root);
        // We match against only the nodes which have already been matched,
        // since a combinator is meant to refine a previous selector.
        switch (type) {
          // The + combinator
          case ADJACENT_SIBLING:
            results = matchAdjacentSiblings(results, _predicate, root);
            break;
          // The ~ combinator
          case GENERAL_SIBLING:
            results = matchGeneralSibling(results, _predicate, root);
            break;
          // The > combinator
          case CHILD:
            results = matchDirectChild(results, _predicate);
            break;
          // The ' ' (whitespace) combinator
          case DESCENDANT:
            {
              results = matchDescendant(results, _predicate);
              break;
            }
          default:
            throw new Error('Unknown combinator selector: ' + String(type));
        }
      }
      index += 1;
    }
  } else {
    throw new TypeError('Enzyme::Selector expects a string, object, or Component Constructor');
  }
  return results;
}

function reduceTreesBySelector(selector, roots) {
  var results = roots.map(function (n) {
    return reduceTreeBySelector(selector, n);
  });
  return unique((0, _arrayPrototype2['default'])(results, 1));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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
//# sourceMappingURL=selectors.js.map