"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mkdirp_1 = __importDefault(require("mkdirp"));
const util_1 = require("util");
const path_1 = require("path");
const utils_1 = require("next-server/dist/server/utils");
const render_1 = require("next-server/dist/server/render");
const fs_1 = require("fs");
const async_sema_1 = __importDefault(require("async-sema"));
const amphtml_validator_1 = __importDefault(require("amphtml-validator"));
const load_components_1 = require("next-server/dist/server/load-components");
const envConfig = require('next-server/config');
const mkdirp = util_1.promisify(mkdirp_1.default);
global.__NEXT_DATA__ = {
    nextExport: true
};
process.on('message', async ({ distDir, buildId, exportPaths, exportPathMap, outDir, renderOpts, serverRuntimeConfig, concurrency }) => {
    const sema = new async_sema_1.default(concurrency, { capacity: exportPaths.length });
    try {
        const work = async (path) => {
            await sema.acquire();
            const { page, query = {} } = exportPathMap[path];
            const ampOpts = { amphtml: Boolean(query.amp), hasAmp: query.hasAmp, ampPath: query.ampPath };
            delete query.hasAmp;
            delete query.ampPath;
            const req = { url: path };
            const res = {};
            envConfig.setConfig({
                serverRuntimeConfig,
                publicRuntimeConfig: renderOpts.runtimeConfig
            });
            if (query.ampOnly) {
                delete query.ampOnly;
                path = utils_1.cleanAmpPath(path);
            }
            // replace /docs/index.amp with /docs.amp
            path = path.replace(/(?<!^)\/index\.amp$/, '.amp');
            let htmlFilename = `${path}${path_1.sep}index.html`;
            const pageExt = path_1.extname(page);
            const pathExt = path_1.extname(path);
            // Make sure page isn't a folder with a dot in the name e.g. `v1.2`
            if (pageExt !== pathExt && pathExt !== '') {
                // If the path has an extension, use that as the filename instead
                htmlFilename = path;
            }
            else if (path === '/') {
                // If the path is the root, just use index.html
                htmlFilename = 'index.html';
            }
            const baseDir = path_1.join(outDir, path_1.dirname(htmlFilename));
            const htmlFilepath = path_1.join(outDir, htmlFilename);
            await mkdirp(baseDir);
            const components = await load_components_1.loadComponents(distDir, buildId, page);
            const html = await render_1.renderToHTML(req, res, page, query, Object.assign({}, components, renderOpts, ampOpts));
            if (ampOpts.amphtml) {
                const validator = await amphtml_validator_1.default.getInstance();
                const result = validator.validateString(html);
                const errors = result.errors.filter(e => e.severity === 'ERROR');
                const warnings = result.errors.filter(e => e.severity !== 'ERROR');
                if (warnings.length || errors.length) {
                    process.send({
                        type: 'amp-validation',
                        payload: {
                            page,
                            result: {
                                errors,
                                warnings
                            }
                        }
                    });
                }
            }
            await new Promise((resolve, reject) => fs_1.writeFile(htmlFilepath, html, 'utf8', err => (err ? reject(err) : resolve())));
            process.send({ type: 'progress' });
            sema.release();
        };
        await Promise.all(exportPaths.map(work));
        process.send({ type: 'done' });
    }
    catch (err) {
        console.error(err);
        process.send({ type: 'error', payload: err });
    }
});
