import moment from 'moment'

export const productTypes =
{
  DYNAMIC: 1,
  STATIC: 2,
  COMPOSITE: 5,
  KIT: 14
}

export const getIsNGProduct = (product) => {
  const { Type, Attributes } = product

  switch (Type) {
    case productTypes.KIT:
      return true
    case productTypes.STATIC:
      const NGCompatible = Attributes.find(attribute => attribute.Name === 'NGCompatible')
      return NGCompatible && NGCompatible.Value.toLowerCase() === 'true'
    default:
      return false
  }
}

export const isDate = function (date) {
  if (typeof date === 'string') {
    if (date && (date.includes(':') || date.includes('-'))) {
      const dateToday = moment(new Date()).format('YYYY-MM-DD')

      const d = date.includes(':') && !date.includes('-')
        ? new Date(`${dateToday}T${date}`)
        : new Date(date)

      return !isNaN(d.valueOf())
    }
  }
  return false
}

const convertDatesForState = (value, modes) => {
  const { showDateTime, showTime } = modes
  const date = moment(new Date()).format('YYYY-MM-DD')

  Date.prototype.addHours = function (h) {
    this.setHours(this.getHours() + h)
    return this
  }

  const dateFromProps =
    showDateTime && showTime
      ? value
      : showTime
        ? `${date}T${value}`
        : `${value}T00:00:00`

  return moment(dateFromProps).toDate()
}

export const convertDate = (str_) => {
  const showTime = str_.includes(':')
  const showDateTime = str_.includes('/') || str_.includes('-')

  const nodes = { showTime, showDateTime }
  const str = convertDatesForState(str_, nodes)

  const date = moment(str).format('YYYY-MM-DD')
  const time = moment(str).format('HH:mm')

  const clientNow = moment().format('YYYY-MM-DDTHH:mm:ss')

  return { date, time, clientNow }
}

export const convertProductPropertiesFormIntoArray = (currentProductProperties, excelPricingEnabled) => {
  const propertiesArray = []
  const { formData = {}, JSONSchema: { definitions } } = currentProductProperties
  const propertiesNames = Object.keys(formData)

  for (const name of propertiesNames) {
    const propertyDefinition = definitions[name]
    if (formData[name]) {
      if (isDate(formData[name])) {
        propertiesArray.push({ id: propertyDefinition.custom.id, value: convertDate(formData[name]) })
      } else {
        propertiesArray.push({ id: propertyDefinition.custom.id, value: formData[name] })
      }
    } else {
      propertiesArray.push({ id: propertyDefinition.custom.id, value: '' })
    }
  }
  return propertiesArray.map((p) => ({ ...p }))
}

function isObject (item) {
  return (item && typeof item === 'object' && !Array.isArray(item))
}
export const mergeDeep = (target, ...sources) => {
  if (!sources.length) return target
  const source = sources.shift()

  if (isObject(target) && isObject(source)) {
    for (const key in source) {
      if (isObject(source[key])) {
        if (!target[key]) Object.assign(target, { [key]: {} })
        mergeDeep(target[key], source[key])
      } else {
        Object.assign(target, { [key]: source[key] })
      }
    }
  }

  return mergeDeep(target, ...sources)
}

export const PriceNotUpdatedReason = {
  hiddenError: 'hidden error',
  visibleError: 'visible error',
  noUpdates: 'nothing was updated',
  notAffectPriceProperty: 'updated property does not affect price'
}
export const shouldPriceBeUpdatedDespiteErrors = (hiddenErrors = [], visibleErrors = [], excelPricingEnabled = false) => {
  if (visibleErrors && visibleErrors.length > 0 && !!visibleErrors.find((e) => e.affectPrice)) {
    return { result: false, error: PriceNotUpdatedReason.visibleError }
  }

  if (excelPricingEnabled) {
    return { result: true }
  }

  let hiddenErrorsAffectPrice = false
  if (hiddenErrors && hiddenErrors.length > 0) {
    hiddenErrorsAffectPrice = !!hiddenErrors.find((e) => e.affectPrice)
  }

  return !hiddenErrorsAffectPrice ? { result: true } : { result: false, error: PriceNotUpdatedReason.hiddenError }
}

let prevDefs = {}
export const shouldPriceBeUpdatedByUpdates = (newProps, prevProps, schema) => {
  const updatedProperties = Object.keys(newProps)
    .filter((propertyName) => newProps[propertyName] !== prevProps[propertyName])

  if (updatedProperties.length === 0) {
    return { result: false, error: PriceNotUpdatedReason.noUpdates }
  }
  const { definitions } = schema

  let shouldUpdatePrice = false
  const filteredDefs = Object.keys(schema.properties)
    .reduce((acc, next) => ({ ...acc, [next]: schema.definitions[next] }), {})

  if (Object.keys(prevDefs).length !== Object.keys(filteredDefs).length) {
    shouldUpdatePrice = true
  }
  prevDefs = filteredDefs
  for (const propertyName of updatedProperties) {
    if (definitions[propertyName].custom.affectPrice) {
      shouldUpdatePrice = true
      break
    }
  }

  if (!shouldUpdatePrice) {
    return { result: false, error: PriceNotUpdatedReason.notAffectPriceProperty }
  }

  return { result: true }
}

export const shouldPriceBeUpdated = (newProps, prevProps, schema, hiddenErrors = [], visibleErrors = [], excelPricingEnabled = false) => {
  const shouldPriceBeUpdatedByUpdatesResult = shouldPriceBeUpdatedByUpdates(newProps, prevProps, schema)
  const shouldPriceBeUpdatedDespiteErrorsResult = shouldPriceBeUpdatedDespiteErrors(hiddenErrors, visibleErrors, excelPricingEnabled)

  if (!shouldPriceBeUpdatedByUpdatesResult.result) {
    return shouldPriceBeUpdatedByUpdatesResult
  }

  return shouldPriceBeUpdatedDespiteErrorsResult
}
