const fs = require('fs')
const path = require('path')
const { glob } = require('glob')
const jsdoc2md = require('jsdoc-to-markdown')

const createdMdFiles = []

function loadFileAndDependencies (filePath, loadedFiles = new Set()) {
  if (loadedFiles.has(filePath)) {
    return ''
  }
  loadedFiles.add(filePath)

  let fileContent = ''
  try {
    fileContent = fs.readFileSync(filePath, 'utf8')

    const importRegex = /require\s*\(\s*['"](.+?)['"]\s*\)/g
    let match
    while ((match = importRegex.exec(fileContent)) !== null) {
      const importedFilePath = match[1]
      const fullPath = path.resolve(path.dirname(filePath), importedFilePath)
      fileContent += loadFileAndDependencies(fullPath, loadedFiles)
    }
  } catch (error) {
    if (error.code !== 'ENOENT') {
      throw error
    }
  }

  return fileContent
}

async function processFilesInDirectory (directoryPath, outputFolder) {
  const jsfiles = await glob('./src/**/*.js', { ignore: 'node_modules/**' })
  for (const file of jsfiles) {
    const mdFile = file.replace(/^src\//, 'uStoreProvider-').replace(/\//g, '-').replace(/\.js$/, '.md')
    const fileContent = loadFileAndDependencies(file)
    if (!fileContent) continue

    const output = await jsdoc2md.render({ source: fileContent })
    try {
      if (output) {
        fs.writeFileSync(`${outputFolder}/${mdFile}`, output)
        console.log(`Documentation generated for ${mdFile}`)
        createdMdFiles.push(mdFile)
      }
    } catch (err) {
      console.error(`Error during conversion: ${file}:`, err)
    }
  }
}

const inputFolderPath = 'src'
const outputFolderPath = 'docs-md'
const uStoreProviderFilePath = `${outputFolderPath}/uStoreProvider-UStoreProvider.md`

if (!fs.existsSync(outputFolderPath)) {
  fs.mkdirSync(outputFolderPath)
  console.log(`Directory '${outputFolderPath}' created.`)
} else {
  console.log(`Directory '${outputFolderPath}' already exists deleting all files in it.`)
  fs.rmdirSync(outputFolderPath, { recursive: true })
  fs.mkdirSync(outputFolderPath)
}

processFilesInDirectory(inputFolderPath, outputFolderPath + '/')
  .then(() => {
    console.log('Documentation generation completed.')

    const links = createdMdFiles.filter(file => file !== 'uStoreProvider-UStoreProvider.md').map(file => `- [${file.replace(/(\.md)|(uStoreProvider-)\//, '').replace(/-/g, ' ')}](https://github.com/XMPieLab/uStore-NG/wiki/${file.replace(/\.md$/, '')})`).join('\n')
    const templateContent = fs.readFileSync(uStoreProviderFilePath, 'utf8')
    const filledTemplateContent = templateContent.replace('{{links}}', links)
    fs.writeFileSync(uStoreProviderFilePath, filledTemplateContent)
  })
  .catch(err => console.error('Error during documentation generation:', err))
