package com.xuelang.spos.store.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.xuelang.spos.common.AppUtils;
import com.xuelang.spos.config.GlobalEnv;
import com.xuelang.spos.oss.constant.AppStatus;
import com.xuelang.spos.oss.dao.AliyunOssDao;
import com.xuelang.spos.oss.service.AppInformationService;
import com.xuelang.spos.store.data.DownloadRel;
import com.xuelang.spos.store.data.DownloadRelList;
import com.xuelang.spos.store.exception.DeployDockerException;
import com.xuelang.spos.store.exception.DownloadDockerException;
import com.xuelang.spos.store.vo.ActionVo;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author: moyan.zk
 * @author: xiongw.fnst@cn.fujitsu.com
 * @date:2019/6/26
 * @date:2019/07/08
 */
@Slf4j
@Service
public class AppActionServiceImpl implements AppActionService {
    @Autowired
    private AliyunOssDao ossDao;
    @Autowired
    private AppInformationService informationService;
    @Autowired
    private GlobalEnv globalEnv;

    private String urlHelper(String host, String path) {
        return String.format("http://%s/%s", host, path);
    }

    private String executeRequest(Request request) {
        OkHttpClient client = new OkHttpClient();
        try {
            Response response = client.newCall(request).execute();
            String content = response.body().string();
            log.info("execute url: {}, response: {}", request.url(), content);
            return content;
        } catch (IOException e) {
            log.error("executeRequest: {}", AppUtils.stackTraceAsString(e));
            throw new RuntimeException(e);
        } finally {
            client.connectionPool().evictAll();
        }
    }

    private String queryAppInfo(String id) {
        JSONObject data = buildAppIdData(id);
        String url = urlHelper(globalEnv.getSpHost(), "app/get");
        Request request = buildRequest(url, data);
        return executeRequest(request);
    }

    private String queryGraphFromOss(String appInfo) {
        JSONObject app = JSON.parseObject(appInfo);
        String ossKey = app.getJSONObject("appInfo").getString("data");
        byte[] bytes = ossDao.getBytes(ossKey);
        // 转为无符号位整数
        int[] bufferNew = new int[bytes.length];
        for (int i = 0; i < bytes.length; i++) {
            int u = bytes[i] & 0xFF;
            bufferNew[i] = u;
        }
        // 解析
        ScriptEngineManager manager = new ScriptEngineManager();
        ScriptEngine engine = manager.getEngineByName("nashorn");
        try {
            InputStream inputStream = this.getClass().getResourceAsStream("/script/uint8.js");
            engine.eval(new InputStreamReader(inputStream));
        } catch (Exception e) {
            log.error(AppUtils.stackTraceAsString(e));
            throw new RuntimeException(e);
        }
        Invocable invocable = (Invocable) engine;
        try {
            return invocable
                    .invokeFunction("decompressFromUint8Array", bufferNew)
                    .toString();
        } catch (Exception e) {
            log.error("decompressFromUint8Array fail: {}", AppUtils.stackTraceAsString(e));
            throw new RuntimeException(e);
        }
    }

    private String dockerRepoUrl(String repo) {
        return String.format("%s/%s", globalEnv.getSpDockerRegistryHost(), repo);
    }

    private List<String> extractDownloadRepoUrls(String fromOss) {
        List<String> urls = new ArrayList<>();
        JSONObject processes = JSON.parseObject(fromOss).getJSONObject("processes");
        if (processes.size() == 0)
            return urls;

        Map mapTypes = JSON.parseObject(processes.toJSONString());
        // 获取下载地址
        for (Object key : mapTypes.keySet()) {
            JSONObject jsonObject = (JSONObject) mapTypes.get(key);
            String dockerRepo = jsonObject
                    .getJSONObject("metadata")
                    .getJSONObject("def")
                    .getString("dockerRepo");
            if (StringUtils.isNotBlank(dockerRepo) &&
                    !urls.contains(dockerRepoUrl(dockerRepo))
            ) {
                urls.add(dockerRepoUrl(dockerRepo));
            }
        }
        return urls;
    }

    private String requestDownload(List<String> repoUrls) {
        RequestBody body = RequestBody.create(
                MediaType.parse("application/json; charset=utf-8"), JSON.toJSONString(repoUrls)
        );
        Request request = new Request.Builder()
                .url(urlHelper(globalEnv.getSpServiceDockerRegistryUrl(), "downloads"))
                .post(body)
                .build();
        String content = executeRequest(request);
        JSONObject jsonObject = JSON.parseObject(content);
        return jsonObject.getString("download_id");
    }

    @Override
    public ActionVo download(String id) {
        String appInfo = queryAppInfo(id);
        String fromOss = queryGraphFromOss(appInfo);
        List<String> repoUrls = extractDownloadRepoUrls(fromOss);
        String downloadId = requestDownload(repoUrls);

        log.info("download app: {}, downloadId: {}", id, downloadId);
        DownloadRelList.addRel(id, downloadId);

        AppStatus appStatus = AppStatus.DOWNLOADING;
        informationService.updateAppStatus(id, appStatus);

        ActionVo vo = new ActionVo();
        vo.setId(id);
        vo.setDownloadId(downloadId);
        vo.setStatus(appStatus.getStatus());
        return vo;
    }

    @Override
    public ActionVo downloadProgress(String id) throws DownloadDockerException {
        DownloadRel rel = DownloadRelList.queryRelByAppId(id);
        String path = "progresses/" + rel.getDownloadId();
        String url = urlHelper(globalEnv.getSpServiceDockerRegistryUrl(), path);

        Request request = buildRequest(url);
        String content = executeRequest(request);

        JSONObject allProgress = JSON.parseObject(content);
        int process = 0;
        for (String key : allProgress.keySet()) {
            String value = (String) allProgress.get(key);
            if (value.startsWith("E-")) {
                throw new DownloadDockerException(value);
            }
            process = process + Integer.parseInt(value.replace("%", ""));
        }
        process = process / allProgress.size();

        ActionVo vo = new ActionVo();
        vo.setId(id);
        vo.setProgress(process);
        AppStatus status = process == 100 ?
                AppStatus.INSTALLED : AppStatus.DOWNLOADING;
        informationService.updateAppStatus(id, status);
        vo.setStatus(status.getStatus());
        return vo;
    }

    @Override
    public ActionVo deploy(String id) {
        String url = urlHelper(globalEnv.getSpHost(), "predict/deploy");
        JSONObject data = buildAppIdData(id);

        Request request = buildRequest(url, data);
        executeRequest(request);
        log.info("deployed app: {}", id);

        ActionVo vo = new ActionVo();
        vo.setId(id);
        return vo;
    }

    @Override
    public ActionVo deployStatus(String id) throws DeployDockerException {
        String url = urlHelper(globalEnv.getSpHost(), "predict/status");

        JSONObject data = buildAppIdData(id);
        Request request = buildRequest(url, data);
        String content = executeRequest(request);

        String deployStatus = JSON.parseObject(content)
                .getJSONObject("map")
                .getString("status");
        if (StringUtils.isBlank(deployStatus)) {
            throw new DeployDockerException(id);
        }

        ActionVo vo = new ActionVo();
        vo.setId(id);
        vo.setDeployStatus(deployStatus);
        return vo;
    }

    private JSONObject buildAppIdData(String id) {
        JSONObject data = new JSONObject();
        data.put("id", id);
        return data;
    }

    private Request buildRequest(String url) {
        return buildRequest(url, null);
    }

    private Request buildRequest(String url, JSONObject jsonObject) {
        Request.Builder requestBuilder = new Request.Builder()
                .header("set-cookie", globalEnv.getSpSid())
                .url(url);
        if (jsonObject != null) {
            RequestBody body = RequestBody.create(
                    MediaType.parse("application/json; charset=utf-8"), jsonObject.toJSONString()
            );
            requestBuilder.post(body);
        }
        return requestBuilder.build();
    }

    @Override
    public ActionVo uninstall(String id) {
        JSONObject data = buildAppIdData(id);
        String url = urlHelper(globalEnv.getSpHost(), "predict/release");

        Request request = buildRequest(url, data);
        executeRequest(request);

        informationService.updateAppStatus(id, AppStatus.CAN_INSTALL);
        ActionVo vo = new ActionVo();
        vo.setId(id);
        vo.setStatus(AppStatus.CAN_INSTALL.getStatus());
        return vo;
    }
}
