-- Conduct a QA check.

drop view if exists distinct_offsets cascade;
create view distinct_offsets as
with v as (
  select distinct std_offset as the_offset from approved_timezone_names
  union
  select distinct dst_offset as the_offset from approved_timezone_names
  )
select distinct the_offset from v;

-- There's no \or (relative spool) metacommand.
-- So when this file is invoked by "0.sql" on its parent directory,
-- the spool file path must be given relative to that parent parent directory.

\o set-timezone-and-at-timezone-encapsulations/spool-distinct-offsets-from-approved-timezone-names.txt

do $body$
declare
  n constant int not null := (select count(the_offset) from distinct_offsets);
begin
  assert n = 39, 'Assert failed';
end;
$body$;

do $body$
begin
  -- Prove that distinct_offsets.the_offset is a "pure seconds" interval .
  declare
    x int not null := (
      with
        v as (select interval_mm_dd_ss(the_offset) as i from distinct_offsets)
      select
        (select count(*) as "count w/ non-zero mm" from v where (i).mm <> 0::double precision) +
        (select count(*) as "count w/ non-zero dd" from v where (i).dd <> 0::double precision)
      );
  begin
    assert x = 0::double precision, 'assert #1 failed';
  end;

  -- Prove that the ss value of distinct_offsets.the_offset is an integral no. of quarter-hours.
  declare
    x numeric not null := (
      with
        v as (select interval_mm_dd_ss(the_offset) as i from distinct_offsets)
      select sum(mod((i).ss::numeric, 60*15::numeric)) from v
      );
  begin
    assert x = 0::numeric, 'assert #2 failed';
  end;
end;
$body$;

-- List all distinct values of the_offset
with
  v1 as (select the_offset as i from distinct_offsets)
select
  extract(hour   from i) as "hour",
  extract(minute from i) as "minute"
from v1
order by 1, 2;

-- List the facts about timezones where the_offset isn't
-- an integral no. of hours.
with
  v as (
    select the_offset from distinct_offsets
    where mod((interval_mm_dd_ss(the_offset)).ss::numeric, 60*60::numeric) <> 0::numeric)
select
  name,
  lpad(std_offset::text, 9) as "std offset",
  lpad(dst_offset::text, 9) as "dst offset"
from approved_timezone_names
where
  std_offset in (select the_offset from v) or
  dst_offset in (select the_offset from v)
order by std_offset, dst_offset, name;
\o

drop view if exists distinct_offsets cascade;
