/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.zip.Adler32;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.IndexFormatTooNewException;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.index.SegmentCommitInfo;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.BufferedChecksum;
import org.apache.lucene.store.ByteArrayDataInput;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefBuilder;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.Version;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.lucene.store.ByteArrayIndexInput;
import org.elasticsearch.common.lucene.store.InputStreamIndexInput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.Callback;
import org.elasticsearch.common.util.SingleObjectCache;
import org.elasticsearch.common.util.concurrent.AbstractRefCounted;
import org.elasticsearch.common.util.concurrent.RefCounted;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.settings.IndexSettingsService;
import org.elasticsearch.index.shard.AbstractIndexShardComponent;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.DirectoryService;
import org.elasticsearch.index.store.LegacyVerification;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.index.store.StoreStats;
import org.elasticsearch.index.store.VerifyingIndexOutput;

public class Store
extends AbstractIndexShardComponent
implements Closeable,
RefCounted {
    static final String CODEC = "store";
    static final int VERSION_WRITE_THROWABLE = 2;
    static final int VERSION_STACK_TRACE = 1;
    static final int VERSION_START = 0;
    static final int VERSION = 2;
    static final String CORRUPTED = "corrupted_";
    public static final String INDEX_STORE_STATS_REFRESH_INTERVAL = "index.store.stats_refresh_interval";
    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private final StoreDirectory directory;
    private final ReentrantReadWriteLock metadataLock = new ReentrantReadWriteLock();
    private final ShardLock shardLock;
    private final OnClose onClose;
    private final SingleObjectCache<StoreStats> statsCache;
    private final AbstractRefCounted refCounter = new AbstractRefCounted("store"){

        @Override
        protected void closeInternal() {
            Store.this.closeInternal();
        }
    };
    public static final String CHECKSUMS_PREFIX = "_checksums-";

    public Store(ShardId shardId, Settings indexSettings, DirectoryService directoryService, ShardLock shardLock) throws IOException {
        this(shardId, indexSettings, directoryService, shardLock, OnClose.EMPTY);
    }

    @Inject
    public Store(ShardId shardId, IndexSettingsService indexSettingsService, DirectoryService directoryService, ShardLock shardLock, OnClose onClose) throws IOException {
        this(shardId, indexSettingsService.getSettings(), directoryService, shardLock, onClose);
    }

    public Store(ShardId shardId, Settings indexSettings, DirectoryService directoryService, ShardLock shardLock, OnClose onClose) throws IOException {
        super(shardId, indexSettings);
        this.directory = new StoreDirectory(directoryService.newDirectory(), Loggers.getLogger("index.store.deletes", indexSettings, shardId, new String[0]));
        this.shardLock = shardLock;
        this.onClose = onClose;
        TimeValue refreshInterval = indexSettings.getAsTime(INDEX_STORE_STATS_REFRESH_INTERVAL, TimeValue.timeValueSeconds(10L));
        this.statsCache = new StoreStatsCache(refreshInterval, (Directory)this.directory, directoryService);
        this.logger.debug("store stats are refreshed with refresh_interval [{}]", refreshInterval);
        assert (onClose != null);
        assert (shardLock != null);
        assert (shardLock.getShardId().equals(shardId));
    }

    public Directory directory() {
        this.ensureOpen();
        return this.directory;
    }

    public SegmentInfos readLastCommittedSegmentsInfo() throws IOException {
        this.failIfCorrupted();
        try {
            return Store.readSegmentsInfo(null, this.directory());
        }
        catch (CorruptIndexException ex) {
            this.markStoreCorrupted((IOException)((Object)ex));
            throw ex;
        }
    }

    private static SegmentInfos readSegmentsInfo(IndexCommit commit, Directory directory) throws IOException {
        assert (commit == null || commit.getDirectory() == directory);
        try {
            return commit == null ? Lucene.readSegmentInfos(directory) : Lucene.readSegmentInfos(commit);
        }
        catch (EOFException eof) {
            throw new CorruptIndexException("Read past EOF while reading segment infos", "commit(" + commit + ")", (Throwable)eof);
        }
        catch (IOException exception) {
            throw exception;
        }
        catch (Exception ex) {
            throw new CorruptIndexException("Hit unexpected exception while reading segment infos", "commit(" + commit + ")", (Throwable)ex);
        }
    }

    final void ensureOpen() {
        if (this.refCounter.refCount() <= 0) {
            throw new AlreadyClosedException("store is already closed");
        }
    }

    public MetadataSnapshot getMetadataOrEmpty() throws IOException {
        try {
            return this.getMetadata(null);
        }
        catch (IndexNotFoundException indexNotFoundException) {
        }
        catch (FileNotFoundException | NoSuchFileException ex) {
            this.logger.info("Failed to open / find files while reading metadata snapshot", new Object[0]);
        }
        return MetadataSnapshot.EMPTY;
    }

    public MetadataSnapshot getMetadata() throws IOException {
        return this.getMetadata(null);
    }

    public MetadataSnapshot getMetadata(IndexCommit commit) throws IOException {
        this.ensureOpen();
        this.failIfCorrupted();
        this.metadataLock.readLock().lock();
        try {
            MetadataSnapshot metadataSnapshot = new MetadataSnapshot(commit, (Directory)this.directory, this.logger);
            return metadataSnapshot;
        }
        catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException ex) {
            this.markStoreCorrupted((IOException)ex);
            throw ex;
        }
        finally {
            this.metadataLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void renameTempFilesSafe(Map<String, String> tempFileMap) throws IOException {
        Object[] entries = tempFileMap.entrySet().toArray(new Map.Entry[tempFileMap.size()]);
        ArrayUtil.timSort((Object[])entries, (Comparator)new Comparator<Map.Entry<String, String>>(){

            @Override
            public int compare(Map.Entry<String, String> o1, Map.Entry<String, String> o2) {
                String left = o1.getValue();
                String right = o2.getValue();
                if (left.startsWith("segments") || right.startsWith("segments")) {
                    if (!left.startsWith("segments")) {
                        return -1;
                    }
                    if (!right.startsWith("segments")) {
                        return 1;
                    }
                }
                return left.compareTo(right);
            }
        });
        this.metadataLock.writeLock().lock();
        try (Lock writeLock = this.directory().obtainLock("write.lock");){
            for (Object entry : entries) {
                String tempFile = (String)entry.getKey();
                String origFile = (String)entry.getValue();
                try {
                    this.directory.deleteFile(origFile);
                }
                catch (FileNotFoundException | NoSuchFileException iOException) {
                }
                catch (Throwable ex) {
                    this.logger.debug("failed to delete file [{}]", ex, origFile);
                }
                this.renameFile(tempFile, origFile);
                String remove = tempFileMap.remove(tempFile);
                assert (remove != null);
            }
        }
        finally {
            this.metadataLock.writeLock().unlock();
        }
    }

    public StoreStats stats() throws IOException {
        this.ensureOpen();
        return this.statsCache.getOrRefresh();
    }

    public void renameFile(String from, String to) throws IOException {
        this.ensureOpen();
        this.directory.renameFile(from, to);
    }

    @Override
    public final void incRef() {
        this.refCounter.incRef();
    }

    @Override
    public final boolean tryIncRef() {
        return this.refCounter.tryIncRef();
    }

    @Override
    public final void decRef() {
        this.refCounter.decRef();
    }

    @Override
    public void close() {
        if (this.isClosed.compareAndSet(false, true)) {
            this.decRef();
            this.logger.debug("store reference count on close: " + this.refCounter.refCount(), new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeInternal() {
        try {
            try {
                this.directory.innerClose();
            }
            finally {
                this.onClose.handle(this.shardLock);
            }
        }
        catch (IOException e) {
            try {
                this.logger.debug("failed to close directory", e, new Object[0]);
            }
            catch (Throwable throwable) {
                IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.shardLock});
                throw throwable;
            }
            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.shardLock});
        }
        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.shardLock});
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static MetadataSnapshot readMetadataSnapshot(Path indexLocation, ESLogger logger) throws IOException {
        try (SimpleFSDirectory dir2222 = new SimpleFSDirectory(indexLocation);){
            Store.failIfCorrupted((Directory)dir2222, new ShardId("", 1));
            MetadataSnapshot metadataSnapshot = new MetadataSnapshot(null, (Directory)dir2222, logger);
            return metadataSnapshot;
        }
        catch (IndexNotFoundException dir2222) {
            return MetadataSnapshot.EMPTY;
        }
        catch (FileNotFoundException | NoSuchFileException ex) {
            logger.info("Failed to open / find files while reading metadata snapshot", new Object[0]);
        }
        return MetadataSnapshot.EMPTY;
    }

    public static boolean canOpenIndex(ESLogger logger, Path indexLocation) throws IOException {
        try {
            Store.tryOpenIndex(indexLocation);
        }
        catch (Exception ex) {
            logger.trace("Can't open index for path [{}]", ex, indexLocation);
            return false;
        }
        return true;
    }

    public static void tryOpenIndex(Path indexLocation) throws IOException {
        try (SimpleFSDirectory dir = new SimpleFSDirectory(indexLocation);){
            Store.failIfCorrupted((Directory)dir, new ShardId("", 1));
            Lucene.readSegmentInfos((Directory)dir);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public IndexOutput createVerifyingOutput(String fileName, StoreFileMetaData metadata, IOContext context) throws IOException {
        IndexOutput output = this.directory().createOutput(fileName, context);
        boolean success = false;
        try {
            if (metadata.hasLegacyChecksum()) {
                this.logger.debug("create legacy adler32 output for {}", fileName);
                return new LegacyVerification.Adler32VerifyingIndexOutput(output, metadata.checksum(), metadata.length());
            }
            if (metadata.checksum() == null) {
                this.logger.debug("create legacy length-only output for {}", fileName);
                return new LegacyVerification.LengthVerifyingIndexOutput(output, metadata.length());
            }
            assert (metadata.writtenBy() != null);
            if ($assertionsDisabled) return new LuceneVerifyingIndexOutput(metadata, output);
            if (metadata.writtenBy().onOrAfter(Version.LUCENE_4_8)) return new LuceneVerifyingIndexOutput(metadata, output);
            throw new AssertionError();
        }
        catch (Throwable throwable) {
            if (success) throw throwable;
            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{output});
            throw throwable;
        }
    }

    public static void verify(IndexOutput output) throws IOException {
        if (output instanceof VerifyingIndexOutput) {
            ((VerifyingIndexOutput)output).verify();
        }
    }

    public IndexInput openVerifyingInput(String filename, IOContext context, StoreFileMetaData metadata) throws IOException {
        if (metadata.hasLegacyChecksum() || metadata.checksum() == null) {
            this.logger.debug("open legacy input for {}", filename);
            return this.directory().openInput(filename, context);
        }
        assert (metadata.writtenBy() != null);
        assert (metadata.writtenBy().onOrAfter(Version.LUCENE_4_8_0));
        return new VerifyingIndexInput(this.directory().openInput(filename, context));
    }

    public static void verify(IndexInput input) throws IOException {
        if (input instanceof VerifyingIndexInput) {
            ((VerifyingIndexInput)input).verify();
        }
    }

    public boolean checkIntegrityNoException(StoreFileMetaData md) {
        return Store.checkIntegrityNoException(md, this.directory());
    }

    public static boolean checkIntegrityNoException(StoreFileMetaData md, Directory directory) {
        try {
            Store.checkIntegrity(md, directory);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static void checkIntegrity(StoreFileMetaData md, Directory directory) throws IOException {
        try (IndexInput input = directory.openInput(md.name(), IOContext.READONCE);){
            Object checksum;
            if (input.length() != md.length()) {
                throw new CorruptIndexException("expected length=" + md.length() + " != actual length: " + input.length() + " : file truncated?", (DataInput)input);
            }
            if (md.writtenBy() != null && md.writtenBy().onOrAfter(Version.LUCENE_4_8_0)) {
                checksum = Store.digestToString(CodecUtil.checksumEntireFile((IndexInput)input));
                if (!((String)checksum).equals(md.checksum())) {
                    throw new CorruptIndexException("inconsistent metadata: lucene checksum=" + (String)checksum + ", metadata checksum=" + md.checksum(), (DataInput)input);
                }
            } else if (md.hasLegacyChecksum()) {
                int bytesToRead;
                checksum = new Adler32();
                byte[] buffer = new byte[md.length() > 4096L ? 4096 : (int)md.length()];
                long len = input.length();
                for (long read = 0L; len > read; read += (long)bytesToRead) {
                    long bytesLeft = len - read;
                    bytesToRead = bytesLeft < (long)buffer.length ? (int)bytesLeft : buffer.length;
                    input.readBytes(buffer, 0, bytesToRead, false);
                    checksum.update(buffer, 0, bytesToRead);
                }
                String adler32 = Store.digestToString(checksum.getValue());
                if (!adler32.equals(md.checksum())) {
                    throw new CorruptIndexException("checksum failed (hardware problem?) : expected=" + md.checksum() + " actual=" + adler32, (DataInput)input);
                }
            }
        }
    }

    public boolean isMarkedCorrupted() throws IOException {
        String[] files;
        this.ensureOpen();
        for (String file : files = this.directory().listAll()) {
            if (!file.startsWith(CORRUPTED)) continue;
            return true;
        }
        return false;
    }

    public void removeCorruptionMarker() throws IOException {
        String[] files;
        this.ensureOpen();
        Directory directory = this.directory();
        IOException firstException = null;
        for (String file : files = directory.listAll()) {
            if (!file.startsWith(CORRUPTED)) continue;
            try {
                directory.deleteFile(file);
            }
            catch (IOException ex) {
                if (firstException == null) {
                    firstException = ex;
                    continue;
                }
                firstException.addSuppressed(ex);
            }
        }
        if (firstException != null) {
            throw firstException;
        }
    }

    public void failIfCorrupted() throws IOException {
        this.ensureOpen();
        Store.failIfCorrupted((Directory)this.directory, this.shardId);
    }

    private static final void failIfCorrupted(Directory directory, ShardId shardId) throws IOException {
        String[] files = directory.listAll();
        ArrayList<CorruptIndexException> ex = new ArrayList<CorruptIndexException>();
        for (String file : files) {
            if (!file.startsWith(CORRUPTED)) continue;
            try (ChecksumIndexInput input = directory.openChecksumInput(file, IOContext.READONCE);){
                int version = CodecUtil.checkHeader((DataInput)input, (String)CODEC, (int)0, (int)2);
                if (version == 2) {
                    int size = input.readVInt();
                    byte[] buffer = new byte[size];
                    input.readBytes(buffer, 0, buffer.length);
                    StreamInput in = StreamInput.wrap(buffer);
                    Object t = in.readThrowable();
                    if (t instanceof CorruptIndexException) {
                        ex.add((CorruptIndexException)((Object)t));
                    } else {
                        ex.add(new CorruptIndexException(((Throwable)t).getMessage(), "preexisting_corruption", t));
                    }
                } else {
                    assert (version == 0 || version == 1);
                    String msg = input.readString();
                    StringBuilder builder = new StringBuilder(shardId.toString());
                    builder.append(" Preexisting corrupted index [");
                    builder.append(file).append("] caused by: ");
                    builder.append(msg);
                    if (version == 1) {
                        builder.append(System.lineSeparator());
                        builder.append(input.readString());
                    }
                    ex.add(new CorruptIndexException(builder.toString(), "preexisting_corruption"));
                }
                CodecUtil.checkFooter((ChecksumIndexInput)input);
            }
        }
        if (!ex.isEmpty()) {
            ExceptionsHelper.rethrowAndSuppress(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanupAndVerify(String reason, MetadataSnapshot sourceMetaData) throws IOException {
        this.metadataLock.writeLock().lock();
        try (Lock writeLock = this.directory.obtainLock("write.lock");){
            StoreDirectory dir = this.directory;
            for (String existingFile : dir.listAll()) {
                if (Store.isAutogenerated(existingFile) || sourceMetaData.contains(existingFile)) continue;
                try {
                    dir.deleteFile(reason, existingFile);
                }
                catch (IOException ex) {
                    if (existingFile.startsWith("segments") || existingFile.equals("segments.gen")) {
                        throw new IllegalStateException("Can't delete " + existingFile + " - cleanup failed", ex);
                    }
                    this.logger.debug("failed to delete file [{}]", ex, existingFile);
                }
            }
            MetadataSnapshot metadataOrEmpty = this.getMetadata();
            this.verifyAfterCleanup(sourceMetaData, metadataOrEmpty);
        }
        finally {
            this.metadataLock.writeLock().unlock();
        }
    }

    final void verifyAfterCleanup(MetadataSnapshot sourceMetaData, MetadataSnapshot targetMetaData) {
        RecoveryDiff recoveryDiff = targetMetaData.recoveryDiff(sourceMetaData);
        if (recoveryDiff.identical.size() != recoveryDiff.size()) {
            if (recoveryDiff.missing.isEmpty()) {
                for (StoreFileMetaData meta : recoveryDiff.different) {
                    StoreFileMetaData local = targetMetaData.get(meta.name());
                    StoreFileMetaData remote = sourceMetaData.get(meta.name());
                    boolean same = local.isSame(remote);
                    boolean hashAndLengthEqual = local.checksum() == null && remote.checksum() == null && local.hash().equals((Object)remote.hash()) && local.length() == remote.length();
                    boolean consistent = hashAndLengthEqual || same;
                    if (consistent) continue;
                    this.logger.debug("Files are different on the recovery target: {} ", recoveryDiff);
                    throw new IllegalStateException("local version: " + local + " is different from remote version after recovery: " + remote, null);
                }
            } else {
                this.logger.debug("Files are missing on the recovery target: {} ", recoveryDiff);
                throw new IllegalStateException("Files are missing on the recovery target: [different=" + recoveryDiff.different + ", missing=" + recoveryDiff.missing + ']', null);
            }
        }
    }

    public int refCount() {
        return this.refCounter.refCount();
    }

    public static boolean isChecksum(String name) {
        return name.startsWith(CHECKSUMS_PREFIX) || name.endsWith(".cks");
    }

    public static boolean isAutogenerated(String name) {
        return "write.lock".equals(name) || Store.isChecksum(name);
    }

    public static String digestToString(long digest) {
        return Long.toString(digest, 36);
    }

    public void deleteQuiet(String ... files) {
        for (String file : files) {
            try {
                this.directory().deleteFile(file);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
    }

    public void markStoreCorrupted(IOException exception) throws IOException {
        this.ensureOpen();
        if (!this.isMarkedCorrupted()) {
            String uuid = CORRUPTED + Strings.randomBase64UUID();
            try (IndexOutput output = this.directory().createOutput(uuid, IOContext.DEFAULT);){
                CodecUtil.writeHeader((DataOutput)output, (String)CODEC, (int)2);
                BytesStreamOutput out = new BytesStreamOutput();
                out.writeThrowable(exception);
                BytesReference bytes = out.bytes();
                output.writeVInt(bytes.length());
                output.writeBytes(bytes.array(), bytes.arrayOffset(), bytes.length());
                CodecUtil.writeFooter((IndexOutput)output);
            }
            catch (IOException ex) {
                this.logger.warn("Can't mark store as corrupted", ex, new Object[0]);
            }
            this.directory().sync(Collections.singleton(uuid));
        }
    }

    private static class StoreStatsCache
    extends SingleObjectCache<StoreStats> {
        private final Directory directory;
        private final DirectoryService directoryService;

        public StoreStatsCache(TimeValue refreshInterval, Directory directory, DirectoryService directoryService) throws IOException {
            super(refreshInterval, new StoreStats(StoreStatsCache.estimateSize(directory), directoryService.throttleTimeInNanos()));
            this.directory = directory;
            this.directoryService = directoryService;
        }

        @Override
        protected StoreStats refresh() {
            try {
                return new StoreStats(StoreStatsCache.estimateSize(this.directory), this.directoryService.throttleTimeInNanos());
            }
            catch (IOException ex) {
                throw new ElasticsearchException("failed to refresh store stats", (Throwable)ex, new Object[0]);
            }
        }

        private static long estimateSize(Directory directory) throws IOException {
            String[] files;
            long estimatedSize = 0L;
            for (String file : files = directory.listAll()) {
                try {
                    estimatedSize += directory.fileLength(file);
                }
                catch (FileNotFoundException | AccessDeniedException | NoSuchFileException iOException) {
                    // empty catch block
                }
            }
            return estimatedSize;
        }
    }

    public static interface OnClose
    extends Callback<ShardLock> {
        public static final OnClose EMPTY = new OnClose(){

            @Override
            public void handle(ShardLock Lock2) {
            }
        };
    }

    static class VerifyingIndexInput
    extends ChecksumIndexInput {
        private final IndexInput input;
        private final Checksum digest;
        private final long checksumPosition;
        private final byte[] checksum = new byte[8];
        private long verifiedPosition = 0L;

        public VerifyingIndexInput(IndexInput input) {
            this(input, (Checksum)new BufferedChecksum((Checksum)new CRC32()));
        }

        public VerifyingIndexInput(IndexInput input, Checksum digest) {
            super("VerifyingIndexInput(" + input + ")");
            this.input = input;
            this.digest = digest;
            this.checksumPosition = input.length() - 8L;
        }

        public byte readByte() throws IOException {
            long pos = this.input.getFilePointer();
            byte b = this.input.readByte();
            if (++pos > this.verifiedPosition) {
                if (pos <= this.checksumPosition) {
                    this.digest.update(b);
                } else {
                    this.checksum[(int)(pos - this.checksumPosition - 1L)] = b;
                }
                this.verifiedPosition = pos;
            }
            return b;
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            long pos = this.input.getFilePointer();
            this.input.readBytes(b, offset, len);
            if (pos + (long)len > this.verifiedPosition) {
                int alreadyVerified = (int)Math.max(0L, this.verifiedPosition - pos);
                if (pos < this.checksumPosition) {
                    if (pos + (long)len < this.checksumPosition) {
                        this.digest.update(b, offset + alreadyVerified, len - alreadyVerified);
                    } else {
                        int checksumOffset = (int)(this.checksumPosition - pos);
                        if (checksumOffset - alreadyVerified > 0) {
                            this.digest.update(b, offset + alreadyVerified, checksumOffset - alreadyVerified);
                        }
                        System.arraycopy(b, offset + checksumOffset, this.checksum, 0, len - checksumOffset);
                    }
                } else {
                    assert (pos - this.checksumPosition < 8L);
                    System.arraycopy(b, offset, this.checksum, (int)(pos - this.checksumPosition), len);
                }
                this.verifiedPosition = pos + (long)len;
            }
        }

        public long getChecksum() {
            return this.digest.getValue();
        }

        public void seek(long pos) throws IOException {
            if (pos < this.verifiedPosition) {
                this.input.seek(pos);
            } else if (this.verifiedPosition > this.getFilePointer()) {
                this.input.seek(this.verifiedPosition);
                this.skipBytes(pos - this.verifiedPosition);
            } else {
                this.skipBytes(pos - this.getFilePointer());
            }
        }

        public void close() throws IOException {
            this.input.close();
        }

        public long getFilePointer() {
            return this.input.getFilePointer();
        }

        public long length() {
            return this.input.length();
        }

        public IndexInput clone() {
            throw new UnsupportedOperationException();
        }

        public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
            throw new UnsupportedOperationException();
        }

        public long getStoredChecksum() {
            return new ByteArrayDataInput(this.checksum).readLong();
        }

        public long verify() throws CorruptIndexException {
            long storedChecksum = this.getStoredChecksum();
            if (this.getChecksum() == storedChecksum) {
                return storedChecksum;
            }
            throw new CorruptIndexException("verification failed : calculated=" + Store.digestToString(this.getChecksum()) + " stored=" + Store.digestToString(storedChecksum), (DataInput)this);
        }
    }

    static class LuceneVerifyingIndexOutput
    extends VerifyingIndexOutput {
        private final StoreFileMetaData metadata;
        private long writtenBytes;
        private final long checksumPosition;
        private String actualChecksum;
        private final byte[] footerChecksum = new byte[8];

        LuceneVerifyingIndexOutput(StoreFileMetaData metadata, IndexOutput out) {
            super(out);
            this.metadata = metadata;
            this.checksumPosition = metadata.length() - 8L;
        }

        @Override
        public void verify() throws IOException {
            String footerDigest = null;
            if (this.metadata.checksum().equals(this.actualChecksum) && this.writtenBytes == this.metadata.length()) {
                ByteArrayIndexInput indexInput = new ByteArrayIndexInput("checksum", this.footerChecksum);
                footerDigest = Store.digestToString(indexInput.readLong());
                if (this.metadata.checksum().equals(footerDigest)) {
                    return;
                }
            }
            throw new CorruptIndexException("verification failed (hardware problem?) : expected=" + this.metadata.checksum() + " actual=" + this.actualChecksum + " footer=" + footerDigest + " writtenLength=" + this.writtenBytes + " expectedLength=" + this.metadata.length() + " (resource=" + this.metadata.toString() + ")", "VerifyingIndexOutput(" + this.metadata.name() + ")");
        }

        @Override
        public void writeByte(byte b) throws IOException {
            long writtenBytes;
            if ((writtenBytes = this.writtenBytes++) >= this.checksumPosition) {
                long indexLong;
                if (writtenBytes == this.checksumPosition) {
                    this.readAndCompareChecksum();
                }
                if ((long)((int)(indexLong = writtenBytes - this.checksumPosition)) != indexLong) {
                    throw new ArithmeticException("integer overflow");
                }
                int index = (int)indexLong;
                if (index < this.footerChecksum.length) {
                    this.footerChecksum[index] = b;
                    if (index == this.footerChecksum.length - 1) {
                        this.verify();
                    }
                } else {
                    this.verify();
                    throw new AssertionError((Object)("write past EOF expected length: " + this.metadata.length() + " writtenBytes: " + writtenBytes));
                }
            }
            this.out.writeByte(b);
        }

        private void readAndCompareChecksum() throws IOException {
            this.actualChecksum = Store.digestToString(this.getChecksum());
            if (!this.metadata.checksum().equals(this.actualChecksum)) {
                throw new CorruptIndexException("checksum failed (hardware problem?) : expected=" + this.metadata.checksum() + " actual=" + this.actualChecksum + " (resource=" + this.metadata.toString() + ")", "VerifyingIndexOutput(" + this.metadata.name() + ")");
            }
        }

        @Override
        public void writeBytes(byte[] b, int offset, int length) throws IOException {
            if (this.writtenBytes + (long)length > this.checksumPosition) {
                for (int i = 0; i < length; ++i) {
                    this.writeByte(b[offset + i]);
                }
            } else {
                this.out.writeBytes(b, offset, length);
                this.writtenBytes += (long)length;
            }
        }
    }

    public static final class LegacyChecksums {
        private final Map<String, String> legacyChecksums = new HashMap<String, String>();

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void add(StoreFileMetaData metaData) throws IOException {
            if (metaData.hasLegacyChecksum()) {
                LegacyChecksums legacyChecksums = this;
                synchronized (legacyChecksums) {
                    this.legacyChecksums.put(metaData.name(), metaData.checksum());
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void write(Store store) throws IOException {
            StoreDirectory storeDirectory = store.directory;
            synchronized (storeDirectory) {
                Tuple<Map<String, String>, Long> tuple = MetadataSnapshot.readLegacyChecksums((Directory)store.directory);
                tuple.v1().putAll(this.legacyChecksums);
                if (!tuple.v1().isEmpty()) {
                    this.writeChecksums((Directory)store.directory, tuple.v1(), tuple.v2());
                }
            }
        }

        synchronized void writeChecksums(Directory directory, Map<String, String> checksums, long lastVersion) throws IOException {
            long nextVersion = Math.max(lastVersion + 1L, System.currentTimeMillis());
            String checksumName = Store.CHECKSUMS_PREFIX + nextVersion;
            try (IndexOutput output = directory.createOutput(checksumName, IOContext.DEFAULT);){
                output.writeInt(0);
                output.writeStringStringMap(checksums);
            }
            directory.sync(Collections.singleton(checksumName));
            MetadataSnapshot.cleanLegacyChecksums(directory, nextVersion);
        }

        public void clear() {
            this.legacyChecksums.clear();
        }

        public void remove(String name) {
            this.legacyChecksums.remove(name);
        }
    }

    public static final class RecoveryDiff {
        public final List<StoreFileMetaData> identical;
        public final List<StoreFileMetaData> different;
        public final List<StoreFileMetaData> missing;

        RecoveryDiff(List<StoreFileMetaData> identical, List<StoreFileMetaData> different, List<StoreFileMetaData> missing) {
            this.identical = identical;
            this.different = different;
            this.missing = missing;
        }

        public int size() {
            return this.identical.size() + this.different.size() + this.missing.size();
        }

        public String toString() {
            return "RecoveryDiff{identical=" + this.identical + ", different=" + this.different + ", missing=" + this.missing + '}';
        }
    }

    public static final class MetadataSnapshot
    implements Iterable<StoreFileMetaData>,
    Writeable<MetadataSnapshot> {
        private static final ESLogger logger = Loggers.getLogger(MetadataSnapshot.class);
        private static final Version FIRST_LUCENE_CHECKSUM_VERSION = Version.LUCENE_4_8;
        private final ImmutableMap<String, StoreFileMetaData> metadata;
        public static final MetadataSnapshot EMPTY = new MetadataSnapshot();
        private final ImmutableMap<String, String> commitUserData;
        private final long numDocs;
        private static final String DEL_FILE_EXTENSION = "del";
        private static final String LIV_FILE_EXTENSION = "liv";
        private static final String FIELD_INFOS_FILE_EXTENSION = "fnm";
        private static final String SEGMENT_INFO_EXTENSION = "si";

        public MetadataSnapshot(Map<String, StoreFileMetaData> metadata, Map<String, String> commitUserData, long numDocs) {
            ImmutableMap.Builder metaDataBuilder = ImmutableMap.builder();
            this.metadata = metaDataBuilder.putAll(metadata).build();
            ImmutableMap.Builder commitUserDataBuilder = ImmutableMap.builder();
            this.commitUserData = commitUserDataBuilder.putAll(commitUserData).build();
            this.numDocs = numDocs;
        }

        MetadataSnapshot() {
            this.metadata = ImmutableMap.of();
            this.commitUserData = ImmutableMap.of();
            this.numDocs = 0L;
        }

        MetadataSnapshot(IndexCommit commit, Directory directory, ESLogger logger) throws IOException {
            LoadedMetadata loadedMetadata = MetadataSnapshot.loadMetadata(commit, directory, logger);
            this.metadata = loadedMetadata.fileMetadata;
            this.commitUserData = loadedMetadata.userData;
            this.numDocs = loadedMetadata.numDocs;
            assert (this.metadata.isEmpty() || this.numSegmentFiles() == 1) : "numSegmentFiles: " + this.numSegmentFiles();
        }

        public MetadataSnapshot(StreamInput in) throws IOException {
            int num;
            int size = in.readVInt();
            ImmutableMap.Builder metadataBuilder = ImmutableMap.builder();
            for (int i = 0; i < size; ++i) {
                StoreFileMetaData meta = StoreFileMetaData.readStoreFileMetaData(in);
                metadataBuilder.put((Object)meta.name(), (Object)meta);
            }
            ImmutableMap.Builder commitUserDataBuilder = ImmutableMap.builder();
            for (int i = num = in.readVInt(); i > 0; --i) {
                commitUserDataBuilder.put((Object)in.readString(), (Object)in.readString());
            }
            this.commitUserData = commitUserDataBuilder.build();
            this.metadata = metadataBuilder.build();
            this.numDocs = in.readLong();
            assert (this.metadata.isEmpty() || this.numSegmentFiles() == 1) : "numSegmentFiles: " + this.numSegmentFiles();
        }

        public long getNumDocs() {
            return this.numDocs;
        }

        static LoadedMetadata loadMetadata(IndexCommit commit, Directory directory, ESLogger logger) throws IOException {
            long numDocs;
            ImmutableMap.Builder commitUserDataBuilder;
            ImmutableMap.Builder builder;
            block24: {
                builder = ImmutableMap.builder();
                Map<String, String> checksumMap = MetadataSnapshot.readLegacyChecksums(directory).v1();
                commitUserDataBuilder = ImmutableMap.builder();
                try {
                    long length;
                    SegmentInfos segmentCommitInfos = Store.readSegmentsInfo(commit, directory);
                    numDocs = Lucene.getNumDocs(segmentCommitInfos);
                    commitUserDataBuilder.putAll(segmentCommitInfos.getUserData());
                    Version maxVersion = Version.LUCENE_4_0;
                    for (SegmentCommitInfo info : segmentCommitInfos) {
                        Version version = info.info.getVersion();
                        if (version == null) {
                            throw new IllegalArgumentException("expected valid version value: " + info.info.toString());
                        }
                        if (version.onOrAfter(maxVersion)) {
                            maxVersion = version;
                        }
                        for (String file : info.files()) {
                            String legacyChecksum = checksumMap.get(file);
                            if (version.onOrAfter(FIRST_LUCENE_CHECKSUM_VERSION)) {
                                MetadataSnapshot.checksumFromLuceneFile(directory, file, (ImmutableMap.Builder<String, StoreFileMetaData>)builder, logger, version, SEGMENT_INFO_EXTENSION.equals(IndexFileNames.getExtension((String)file)));
                                continue;
                            }
                            builder.put((Object)file, (Object)new StoreFileMetaData(file, directory.fileLength(file), legacyChecksum, version));
                        }
                    }
                    String segmentsFile = segmentCommitInfos.getSegmentsFileName();
                    String legacyChecksum = checksumMap.get(segmentsFile);
                    if (maxVersion.onOrAfter(FIRST_LUCENE_CHECKSUM_VERSION)) {
                        MetadataSnapshot.checksumFromLuceneFile(directory, segmentsFile, (ImmutableMap.Builder<String, StoreFileMetaData>)builder, logger, maxVersion, true);
                        break block24;
                    }
                    BytesRefBuilder fileHash = new BytesRefBuilder();
                    try (IndexInput in = directory.openInput(segmentsFile, IOContext.READONCE);){
                        length = in.length();
                        MetadataSnapshot.hashFile(fileHash, new InputStreamIndexInput(in, length), length);
                    }
                    builder.put((Object)segmentsFile, (Object)new StoreFileMetaData(segmentsFile, length, legacyChecksum, maxVersion, fileHash.get()));
                }
                catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException | IndexNotFoundException ex) {
                    throw ex;
                }
                catch (Throwable ex) {
                    try {
                        logger.warn("failed to build store metadata. checking segment info integrity (with commit [{}])", ex, commit == null ? "no" : "yes");
                        Lucene.checkSegmentInfoIntegrity(directory);
                    }
                    catch (CorruptIndexException | IndexFormatTooNewException | IndexFormatTooOldException cex) {
                        cex.addSuppressed(ex);
                        throw cex;
                    }
                    catch (Throwable throwable) {
                        // empty catch block
                    }
                    throw ex;
                }
            }
            return new LoadedMetadata((ImmutableMap<String, StoreFileMetaData>)builder.build(), (ImmutableMap<String, String>)commitUserDataBuilder.build(), numDocs);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        static Tuple<Map<String, String>, Long> readLegacyChecksums(Directory directory) throws IOException {
            Directory directory2 = directory;
            synchronized (directory2) {
                long lastFound = -1L;
                for (String name : directory.listAll()) {
                    long current;
                    if (!Store.isChecksum(name) || (current = Long.parseLong(name.substring(Store.CHECKSUMS_PREFIX.length()))) <= lastFound) continue;
                    lastFound = current;
                }
                if (lastFound <= -1L) {
                    return new Tuple<Map<String, String>, Long>(new HashMap(), -1L);
                }
                try (IndexInput indexInput = directory.openInput(Store.CHECKSUMS_PREFIX + lastFound, IOContext.READONCE);){
                    indexInput.readInt();
                    Tuple<Map<String, String>, Long> tuple = new Tuple<Map<String, String>, Long>(indexInput.readStringStringMap(), lastFound);
                    return tuple;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static void cleanLegacyChecksums(Directory directory, long newVersion) throws IOException {
            Directory directory2 = directory;
            synchronized (directory2) {
                for (String name : directory.listAll()) {
                    long current;
                    if (!Store.isChecksum(name) || (current = Long.parseLong(name.substring(Store.CHECKSUMS_PREFIX.length()))) >= newVersion) continue;
                    try {
                        directory.deleteFile(name);
                    }
                    catch (IOException ex) {
                        logger.debug("can't delete old checksum file [{}]", ex, name);
                    }
                }
            }
        }

        private static void checksumFromLuceneFile(Directory directory, String file, ImmutableMap.Builder<String, StoreFileMetaData> builder, ESLogger logger, Version version, boolean readFileAsHash) throws IOException {
            BytesRefBuilder fileHash = new BytesRefBuilder();
            try (IndexInput in = directory.openInput(file, IOContext.READONCE);){
                String checksum;
                long length;
                try {
                    length = in.length();
                    if (length < (long)CodecUtil.footerLength()) {
                        throw new CorruptIndexException("Can't retrieve checksum from file: " + file + " file length must be >= " + CodecUtil.footerLength() + " but was: " + in.length(), (DataInput)in);
                    }
                    if (readFileAsHash) {
                        VerifyingIndexInput verifyingIndexInput = new VerifyingIndexInput(in);
                        MetadataSnapshot.hashFile(fileHash, new InputStreamIndexInput((IndexInput)verifyingIndexInput, length), length);
                        checksum = Store.digestToString(verifyingIndexInput.verify());
                    } else {
                        checksum = Store.digestToString(CodecUtil.retrieveChecksum((IndexInput)in));
                    }
                }
                catch (Throwable ex) {
                    logger.debug("Can retrieve checksum from file [{}]", ex, file);
                    throw ex;
                }
                builder.put((Object)file, (Object)new StoreFileMetaData(file, length, checksum, version, fileHash.get()));
            }
        }

        public static BytesRef hashFile(Directory directory, String file) throws IOException {
            BytesRefBuilder fileHash = new BytesRefBuilder();
            try (IndexInput in = directory.openInput(file, IOContext.READONCE);){
                MetadataSnapshot.hashFile(fileHash, new InputStreamIndexInput(in, in.length()), in.length());
            }
            return fileHash.get();
        }

        public static void hashFile(BytesRefBuilder fileHash, InputStream in, long size) throws IOException {
            int len = (int)Math.min(0x100000L, size);
            fileHash.grow(len);
            fileHash.setLength(len);
            int readBytes = Streams.readFully(in, fileHash.bytes(), 0, len);
            assert (readBytes == len) : Integer.toString(readBytes) + " != " + Integer.toString(len);
            assert (fileHash.length() == len) : Integer.toString(fileHash.length()) + " != " + Integer.toString(len);
        }

        @Override
        public Iterator<StoreFileMetaData> iterator() {
            return this.metadata.values().iterator();
        }

        public StoreFileMetaData get(String name) {
            return (StoreFileMetaData)this.metadata.get((Object)name);
        }

        public Map<String, StoreFileMetaData> asMap() {
            return this.metadata;
        }

        public RecoveryDiff recoveryDiff(MetadataSnapshot recoveryTargetSnapshot) {
            ArrayList identical = new ArrayList();
            ArrayList<StoreFileMetaData> different = new ArrayList<StoreFileMetaData>();
            ArrayList<StoreFileMetaData> missing = new ArrayList<StoreFileMetaData>();
            HashMap<String, ArrayList<StoreFileMetaData>> perSegment = new HashMap<String, ArrayList<StoreFileMetaData>>();
            ArrayList<StoreFileMetaData> perCommitStoreFiles = new ArrayList<StoreFileMetaData>();
            for (StoreFileMetaData meta : this) {
                if ("segments.gen".equals(meta.name())) continue;
                String segmentId = IndexFileNames.parseSegmentName((String)meta.name());
                String extension = IndexFileNames.getExtension((String)meta.name());
                assert (!FIELD_INFOS_FILE_EXTENSION.equals(extension) || IndexFileNames.stripExtension((String)IndexFileNames.stripSegmentName((String)meta.name())).isEmpty()) : "FieldInfos are generational but updateable DV are not supported in elasticsearch";
                if ("segments".equals(segmentId) || DEL_FILE_EXTENSION.equals(extension) || LIV_FILE_EXTENSION.equals(extension)) {
                    perCommitStoreFiles.add(meta);
                    continue;
                }
                ArrayList<StoreFileMetaData> perSegStoreFiles = (ArrayList<StoreFileMetaData>)perSegment.get(segmentId);
                if (perSegStoreFiles == null) {
                    perSegStoreFiles = new ArrayList<StoreFileMetaData>();
                    perSegment.put(segmentId, perSegStoreFiles);
                }
                perSegStoreFiles.add(meta);
            }
            ArrayList<StoreFileMetaData> identicalFiles = new ArrayList<StoreFileMetaData>();
            for (List segmentFiles : Iterables.concat(perSegment.values(), Collections.singleton(perCommitStoreFiles))) {
                identicalFiles.clear();
                boolean consistent = true;
                for (StoreFileMetaData meta : segmentFiles) {
                    StoreFileMetaData storeFileMetaData = recoveryTargetSnapshot.get(meta.name());
                    if (storeFileMetaData == null) {
                        consistent = false;
                        missing.add(meta);
                        continue;
                    }
                    if (!storeFileMetaData.isSame(meta)) {
                        consistent = false;
                        different.add(meta);
                        continue;
                    }
                    identicalFiles.add(meta);
                }
                if (consistent) {
                    identical.addAll(identicalFiles);
                    continue;
                }
                different.addAll(identicalFiles);
            }
            RecoveryDiff recoveryDiff = new RecoveryDiff(Collections.unmodifiableList(identical), Collections.unmodifiableList(different), Collections.unmodifiableList(missing));
            assert (recoveryDiff.size() == this.metadata.size() - (this.metadata.containsKey((Object)"segments.gen") ? 1 : 0)) : "some files are missing recoveryDiff size: [" + recoveryDiff.size() + "] metadata size: [" + this.metadata.size() + "] contains  segments.gen: [" + this.metadata.containsKey((Object)"segments.gen") + "]";
            return recoveryDiff;
        }

        public int size() {
            return this.metadata.size();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeVInt(this.metadata.size());
            for (StoreFileMetaData meta : this) {
                meta.writeTo(out);
            }
            out.writeVInt(this.commitUserData.size());
            for (Map.Entry entry : this.commitUserData.entrySet()) {
                out.writeString((String)entry.getKey());
                out.writeString((String)entry.getValue());
            }
            out.writeLong(this.numDocs);
        }

        public Map<String, String> getCommitUserData() {
            return this.commitUserData;
        }

        public boolean contains(String existingFile) {
            return this.metadata.containsKey((Object)existingFile);
        }

        public StoreFileMetaData getSegmentsFile() {
            for (StoreFileMetaData file : this) {
                if (!file.name().startsWith("segments")) continue;
                return file;
            }
            assert (this.metadata.isEmpty());
            return null;
        }

        private final int numSegmentFiles() {
            int count = 0;
            for (StoreFileMetaData file : this) {
                if (!file.name().startsWith("segments")) continue;
                ++count;
            }
            return count;
        }

        public String getSyncId() {
            return (String)this.commitUserData.get((Object)"sync_id");
        }

        @Override
        public MetadataSnapshot readFrom(StreamInput in) throws IOException {
            return new MetadataSnapshot(in);
        }

        static class LoadedMetadata {
            final ImmutableMap<String, StoreFileMetaData> fileMetadata;
            final ImmutableMap<String, String> userData;
            final long numDocs;

            LoadedMetadata(ImmutableMap<String, StoreFileMetaData> fileMetadata, ImmutableMap<String, String> userData, long numDocs) {
                this.fileMetadata = fileMetadata;
                this.userData = userData;
                this.numDocs = numDocs;
            }
        }
    }

    private static final class StoreDirectory
    extends FilterDirectory {
        private final ESLogger deletesLogger;

        StoreDirectory(Directory delegateDirectory, ESLogger deletesLogger) throws IOException {
            super(delegateDirectory);
            this.deletesLogger = deletesLogger;
        }

        public void close() throws IOException {
            assert (false) : "Nobody should close this directory except of the Store itself";
        }

        public void deleteFile(String msg, String name) throws IOException {
            this.deletesLogger.trace("{}: delete file {}", msg, name);
            super.deleteFile(name);
        }

        public void deleteFile(String name) throws IOException {
            this.deleteFile("StoreDirectory.deleteFile", name);
        }

        private void innerClose() throws IOException {
            super.close();
        }

        public String toString() {
            return "store(" + this.in.toString() + ")";
        }
    }
}

