import sys
import re
import os
import json
import xbmc, xbmcgui, xbmcaddon, xbmcplugin
from xbmcgui import ListItem
from xbmcvfs import translatePath
from routing import Plugin

from urllib.parse import urlparse, urlencode, urljoin
from datetime import datetime, timedelta
from dateutil.parser import parse
from dateutil.tz import gettz, tzlocal

from resources.lib.lazymlb import LazyMLB
from resources.lib.lazynhl import LazyNHL

addon = xbmcaddon.Addon()
plugin = Plugin()
plugin.name = addon.getAddonInfo("name")


USER_DATA_DIR = translatePath(addon.getAddonInfo("profile"))
ADDON_DATA_DIR = translatePath(addon.getAddonInfo("path"))
image = os.path.join(ADDON_DATA_DIR, "resources", "icon.png")
if not os.path.exists(USER_DATA_DIR):
    os.makedirs(USER_DATA_DIR)

config = {}
if addon.getSettingBool("http_proxy"):
    proxy_user = addon.getSetting("proxy_user")
    proxy_pass = addon.getSetting("proxy_pass")
    proxy_url = addon.getSetting("proxy_url")
    if proxy_user and proxy_pass:
        p_url = proxy_url.split("//")
        config["http_proxy"] = "//".join([p_url[0], "{0}:{1}@{2}".format(proxy_user, proxy_pass, p_url[1])])
    else:
        config["http_proxy"] = proxy_url

api_tz = gettz("America/Los_Angeles")
lmm = LazyMLB(USER_DATA_DIR, config)
lmn = LazyNHL(USER_DATA_DIR, config)

try:
    locale_timezone = json.loads(
        xbmc.executeJSONRPC(
            '{"jsonrpc": "2.0", "method": "Settings.GetSettingValue", "params": {"setting": "locale.timezone"}, "id": 1}'
        )
    )
    if "result" in locale_timezone:
        if locale_timezone["result"]["value"]:
            local_tzinfo = gettz(locale_timezone["result"]["value"])
        else:
            local_tzinfo = tzlocal()
    else:
        local_tzinfo = tzlocal()
except:
    local_tzinfo = ""

local_format = xbmc.getRegion("time").replace(":%S", "")

player_headers = {
    "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/103.0.0.0 Safari/537.36",
}


def replace_playlist(url, playlist):
    if "master" in url:
        _url = urlparse(url)
        return urlparse(urljoin(_url._replace(scheme="").geturl(), playlist))._replace(scheme=_url.scheme).geturl()
    else:
        return url


@plugin.route("/")
def root():
    list_items = []

    li = ListItem("MLB", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(mlb)
    list_items.append((url, li, True))

    li = ListItem("NHL", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(nhl)
    list_items.append((url, li, True))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/mlb")
def mlb():
    list_items = []

    li = ListItem("MLB Today", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(mlb_today)
    list_items.append((url, li, True))

    li = ListItem("MLB Yesterday", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(mlb_yday)
    list_items.append((url, li, True))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/mlb_today")
def mlb_today():
    date = datetime.now(api_tz)
    plugin.args["date"] = [date.strftime("%Y-%m-%d")]
    return mlb_date()


@plugin.route("/mlb_yday")
def mlb_yday():
    date = datetime.now(api_tz) - timedelta(days=1)
    plugin.args["date"] = [date.strftime("%Y-%m-%d")]
    return mlb_date()


@plugin.route("/mlb_date")
def mlb_date():
    date = plugin.args["date"][0]
    list_items = []

    li_date = (parse(date) - timedelta(days=1)).strftime("%Y-%m-%d")
    li = ListItem(f"<< Previous Day {li_date}", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(mlb_date, date=li_date)
    list_items.append((url, li, True))

    for game in lmm.get_games_date(date):
        time = parse(game.game_date).astimezone(local_tzinfo).strftime(local_format)
        title = f"[{time}] {game.away_team} @ {game.home_team}"
        li = ListItem(title, offscreen=True)
        li.setArt({"thumb": image, "icon": image})
        url = plugin.url_for(mlb_game, game_pk=game.game_pk)
        list_items.append((url, li, True))

    li_date = (parse(date) + timedelta(days=1)).strftime("%Y-%m-%d")
    li = ListItem(f">> Next Day {li_date}", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(mlb_date, date=li_date)
    list_items.append((url, li, True))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.setContent(plugin.handle, "videos")
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/mlb_game")
def mlb_game():
    game_pk = int(plugin.args["game_pk"][0])
    list_items = []
    game = lmm.get_game(game_pk)
    content = lmm.get_game_content(game_pk)

    for epg in content["media"]["epg"]:
        for media in epg["items"]:
            print(media)
            if epg["title"] == "MLBTV":
                if media["mediaState"] == "MEDIA_ARCHIVE":
                    state = "vod"
                else:
                    state = "live"
                _type = media.get("mediaFeedType", media.get("mediaFeedSubType"))
                _desc = media.get("description", media.get("callLetters"))
                title = f"{game.away_team} @ {game.home_team}"
                li_title = f'{epg["title"]} {_type} {_desc} {media["id"]}'
                li = ListItem(li_title, offscreen=True)
                li.setArt({"thumb": image, "icon": image})
                li.setProperty("IsPlayable", "true")
                li.setInfo(type="Video", infoLabels={"Title": title, "mediatype": "video"})
                url = plugin.url_for(
                    mlb_play_media,
                    state=state,
                    media_id=media["id"],
                    date=game.official_date,
                    game_date=game.game_date,
                    feed=_type,
                    team=game.home_team,
                )
                list_items.append((url, li, False))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.setContent(plugin.handle, "videos")
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/mlb_play_media")
def mlb_play_media():
    state = plugin.args["state"][0]
    media_id = int(plugin.args["media_id"][0])
    date = plugin.args["date"][0]
    game_date = plugin.args["game_date"][0]
    feed = plugin.args["feed"][0]
    team = plugin.args["team"][0]
    path = "/lazy.m3u8"
    hls_params = {}
    _time = parse(game_date).strftime("%Y-%m-%d %H:%M:%S")
    media_roy = lmm.get_media_roy(_time, team, feed)
    if media_roy:
        hls_params["__hls_origin_url"] = media_roy[0]
        hls_params["__hls_replace_key"] = json.dumps(media_roy[1])
    else:
        media = lmm.get_media(date, media_id)
        re_exp = re.search("exp=(\d+)", media)
        if re_exp:
            exp = int(re_exp.group(1))
            if datetime.utcnow() > datetime.fromtimestamp(exp):
                dialog = xbmcgui.Dialog()
                dialog.notification(plugin.name, "Link Expired", xbmcgui.NOTIFICATION_ERROR, 3000)
                xbmcplugin.setResolvedUrl(plugin.handle, False, ListItem())
                return
        else:
            dialog = xbmcgui.Dialog()
            dialog.notification(plugin.name, media, xbmcgui.NOTIFICATION_ERROR, 3000)
            xbmcplugin.setResolvedUrl(plugin.handle, False, ListItem())
            return
        hls_params["__hls_origin_url"] = media
        hls_params["__hls_lazy_server"] = lmm.key_server
    if "http_proxy" in config:
        path = "/lazy_proxy.m3u8"
        hls_params["__hls_http_proxy"] = config["http_proxy"]
    if addon.getSettingInt("mlb_stream_type") == 0:
        if state == "live":
            hls_params["__hls_bitrate"] = addon.getSettingInt("mlb_bitrate_live")
        else:
            hls_params["__hls_bitrate"] = addon.getSettingInt("mlb_bitrate_vod")

    if state == "live":
        if addon.getSettingInt("mlb_playlist_type") == 0:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_complete.m3u8"
            )
        elif addon.getSettingInt("mlb_playlist_type") == 1:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_complete_gdfp.m3u8"
            )
        elif addon.getSettingInt("mlb_playlist_type") == 2:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_slide.m3u8"
            )
        elif addon.getSettingInt("mlb_playlist_type") == 3:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_slide_gdfp.m3u8"
            )
    else:
        if addon.getSettingInt("mlb_playlist_type_vod") == 0:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_complete-trimmed.m3u8"
            )
        elif addon.getSettingInt("mlb_playlist_type_vod") == 1:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_complete_gdfp.m3u8"
            )
        elif addon.getSettingInt("mlb_playlist_type_vod") == 10:
            hls_params["__hls_origin_url"] = replace_playlist(
                hls_params["__hls_origin_url"], "master_desktop_complete.m3u8"
            )
    hls_proxy = (
        urlparse(addon.getSetting("hls_proxy"))
        ._replace(
            path=path,
            query=urlencode(hls_params),
        )
        .geturl()
    )
    li = ListItem(path="{0}|{1}".format(hls_proxy, urlencode(player_headers)))
    li.setContentLookup(False)
    if addon.getSettingInt("mlb_stream_type") == 1:
        li.setProperty("inputstream", "inputstream.adaptive")
        li.setProperty("inputstream.adaptive.manifest_type", "hls")
    xbmcplugin.setResolvedUrl(plugin.handle, True, li)


@plugin.route("/nhl")
def nhl():
    list_items = []

    li = ListItem("NHL Today", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(nhl_today)
    list_items.append((url, li, True))

    li = ListItem("NHL Yesterday", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(nhl_yday)
    list_items.append((url, li, True))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/nhl_today")
def nhl_today():
    date = datetime.now(api_tz)
    plugin.args["date"] = [date.strftime("%Y-%m-%d")]
    return nhl_date()


@plugin.route("/nhl_yday")
def nhl_yday():
    date = datetime.now(api_tz) - timedelta(days=1)
    plugin.args["date"] = [date.strftime("%Y-%m-%d")]
    return nhl_date()


@plugin.route("/nhl_date")
def nhl_date():
    date = plugin.args["date"][0]
    list_items = []

    li_date = (parse(date) - timedelta(days=1)).strftime("%Y-%m-%d")
    li = ListItem(f"<< Previous Day {li_date}", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(nhl_date, date=li_date)
    list_items.append((url, li, True))

    for game in lmn.get_games_date(date):
        time = parse(game.game_date).astimezone(local_tzinfo).strftime(local_format)
        title = f"[{time}] {game.away_team} @ {game.home_team}"
        li = ListItem(title, offscreen=True)
        li.setArt({"thumb": image, "icon": image})
        url = plugin.url_for(nhl_game, game_pk=game.game_pk)
        list_items.append((url, li, True))

    li_date = (parse(date) + timedelta(days=1)).strftime("%Y-%m-%d")
    li = ListItem(f">> Next Day {li_date}", offscreen=True)
    li.setArt({"thumb": image, "icon": image})
    url = plugin.url_for(nhl_date, date=li_date)
    list_items.append((url, li, True))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.setContent(plugin.handle, "videos")
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/nhl_game")
def nhl_game():
    game_pk = int(plugin.args["game_pk"][0])
    list_items = []
    game = lmn.get_game(game_pk)
    content = lmn.get_game_content(game_pk)

    for epg in content["media"]["epg"]:
        if epg["title"] == "NHLTV":
            for media in epg["items"]:
                if media["mediaState"] == "MEDIA_ARCHIVE":
                    state = "vod"
                else:
                    state = "live"
                _type = media.get("mediaFeedType", media.get("mediaFeedSubType"))
                _desc = media.get("description", media.get("callLetters"))
                title = f"{game.away_team} @ {game.home_team}"
                li_title = f'{title} {_type} {_desc} {media["mediaPlaybackId"]}'
                li = ListItem(li_title, offscreen=True)
                li.setProperty("IsPlayable", "true")
                li.setArt({"thumb": image, "icon": image})
                li.setInfo(type="Video", infoLabels={"Title": title, "mediatype": "video"})
                url = plugin.url_for(
                    nhl_play_media, state=state, media_id=media["mediaPlaybackId"], date=game.official_date
                )
                list_items.append((url, li, False))

    xbmcplugin.addDirectoryItems(plugin.handle, list_items)
    xbmcplugin.setContent(plugin.handle, "videos")
    xbmcplugin.endOfDirectory(plugin.handle)


@plugin.route("/nhl_play_media")
def nhl_play_media():
    state = plugin.args["state"][0]
    media_id = int(plugin.args["media_id"][0])
    date = plugin.args["date"][0]
    path = "/lazy.m3u8"
    hls_params = {}
    try:
        media66 = lmn.get_media66(media_id)
    except Exception:
        media66 = ""
    if media66:
        hls_params["__hls_origin_url"] = media66[0]
        hls_params["__hls_keys"] = json.dumps(media66[1])
    else:
        media = lmn.get_media(date, media_id)
        re_exp = re.search("exp=(\d+)", media)
        if re_exp:
            exp = int(re_exp.group(1))
            if datetime.utcnow() > datetime.fromtimestamp(exp):
                dialog = xbmcgui.Dialog()
                dialog.notification(plugin.name, "Link Expired", xbmcgui.NOTIFICATION_ERROR, 3000)
                xbmcplugin.setResolvedUrl(plugin.handle, False, ListItem())
                return
        else:
            dialog = xbmcgui.Dialog()
            dialog.notification(plugin.name, media, xbmcgui.NOTIFICATION_ERROR, 3000)
            xbmcplugin.setResolvedUrl(plugin.handle, False, ListItem())
            return
        hls_params["__hls_origin_url"] = media
        hls_params["__hls_lazy_server"] = lmn.key_server
    if "http_proxy" in config:
        path = "/lazy_proxy.m3u8"
        hls_params["__hls_http_proxy"] = config["http_proxy"]
    if addon.getSettingInt("nhl_stream_type") == 0:
        if state == "live":
            hls_params["__hls_bitrate"] = addon.getSettingInt("nhl_bitrate_live")
        else:
            hls_params["__hls_bitrate"] = addon.getSettingInt("nhl_bitrate_vod")
    hls_proxy = (
        urlparse(addon.getSetting("hls_proxy"))
        ._replace(
            path=path,
            query=urlencode(hls_params),
        )
        .geturl()
    )
    li = ListItem(path="{0}|{1}".format(hls_proxy, urlencode(player_headers)))
    li.setContentLookup(False)
    if addon.getSettingInt("nhl_stream_type") == 1:
        li.setProperty("inputstream", "inputstream.adaptive")
        li.setProperty("inputstream.adaptive.manifest_type", "hls")
    xbmcplugin.setResolvedUrl(plugin.handle, True, li)


if __name__ == "__main__":
    plugin.run(sys.argv)
